/*
    rommetadatafactory.cpp - Singleton factory classes that create correct RomMetaData.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "rommetadatafactory.h"

// Qt includes
#include <qmap.h>

// KDE includes
#include <kdebug.h>
#include <kstaticdeleter.h>

// Kamefu includes
#include "rommetadata.h"
#include "defaultrommetadata.h"
#include "ziprommetadata.h"

namespace Kamefu {

static KStaticDeleter<RomMetaDataFactory> romMetaDataFactoryDeleter;

RomMetaDataFactory *RomMetaDataFactory::s_self = 0L;
RomMetaDataFactory *RomMetaDataFactory::self()
{
	if(!s_self)
	{
		romMetaDataFactoryDeleter.setObject(s_self, new RomMetaDataFactory);
	}

	return s_self;
}

class RomMetaDataFactory::Private
{
public:
	Private()
	 : zipHandler(new ZipRomMetaData), defaultHandler(new DefaultRomMetaData)
	{}
	
	~Private()
	{
		delete zipHandler;
		delete defaultHandler;
		
		QMap<QString, RomMetaData*>::const_iterator it, itEnd = romMetaDataMap.constEnd();
		for(it = romMetaDataMap.constBegin(); it != itEnd; ++it)
		{
			delete it.data();
		}
	}

	QMap<QString, RomMetaData*> romMetaDataMap;
	ZipRomMetaData *zipHandler;
	DefaultRomMetaData *defaultHandler;
};

RomMetaDataFactory::RomMetaDataFactory()
	: d(new Private())
{}

RomMetaDataFactory::~RomMetaDataFactory()
{
	kdDebug() << k_funcinfo << endl;
	delete d;
}

void RomMetaDataFactory::addRomMetaDataFactory(const QString &mimeType, RomMetaData *newRomMetaData)
{
	d->romMetaDataMap.insert(mimeType, newRomMetaData);
}

bool RomMetaDataFactory::hasMimeType(const QString &mimeType)
{
	QMap<QString, RomMetaData*>::const_iterator it, itEnd = d->romMetaDataMap.constEnd();
	for(it = d->romMetaDataMap.constBegin(); it != itEnd; ++it)
	{
		if(it.key() == mimeType)
		{
			return true;
		}
	}

	return false;
}

RomMetaData *RomMetaDataFactory::findSuitableRomMetaData(const QString &path)
{
	QMap<QString, RomMetaData*>::const_iterator it, itEnd = d->romMetaDataMap.constEnd();
	for(it = d->romMetaDataMap.constBegin(); it != itEnd; ++it)
	{
		RomMetaData *tempMetaData = static_cast<RomMetaData*>(it.data());
		if(tempMetaData->canHandle(path))
		{
			return tempMetaData;
		}
	}

	return 0;
}

RomMetaData *RomMetaDataFactory::createRomMetaData(const QString &path, const QString &mimeType)
{
	// When it's a ZIP file, check if a RomMetaData can handle it before.
	if( mimeType == QString::fromUtf8("application/x-zip") )
	{
		RomMetaData *suitableRomMetaData = findSuitableRomMetaData(path);
		if(suitableRomMetaData)
		{
			kdDebug() << "Returning a special RomMetaData for file: " << path << endl;
			return suitableRomMetaData;
		}
		else
		{
			kdDebug() << k_funcinfo << "Returning ZIP file handler RomMetaData." << endl;	
			return d->zipHandler;
		}
	}
	else if( hasMimeType(mimeType) )
	{
		kdDebug() << k_funcinfo << "Returning RomMetaData for mimetype: " << mimeType << endl;
		return d->romMetaDataMap[mimeType];
	}
	else
	{
		kdDebug() << k_funcinfo << "Returning default RomMetaData." << endl;
		return d->defaultHandler;
	}
	
	return 0;
}


}
