/*
    rommetainformation.h - Data class used to exchange ROM Meta Information.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#ifndef KAMEFUROMMETAINFORMATION_H
#define KAMEFUROMMETAINFORMATION_H

#include <ksharedptr.h>
#include <qstring.h>
#include <kdemacros.h>

namespace Kamefu 
{

/**
 * This is data class used to exchange ROM Meta information between classes.
 * This class is implicit shared.
 *
 * Simple creation of a new RomMetaInformation.
 * @code
 * Kamefu::RomMetaInformation romMetaInfo;
 * @endcode
 *
 * @author Michaël Larouche <michael.larouche@kdemtail.net>
 */
class KDE_EXPORT RomMetaInformation
{

public:
	/**
	 * @brief Construct a empty ROMMetaInformation object
	 */
	RomMetaInformation();
	/**
	 * d-tor.
	 */
	~RomMetaInformation();
	/**
	 * @brief Copy constructor
	 * This do not create a deep copy, just make a reference.
	 */
	RomMetaInformation(const RomMetaInformation &copy);
	/**
	 * @brief Copy-assignment operator.
	 * Mostly the same as the copy constructor.
	 */
	RomMetaInformation &operator=(const RomMetaInformation &other);

	/**
	 * @brief Is this RomMetaInformation is valid.
	 * A valid RomMetaInformation has either a MD5 hash, an URL or an editor name.
	 * @return true is the RomMetaInformation is valid.
	 */
	bool isValid() const;

	/**
	 * @brief Get ROM url.
	 * This the complete path where the ROM is located on disk.
	 * @return ROM url.
	 */
	QString url() const;
	/**
	 * @brief Set ROM url.
	 * This the complete path where the ROM is located on disk.
	 * @param url ROM url.
	 */
	void setUrl(const QString &url);

	/**
	 * @brief Get ROM name.
	 * This is a name defined by the user.
	 * @return ROM name.
	 */
	QString romName() const;
	/**
	 * @brief Set ROM name.
	 * This is a name defined by the user.
	 * @param romName ROM name.
	 */
	void setRomName(const QString &romName);

	/**
	 * @brief Get system name
	 * This is the system name. Came from Kamefu::SystemManager
	 * @return system name
	 */
	QString system() const;
	/**
	 * @brief Set system name.
	 * This is the system name. Came from Kamefu::SystemManager.
	 * @param system system name.
	 */
	void setSystem(const QString &system);

	/**
	 * @brief Get editor/developer name
	 * In GUI, this is labeled developer, it is the company who made the game.
	 * @return the developer name.
	 */
	QString editor() const;
	/**
	 * @brief Set editor/developer name
	 * In GUI, this is labeled developer, it is the company who made the game
	 * @param editor editor/developer name.
	 */
	void setEditor(const QString &editor);

	/**
	 * @brief Get MD5 hash.
	 * @return MD5 hash.
	 */
	QString md5Hash() const;
	/**
	 * @brief Set MD5 hash.
	 * @param md5Hash MD5 hash.
	 */
	void setMd5Hash(const QString &md5Hash);

	/**
	 * @brief Get the year of making.
	 * @return the year.
	 */
	QString year() const;
	/**
	 * @brief Set the year
	 * @param year Year.
	 */
	void setYear(const QString &year);

	/**
	 * @brief Get the country.
	 * @return the country.
	 */
	QString country() const;
	/**
	 * @brief Set the country.
	 * @param country the country.
	 */
	void setCountry(const QString &country);

	/**
	 * @brief Get the genre.
	 * @param genre the genre.
	 */
	QString genre() const;
	void setGenre(const QString &genre);

	/**
	 * @brief Get the titlescreen path.
	 * @return the titlescreen path.
	 */
	QString titlescreenPath() const;
	/**
	 * @brief Set the titlescreen path.
	 * @param titlescreenPath the titlescreen path.
	 */
	void setTitlescreenPath(const QString &titlescreenPath);

	/**
	 * @brief Get the screenshot path.
	 * @return the screenshot path.
	 */
	QString screenshotPath() const;
	/**
	 * @brief Set the screenshot path.
	 * @param screenshotPath the screenshot path.
	 */
	void setScreenshotPath(const QString &screenshotPath);

private:
	class Private;
	KSharedPtr<Private> d;
};

}

#endif
