/*
    Copyright (C) 2005 by Ryan Schultz <schultz.ryan@gmail.com>
	GNU GPL v2 
*/

#include <stdlib.h>
#include <kapplication.h>
#include <klocale.h>
#include <kconfig.h>
#include <qcheckbox.h>
#include <qcolor.h>
#include <qfontdatabase.h>
#include <qlabel.h>
#include <qrect.h>
#include <qpoint.h>
#include <qspinbox.h>
#include <kcolorcombo.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qpainter.h>
#include <kglobal.h>
#include <kfontcombo.h>
#include <qradiobutton.h>
#ifdef DEBUG
#include <iostream>
#endif
#include "kanji.h"
#include "kanjisaver.h"
#include "kanjisaverui.h"


//! libkscreensaver interface
extern "C"
{
    const char *kss_applicationName = "kanjisaver.kss";
    const char *kss_description = I18N_NOOP( "Kanjisaver" );
    const char *kss_version = "0.9.9";

    kanjisaver *kss_create( WId id )
    {
        KGlobal::locale()->insertCatalogue("kanjisaver");
        return new kanjisaver( id );
    }

    QDialog *kss_setup()
    {
        KGlobal::locale()->insertCatalogue("kanjisaver");
        return new kanjisaverSetup();
    }
}

// ------- Setup Program --------------------------------
//! dialog to setup screen saver parameters
kanjisaverSetup::kanjisaverSetup( QWidget *parent, const char *name )
        : kanjisaverUI( parent, name, TRUE )
{
    if(UsableFontList.empty()) {
    	QFontDatabase fdb;
		QStringList KatakanaFonts(fdb.families(QFont::Katakana));
		QStringList HiraganaFonts(fdb.families(QFont::Hiragana));
		QStringList KanjiFonts(fdb.families(QFont::Han));
    
    // find all fonts that have both charsets, and are smoothly scalable
    	for(QStringList::Iterator s = HiraganaFonts.begin(); s != HiraganaFonts.end(); ++s) 
    		if(fdb.isSmoothlyScalable(*s) && !(KatakanaFonts.grep(*s).empty()) && !(KanjiFonts.grep(*s).empty())) {
			UsableFontList+=*s;
		};
    }
    		
    if(!(UsableFontList.empty())) {
		KanaFontSelector->setFonts(UsableFontList);
		KanjiFontSelector->setFonts(UsableFontList);
	}
    
    //Connect your signals and slots here to configure the screen saver.
    connect( (QObject *)OkayPushButton, SIGNAL( released() ),
             SLOT( slotOkPressed() ) );
    connect( (QObject *)CancelPushButton, SIGNAL( released() ),
             SLOT( slotCancelPressed() ) );
	     
    
    readSettings();     

}


//! read settings from config file
void kanjisaverSetup::readSettings()
{
    KConfig *config = KGlobal::config();
    config->setGroup( "Settings" );
    // Add your config options here...
    KanaFontSelector->setCurrentFont( config->readEntry("KanaFontFamily", UsableFontList[0]) );
    KanjiFontSelector->setCurrentFont( config->readEntry("KanjiFontFamily", UsableFontList[0]) );
    ReadingToUse = config->readNumEntry("ReadingToUse",0);
    TranslationOrReading = config->readNumEntry("TranslationOrReading", 0);
    JLPTSet = config->readNumEntry("JLPTSet",4);
	KanaColor = config->readColorEntry("KanaColor", &Qt::white);
	KanjiColor = config->readColorEntry("KanjiColor", &Qt::white);
	TransReadColor = config->readColorEntry("TransReadColor", &Qt::white);
	BackgroundColor = config->readColorEntry("BackgroundColor", &Qt::black);
    DisplayTime = config->readNumEntry("DisplayTime", 10);

    switch(ReadingToUse) {
		case 1: Kun->setChecked(true);
			break;
		case 0: On->setChecked(true);
			break;
    };

	switch(TranslationOrReading){
		case 1: Reading->setChecked(true);
			break;
		case 0: Translation->setChecked(true);
			break;
    }; 

	jlptSpin->setValue(JLPTSet);
	timeSpin->setValue(DisplayTime);

	kanjiColorCombo->setColor( KanjiColor );
	kanaColorCombo->setColor( KanaColor );
	transreadColorCombo->setColor( TransReadColor );
	backgroundColorCombo->setColor( BackgroundColor );
	
}

//! Ok pressed - save settings and exit
void kanjisaverSetup::slotOkPressed()
{
    KConfig *config = KGlobal::config();
    config->setGroup( "Settings" );
    
    ReadingToUse=(Kun->isChecked()?1:0);
    TranslationOrReading=(Reading->isChecked()?1:0);
    config->writeEntry( "KanaFontFamily", KanaFontSelector->currentFont() );
    config->writeEntry( "KanjiFontFamily", KanjiFontSelector->currentFont() );
    config->writeEntry( "TranslationOrReading", TranslationOrReading);
    config->writeEntry( "ReadingToUse", ReadingToUse); 
    config->writeEntry( "JLPTSet", jlptSpin->value()); 
	config->writeEntry( "KanaColor", kanaColorCombo->color());
	config->writeEntry( "KanjiColor", kanjiColorCombo->color());
	config->writeEntry( "TransReadColor", transreadColorCombo->color());
	config->writeEntry( "BackgroundColor", backgroundColorCombo->color());
    config->writeEntry( "DisplayTime", timeSpin->value()); 

    config->sync();
    accept();
}

void kanjisaverSetup::slotCancelPressed()
{
    reject();
}
//-------------------------------------------------------


//----- Screensaver -------------------------------------

kanjisaver::kanjisaver( WId id ) : KScreenSaver( id )
{
	QFontDatabase fdb;
	QStringList KatakanaFonts(fdb.families(QFont::Katakana));
	QStringList HiraganaFonts(fdb.families(QFont::Hiragana));
	QStringList KanjiFonts(fdb.families(QFont::Han));
    
    // find all fonts that have both charsets, and are smoothly scalable
	for(QStringList::Iterator s = HiraganaFonts.begin(); s != HiraganaFonts.end(); ++s) 
		if(fdb.isSmoothlyScalable(*s) && !(KatakanaFonts.grep(*s).empty()) && !(KanjiFonts.grep(*s).empty())) {
		UsableFontList+=*s;
	};

    srand(time(0));
    readSettings();
	init();
    blank();
}

kanjisaver::~kanjisaver()
{
}

//! read configuration settings from config file
void kanjisaver::readSettings()
{
    KConfig *config = KGlobal::config();
    config->setGroup( "Settings" );
    // Add your config options here...
    KanaFontName = config->readEntry("KanaFontFamily", UsableFontList[0]);
    KanjiFontName = config->readEntry("KanjiFontFamily", UsableFontList[0]);
    TranslationOrReading = config->readNumEntry("TranslationOrReading", 0);
    ReadingToUse = config->readNumEntry("ReadingToUse",0);
    JLPTSet = config->readNumEntry("JLPTSet",4);
	KanaColor = config->readColorEntry("KanaColor", &Qt::white);
	KanjiColor = config->readColorEntry("KanjiColor", &Qt::white);
	TransReadColor = config->readColorEntry("TransReadColor", &Qt::white);
	BackgroundColor = config->readColorEntry("BackgroundColor", &Qt::black);
    DisplayTime = config->readNumEntry("DisplayTime", 10);
}

void kanjisaver::init()
{
	if(UsableFontList.empty()) {
    	QMessageBox mb(tr2i18n("Kanjisaver"), tr2i18n("There are no usable (that is, capable of Hiragana, Katakana, and Kanji and\n soft-scalable) fonts on your system. You must get and install\nsome before using this screensaver."),
	QMessageBox::Critical,QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
	if(mb.exec()==QMessageBox::Ok) exit(-1);
    }
    
    kanaFont = new QFont(KanaFontName);
    kanjiFont = new QFont(KanjiFontName);
    
    // use the systemwide default font for the romaji
    romajiFont = new QFont();
    
    int myheight = height();
    
    // kana are shown in 1/4 of screen height
    kanaFont->setPixelSize(myheight/4);
    
	// kanji are shown in 1/4 of screen height
    kanjiFont->setPixelSize(myheight/4);
    
    // romaji are shown in 1/10 of screen height
    romajiFont->setPixelSize(myheight/10);
    
    setBackgroundColor( BackgroundColor );

}

void kanjisaver::blank()
{
	erase();

    timer = new QTimer();
    connect(timer, SIGNAL(timeout()), SLOT(draw_kanji())); 

	// the small value here will be updated by draw_kanji
	// this allows the first kanji to appear snappily
    timer->start(1, FALSE);
}

void kanjisaver::draw_kanji()
{
    QPainter paint(this);
	
    int mywidth = width();
    int myheight = height();
    int character = 0;
	kanjitransInfo* kanjitable;

	// yeah, this is ugly
	switch(JLPTSet){
		case 4: 
					kanjitable = kanjitable_jlpt4;
					character = rand()%(sizeof(kanjitable_jlpt4)/sizeof(kanjitable_jlpt4[0]));
					break;
				
		case 3: 
					kanjitable = kanjitable_jlpt3;
					character = rand()%(sizeof(kanjitable_jlpt3)/sizeof(kanjitable_jlpt3[0]));
					break;
				
		case 2: 
					kanjitable = kanjitable_jlpt2;
					character = rand()%(sizeof(kanjitable_jlpt2)/sizeof(kanjitable_jlpt2[0]));
					break;
				
		case 1: 
					kanjitable = kanjitable_jlpt1;
					character = rand()%(sizeof(kanjitable_jlpt1)/sizeof(kanjitable_jlpt1[0]));
					break;
				
		default: 
					kanjitable = kanjitable_jlpt4;
					character = rand()%(sizeof(kanjitable_jlpt4)/sizeof(kanjitable_jlpt4[0]));
					break;
				
	}
	
	paint.setClipping(false);
    paint.eraseRect(0,0,mywidth,myheight);        
   
	/// draw the kanji
    paint.setPen( KanjiColor );
    paint.setFont(*kanjiFont);
    paint.drawText( (mywidth/3), (myheight/3),  (mywidth/3), (myheight/3),
                   Qt::AlignCenter|Qt::DontClip,
		   QString::fromUtf8(kanjitable[character].pKanji, -1));
	
	printf("%s\n", kanjitable[character].pKanji);

	/// draw the reading
    paint.setPen( KanaColor );
	kanaFont->setPixelSize(myheight/15);
	paint.setFont(*kanaFont);
	if (ReadingToUse == 0) { // On
		
		int text_len = strlen(kanjitable[character].pOn);
		if (text_len < 20){
			kanaFont->setPixelSize(myheight/15);
		}
		else if (text_len < 28) {
			kanaFont->setPixelSize(myheight/20);
		}
		else if (text_len < 36) {
			kanaFont->setPixelSize(myheight/26);
		}
		else if (text_len < 42) {
			kanaFont->setPixelSize(myheight/32);
		}
		else if (text_len < 49) {
			kanaFont->setPixelSize(myheight/38);
		}
		else if (text_len < 55) {
			kanaFont->setPixelSize(myheight/43);
		}
		else {
			kanaFont->setPixelSize(myheight/48);
		}
		paint.setFont(*kanaFont);
	
		paint.drawText( 
			QRect::QRect(
					QPoint::QPoint(0 + mywidth/30, myheight/20),
					QPoint::QPoint(mywidth, myheight/10)
					),
			Qt::AlignLeft|Qt::AlignVCenter|Qt::DontClip,
		   QString::fromUtf8(kanjitable[character].pOn, -1));

		printf("%s\n", kanjitable[character].pOn);

	}
    else if (ReadingToUse == 1) { // Kun
		int text_len = strlen(kanjitable[character].pKun)/3;
		if (text_len < 20){
			kanaFont->setPixelSize(myheight/15);
		}
		else if (text_len < 28) {
			kanaFont->setPixelSize(myheight/19);
		}
		else if (text_len < 37) {
			kanaFont->setPixelSize(myheight/25);
		}
		else if (text_len < 46) {
			kanaFont->setPixelSize(myheight/30);
		}
		else if (text_len < 52) {
			kanaFont->setPixelSize(myheight/35);
		}
		else if (text_len < 57) {
			kanaFont->setPixelSize(myheight/41);
		}
		else {
			kanaFont->setPixelSize(myheight/47);
		}

		paint.setFont(*kanaFont);

				paint.drawText( 
			QRect::QRect(
					QPoint::QPoint(0 + mywidth/30, myheight/20),
					QPoint::QPoint(mywidth, myheight/10)
					),
			Qt::AlignLeft|Qt::AlignVCenter|Qt::DontClip,
		   QString::fromUtf8(kanjitable[character].pKun, -1));

			printf("%s\n", kanjitable[character].pKun);
	} 
	

	/// draw the english/romaji
    paint.setPen( TransReadColor );

	// set a default font style
	romajiFont->setPixelSize(myheight/20);
	paint.setFont(*romajiFont);
	
	if (TranslationOrReading == 0) { // Translation

		// font scaling for translation text (needs split out into a function)
		int text_len = strlen(kanjitable[character].pEnglish);
		if (text_len < 45){
			romajiFont->setPixelSize(myheight/20);
		}
		else if (text_len < 58) {
			romajiFont->setPixelSize(myheight/24);
		}
		else if (text_len < 70) {
			romajiFont->setPixelSize(myheight/28);
		}
		else if (text_len < 78) {
			romajiFont->setPixelSize(myheight/32);
		}
		else if (text_len < 88) {
			romajiFont->setPixelSize(myheight/35);
		}
		else if (text_len < 98) {
			romajiFont->setPixelSize(myheight/38);
		}
		else {
			romajiFont->setPixelSize(myheight/43);
		}
		
		paint.setFont(*romajiFont);

			paint.drawText( 
				QRect::QRect(
					QPoint::QPoint(0, 6*(myheight/7)),
					QPoint::QPoint(mywidth - mywidth/30, myheight)
					),
					Qt::AlignRight|Qt::AlignVCenter|Qt::DontClip,
		   QString::fromUtf8(kanjitable[character].pEnglish, -1));
			

			printf("%s\n", kanjitable[character].pEnglish);
	}
	else if (TranslationOrReading == 1) { // Reading
		if (ReadingToUse == 0) { // On

			// font scaling for translation text
			int text_len = strlen(kanjitable[character].pOnRomaji);
			if (text_len < 45){
				romajiFont->setPixelSize(myheight/20);
			}
			else if (text_len < 58) {
				romajiFont->setPixelSize(myheight/24);
			}
			else if (text_len < 70) {
				romajiFont->setPixelSize(myheight/28);
			}
			else if (text_len < 78) {
				romajiFont->setPixelSize(myheight/32);
			}
			else if (text_len < 88) {
				romajiFont->setPixelSize(myheight/35);
			}
			else if (text_len < 98) {
				romajiFont->setPixelSize(myheight/38);
			}
			else {
				romajiFont->setPixelSize(myheight/43);
			}

			paint.setFont(*romajiFont);

			paint.drawText( 
					QRect::QRect(
							QPoint::QPoint(0, 6*(myheight/7)),
							QPoint::QPoint(mywidth - mywidth/30, myheight)
							),
					Qt::AlignRight|Qt::AlignVCenter|Qt::DontClip,
				   QString::fromUtf8(kanjitable[character].pOnRomaji, -1));
		
			printf("%s\n", kanjitable[character].pOnRomaji);
		
		}
		else if (ReadingToUse == 1) { // Kun

			// font scaling for translation text
			int text_len = strlen(kanjitable[character].pKunRomaji);
			if (text_len < 45){
				romajiFont->setPixelSize(myheight/20);
			}
			else if (text_len < 58) {
				romajiFont->setPixelSize(myheight/24);
			}
			else if (text_len < 70) {
				romajiFont->setPixelSize(myheight/28);
			}
			else if (text_len < 78) {
				romajiFont->setPixelSize(myheight/32);
			}
			else if (text_len < 88) {
				romajiFont->setPixelSize(myheight/35);
			}
			else if (text_len < 98) {
				romajiFont->setPixelSize(myheight/38);
			}
			else {
				romajiFont->setPixelSize(myheight/43);
			}

			paint.setFont(*romajiFont);

			paint.drawText( 
					QRect::QRect(
							QPoint::QPoint(0, 6*(myheight/7)),
							QPoint::QPoint(mywidth - mywidth/30, myheight)
							),
					Qt::AlignRight|Qt::AlignVCenter|Qt::DontClip,
				   QString::fromUtf8(kanjitable[character].pKunRomaji, -1));
			
			printf("%s\n", kanjitable[character].pKunRomaji);
		} 
	}

	printf("----------------------\n");
	timer->changeInterval(DisplayTime*1000);

}

void kanjisaverSetup::FontChanged()
{
	KanaFontSample->setFont(QFont(KanaFontSelector->currentFont(), 12,0,0));
	KanjiFontSample->setFont(QFont(KanjiFontSelector->currentFont(), 12,0,0));
}



