/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *   Laurent Montel : montel@kde.org                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <assert.h>
#include <qlayout.h>
#include <klocale.h>
#include <kglobal.h>
#include <kparts/genericfactory.h>
#include <qvariant.h>
#include <qlistbox.h>
#include <qtabwidget.h>
#include <qwidget.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qlistbox.h>
#include <qcheckbox.h>
#include <qeventloop.h>
#include <qlineedit.h>

#include <kjanuswidget.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <knuminput.h>
#include <kconfig.h>
#include <keditlistbox.h>
#include <kfiledialog.h>
#include <qvbox.h>
#include <kurlrequester.h>
#include "katcontrol.h"
#include "languagemanagement.h"
#include "katengine.h"
#include "dlgnewcatalog.h"
#include <kdeversion.h>
#include <kmessagebox.h>
#include <klineedit.h>
#include <kregexpeditorinterface.h>
#include <kparts/componentfactory.h>

typedef KGenericFactory<katcontrol, QWidget> katcontrolFactory;
K_EXPORT_COMPONENT_FACTORY( kcm_katcontrol, katcontrolFactory( "kat" ) )

static inline QPixmap loadIcon( const char* name ) {
  return KGlobal::instance()->iconLoader()
    ->loadIcon( QString::fromLatin1(name), KIcon::NoGroup, KIcon::SizeMedium );
}


katRegexpWidget::katRegexpWidget( QWidget* parent, char* name )
    : QWidget( parent, name ),
      m_regexpDialogQueryDone( false ),
      m_regexpDialog( 0L )
{
    init();
}

katRegexpWidget::~katRegexpWidget()
{
}


void katRegexpWidget::init()
{
    QHBoxLayout *layout = new QHBoxLayout( this );
    m_editLinePattern = new KLineEdit( this );
    layout->addWidget( m_editLinePattern );

    m_editPattern = new QPushButton( i18n( "Edit..." ), this );
    layout->addWidget( m_editPattern );

    connect( m_editPattern, SIGNAL( clicked () ), this, SLOT( slotEditPattern() ) );
}


void katRegexpWidget::slotEditPattern()
{
    if ( !m_regexpDialogQueryDone )
    {
        m_regexpDialogQueryDone=true;
        m_regexpDialog = KParts::ComponentFactory::createInstanceFromQuery<QDialog>( "KRegExpEditor/KRegExpEditor", QString::null, this );
    }
    if ( m_regexpDialog )
    {
        KRegExpEditorInterface *iface = static_cast<KRegExpEditorInterface *>( m_regexpDialog->qt_cast( "KRegExpEditorInterface" ) );
        assert( iface );

        iface->setRegExp( pattern() );
        if ( m_regexpDialog->exec() == QDialog::Accepted )
            setPattern( iface->regExp() );
    }
    else
    {
        KMessageBox::error( this, i18n( "KRegExpEditor not found, please install kdeutils package" ) );
        m_editPattern->setEnabled( false );
    }
}

void katRegexpWidget::setPattern( const QString &_pattern)
{
    m_editLinePattern->setText( _pattern );
}

QString katRegexpWidget::pattern() const
{
    return m_editLinePattern->text();
}

katcontrol::katcontrol( QWidget* parent, const char* name, const QStringList& )
    : KCModule( parent, name ), myAboutData( 0 )
{
    katConf = new KConfig( "katrc" );
    QHBoxLayout* layout = new QHBoxLayout( this );
    m_page = new KJanusWidget( this, "janus_widget", KJanusWidget::IconList );
    layout->addWidget( m_page );

    QFrame* page = m_page->addPage( i18n( "Catalog" ),
                                    i18n( "Configure Catalog" ),
                                    loadIcon( "kat_database" ) );
    m_catalogPage=new catalogPage( page );
    connect( m_catalogPage, SIGNAL( configChanged() ), SLOT( configChanged() ) );

    page = m_page->addPage( i18n( "Language" ),
                            i18n( "Configure Language" ),
                            loadIcon( "kat_locale" ) );
    m_languagePage=new languagePage( page );
    connect( m_languagePage, SIGNAL( configChanged() ), SLOT( configChanged() ) );

    page = m_page->addPage( i18n( "Advanced" ),
                            i18n( "Advanced" ),
                            loadIcon( "configure" ) );
    m_advancedPage=new advancedPage( page );
    connect( m_advancedPage, SIGNAL( configChanged() ), SLOT( configChanged() ) );

    page = m_page->addPage( i18n( "Misc" ),
                            i18n( "Misc" ),
                            loadIcon( "misc" ) );
    m_miscPage=new miscPage( page );
    connect( m_miscPage, SIGNAL( configChanged() ), SLOT( configChanged() ) );

    load();
}

katcontrol::~katcontrol()
{
    save();
    katConf->sync();
    delete katConf;
}

void katcontrol::load()
{
    m_languagePage->load();
    m_catalogPage->load();
    m_advancedPage->load( katConf );
    m_miscPage->load( katConf );
}

void katcontrol::defaults()
{
    m_languagePage->defaults();
    m_catalogPage->defaults();
    m_advancedPage->defaults();
    m_miscPage->defaults();
    emit changed( true );
}

void katcontrol::save()
{
    m_languagePage->save();
    m_catalogPage->save();
    m_advancedPage->save( katConf );
    m_miscPage->save( katConf );
    emit changed( true );
}

int katcontrol::buttons()
{
    return KCModule::Default | KCModule::Apply | KCModule::Help;
}

void katcontrol::configChanged()
{
    // insert your saving code here...
    emit changed(true);
}

QString katcontrol::quickHelp() const
{
    return i18n( "Helpful information about the katcontrol module." );
}

catalogPage::catalogPage( QWidget* parent, char* name)
    : QWidget( parent, name )
{
    QGridLayout * grid = new QGridLayout(parent, 7, 2,
                                         KDialog::marginHint(),
                                         KDialog::spacingHint());
    m_catalogList = new QListBox( parent );
    grid->addMultiCellWidget( m_catalogList, 0, 6, 0, 0 );

    m_addCatalog = new QPushButton( i18n("&Add..."), parent);
    grid->addWidget( m_addCatalog, 0, 1 );

    m_deleteCatalog = new QPushButton( i18n("&Delete"), parent);
    grid->addWidget( m_deleteCatalog, 1, 1 );

    connect( m_addCatalog, SIGNAL( pressed () ), this, SLOT(slotAddCatalog() ) );
    connect( m_deleteCatalog, SIGNAL( pressed () ), this, SLOT( slotDeleteCatalog() ) );
    connect( m_catalogList, SIGNAL( selectionChanged () ), this, SLOT( slotUpdateButton() ) );
    m_ke = new KatEngine();
    m_catalogs = m_ke->readCatalogs();
    slotUpdateButton();
}

catalogPage::~catalogPage()
{
    delete m_ke;
}

void catalogPage::slotUpdateButton()
{
    m_deleteCatalog->setEnabled( m_catalogList->currentItem()> -1 );
}

void catalogPage::slotAddCatalog()
{
    DlgNewCatalog *nc = new DlgNewCatalog( this );
    if ( nc->exec() == QDialog::Accepted )
    {
        KatCatalog *cat = nc->newCatalog();
        // Write the catalog to disk
        m_ke->addCatalog( cat );

        QByteArray data;
        QDataStream arg( data, IO_WriteOnly );

        arg << cat->catalogId();
        addCatalogId( cat->catalogId() );

        m_catalogs = m_ke->readCatalogs();
        kapp->dcopClient()->emitDCOPSignal( "CatalogPage::CatalogAdded",
                                            "Catalog_added(int)",
                                            data );
        m_catalogList->insertItem( cat->name() );
        slotUpdateButton();
        emit configChanged();
    }
    //TODO fixme: reactivate add button
    m_addCatalog->setOn( false );
    delete nc;
}

void catalogPage::slotDeleteCatalog()
{
    KatCatalog* cat = m_ke->getCatalog( m_catalogList->currentText () );

    // delete it
    if ( cat )
    {
        kdDebug() << " delete catalog: "<< m_catalogList->currentText() << endl;
        QByteArray data;
        QDataStream arg( data, IO_WriteOnly );

        arg << cat->catalogId();

        m_ke->deleteCatalog( cat );
        deleteCatalogId( cat->catalogId() );
        m_catalogs = m_ke->readCatalogs();
        kapp->dcopClient()->emitDCOPSignal( "CatalogPage::CatalogDeleted",
                                            "Catalog_deleted(int)",
                                            data );
        m_catalogList->removeItem( m_catalogList->currentItem () );
    }
    slotUpdateButton();
    emit configChanged();
}


void catalogPage::addCatalogId( int catalogId )
{
    // Send addCatalog
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << catalogId;

    if (kapp->dcopClient()->call( "kded", "katd", "addCatalog(int)", data, replyType, replyData ) && replyType=="bool" ) {
        QDataStream reply(replyData, IO_ReadOnly);
        bool result;
        reply >> result;
        if ( !result )
            KMessageBox::error(this, i18n("Unable to add catalog."));

    }
    else
    {
            kdDebug() << "addCatalog() on kded returned an unexpected type of reply: " << replyType << endl;
    }

}

void catalogPage::deleteCatalogId( int catalogId )
{
    // Send deleteCatalog
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << catalogId;
    if (kapp->dcopClient()->call( "kded", "katd", "deleteCatalog(int)", data, replyType, replyData ) && replyType=="bool") {
        QDataStream reply(replyData, IO_ReadOnly);
        bool result;
        reply >> result;
        if ( !result )
            KMessageBox::error(this, i18n("Unable to delete catalog."));
    }
    else
    {
        kdDebug() << "deleteCatalog() on kded returned an unexpected type of reply: " << replyType << endl;
    }

}


void catalogPage::load()
{
    //kdDebug() << " m_catalogs.count(): "<< m_catalogs.count() << endl;
    QStringList lst;
    QPtrList<KatCatalog>::iterator end( m_catalogs.end() );
    for ( QPtrList<KatCatalog>::iterator it = m_catalogs.begin(); it != end; ++it )
    {
        KatCatalog* cat = *it;
        lst << cat->name();
    }

    m_catalogList->insertStringList( lst );
    slotUpdateButton();
}

void catalogPage::save()
{
    //TODO
    //send dcop signal
    //kapp->dcopClient()->emitDCOPSignal( "CatalogPage::CatalogChanged",
    //                                    "Catalog_ConfigChanged()",
    //                                    QByteArray() );
}

void catalogPage::defaults()
{
    //nothing
}

advancedPage::advancedPage( QWidget* parent, char* name )
    : QWidget( parent, name )
{
    QVBoxLayout* layout = new QVBoxLayout( parent );
    m_schedulerLoad = new KIntNumInput( 20, parent );
    layout->addWidget( m_schedulerLoad );
    m_schedulerLoad->setLabel( i18n( "Scheduler load:" ) );
    m_schedulerLoad->setRange( 1, 100 );
    m_schedulerLoad->setSuffix( i18n( "%" ) );
    connect( m_schedulerLoad, SIGNAL(  valueChanged(int) ), SIGNAL( configChanged() ) );

    QWhatsThis::add( m_schedulerLoad,
        i18n( "Scheduler load ( default 20 )- This value can be between 1 and 100. "
        "A scheduler load of 20 implies that 20% of CPU is used for indexing. "
        "This is calculated based on time taken for last index job. Say the last index job took 20secs. "
        "The next index job is scheduled to resume after 80secs.( this is wait time )" ) );

    m_schedulerWaitBeforeNextJob = new KIntNumInput( 120, parent );
    layout->addWidget( m_schedulerWaitBeforeNextJob );
    m_schedulerWaitBeforeNextJob->setLabel( i18n( "Wait in seconds before next scheduler job:" ) );
    m_schedulerWaitBeforeNextJob->setRange( 1, 1000 );
    connect( m_schedulerWaitBeforeNextJob, SIGNAL(  valueChanged(int) ), SIGNAL( configChanged() ) );
    m_schedulerWaitBeforeNextJob->setSuffix( i18n( "s" ) );
    QWhatsThis::add( m_schedulerWaitBeforeNextJob,
        i18n( "Scheduler maximum wait time  ( 120 secs. ) - Schedule the next job no later than this. "
        "Say the previous job took 60secs. the next job will scheduled after 240secs. "
        "This introduces considerable delay and so we limit that so as not to let the index time lag." ) );
#if 0
    //TODO fix range/value
    //TODO add QWhatsThis
    m_schedulerNbFileIndexed = new KIntNumInput( 5, parent );
    m_schedulerNbFileIndexed->setLabel( i18n( "Number of files to index in one batch." ) );
    m_schedulerNbFileIndexed->setRange( 1, 50 );
    connect( m_schedulerNbFileIndexed, SIGNAL(  valueChanged(int) ), SIGNAL( configChanged() ) );
    layout->addWidget( m_schedulerNbFileIndexed );
#endif
#if KDE_IS_VERSION(3,3,90)
    KURLRequester *line = new KURLRequester;
    line->fileDialog()->setMode(  KFile::Directory );
    KEditListBox::CustomEditor customLineEdit(  line, line->lineEdit() );
    m_excludeFolders = new KEditListBox( i18n( "Exclude Folders" ),customLineEdit, parent );
    m_excludeFolders->setButtons( KEditListBox::Add |KEditListBox::Remove );
#else
    m_excludeFolders = new KEditListBox( i18n( "Exclude Folders" ), parent, 0, false, KEditListBox::Add |KEditListBox::Remove );
#endif
    layout->addWidget( m_excludeFolders );

    connect( m_excludeFolders->addButton(), SIGNAL( pressed () ), this, SIGNAL(configChanged() ) );
    connect( m_excludeFolders->removeButton(), SIGNAL( pressed () ), this, SIGNAL(configChanged() ) );

#if KDE_IS_VERSION(3,3,90)
    katRegexpWidget *lineRegExp = new katRegexpWidget( parent );
    KEditListBox::CustomEditor customLineEditReg(  lineRegExp, lineRegExp->lineEdit() );
    m_excludeFiles= new KEditListBox( i18n( "Exclude Files" ), customLineEditReg, parent );
    m_excludeFiles->setButtons( KEditListBox::Add | KEditListBox::Remove );
#else
    m_excludeFiles= new KEditListBox( i18n( "Exclude Files" ), parent, 0, false, KEditListBox::Add |KEditListBox::Remove );
#endif

    layout->addWidget( m_excludeFiles );
    connect( m_excludeFiles->addButton(), SIGNAL( pressed () ), this, SIGNAL(configChanged() ) );
    connect( m_excludeFiles->removeButton(), SIGNAL( pressed () ), this, SIGNAL(configChanged() ) );
}

void advancedPage::load( KConfig* conf )
{
    conf->setGroup( "Daemon" );

    m_schedulerLoad->setValue( conf->readNumEntry( "scheduler load", 20 ) );
    m_schedulerWaitBeforeNextJob->setValue( conf->readNumEntry("scheduler wait next job", 120 ) );
#if 0
    m_schedulerNbFileIndexed->setValue( conf->readNumEntry( "scheduler number files indexed", 5 ) );
#endif
    m_excludeFolders->listBox()->insertStringList( conf->readPathEntry( "Exclude Folders" ) );
    m_excludeFiles->listBox()->insertStringList( conf->readPathEntry( "Exclude Files" ) );
}

void advancedPage::save( KConfig* conf )
{
    conf->setGroup( "Daemon" );
    conf->writeEntry( "scheduler load", m_schedulerLoad->value() );
    conf->writeEntry( "scheduler wait next job", m_schedulerWaitBeforeNextJob->value() );
#if 0
    conf->writeEntry( "scheduler number files indexed", m_schedulerNbFileIndexed->value() );
#endif
    conf->writePathEntry( "Exclude Folders", m_excludeFolders->items() );
    conf->writePathEntry( "Exclude Files", m_excludeFiles->items() );


    // Send addCatalog
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << m_schedulerLoad->value();
    arg << m_schedulerWaitBeforeNextJob->value();
    arg << m_excludeFolders->items();
    arg << m_excludeFiles->items();
    //TODO add test and i18n after 0.6.0
    kapp->dcopClient()->call( "kded", "katd", "changeIdleLoadMaxWait(int,int,QStringList,QStringList)", data, replyType, replyData );
}

void advancedPage::defaults()
{
    m_schedulerLoad->setValue( 20 );
    m_schedulerWaitBeforeNextJob->setValue( 120 );
    m_excludeFolders->listBox()->clear();
    m_excludeFiles->listBox()->clear();
#if 0
    m_schedulerNbFileIndexed->setValue( 5 );
#endif
}

languagePage::languagePage( QWidget *parent, char *name )
{
    ( void )new languageManagement( parent );
}

void languagePage::load()
{
    //nothing
}

void languagePage::save()
{
    //nothing
}

void languagePage::defaults()
{
    //nothing
}

miscPage::miscPage( QWidget* parent, char* name )
    : QWidget( parent, name )
{

    QGridLayout * grid = new QGridLayout(parent, 7, 1,
                                         KDialog::marginHint(),
                                         KDialog::spacingHint());

    m_showSplashScreen = new QCheckBox( i18n( "Show splashscreen" ), parent );
    grid->addWidget( m_showSplashScreen,0,0 );
}

void miscPage::load( KConfig *_cfg )
{
    _cfg->setGroup( "General" );
    m_showSplashScreen->setChecked( _cfg->readBoolEntry( "ShowSplashScreen", true ) );
}

void miscPage::save( KConfig *_cfg )
{
    _cfg->setGroup( "General" );
    _cfg->writeEntry( "ShowSplashScreen", m_showSplashScreen->isChecked() );

}

void miscPage::defaults()
{
    m_showSplashScreen->setChecked( true );
}


#include "katcontrol.moc"
