/*
 * preferencesdialog.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include "preferencesdialog.hh"
#ifdef HAVE_SPELL
#include "spell.hh"
#endif
#include "macros.h"
#include <gtkmm/stock.h>
#include "mdi.hh"
#include "utils.hh"
#include <gtkmm/enums.h>
#include <gtkmm/colorselection.h>
#include <gtkmm/fontselection.h>
#include "dict.hh"
#include "dialogs.hh"

char *sections[] = {
  _("General"),
  _("Interface"),
  _("Tabs"),
  _("Editor"),
  _("Encodings"),
  _("File Save"),
#ifdef HAVE_SPELL
  _("Spell Checker"),
#endif
#ifdef ENABLE_PRINT
  _("Print"),
#endif
  _("Dictionary"),
  _("Remote"),
  _("Advanced"),
  NULL
};

PreferencesDialog::PreferencesDialog(Conf& conf, Encodings& enc) :
  _conf(conf),
  _enc(enc),
  apply(Gtk::Stock::APPLY),
  undono_adj(0,0,100),
  exec_adj(0,0,100),
  undo_closed_adj(0,0,100),
  recentno_adj(0,0,100),
#ifdef ENABLE_PRINT
  dpi_x_adj(0,0,0),
  dpi_y_adj(0,0,0),
#endif
  dict_port_adj(0,0,0),
  dict_timeout_adj(0,0,0),
  locations_size_adj(0,0,0),
  x_adj(0,0,0),
  y_adj(0,0,0),
  w_adj(0,0,0),
  h_adj(0,0,0)
{
  dialog.set_title(_("Preferences"));
  dialog.set_modal(true);

  Gtk::VBox *vbox = dialog.get_vbox();
  vbox->pack_start(box);

  sw.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  box.pack_start(sw);

  // Now for the tree view.
  record.add(section);
  record.add(number);
  store = Gtk::ListStore::create(record);
  treeview.set_model(store);
  treeview.append_column(_("Options for"), section);
  treeview.set_headers_visible(false);

  sw.add(treeview);
  sw.set_size_request(150, 300);
  selection = treeview.get_selection();
  selection->signal_changed().connect(sigc::mem_fun(*this, &PreferencesDialog::selection_signal_changed_cb));
  char **tmp = sections;
  int i = 0;
  while (*tmp)
    {
      Gtk::TreeModel::Row row = *(store->append ());
      row[number] = i;
      row[section] = _(*tmp);
      ++tmp;
      ++i;
    }

  box.pack_start(notebook);
  notebook.set_show_tabs(false);

  _general.set_spacing(5);
  _interface.set_spacing(5);
  _tabs.set_spacing(5);
  _editor.set_spacing(5);
  _encodings.set_spacing(5);
  _file_save.set_spacing(5);
#ifdef HAVE_SPELL
  _spell_checker.set_spacing(5);
#endif
#ifdef ENABLE_PRINT
  _print.set_spacing(5);
#endif
  _dictionary.set_spacing(5);
  _remote.set_spacing(5);
  _advanced.set_spacing(5);

  _general.set_border_width(10);
  _interface.set_border_width(10);
  _tabs.set_border_width(10);
  _editor.set_border_width(10);
  _encodings.set_border_width(10);
  _file_save.set_border_width(10);
#ifdef HAVE_SPELL
  _spell_checker.set_border_width(10);
#endif
#ifdef ENABLE_PRINT
  _print.set_border_width(10);
#endif
  _dictionary.set_border_width(10);
  _remote.set_border_width(10);
  _advanced.set_border_width(10);

  _general.set_homogeneous(false);

  general();
  interface();
  tabs();
  editor();
  encodings();
  file_save();
#ifdef HAVE_SPELL
  spell_checker();
#endif
#ifdef ENABLE_PRINT
  print();
#endif
  dictionary();
  remote();
  advanced();

  dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
  Gtk::HButtonBox* box = dialog.get_action_area();
  box->pack_start(apply);
  //  dialog.add_button(Gtk::Stock::APPLY, Gtk::RESPONSE_APPLY);
  dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  apply.signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::apply_clicked_cb));
}

PreferencesDialog::~PreferencesDialog()
{

}

bool PreferencesDialog::run()
{
  dialog.show_all();
  if (dialog.run() == Gtk::RESPONSE_OK)
    {
      repopulate_conf();
      return true;
    }
  return false;
}

void PreferencesDialog::general()
{
  notebook.append_page(_general);

  undo.set_use_underline();
  undo.set_label(_("_Enable undo, redo\t"));
  undo_closed.set_use_underline();
  undo_closed.set_label(_("Keep a history of recently closed documents"));

  undo_label.set_text(_("Undo history size\n(0 for unlimited)"));
  exec_label.set_text(_("Executed commands history size\n(0 for unlimited)"));
  undo_closed_label.set_text(_("Closed documents history size\n(0 for unlimited)"));

  undono.set_adjustment(undono_adj);
  exec_cmd_size.set_adjustment(exec_adj);
  undo_closedno.set_adjustment(undo_closed_adj);

  _general.pack_start(undo, false, false);
  _general.pack_start(general_table1, false, false);
  _general.pack_start(undo_closed, false, false);
  _general.pack_start(general_table2, false, false);

  general_table1.set_col_spacing(0, 5);
  general_table1.attach(undo_label, 0, 1, 0, 1, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  general_table1.attach(undono, 1, 2, 0, 1, Gtk::AttachOptions(Gtk::EXPAND|Gtk::FILL), Gtk::AttachOptions(Gtk::SHRINK));
  general_table1.attach(exec_label, 0, 1, 1, 2, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  general_table1.attach(exec_cmd_size, 1, 2, 1, 2, Gtk::AttachOptions(Gtk::EXPAND|Gtk::FILL), Gtk::AttachOptions(Gtk::SHRINK));

  general_table2.set_col_spacing(0, 5);
  general_table2.attach(undo_closed_label, 0, 1, 2, 3, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  general_table2.attach(undo_closedno, 1, 2, 2, 3, Gtk::AttachOptions(Gtk::EXPAND|Gtk::FILL), Gtk::AttachOptions(Gtk::SHRINK));

  undono_adj.set_value(_conf.get("undono", 0));
  exec_adj.set_value(_conf.get("exec_cmd_size", 10));
  undo_closed_adj.set_value(_conf.get("undo_closedno", 5));
  undo.set_active(_conf.get("undo", true));
  undo_closed.set_active(_conf.get("undo_closed", true));

  undo.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::undo_toggled_cb));
  undo_closed.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::undo_closed_toggled_cb));
  undo.toggled();
  undo_closed.toggled();
}

void PreferencesDialog::interface()
{
  notebook.append_page(_interface);

  toolbar.set_use_underline();
  extended_toolbar.set_use_underline();
  statusbar.set_use_underline();
  recent.set_use_underline();
  showclose.set_use_underline();
  extra_buttons.set_use_underline();

  extra_buttons.set_label(_("Show the extra _buttons"));
  toolbar.set_label(_("Show the _Main Toolbar"));
  extended_toolbar.set_label(_("Show the _Extended Toolbar"));
  statusbar.set_label(_("Show the _Statusbar"));
  recent.set_label(_("Enable re_cent menu"));
  showclose.set_label(_("Show the _close button on each tab label"));
  recentno_label.set_text(_("No. of recent entries\n(0 for unlimited)"));
  toolbartype_label.set_text(_("Show the toolbar as"));
  toolbartype.append_text(_("Text only"));
  toolbartype.append_text(_("Icons only"));
  toolbartype.append_text(_("Both"));
  toolbartype.append_text(_("Text beside icons"));

  _interface.pack_start(toolbar, false, false);
  _interface.pack_start(extended_toolbar, false, false);
  _interface.pack_start(statusbar, false, false);
  _interface.pack_start(recent, false, false);
  _interface.pack_start(interface_table, false, false);

  interface_table.set_col_spacing(0, 5);
  interface_table.attach(recentno_label, 0, 1, 0, 1, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  interface_table.attach(recentno, 1, 2, 0, 1, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  interface_table.attach(toolbartype_label, 0, 1, 1, 2, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  interface_table.attach(toolbartype, 1, 2, 1, 2, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  _interface.pack_start(showclose, false, false);
  _interface.pack_start(extra_buttons, false, false);

  extra_buttons.set_active(_conf.get("extra_buttons", true));
  recentno_adj.set_value(_conf.get("recentno", 10));
  recentno.set_adjustment(recentno_adj);
  toolbar.set_active(_conf.get("toolbar", true));
  extended_toolbar.set_active(_conf.get("extended_toolbar", true));
  statusbar.set_active(_conf.get("statusbar", true));
  recent.set_active(_conf.get("recent", true));
  showclose.set_active(_conf.get("showclose", true));
  std::string type = _conf.get("toolbartype","both");
  if (type == "text")
    toolbartype.set_active_text(_("Text only"));
  else if (type == "icons")
    toolbartype.set_active_text(_("Icons only"));
  else if (type == "both_horiz")
    toolbartype.set_active_text(_("Text beside icons"));
  else
    toolbartype.set_active_text(_("Both"));

  toolbar.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::toolbar_toggled_cb));
  recent.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::recent_toggled_cb));
  toolbar.toggled();
  recent.toggled();
}

void PreferencesDialog::tabs()
{
  notebook.append_page(_tabs);

  showtabs.set_use_underline();
  tabsmenu.set_use_underline();
  scrolltabs.set_use_underline();
  showtabs.set_label(_("_Show tabs"));
  tabsmenu.set_label(_("_Enable tabs menu"));
  scrolltabs.set_label(_("Scro_ll tabs"));
  tabspos_label.set_text(_("Show the tabs at the"));
  tabspos.append_text(_("Top"));
  tabspos.append_text(_("Bottom"));
  tabspos.append_text(_("Right"));
  tabspos.append_text(_("Left"));

  readonly_label.set_text(_("Color for read only tabs"));
  modified_label.set_text(_("Color for modified tabs"));
  normal_label.set_text(_("Color for normal tabs"));

  Gdk::Color readonly_color, modified_color, normal_color;

  readonly_color.set_red(_conf.get("readonly_red", 0));
  readonly_color.set_green(_conf.get("readonly_green", 0));
  readonly_color.set_blue(_conf.get("readonly_blue", 65535));

  modified_color.set_red(_conf.get("modified_red", 65535));
  modified_color.set_green(_conf.get("modified_green", 0));
  modified_color.set_blue(_conf.get("modified_blue", 0));

  normal_color.set_red(_conf.get("normal_red", 0));
  normal_color.set_green(_conf.get("normal_green", 0));
  normal_color.set_blue(_conf.get("normal_blue", 0));

  readonly.set_color(readonly_color);
  modified.set_color(modified_color);
  normal.set_color(normal_color);

  readonly.set_use_alpha(false);
  normal.set_use_alpha(false);
  modified.set_use_alpha(false);

  _tabs.pack_start(showtabs, false, false);
  _tabs.pack_start(tabsmenu, false, false);
  _tabs.pack_start(scrolltabs, false, false);
  _tabs.pack_start(tabs_table, true, true);

  tabs_table.attach(tabspos_label, 0, 1, 0, 1, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(tabspos, 1, 2, 0, 1, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(readonly_label, 0, 1, 1, 2, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(readonly, 1, 2, 1, 2, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(modified_label, 0, 1, 3, 4, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(modified, 1, 2, 3, 4, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(normal_label, 0, 1, 5, 6, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));
  tabs_table.attach(normal, 1, 2, 5, 6, Gtk::AttachOptions(Gtk::FILL|Gtk::EXPAND), Gtk::AttachOptions(Gtk::SHRINK));

  showtabs.set_active(_conf.get("showtabs", true));
  tabsmenu.set_active(_conf.get("tabsmenu", true));
  scrolltabs.set_active(_conf.get("scrolltabs", true));
  switch(_conf.get("tabspos", TABS_POS_TOP))
    {
    case TABS_POS_BOTTOM:
      tabspos.set_active_text(_("Bottom"));
      break;
    case TABS_POS_RIGHT:
      tabspos.set_active_text(_("Right"));
      break;
    case TABS_POS_LEFT:
      tabspos.set_active_text(_("Left"));
      break;
    default:
      tabspos.set_active_text(_("Top"));
      break;
    }

  showtabs.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::showtabs_toggled_cb));
  showtabs.toggled();
}

void PreferencesDialog::editor()
{
  notebook.append_page(_editor);

  textwrap.set_use_underline();
  linenumbers.set_use_underline();
  default_font.set_use_underline();
#ifdef ENABLE_HIGHLIGHT
  highlight_auto.set_use_underline();
#endif

  textwrap.set_label(_("Text _Wrapping is on by default"));
  linenumbers.set_label(_("Show _Line numbers by default"));
  default_font.set_label(_("Use the default _font"));
#ifdef ENABLE_HIGHLIGHT
  highlight_auto.set_label(_("_Automatic syntax highlighting for known file types"));
#endif
  linenumbers_label.set_text(_("Line numbers position"));
  linenumbers_pos.append_text(_("Right"));
  linenumbers_pos.append_text(_("Left"));
  linenumbers_pos.append_text(_("Both"));
  font_frame.set_label(_("Font"));

  _editor.pack_start(textwrap, false, false);
  _editor.pack_start(linenumbers, false, false);
  _editor.pack_start(ed_box1, false, false);
  _editor.pack_start(font_frame, false, false);
#ifdef ENABLE_HIGHLIGHT
  _editor.pack_start(highlight_auto, false, false);
#endif

  font_frame.add(ed_box2);
  ed_box2.pack_start(default_font);
  ed_box2.pack_start(font);
  ed_box1.pack_start(linenumbers_label);
  ed_box1.pack_start(linenumbers_pos);

  textwrap.set_active(_conf.get("textwrap", true));
  linenumbers.set_active(_conf.get("linenumbers", false));
  default_font.set_active(_conf.get("default_font", true));
#ifdef ENABLE_HIGHLIGHT
  highlight_auto.set_active(_conf.get("highlight_auto", true));
#endif

  bool r = _conf.get("numbers_right", true);
  bool l = _conf.get("numbers_left", true);
  if ((l) && (r))
    linenumbers_pos.set_active_text(_("Both"));
  else if (l)
    linenumbers_pos.set_active_text(_("Left"));
  else if (r)
    linenumbers_pos.set_active_text(_("Right"));
  else
    linenumbers_pos.set_active_text(_("Both"));

  std::string _df = Utils::katoob_get_default_font();
  std::string def_font = _conf.get("font", _df);
  font.set_font_name(def_font);

  linenumbers.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::linenumbers_toggled_cb));
  default_font.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::default_font_toggled_cb));
  linenumbers.toggled();
  default_font.toggled();
}

void PreferencesDialog::encodings()
{
  notebook.append_page(_encodings);

  locale_enc.set_use_underline();
  locale_enc.set_label(_("_Use the locale encoding"));
  saved_enc_label.set_text(_("Specify an encoding"));
  save_enc_label.set_text(_("Specify an encoding"));

  locale_enc.set_active(_conf.get("locale_enc", false));

  saved_frame.set_label(_("File open encoding"));
  save_frame.set_label(_("File save encoding"));

  _encodings.pack_start(saved_frame, false, false);
  _encodings.pack_start(save_frame, false, false);

  saved_frame.add(enc_box3);
  enc_box3.pack_start(locale_enc, false, false);
  enc_box3.pack_start(enc_box1, false, false);
  save_frame.add(enc_box2);
  enc_box1.pack_start(saved_enc_label, false, false);
  enc_box1.pack_start(saved_enc, false, false);

  enc_box2.pack_start(save_enc_label, false, false);
  enc_box2.pack_start(save_enc, false, false);

  for (int i = 0; i < _enc.size(); i++)
    if (i != _enc.utf8())
      saved_enc.append_text(_enc.at(i));

  for (int i = 0; i < _enc.size(); i++)
    save_enc.append_text(_enc.at(i));

  saved_enc.set_active(_enc.default_open());
  save_enc.set_active(_enc.default_save());

  locale_enc.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::locale_enc_toggled_cb));
  locale_enc.toggled();
}

void PreferencesDialog::file_save()
{
  notebook.append_page(_file_save);

  backup.set_use_underline();
  backup.set_label(_("_Backup files before saving."));
  backup_label.set_text(_("Backup extension"));

  _file_save.pack_start(backup, false, false);
  _file_save.pack_start(fs_box1, false, false);
  fs_box1.pack_start(backup_label);
  fs_box1.pack_start(backup_ext);

  backup.set_active(_conf.get("backup", true));
  backup_ext.set_text(_conf.get("backup_ext", "~"));

  backup.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::backup_toggled_cb));
  backup.toggled();
}

#ifdef HAVE_SPELL
void PreferencesDialog::spell_checker()
{
  notebook.append_page(_spell_checker);

  spell_check.set_use_underline();

  spell_check.set_label(_("_Autocheck Spelling"));
  //  misspelled_label.set_text(_("Misspelled words color"));
  default_dict_label.set_text(_("Choose the default dictionary"));

  std::vector<std::string> dicts;
  katoob_spell_list_available(dicts);

  for (unsigned x = 0; x < dicts.size(); x++)
    default_dict.append_text(dicts[x]);

  _spell_checker.pack_start(spell_check, false, false);
  //  _spell_checker.pack_start(sc_box1, false, false);
  _spell_checker.pack_start(sc_box2, false, false);
  //  sc_box1.pack_start(misspelled_label);
  //  sc_box1.pack_start(misspelled);
  sc_box2.pack_start(default_dict_label);
  sc_box2.pack_start(default_dict);

  spell_check.set_active(_conf.get("spell_check", true));
  default_dict.set_active_text(_conf.get("default_dict", "en"));
  /*
  Gdk::Color color;
  color.set_red(_conf.get("misspelled_red", 65535));
  color.set_green(_conf.get("misspelled_green", 0));
  color.set_blue(_conf.get("misspelled_blue", 0));

  misspelled.set_color(color);
  misspelled.set_use_alpha(false);
  */
  //  spell_check.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::spell_check_toggled_cb));
  spell_check.toggled();
}
#endif

#ifdef ENABLE_PRINT
void PreferencesDialog::print()
{
  notebook.append_page(_print);

  print_label.set_text(_("Printing font"));

  gui_show_advanced.set_use_underline();
  gui_show_advanced.set_label(_("_Enable advanced options"));

  dpi_x_label.set_use_underline();
  dpi_y_label.set_use_underline();
  dpi_x_label.set_text(_("X Dot Per Inch"));
  dpi_y_label.set_text(_("Y Dot Per Inch"));

  dpi_x_adj.set_lower(150);
  dpi_x_adj.set_upper(1200);
  dpi_x_adj.set_value(_conf.print_get("dpi_x", 150));
  dpi_y_adj.set_lower(150);
  dpi_y_adj.set_upper(1200);
  dpi_y_adj.set_value(_conf.print_get("dpi_y", 150));

  dpi_x.set_adjustment(dpi_x_adj);
  dpi_y.set_adjustment(dpi_y_adj);

  gui_show_advanced.set_active(_conf.print_get("gui_show_advanced", false));

  print_font.set_font_name(_conf.print_get("print_font", "Sans Regular 12"));

  _print.pack_start(p_box2, false, false);
  p_box2.pack_start(print_label, false, false);
  p_box2.pack_start(print_font, false, false);

  _print.pack_start(gui_show_advanced, false, false);

  _print.pack_start(print_frame, false, false);

  print_frame.add(print_table);
  print_table.attach(dpi_x_label, 0, 1, 0, 1);
  print_table.attach(dpi_x, 1, 2, 0, 1);
  print_table.attach(dpi_y_label, 0, 1, 1, 2);
  print_table.attach(dpi_y, 1, 2, 1, 2);

  gui_show_advanced.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::gui_show_advanced_toggled_cb));
  gui_show_advanced.toggled();
}
#endif

void PreferencesDialog::dictionary()
{
  notebook.append_page(_dictionary);

  dict.set_use_underline();
  dict.set_label(_("_Enable the dictionary."));
  dict_host_label.set_text(_("Host"));
  list_dicts.set_label(_("_List dictionaries"));
  list_dicts.set_use_underline();
  dict_port_label.set_label(_("Port"));
  dict_timeout_label.set_label(_("Timeout"));

  dict_timeout_adj.set_lower(1);
  dict_timeout_adj.set_upper(100);
  dict_timeout_adj.set_value(_conf.get("dict_timeout", 1));

  dict_port_adj.set_lower(0);
  dict_port_adj.set_upper(65535);
  dict_port_adj.set_value(_conf.get("dict_port", 2628));

  dict_port.set_adjustment(dict_port_adj);
  dict_timeout.set_adjustment(dict_timeout_adj);

  dict.set_active(_conf.get("dict", true));
  dict_host.set_text(_conf.get("dict_host", "dict.arabeyes.org"));

  dict_name_label.set_text(_("Dictionary"));
  dict_sw.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  dict_sw.add(dicts);

  dict_record.add(dict_name);
  dict_record.add(dict_description);
  dict_store = Gtk::ListStore::create(dict_record);
  dicts.set_model(dict_store);
  dicts.append_column(_("Dictionary"), dict_name);
  dicts.append_column(_("Description"), dict_description);
  dict_selection = dicts.get_selection();

  Gtk::TreeModel::Row row = *(dict_store->append ());
  row[dict_name] = _conf.get("dict_db", "arabic");
  row[dict_description] = _("Not Available.");
  dict_selection->select(row);

  _dictionary.pack_start(dict, false, false);
  _dictionary.pack_start(dictionary_table);
  dictionary_table.set_col_spacing(0, 5);

  dictionary_table.attach(dict_host_label, 0, 1, 0, 1, Gtk::AttachOptions(Gtk::SHRINK));
  dictionary_table.attach(dict_host, 1, 2, 0, 1);
  dictionary_table.attach(list_dicts, 2, 3, 0, 1, Gtk::AttachOptions(Gtk::SHRINK), Gtk::AttachOptions(Gtk::SHRINK));
  dictionary_table.attach(dict_port_label, 0, 1, 1, 2, Gtk::AttachOptions(Gtk::SHRINK));
  dictionary_table.attach(dict_port, 1, 2, 1, 2);
  dictionary_table.attach(dict_timeout_label, 0, 1, 2, 3, Gtk::AttachOptions(Gtk::SHRINK));
  dictionary_table.attach(dict_timeout, 1, 2, 2, 3);
  dictionary_table.attach(dict_name_label, 0, 1, 3, 4, Gtk::AttachOptions(Gtk::SHRINK));
  dictionary_table.attach(dict_sw, 1, 2, 3, 4);

  dict.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::dict_toggled_cb));
  dict.toggled();
  list_dicts.signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::list_dicts_clicked_cb));
}

void PreferencesDialog::remote()
{
  notebook.append_page(_remote);

  _remote.pack_start(open_location_to_active, false, false);
  _remote.pack_start(r_box1, false, false);
  _remote.pack_start(r_box2, false, false);
  r_box1.pack_start(locations_size_label);
  r_box1.pack_start(locations_size);
  r_box2.pack_start(remote_command_label);
  r_box2.pack_start(remote_command);

  remote_command_label.set_text(_("Command"));

  std::string command(_conf.get("remote_command", REMOTE_COMMAND));

  if (command.size() == 0)
    command = REMOTE_COMMAND;
  remote_command.set_text(command);
  locations_size.set_adjustment(locations_size_adj);
  locations_size_adj.set_upper(1000);
  locations_size_adj.set_lower(0);
  locations_size_adj.set_value(_conf.get("locations_size", 10));
  locations_size_label.set_text(_("Number of remote addresses to save\n(0 for unlimited)"));
  open_location_to_active.set_use_underline();
  open_location_to_active.set_active(_conf.get("open_location_to_active", false));
  open_location_to_active.set_label(_("_Insert into the active document enabled by default ?"));
}

void PreferencesDialog::advanced()
{
  notebook.append_page(_advanced);

  x_label.set_text(_("Left"));
  y_label.set_text(_("Top"));
  w_label.set_text(_("Width"));
  h_label.set_text(_("Height"));

  saveonexit.set_label(_("Save all the program _settings on exit"));
  savewinpos.set_label(_("Save the main window _attributes on exit"));

  saveonexit.set_use_underline();
  savewinpos.set_use_underline();

  pos.set_label(_("Window position"));
  size.set_label(_("Window size"));
  x.set_adjustment(x_adj);
  y.set_adjustment(y_adj);
  w.set_adjustment(w_adj);
  h.set_adjustment(h_adj);

  _advanced.pack_start(pos, false, false);
  _advanced.pack_start(size, false, false);
  _advanced.pack_start(savewinpos, false, false);
  _advanced.pack_start(saveonexit, false, false);

  pos.add(pos_table);
  size.add(size_table);

  pos_table.attach(x_label, 0, 1, 0, 1);
  pos_table.attach(x, 1, 2, 0, 1);
  pos_table.attach(y_label, 0, 1, 1, 2);
  pos_table.attach(y, 1, 2, 1, 2);

  size_table.attach(w_label, 0, 1, 0, 1);
  size_table.attach(w, 1, 2, 0, 1);
  size_table.attach(h_label, 0, 1, 1, 2);
  size_table.attach(h, 1, 2, 1, 2);

  Glib::RefPtr<Gdk::Screen> screen = Gdk::Screen::get_default();
  x_adj.set_upper(screen->get_width()-10);
  y_adj.set_upper(screen->get_height()-10);
  h_adj.set_upper(screen->get_height());
  w_adj.set_upper(screen->get_width());
  w_adj.set_lower(400);
  h_adj.set_lower(400);

  x_adj.set_value(_conf.get("x", 50));
  y_adj.set_value(_conf.get("y", 50));
  h_adj.set_value(_conf.get("h", 400));
  w_adj.set_value(_conf.get("w", 500));

  saveonexit.set_active(_conf.get("saveonexit", true));
  savewinpos.set_active(_conf.get("savewinpos", true));

  savewinpos.signal_toggled().connect(sigc::mem_fun(*this, &PreferencesDialog::savewinpos_toggled_cb));
  savewinpos.toggled();
}

void PreferencesDialog::selection_signal_changed_cb()
{
  Gtk::TreeModel::iterator iter = selection->get_selected();
  if (iter)
    {
      Gtk::TreeModel::Row row = *iter;
      int n = row[number];
      notebook.set_current_page(n);
    }
}

void PreferencesDialog::list_dicts_clicked_cb()
{
  std::string host(dict_host.get_text());
  int port = dict_port.get_value_as_int();
  int timeout = dict_timeout.get_value_as_int();
  if (host.size() > 0)
    {
      Dict dict(host, port, timeout);
      std::string error;
      if (dict.ok(error))
	{
	  Books books;
	  if (dict.get_books(books, error))
	    {
	      dict_store->clear();
	      Gtk::TreeModel::Row _row;
	      for (unsigned x = 0; x < books.size(); x++)
		{
		  Gtk::TreeModel::Row row = *(dict_store->append());
		  if (x == 0)
		    _row = row;
		  row[dict_name] = books[x].book;
		  row[dict_description] = books[x].description;
		}
	      dict_selection->select(_row);
	    }
	  else {
	    katoob_error(error);
	  }
	}
      else {
	katoob_error(error);
      }
    }
}

void PreferencesDialog::repopulate_conf()
{
  _conf.set("undo", undo.get_active());
  _conf.set("undo_closed", undo_closed.get_active());
  _conf.set("undono", undono.get_value_as_int());
  _conf.set("exec_cmd_size", exec_cmd_size.get_value_as_int());
  _conf.set("undo_closedno", undo_closedno.get_value_as_int());

  _conf.set("toolbar", toolbar.get_active());
  _conf.set("extended_toolbar", extended_toolbar.get_active());
  _conf.set("statusbar", statusbar.get_active());
  _conf.set("recent", recent.get_active());
  _conf.set("showclose", showclose.get_active());
  _conf.set("recentno", recentno.get_value_as_int());
  _conf.set("extra_buttons", extra_buttons.get_active());

  std::string val;
  switch (toolbartype.get_active_row_number())
    {
    case 0:
      val = "text";
      break;
    case 1:
      val = "icons";
      break;
    case 3:
      val = "both_horiz";
      break;
    default:
      val = "both";
      break;
    }
  _conf.set("toolbartype", val.c_str());

  _conf.set("showtabs", showtabs.get_active());
  _conf.set("tabsmenu", tabsmenu.get_active());
  _conf.set("scrolltabs", scrolltabs.get_active());

  int i;
  switch (tabspos.get_active_row_number())
    {
    case 1:
      i = TABS_POS_BOTTOM;
      break;
    case 2:
      i = TABS_POS_RIGHT;
      break;
    case 3:
      i = TABS_POS_LEFT;
      break;
    default:
      i = TABS_POS_TOP;
      break;
    }
  _conf.set("tabspos", i);
  Gdk::Color color = readonly.get_color();
  _conf.set("readonly_red", color.get_red());
  _conf.set("readonly_green", color.get_green());
  _conf.set("readonly_blue", color.get_blue());
  color = modified.get_color();
  _conf.set("modified_red", color.get_red());
  _conf.set("modified_green", color.get_green());
  _conf.set("modified_blue", color.get_blue());
  color = normal.get_color();
  _conf.set("normal_red", color.get_red());
  _conf.set("normal_green", color.get_green());
  _conf.set("normal_blue", color.get_blue());

  _conf.set("textwrap", textwrap.get_active());
  _conf.set("linenumbers", linenumbers.get_active());
  _conf.set("default_font", default_font.get_active());
  _conf.set("font", font.get_font_name().c_str());
#ifdef ENABLE_HIGHLIGHT
  _conf.set("highlight_auto", highlight_auto.get_active());
#endif

  bool r, l;
  switch (linenumbers_pos.get_active_row_number())
    {
    case 0:
      l = false;
      r = true;
      break;
    case 1:
      l = true;
      r = false;
      break;
    default:
      l = true;
      r = true;
      break;
    }
  _conf.set("numbers_right", r);
  _conf.set("numbers_left", l);

  _conf.set("save_enc", _enc.get(save_enc.get_active_row_number()).c_str());
  _conf.set("saved_enc", _enc.get(saved_enc.get_active_row_number()).c_str());
  _conf.set("locale_enc", locale_enc.get_active());
  _conf.defaults(_enc);

  _conf.set("backup", backup.get_active());
  if (backup_ext.get_text().size() == 0)
    backup_ext.set_text("~");
  _conf.set("backup_ext", backup_ext.get_text().c_str());

#ifdef HAVE_SPELL
  _conf.set("spell_check", spell_check.get_active());
  _conf.set("default_dict", default_dict.get_active_text().c_str());
#endif
  /*
  color = misspelled.get_color();
  _conf.set("misspelled_red", color.get_red());
  _conf.set("misspelled_green", color.get_green());
  _conf.set("misspelled_blue", color.get_blue());
  */
#ifdef ENABLE_PRINT
  _conf.print_set("gui_show_advanced", gui_show_advanced.get_active());
  _conf.print_set("dpi_x", dpi_x.get_value_as_int());
  _conf.print_set("dpi_y", dpi_y.get_value_as_int());
  _conf.print_set("print_font", print_font.get_font_name().c_str());
#endif
  _conf.set("dict", dict.get_active());
  _conf.set("dict_host", dict_host.get_text().c_str());
  _conf.set("dict_port", dict_port.get_value_as_int());
  _conf.set("dict_timeout", dict_timeout.get_value_as_int());

  Gtk::TreeModel::iterator iter = dict_selection->get_selected();
  if (iter)
    {
      Gtk::TreeModel::Row row = *iter;
      Glib::ustring word = row[dict_name];
      _conf.set("dict_db", word.c_str());
    }

  _conf.set("open_location_to_active", open_location_to_active.get_active());
  _conf.set("locations_size", locations_size.get_value_as_int());
  _conf.set("remote_command", remote_command.get_text().c_str());

  _conf.set("saveonexit", saveonexit.get_active());
  _conf.set("savewinpos", savewinpos.get_active());
  _conf.set("x", x.get_value_as_int());
  _conf.set("y", y.get_value_as_int());
  _conf.set("w", w.get_value_as_int());
  _conf.set("h", h.get_value_as_int());

  _conf.adjust_lists();
}
