#ifndef _VMSTATE_H // -*-C++-*-
#define _VMSTATE_H

/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

using namespace std;

#ifndef WIN32
#include <pthread.h>
#define GC_PTHREADS
#else
#include <io.h>
#define GC_WIN32_THREADS
#endif
#include <gc/gc_cpp.h>
#include <setjmp.h>

#include <map>
#include <stack>

#include "Heap.h"

// Defined by compiling Builtins.k, so obviously any binary of a kaya
// program needs to be linked to at least Builtins.o!
#define PROJECT_FROM_NON_UNION _ns__D_Builtins__D_Project_UN_From_UN_Non_UN_Union
#define LOOKUP_FROM_NON_ARRAY _ns__D_Builtins__D_Lookup_UN_From_UN_Non_UN_Array
#define NEGATIVE_ARRAY_INDEX _ns__D_Builtins__D_Negative_UN_Array_UN_Index
#define WRONG_CONSTRUCTOR _ns__D_Builtins__D_Wrong_UN_Constructor
#define GETTING_TAG_FROM_NON_UNION _ns__D_Builtins__D_Getting_UN_Tag_UN_From_UN_Non_UN_Union
#define CALL_STACK_OVERFLOW _ns__D_Builtins__D_Call_UN_Stack_UN_Overflow
#define INVALID_MARSHALLING_ID _ns__D_Builtins__D_Invalid_UN_Marshalling_UN_ID
#define INVALID_FUNCTION_TABLE_HASH _ns__D_Builtins__D_Invalid_UN_Function_UN_Table_UN_Hash
#define CANT_MARSHAL_EXCEPTIONS _ns__D_Builtins__D_Cant_UN_Marshal_UN_Exceptions
#define CANT_REFLECT_EXCEPTIONS _ns__D_Builtins__D_Cant_UN_Reflect_UN_Exceptions
#define INVALID_VALUE _ns__D_Builtins__D_Invalid_UN_Value
#define NOT_IMPLEMENTED _ns__D_Builtins__D_Not_UN_Implemented
#define INVALID_CIRCULAR_STRUCTURE _ns__D_Builtins__D_Invalid_UN_Circular_UN_Structure
#define UNKNOWN_FUNCTION_ID _ns__D_Builtins__D_Unknown_UN_Function_UN_ID
#define DIVIDE_BY_ZERO _ns__D_Builtins__D_Divide_UN_By_UN_Zero

extern char* PROJECT_FROM_NON_UNION;
extern char* LOOKUP_FROM_NON_ARRAY;
extern char* NEGATIVE_ARRAY_INDEX;
extern char* WRONG_CONSTRUCTOR;
extern char* GETTING_TAG_FROM_NON_UNION;
extern char* CALL_STACK_OVERFLOW;
extern char* INVALID_MARSHALLING_ID;
extern char* INVALID_FUNCTION_TABLE_HASH;
extern char* INVALID_VALUE;
extern char* NOT_IMPLEMENTED;
extern char* INVALID_CIRCULAR_STRUCTURE;
extern char* CANT_MARSHAL_EXCEPTIONS;
extern char* CANT_REFLECT_EXCEPTIONS;
extern char* UNKNOWN_FUNCTION_ID;
extern char* DIVIDE_BY_ZERO;

/// Function map is global to all threads.
//extern map<int,func> m_funmap;

void initFunMap(int sz, int fmhash);
void addToFunMap(int id, func fn);
func getFn(int id);
int getFnID(func fn);
int getFnHash();

class Exception;

class VMState: public gc {
    friend class Exception;
public:
    VMState(bool panic=false);
    
    Value* doPop() {
	Value* tmp=*(--m_stackptr);
	*m_stackptr=NULL;
	return tmp;
    }

  // sometimes just shortening the stack so this is faster than doPop()
    void discardPop() {
        *(--m_stackptr) = NULL;
    }
  
    void push(Value* val) {
	// FIXME: Grow stack if necessary? Might be quite an overhead... maybe 
	// make it an option.
	*m_stackptr = val;
	++m_stackptr;
    }

    // Next ones are for efficiency
    void push2(Value* val, Value* val2);
    void push3(Value* val, Value* val2, Value* val3);
    void push4(Value* val, Value* val2, Value* val3,Value *val4);

    // Push then settop is common, so combine.
    void pushsettop(Value* val) {
	val->setPtr(*(--m_stackptr));
	*m_stackptr=NULL;
    }

    // Setting the top as a temp is also common.
    void tmpsettop(int val) {
	(*(--m_stackptr))->setInt(val);
	*m_stackptr=NULL;
    }

    void mkArray(int size);
    void finish();
    bool emptyStack();

/// Replace the top stack item with its ith argument.
/// Let's see if inlining helps
    void projarg(int i, int t)
    {
      Value*& topval = (*(m_stackptr-1));
#ifndef NOCHECK
      if (topval==NULL) {
	  kaya_rtsError(INVALID_VALUE);
      }
      if (topval->getType()!=KVT_UNION) {
	  kaya_rtsError(PROJECT_FROM_NON_UNION);
      }
#endif
      Union* top = (Union*)topval->getRaw();
#ifndef NOCHECK
      if (U_TAG(top)!=t) {
	  kaya_rtsError(WRONG_CONSTRUCTOR);
      }
      if (U_ARITY(top) <= i) {
	  kaya_rtsError(INVALID_VALUE);
      }
#endif
      topval=top->args[i];
      //    push(arg);
    }

    void goToIndex();

    void pushToIndex(Value* v)
    {
	int idx = v->getInt();
	(*(m_stackptr-1)) = (*(m_stackptr-1))->lookup(this,idx);
    }

/** Replace the top stack item with the contents of the second stack item,
    then remove both (mutation, not evaluation) */
    void setTop()
    {
	(*(m_stackptr-1))->setPtr((*(m_stackptr-2)));
	(*(m_stackptr-2)) = (*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

// Add the top stack value to the next one, then remove both.
    void addTop()
    {
	(*(m_stackptr-2))->addVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void subTop()
    {
	(*(m_stackptr-2))->subVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void mulTop()
    {
	(*(m_stackptr-2))->mulVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void divTop()
    {
	(*(m_stackptr-2))->divVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void appendTop()
    {
	// append top stack item onto top-1 stack item
	wchar_t* topstr = (*(--m_stackptr))->getString()->getVal();
	(*(m_stackptr-1))->getString()->append(topstr);
	*m_stackptr = NULL;
    }

    void appendTopInt()
    {
	// append top stack item onto top-1 stack item
	wchar_t topstr = (wchar_t)((*(--m_stackptr))->getInt());
	(*(m_stackptr-1))->getString()->append(topstr);
	*m_stackptr = NULL;
    }

    /// Return the top stack item.
    Value* topItem() {
	return (*(m_stackptr-1));
    }

    
    void pushglobal(Value**& globtable, int i);
    void createglobal(wchar_t* modid,int i);

    int tag();

    void doAppend();
    void doAppendChar(wchar_t x);
    void doAppendStr(wchar_t* x);
    void doAppendStr(char* x);
    void doEqExcept(bool inv);
    void doEqString(bool inv);
    void doEqString(bool inv, wchar_t* str);
    void doEqString(bool inv, char* str);
    bool doTopStrEq(wchar_t* str);
    bool doTopStrEq(char* str);

    void str2int();
    void int2str();
    void real2str();
    void str2real();
    void str2chr();
    void chr2str();
    void bool2str();
    void real2int();
    void int2real();

    void newjmp_buf();

    void kaya_throw(wchar_t* msg, int code);
    void kaya_throw(char* msg, int code);
    void kaya_internalError(int code);
    void kaya_rtsError(char* exceptionID);
    void throw_ex();
    jmp_buf* top_ex();
    void tried();
    void restore();

    void getindex();

    void pushgetindex(Value* v) {
	int idx = v->getInt();
	--m_stackptr;
	Value* array = *m_stackptr;
	Value* el = array->lookup(this,idx);
	push(el);
    }

    Value* mkstr(wchar_t* str);
    Value* mkint(void* i);

    void lineno(const char* src, int ln); 
    void lineno(const wchar_t* src, int ln); 
    void pushBT(char* src, char* mod, int ln);
    void pushBT(wchar_t* src, wchar_t* mod, int ln);
    void popBT();

    void memstat();

    int maxMemUsage() { 
	return m_maxmem; 
    }

    // Implement print via write
    void writestdout();

    void returnVal(Value* ret) { 
	m_return.setPtr(ret); 
    }

    Value* getReturnVal() { return &m_return; }

    // Throw divide by zero exception
    void divideByZero() {
	kaya_rtsError(DIVIDE_BY_ZERO);
    }

private:
    Value** m_valstack;
    Value** m_stackptr;
    CallStackEntry** m_btptr;

//    int m_stacksize;

    int m_stackalloc;
    int m_globalloc;

    struct StackData {
	Value** stackptr;
	CallStackEntry** csptr;
    };

    stack<jmp_buf*> m_except_stack;
    stack<StackData> m_stack_stack; // Stack size when an exception happens.

    // Tracing details
    CallStackEntry** m_backtrace;
    wchar_t* m_sourcefile;
    int m_lineno;

    int m_maxmem;

    Value m_return; // Place to put return values (sort of a register).
};

#endif // whole file
