/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include <gc/gc_allocator.h>

#include "sizes.h"
#include <iostream>
#include "Heap.h"
#include <wchar.h>
#include <stdio.h>
#include <assert.h>
#include "ValueFuns.h"
#include "stdfuns.h"
#include "Array.h"
#include "VMState.h"
#include "Closure.h"
#include "VM.h"
#include <cstring>

// cached values
Value* zero = MKVAL((void*)0,KVT_INT);
Value* one = MKVAL((void*)1,KVT_INT);
Value* minusone = MKVAL((void*)(-1),KVT_INT);

wchar_t* emptyString() {
  wchar_t* buf = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t));
  buf[0] = '\0';
  return buf;
}


String* Value::getString()
{
    /// TODO - throw if it's not a string..
    if (m_val!=NULL)
	return (String*)m_val;
    else
	return NEWSTRING(emptyString());
}

Exception* Value::getExcept()
{
    /// TODO - throw if it's not an exception.
    if (m_val!=NULL) {
	if (m_funtable!=KVT_EXCEPTION) {
	    assert(false);
	    return NULL;
	}
	return (Exception*)m_val;
    }
    else {
	assert(false);
	return NULL;
    }
}

void Value::runClosure(VMState* vm)
{
//    cout << m_val << endl;
    ((Closure*)m_val)->run(vm);
//    ((func)m_val)(vm);
}

Value* Value::addClosureArgs(VMState *vm, kint i, bool lam)
{
    // Copy the closure first, because it may be used again in the same
    // way, and we want it to be in the same state then.
    m_val = (void*)(new Closure(*((Closure*)m_val)));
    ((Closure*)m_val)->addArgs(vm,i,lam);
    return this;
}

Array* Value::getArray()
{
    /// TODO - throw if it's not an array.
    if (m_val!=NULL)
	return (Array*)m_val;
    else
	return new Array();
}

Union* Value::getUnion()
{
    /// TODO - throw if it's not a union.
    if (m_val!=NULL)
	return (Union*)m_val;
    assert(false);
    return NULL; // Can't happen.
}

void Value::int2str()
{
    kint v=getInt();
    wchar_t buf[50]; // *Ick*
    SWPRINTF(buf,50,DIGITFORMAT,v);
    setString(NEWSTRING(buf));
}

void Value::int2str(StringPool* sp)
{
    kint v=getInt();
    wchar_t buf[50]; // *Ick*
    SWPRINTF(buf,50,DIGITFORMAT,v);
    setString(NEWPSTRING(sp,buf));
}

void Value::real2str()
{
    double v=getReal();
    wchar_t buf[100]; // *Ick*
    SWPRINTF(buf,50,L"%g",v);
    setString(NEWSTRING(buf));
    m_funtable = KVT_STRING;
}

void Value::real2str(StringPool* sp)
{
    double v=getReal();
    wchar_t buf[100]; // *Ick*
    SWPRINTF(buf,50,L"%g",v);
    setString(NEWPSTRING(sp,buf));
    m_funtable = KVT_STRING;
}

void Value::str2int()
{
    String* s=getString();
    kint val=(kint)(wcstol(s->getVal(),NULL,10));
    setInt(val);
    m_funtable = KVT_INT;
}

void Value::str2real()
{
    String* s=getString();
    double val=wcstod(s->getVal(),NULL);
    setReal(val);
    m_funtable = KVT_REAL;
}

void Value::str2real(RealPool* r)
{
    String* s=getString();
    double val=wcstod(s->getVal(),NULL);
    setReal(r,val);
    m_funtable = KVT_REAL;
}


void Value::chr2str()
{
  //CIM: wchar_t is an integer type storing the UCS position
    wchar_t v=(wchar_t)getInt();
    setString(NEWSTRING(v));
    m_funtable = KVT_STRING;
}

void Value::chr2str(StringPool* s)
{
  //CIM: wchar_t is an integer type storing the UCS position
    wchar_t v=(wchar_t)getInt();
    setString(NEWPSTRING(s,v));
    m_funtable = KVT_STRING;
}


void Value::bool2str()
{
    kint v=getInt();
    if (v==1) { setString(NEWSTRING(L"true")); }
    else { setString(NEWSTRING(L"false")); }
}

void Value::bool2str(StringPool* s)
{
    kint v=getInt();
    if (v==1) { setString(NEWPSTRING(s,L"true")); }
    else { setString(NEWPSTRING(s,L"false")); }
}

void Value::str2chr()
{
    assert(false); // Makes no sense!
}

void Value::int2real()
{
    kint v=getInt();
    setReal((double)v);
    m_funtable = KVT_REAL;
}

void Value::int2real(RealPool* rp)
{
    kint v=getInt();
    setReal(rp,(double)v);
    m_funtable = KVT_REAL;
}

void Value::real2int()
{
    double v=getReal();
    setInt((kint)v);
    m_funtable = KVT_INT;
}

void Value::setReal(double v)
{
    m_val=(void*)(new Real(v));
    m_funtable = KVT_REAL;
}

void Value::setReal(RealPool* r, double v)
{
    m_val=(void*)(new(r) Real(v));
    m_funtable = KVT_REAL;
}

void Value::setFunc(Closure* f)
{
    m_val=(void*)f;
    m_funtable = KVT_FUNC;
}

void Value::setArray(Array* a)
{
    m_val=(void*)a;
    m_funtable = KVT_ARRAY;
}


void Value::project(VMState* vm, kint i)
{
    if (vm->topItem()->getType()!=KVT_UNION) {
	vm->kaya_rtsError(PROJECT_FROM_NON_UNION);
    }

    Union* top=(Union*)(vm->topItem()->m_val);
    Value* arg=top->args[i];
    m_val=arg->m_val;
    m_funtable = arg->m_funtable;
}

Value* Value::lookup(VMState* vm, kint i)
{
#ifndef NOCHECK
    if (m_val && getType()!=KVT_ARRAY) {
//	cout << m_val << "," << getType() << endl;
	vm->kaya_rtsError(LOOKUP_FROM_NON_ARRAY);
    }
#endif
    if (i<0) {
	vm->kaya_rtsError(NEGATIVE_ARRAY_INDEX);
    }

    Array* v = (Array*)m_val;
    if (v==NULL)
    {
	v = new(vm->getAPool()) Array();
	m_val=v;
	m_funtable = KVT_ARRAY;
    }

    kint extras = i-v->size();
    if (extras>=0) {
      //	v->expectedSize(i);
	for(int l=extras;l>=0;--l) {
	  Value* n = MKPVAL(vm->getVPool(),NULL, KVT_NULL);
	  v->push_back(n);
	  if (l == 0) {
	    return n;
	  }
	}
    }

    return v->lookup(i);
}

kint Value::length()
{
    Array* v = (Array*)m_val;
    if (v==NULL)
    {
	v = new Array();
	m_val=v;
    }
    return v->size();
}

// for returns from foreign function calls
// it can be a significant memory saving, though doesn't make much
// difference to speed
String::String(VMState* vm)
{
    m_offset=0;
    m_len = wcslen(vm->strbuffer);
    m_alloc = vm->strbuflen;
    m_str=vm->strbuffer;
}

String::String(const wchar_t* val)
{
    m_offset=0;
    if (val!=NULL) {
        m_len = wcslen(val);
	m_alloc = m_len*2;
	m_str=(wchar_t*)KMALLOC_BLOCK(m_alloc*sizeof(wchar_t));
	wcscpy(m_str,val);
    }
    else {
	m_len=0;
	m_alloc = 128;
	m_str=(wchar_t*)KMALLOC_BLOCK(m_alloc*sizeof(wchar_t));
//	strcpy(m_str,"(none)"); // Debugging purposes
	//wcscpy(m_str,emptyString());
	m_str[0] = '\0';
    }
}

String::String(wchar_t val)
{
    m_alloc = 16;
    m_offset = 0;
    m_len = 1;
    m_str=(wchar_t*)KMALLOC_BLOCK(m_alloc*sizeof(wchar_t));
    m_str[0]=val;
    m_str[1]='\0';
}

String::String(kint len)
{
    m_alloc = len+1;
    m_offset = 0;
    m_len = 0;
    m_str=(wchar_t*)KMALLOC_BLOCK(m_alloc*sizeof(wchar_t));
    m_str[0]='\0';
}

void String::append(String* s)
{
    // reallocation is safe with strings, because there will be no
    // references into the middle.
    m_len += s->length();
    if (m_len>=M_ALLOC) {
	m_alloc = (m_len+8)*2; //(newlen+10)*2;
//	cout << "Realloc " << m_alloc << " to " << wctostr(m_str) << " + " << wctostr(s->getVal()) << endl;
	wchar_t* newstr = (wchar_t*)KMALLOC_BLOCK(sizeof(wchar_t)*m_alloc);
	wcscpy(newstr,M_STR);
	m_str=newstr;
	m_offset=0;
    }
    wcscat(m_str,s->getVal());
}

void String::append(wchar_t c)
{
    if (c == '\0') { return; } // does nothing
    m_len++;
    if (m_len>=M_ALLOC) {
	m_alloc = (m_len+8)*2; //(newlen+10)*2;
//	cout << "Realloc " << m_alloc << " to " << wctostr(m_str) << endl;
	wchar_t* newstr = (wchar_t*)KMALLOC_BLOCK(sizeof(wchar_t)*m_alloc);
	wcscpy(newstr,M_STR);
	m_str=newstr;
	m_offset=0;
    }
    m_str[m_len-1]=c;
    m_str[m_len]='\0';
}

void String::append(const wchar_t* s)
{
    m_len += wcslen(s);
    if (m_len>=M_ALLOC) {
	m_alloc = (m_len+8)*2; //(newlen+10)*2;
//	cout << "Realloc " << m_alloc << " to " << wctostr(m_str) << endl;
	wchar_t* newstr = (wchar_t*)KMALLOC_BLOCK(sizeof(wchar_t)*m_alloc);
	wcscpy(newstr,M_STR);
	m_str=newstr;
	m_offset=0;
    }
    wcscat(m_str,s);
}

void String::getForeign(const char* c) {
  kint clen = strlen(c);
  if (clen >= m_alloc) {
    m_alloc = clen+1;
    cerr << "REALLOC!" << endl;
    wchar_t* newstr = (wchar_t*)KMALLOC_BLOCK(sizeof(wchar_t)*(m_alloc));
    m_str = newstr;
    m_offset = 0;
    m_len = 0;
  } else {
    m_len = 0;
    m_offset = 0;
    M_STR[0] = '\0';
  }
  mkUCS(c,m_str,clen);
  m_len = wcslen(m_str);
}

bool String::eq(String* s)
{
//    cout << "Comparing " << s->getVal() << " and " << m_str << "." << endl;
    return (wcscmp(M_STR,s->getVal())==0);
}

bool String::eq(const wchar_t* s)
{
    return (wcscmp(M_STR,s)==0);
}

kint String::cmp(String* s)
{
//    cout << "Comparing " << s->getVal() << " and " << m_str << "." << endl;
    return (wcscmp(M_STR,s->getVal()));
}

void String::offset(kint inc)
{
    m_offset+=inc;
    m_len-=inc;
    // If it's got too long, reallocate.
    if (m_offset>OFFSET_LIMIT) {
      // CIM: should we recalculate m_alloc at this point?
	wchar_t* newstr = (wchar_t*)KMALLOC_BLOCK(sizeof(wchar_t)*m_alloc);
	wcscpy(newstr,M_STR);
	m_str=newstr;
	m_offset=0;
    }
}

void String::chop(kint inc)
{
    M_STR[m_len-inc]='\0';
    m_len -= inc;
}

Union::Union(VMState* vm, ukint ar, bool getargs)
    :arity(ar)
{
    // Make space for the data, then take <arity> items off the stack and
    // put them into the structure.
//    cout << "Making union with tag " << t << " and arity " << ar << endl;
//    cout << "tag_arity = " << tag_arity << endl;
    if (ar > 0) {
      args = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*ar);
      if (getargs) {
	for(ukint i=0;i<ar;++i) {
	  //	    --stacksize;
	  /// Create a *new* reference, or things break in confusing ways.
	  args[i]=vm->doPop()->cloneTo(vm->getVPool());
	}
      }
    } else {
      // never happens now, hopefully
      args = NULL;
    }
}

bool Union::eq(Union* x, map<Value*, Value*>& done)
{
  //    if (THIS_TAG!=U_TAG(x)) return false;
    for(ukint i=0;i<THIS_ARITY;++i)
    {
	if (!funtable_eq_aux(args[i],x->args[i],done)) { return false; }
    }
    return true;
}

kint Union::cmp(Union* x)
{
  //    if (THIS_TAG<U_TAG(x)) return -1;
  //    if (THIS_TAG>U_TAG(x)) return 1;

    for(ukint i=0;i<THIS_ARITY;++i)
    {
	kint cmp = funtable_compare(NULL,args[i],x->args[i]);
	if (cmp!=0) return cmp;
    }
    return 0;
}

Union* Union::copy(VMState* vm, map<Value*,Value*>& done)
{
    Union* u = new(vm->getUPool()) Union(NULL,THIS_ARITY,false);
    for(ukint i=0;i<THIS_ARITY;++i) {
	Value* copied;
	Value* x = args[i];
	// if x is already copied, return the copied value
	if (done.find(x)!=done.end()) {
	    copied = done[x];
	}
	else {
	    copied = funtable_copy_aux(vm, x, done);
	}
	u->args[i]=copied;
    }
    return u;
}

ukint Union::memusage()
{
    ukint size = sizeof(Union);
    for(ukint x=0;x<THIS_ARITY;++x) {
	size+=funtable_memusage(args[x]);
    }
    return size;
}

Exception::Exception(VMState* vm)
{
    Value* codeval = vm->doPop();
    Value* errval = vm->doPop();
    code = codeval->getInt();
    err = errval->getString();

    throwfile = vm->m_sourcefile;
    throwline = vm->m_lineno;

    CallStackEntry** c = vm->m_btptr-1;
    for(;(c>vm->m_backtrace);--c) {
	CallStackEntry nc(**c);
	backtrace.push_back(nc);
    }
}

Exception::Exception(VMState* vm, const char* t, kint ar):
    tag(t), arity(ar)
{
    err = NEWSTRING(strtowc(t));
    code = 0;
    if (ar > 0) {
	args = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*ar);
	for(int i=0;i<ar;++i) {
	    //	    --stacksize;
	    /// Create a *new* reference, or things break in confusing ways.
	    args[i]=vm->doPop()->clone();
	}
    } else {
	args = NULL;
    }

    throwfile = vm->m_sourcefile;
    throwline = vm->m_lineno;

    CallStackEntry** c = vm->m_btptr-1;
    for(;(c>vm->m_backtrace);--c) {
	CallStackEntry nc(**c);
	backtrace.push_back(nc);
    }
}

void Exception::show()
{
    cout << wctostr(err->getVal()) << endl;
}

bool Exception::eq(Exception* x)
{
    /*   cout << "Comparing..." << endl;
    show();
    x->show();

    cout << ((err->eq(x->err)) && (code==x->code)) << endl;*/
    return (err->eq(x->err)) && (code==x->code);
}

kint Exception::cmp(Exception* x)
{
    return (code-x->code);
}

void Exception::dumpBacktrace()
{
    vector<CallStackEntry>::iterator it = backtrace.begin();
    cout << wctostr(err->getVal());
    if (throwline!=0) {
	cout << " thrown at " << wctostr(throwfile) << ":" << throwline << endl;
    }
    else {
	cout << " thrown" << endl;
    }
    for(int x=0;it!=backtrace.end() && x<=20;++it,++x) {
	if ((*it).line==-1) {
	    cout << "lambda in " << wctostr((*it).fn_name);
	}
	else {
	    cout << wctostr((*it).fn_name);
	}
	if ((it+1)==backtrace.end()) {
	    cout << " [entry point]" << endl;
	}
	else {
	    cout << " called from " << wctostr((*(it+1)).fn_name) << " at " 
		 << wctostr((*it).call_file) << ":" 
		 << (*it).call_line << endl;
	}
	if (x==20) {
	    kint num = backtrace.size()-20;
	    cout << "[... " << num << " more elided...]" << endl;
	}
    }
}

/* Value pools */

ValuePool::ValuePool() {
  refill();
}
void ValuePool::refill() {
  m_pool = (Value*)KMALLOC(sizeof(Value)*MEM_POOLSIZE);
  m_usage = 0;
}
void* ValuePool::next() {
  if (m_usage >= MEM_POOLSIZE) {
    refill();
  }
  void* nextval = (void*)(m_pool+m_usage);
  m_usage++;
  return nextval;
}

UnionPool::UnionPool() {
  refill();
}
void UnionPool::refill() {
  m_pool = (Union*)KMALLOC(sizeof(Union)*MEM_POOLSIZE);
  m_usage = 0;
}
void* UnionPool::next() {
  if (m_usage >= MEM_POOLSIZE) {
    refill();
  }
  void* nextval = (void*)(m_pool+m_usage);
  m_usage++;
  return nextval;
}

StringPool::StringPool() {
  refill();
}
void StringPool::refill() {
  m_pool = (String*)KMALLOC(sizeof(String)*MEM_POOLSIZE);
  m_usage = 0;
}
void* StringPool::next() {
  if (m_usage >= MEM_POOLSIZE) {
    refill();
  }
  void* nextval = (void*)(m_pool+m_usage);
  m_usage++;
  return nextval;
}

RealPool::RealPool() {
  refill();
}
void RealPool::refill() {
  m_pool = (Real*)KMALLOC(sizeof(Real)*MEM_POOLSIZE);
  m_usage = 0;
}
void* RealPool::next() {
  if (m_usage >= MEM_POOLSIZE) {
    refill();
  }
  void* nextval = (void*)(m_pool+m_usage);
  m_usage++;
  return nextval;
}

ArrayPool::ArrayPool() {
  refill();
}
void ArrayPool::refill() {
  m_pool = (Array*)KMALLOC(sizeof(Array)*MEM_POOLSIZE);
  m_usage = 0;
}
void* ArrayPool::next() {
  if (m_usage >= MEM_POOLSIZE) {
    refill();
  }
  void* nextval = (void*)(m_pool+m_usage);
  m_usage++;
  return nextval;
}
