/*
 *  Copyright (C) 2003, 2004 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: MozillaPrivate.cpp 3345 2008-01-05 10:59:45Z kous $
 */

/*
 * Based on MozillaPrivate.cpp in Epiphany-1.5.1.
 *  Copyright (C) 2003, 2004 Marco Pesenti Gritti
 */

#include "mozilla-config.h"

#include "config.h"
#include "kazehakase.h"

#include "MozillaPrivate.h"

#define MOZILLA_INTERNAL_API
#include <nsIServiceManager.h>
#include <nsIComponentManager.h>
#include <nsIPrintSettingsService.h>
#include <nsIPrintOptions.h>
#include <nsWeakPtr.h>
#include <nsPromiseFlatString.h>
#include <nsEscape.h>
#include <nsIDocument.h>
#ifdef MOZ_NSICANVASRENDERINGCONTEXTINTERNAL_HAVE_GETINPUTSTREAM_
#include <nsICanvasRenderingContextInternal.h>
#include <nsIDOMCanvasRenderingContext2D.h>
#endif
#undef MOZILLA_INTERNAL_API
#include <nsISimpleEnumerator.h>
#include <nsISupportsPrimitives.h>
#include <nsIContent.h>
#include <nsIDOMHTMLDocument.h>
#include <nsIDOMHTMLElement.h>
#include <nsIDOMHTMLImageElement.h>
#include <nsIWidget.h>
#include <nsPIDOMWindow.h>
#include <nsIChromeEventHandler.h>
#include <nsIDOMWindowInternal.h>
#include <nsIDOMDocument.h>
#include <nsIDeviceContext.h>
#include <nsIMarkupDocumentViewer.h>
#include <nsIContentViewer.h>


/* IMPORTANT. Put only code that use internal mozilla strings (nsAutoString for
 * example) in this file. Note that you cannot use embed strings here,
 * the header inclusions will conflict.
 */

GList *
MozillaPrivate::GetPrinterList ()
{
	GList *printers = NULL;
	nsresult rv = NS_OK;

	nsCOMPtr<nsIPrintSettingsService> pss =
		do_GetService("@mozilla.org/gfx/printsettings-service;1", &rv);
	NS_ENSURE_SUCCESS(rv, nsnull);

	nsCOMPtr<nsIPrintOptions> po = do_QueryInterface(pss, &rv);
	NS_ENSURE_SUCCESS(rv, nsnull);

	nsCOMPtr<nsISimpleEnumerator> avPrinters;
	rv = po->AvailablePrinters(getter_AddRefs(avPrinters));
	NS_ENSURE_SUCCESS(rv, nsnull);

	PRBool more = PR_FALSE;

	for (avPrinters->HasMoreElements(&more);
	     more == PR_TRUE;
	     avPrinters->HasMoreElements(&more))
	{
		nsCOMPtr<nsISupports> i;
		rv = avPrinters->GetNext(getter_AddRefs(i));
		NS_ENSURE_SUCCESS(rv, nsnull);

		nsCOMPtr<nsISupportsString> printer = do_QueryInterface(i, &rv);
		NS_ENSURE_SUCCESS(rv, nsnull);

		nsAutoString data;
		rv = printer->GetData(data);
		NS_ENSURE_SUCCESS(rv, nsnull);

		printers = g_list_prepend (printers, g_strdup (NS_ConvertUCS2toUTF8 (data).get()));
	}

	return g_list_reverse (printers);
}

gboolean
MozillaPrivate::CreatePrintSettings (nsIPrintSettings **options)
{
	nsCOMPtr<nsIPrintOptions> printService;
	printService = do_GetService("@mozilla.org/gfx/printsettings-service;1");
	
	if (!printService) return FALSE;

	nsresult rv = printService->CreatePrintSettings(options);
	if (NS_FAILED(rv)) return FALSE;

	return TRUE;
}

GdkWindow *
MozillaPrivate::GetGdkWindow (nsIBaseWindow *window)
{
  	nsCOMPtr<nsIWidget> mainWidget;
	window->GetMainWidget(getter_AddRefs(mainWidget));
	GdkWindow *gdk_window;
	gdk_window = NS_STATIC_CAST(GdkWindow *,
			  mainWidget->GetNativeData(NS_NATIVE_WINDOW));

	return gdk_window;
}

gboolean
MozillaPrivate::GetEventReceiver (nsIDOMWindow *domWindow, nsIDOMEventReceiver **receiver)
{
  	// get the private DOM window
	nsCOMPtr<nsPIDOMWindow> domWindowPrivate = do_QueryInterface(domWindow);
  	// and the root window for that DOM window
  	nsCOMPtr<nsIDOMWindowInternal> rootWindow;
#ifndef MOZ_NSPIDOMWINDOW_RETURNNSPIDOMWINDOW_ 
	domWindowPrivate->GetPrivateRoot(getter_AddRefs(rootWindow));
#else
  	rootWindow = domWindowPrivate->GetPrivateRoot();
#endif
  
  	nsCOMPtr<nsPIDOMWindow> piWin;
	piWin = do_QueryInterface(rootWindow);

	nsCOMPtr<nsIChromeEventHandler> chromeHandler;
#ifndef MOZ_NSPIDOMWINDOW_RETURNNSPIDOMWINDOW_ 
	piWin->GetChromeEventHandler(getter_AddRefs(chromeHandler));
#else
	chromeHandler = piWin->GetChromeEventHandler();
#endif
	nsCOMPtr<nsIDOMEventReceiver> er;
	er = do_QueryInterface(chromeHandler);

	*receiver = er;
	NS_ADDREF(*receiver);
	
  	return TRUE;
}

#ifdef MOZ_NSICANVASRENDERINGCONTEXTINTERNAL_HAVE_GETINPUTSTREAM_
#define KZ_CANVAS_WIDTH 1024
#define KZ_CANVAS_HEIGHT 800
gboolean
MozillaPrivate::CreateThumbnail(nsIDOMWindow *domWindow, const gchar *uri,
                                guint last_modified, EggPixbufThumbSize size)
{
	nsresult rv;
	nsCOMPtr<nsIDOMDocument> domDoc;
	domWindow->GetDocument(getter_AddRefs(domDoc));

	nsCOMPtr<nsICanvasRenderingContextInternal> context;

	nsCString ctxString("@mozilla.org/content/canvas-rendering-context;1?id=2d");
	context = do_CreateInstance(nsPromiseFlatCString(ctxString).get(), &rv);

	if (NS_FAILED(rv) || !context) return FALSE;

	context->SetDimensions(size, KZ_THUMB_HEIGHT);
	context->SetCanvasElement(nsnull);

	nsCOMPtr<nsIDOMCanvasRenderingContext2D> domCanvas = do_QueryInterface(context, &rv);
	if (NS_FAILED(rv) || !domCanvas) return FALSE;

	float sx, sy;
	sx = (float) size / KZ_CANVAS_WIDTH;
	sy = (float) KZ_THUMB_HEIGHT / KZ_CANVAS_HEIGHT;
	domCanvas->Scale(sx, sy);

	rv = domCanvas->DrawWindow(domWindow, 0, 0, KZ_CANVAS_WIDTH, KZ_CANVAS_HEIGHT, NS_LITERAL_STRING("rgb(0,0,0)")); 
	if (NS_FAILED(rv)) return FALSE;

	nsCOMPtr<nsIInputStream> imgStream;
	rv = context->GetInputStream(NS_LITERAL_CSTRING("image/png"), EmptyString(), getter_AddRefs(imgStream));
	if (NS_FAILED(rv) || !imgStream) return FALSE;

	// this code wa picked from content/html/content/src/nsHTMLCanvasElement.cpp in firefox-2.0b.
	PRUint32 bufSize;
	rv = imgStream->Available(&bufSize);
	if (NS_FAILED(rv)) return FALSE;

	bufSize += 16;
	PRUint32 imgSize = 0;
	char* imgData = (char*)g_malloc((gulong)bufSize);
	if (!imgData) return FALSE;

	PRUint32 numReadThisTime = 0;
	while ((rv = imgStream->Read(&imgData[imgSize], bufSize - imgSize,
					&numReadThisTime)) == NS_OK && numReadThisTime > 0)
	{
		imgSize += numReadThisTime;
		if (imgSize == bufSize)
		{
			// need a bigger buffer, just double
			bufSize *= 2;
			char* newImgData = (char*)g_realloc(imgData, (gulong)bufSize);
			if (!newImgData)
			{
				g_free(imgData);
				return FALSE;
			}
			imgData = newImgData;
		}
	}

	GdkPixbufLoader *loader = gdk_pixbuf_loader_new_with_mime_type("image/png", NULL);
	if (!loader)
		return FALSE;
	gdk_pixbuf_loader_write(loader, (const guchar *)imgData, (gsize)imgSize, NULL);
	GdkPixbuf *thumbnail = gdk_pixbuf_loader_get_pixbuf(loader);
	gdk_pixbuf_loader_close(loader, NULL);

	if (!thumbnail)
		return FALSE;

        KZ_CREATE_THUMBNAIL(thumbnail, uri, last_modified, size);

	g_object_unref(thumbnail);

	g_free(imgData);
	
	return TRUE;
}
#endif
