/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2002-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-window.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n.h>

#include "kz-marshalers.h"
#include "kz-history-action.h"
#include "kz-zoom-action.h"
#include "gtk-utils.h"
#include "kz-actions.h"
#include "kz-actions-popup.h"
#include "kz-actions-tab.h"
#include "kz-bookmark-bar.h"
#include "kz-web.h"
#include "kz-tab-label.h"
#include "kz-icons.h"
#include "kazehakase.h"
#include "utils.h"
#include "kz-sidebar.h"
#include "kz-feed-info.h"
#include "kz-download-box.h"
#include "kz-bookmark-menu.h"
#include "kz-bookmark-tab-menu.h"
#include "kz-bookmark-file.h"
#include "kz-paned.h"
#include "kz-entry.h"
#include "kz-notebook.h"
#include "kz-proxy-menu.h"
#include "kz-popup-tablist.h"
#include "kz-ext.h"
#include "kz-statusbar.h"
#include "kz-session.h"
#include "kz-web.h"

#define MAX_CLOSED_TABS 10

enum {
	APPEND_TAB_SIGNAL,
	REMOVE_TAB_SIGNAL,
	REORDER_TAB_SIGNAL,
	LAST_SIGNAL
};

typedef struct _KzWindowPrivate	KzWindowPrivate;
struct _KzWindowPrivate
{
	guint       merge_id;

	KzEmbedEventMouse *event;

	/* for popup & gesture */
	KzGesture  *gesture;
	gint start_x, start_y;
	gboolean is_gesture;

	/* for scroll */
	gint is_button3_pressed;
	gint is_button3_scrolled;
	
	/* sidebar */
	gboolean sidebar_was_shown;
};
#define KZ_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_WINDOW, KzWindowPrivate))

static struct {
	const gchar *pos;
	const gchar *tab_act;
	const gchar *sidebar_act;
} positems[] = {
	{"top",    "TabPosTop",    "SidebarPosTop"},
	{"bottom", "TabPosBottom", "SidebarPosBottom"},
	{"left",   "TabPosLeft",   "SidebarPosLeft"},
	{"right",  "TabPosRight",  "SidebarPosRight"},
};

static struct {
	const gint id;
	const gchar *name;
} modifier_map[] = {
	{KZ_ALT_KEY, "alt"},
	{KZ_CTRL_KEY, "ctrl"},
	{KZ_SHIFT_KEY,"shift"},
};

enum {
	TARGET_NETSCAPE_URL,
	TARGET_TEXT_URI_LIST
};

static const GtkTargetEntry drag_targets [] = 
{
        { "_NETSCAPE_URL",        0, TARGET_NETSCAPE_URL},
	{ "text/uri-list",        0, TARGET_TEXT_URI_LIST}
};

static const guint n_drag_targets = G_N_ELEMENTS (drag_targets);

static gboolean key_theme_is_emacs = FALSE;

static gboolean delete_event   (GtkWidget     *widget,
				GdkEventAny   *event);
static gboolean key_press_event(GtkWidget     *widget,
				GdkEventKey   *event);
static void     drag_data_received
			       (GtkWidget *widget,
			        GdkDragContext *context,
				gint x, gint y,
				GtkSelectionData *data,
				guint info,
				guint time);
static gboolean button_release_event
			       (GtkWidget      *widget,
			        GdkEventButton *event);
static gboolean scroll_event   (GtkWidget      *widget,
				GdkEventScroll *event);
/*
 * mozilla doesn't accept these signals, so we connect these funtions to
 * KzWindow instead of KzWeb.
 */
static gboolean motion_notify_event
			       (GtkWidget      *widget,
				GdkEventMotion *event);
static gboolean focus_out_event(GtkWidget       *widget,
			        GdkEventFocus   *event);
static void     destroy        (GtkObject     *object);

static void     kz_window_append_tab     (KzWindow      *kz,
					  GtkWidget     *widget,
					  GtkWidget     *parent);
static void     kz_window_remove_tab     (KzWindow      *kz,
					  GtkWidget     *widget);

static KzWeb   *kz_window_create_web     (KzWindow      *kz);

static void     kz_window_set_cur_web_callbacks  (KzWindow *kz,
						  KzWeb    *web);
static void     kz_window_unset_cur_web_callbacks(KzWindow *kz,
						  KzWeb    *web);
static void     kz_window_set_web_callbacks      (KzWindow *kz,
						  KzWeb    *web);
static void     kz_window_unset_web_callbacks    (KzWindow *kz,
						  KzWeb    *web);

/* callbacks */
static void     cb_profile_global_changed     (KzProfile       *profile,
					       const gchar     *section,
					       const gchar     *key,
					       const gchar     *old_value,
					       KzWindow        *kz);
static void     cb_profile_proxy_changed      (KzProfile       *profile,
					       const gchar     *section,
					       const gchar     *key,
					       const gchar     *old_value,
					       KzWindow        *kz);

static void     cb_profile_gesture_changed    (KzProfile       *profile,
					       const gchar     *section,
					       const gchar     *key,
					       const gchar     *old_value,
					       KzWindow        *kz);
static void     cb_bookmark_bars_insert_child (KzBookmark      *bookmark,
					       KzBookmark      *child,
					       KzBookmark      *sibling,
					       KzWindow        *kz);
static void     cb_bookmark_bars_remove_child (KzBookmark      *bookmark,
					       KzBookmark      *child,
					       KzWindow        *kz);
static void     cb_bookmark_bars_reordered    (KzBookmark      *bookmark,
					       KzWindow        *kz);
static void     cb_smartbookmark_insert_child (KzBookmark      *bookmark,
					       KzBookmark      *child,
					       KzBookmark      *sibling,
					       KzWindow        *kz);
static void     cb_smartbookmark_remove_child (KzBookmark      *bookmark,
					       KzBookmark      *child,
					       KzWindow        *kz);
static void     cb_smartbookmark_reordered    (KzBookmark      *bookmark,
					       KzWindow        *kz);
static void     cb_menu_merge_add_widget      (GtkUIManager    *merge,
					       GtkWidget       *widget,
					       GtkBox          *box);
static void     cb_clipboard_get_text         (GtkClipboard *clipboard,
					       const gchar *text,
					       gpointer data);
static void     cb_notebook_reorder_page      (GtkNotebook     *notebook,
					       GtkWidget       *child,
					       guint            page_num,
					       KzWindow        *kz);
static void     cb_notebook_switch_page       (GtkNotebook     *notebook,
					       GtkNotebookPage *page,
					       guint            page_num,
					       KzWindow        *kz);
static void     cb_notebook_switch_page_after (GtkNotebook     *notebook,
					       GtkNotebookPage *page,
					       guint            page_num,
					       KzWindow        *kz);
static void     cb_gesture_stack_motion       (KzGesture       *gesture,
					       KzGestureMotion  motion,
					       KzWindow        *kz);
static void     cb_sidebar_map                (GtkWidget       *widget,
					       GtkToggleAction *action);
static void     cb_sidebar_unmap              (GtkWidget       *widget,
					       GtkToggleAction *action);
 
/* callbacks for web */
static void     cb_web_title_changed         (KzWeb     *web,
					      const gchar *title,
                                              KzWindow    *kz);
static void     cb_web_location_changed      (KzWeb     *web,
					      const gchar *location,
                                              KzWindow    *kz);
static void     cb_web_link_message          (KzWeb     *web,
						const gchar *message,
						KzWindow    *kz);
static void     cb_web_load_started          (KzWeb     *web,
						KzWindow    *kz);
static void     cb_web_load_finished         (KzWeb     *web,
						KzWindow    *kz);
static void     cb_web_new_window            (KzWeb     *web,
						KzWeb    **new_web,
						KzWindow    *kz);
static void     cb_web_close_tab             (GtkObject   *obj,
						KzWindow    *kz);
static gboolean     cb_web_dom_mouse_click   (KzWeb     *web,
						KzEmbedEventMouse *event,
						KzWindow    *kz);
static gboolean     cb_web_dom_mouse_down    (KzWeb     *web,
						KzEmbedEventMouse *event,
						KzWindow    *kz);
static gboolean     cb_web_dom_mouse_up      (KzWeb *web,
						KzEmbedEventMouse *event,
						KzWindow    *kz);

static gboolean     cb_web_dom_mouse_over    (KzWeb *web,
						KzEmbedEventMouse *event,
						KzWindow    *kz);


static void     gtk_key_theme_changed_cb       (GtkSettings  *settings,
                                                GParamSpec   *pspec,
                                                gpointer dummy);

void kz_window_sync_ui_level (KzWindow *kz);
void kz_window_sync_proxy (KzWindow *kz);


static gint kz_window_signals[LAST_SIGNAL] = {0};


G_DEFINE_TYPE(KzWindow, kz_window, GTK_TYPE_WINDOW)


static void
kz_window_class_init (KzWindowClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *gtk_object_class;
	GtkWidgetClass *widget_class;

	gobject_class = G_OBJECT_CLASS(klass);
	gtk_object_class  = GTK_OBJECT_CLASS(klass);
        widget_class  = GTK_WIDGET_CLASS(klass);

	kz_window_signals[APPEND_TAB_SIGNAL]
		= g_signal_new("append-tab",
			       G_TYPE_FROM_CLASS (klass),
			       G_SIGNAL_RUN_FIRST,
			       G_STRUCT_OFFSET (KzWindowClass, append_tab),
			       NULL, NULL,
			       _kz_marshal_VOID__OBJECT_OBJECT,
			       G_TYPE_NONE, 2,
			       GTK_TYPE_WIDGET, GTK_TYPE_WIDGET);

	kz_window_signals[REMOVE_TAB_SIGNAL]
		= g_signal_new ("remove-tab",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (KzWindowClass, remove_tab),
				NULL, NULL,
				g_cclosure_marshal_VOID__OBJECT,
				G_TYPE_NONE, 1,
				GTK_TYPE_WIDGET);

	kz_window_signals[REORDER_TAB_SIGNAL]
		= g_signal_new ("reorder-tab",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (KzWindowClass, reorder_tab),
				NULL, NULL,
				_kz_marshal_VOID__OBJECT_INT,
				G_TYPE_NONE, 2,
				GTK_TYPE_WIDGET, G_TYPE_INT);

	/* GtkObject signals */
	gtk_object_class->destroy = destroy;

	/* GtkWidget signals */
	widget_class->delete_event         = delete_event;
	widget_class->key_press_event      = key_press_event;
	widget_class->drag_data_received   = drag_data_received;
	widget_class->button_release_event = button_release_event;
	widget_class->scroll_event         = scroll_event;
	widget_class->motion_notify_event  = motion_notify_event;
	widget_class->focus_out_event      = focus_out_event;

	/* KzWindow signals */
	klass->append_tab = kz_window_append_tab;
	klass->remove_tab = kz_window_remove_tab;
	
	g_type_class_add_private(gobject_class, sizeof(KzWindowPrivate));
}

void
kz_window_connect_action (KzWindow *kz, GtkAction *action)
{
	GtkAccelGroup *accel_group;

	accel_group = gtk_ui_manager_get_accel_group(kz->menu_merge);

	gtk_action_set_accel_group(action, accel_group);
	gtk_action_connect_accelerator(action);
}

void
kz_window_disconnect_action (KzWindow *kz, GtkAction *action)
{
	gtk_action_disconnect_accelerator(action);
	gtk_action_set_accel_group(action, NULL);
}

static void
kz_window_setup_action_group (GtkActionGroup *action_group,
			      GtkAccelGroup *accel_group)
{
	GList *node, *action_list;

	action_list = gtk_action_group_list_actions(action_group);

	for (node = action_list; node; node = g_list_next(node))
	{
		GtkAction *action = GTK_ACTION(node->data);
		gtk_action_set_accel_group(action, accel_group);
		gtk_action_connect_accelerator(action);
	}
	g_list_free(action_list);
}

static void
kz_window_setup_action_groups (KzWindow *kz)
{
	GtkAccelGroup *accel_group;

	accel_group = gtk_ui_manager_get_accel_group(kz->menu_merge);

	kz_window_setup_action_group(kz->actions, accel_group);
	kz_window_setup_action_group(kz->popup_actions, accel_group);
	kz_window_setup_action_group(kz->tabpop_actions, accel_group);
}

static void
append_bookmark_bar_widget (KzBookmark *bookmark, KzWindow *kz)
{
	GtkWidget *bar = GTK_WIDGET(kz_bookmark_bar_new(kz, bookmark));
	kz->bookmark_bars = g_list_append(kz->bookmark_bars, bar);
}

static void
kz_window_init (KzWindow *kz)
{
	GtkWidget *menu_box;
	GtkSettings *setting;
	GtkSettings *settings;
	GList *node;
	KzWindowPrivate *priv = KZ_WINDOW_GET_PRIVATE(kz);

	/* init member */
	kz->top_vbox           = gtk_vbox_new(FALSE, 0);

	kz->bookmark_bars_area = gtk_vbox_new(FALSE, 0);
	kz_bookmark_folder_foreach_child(KZ_BOOKMARK_FOLDER(KZ_GET_BAR_BOOKMARK),
					 (GFunc)append_bookmark_bar_widget, kz);

	kz->statusbar          = kz_statusbar_new(kz);
	kz->notebook           = kz_notebook_new(kz);

	kz->feed_info          = g_object_ref(kz_feed_info_new(kz));

	kz->menu_merge         = gtk_ui_manager_new();

	kz->actions            = kz_actions_create_group(kz);
	kz->popup_actions      = kz_actions_popup_create_group(kz);
	kz->tabpop_actions     = kz_actions_tab_popup_create_group(kz);
	
	kz->popup              = kz_popup_preview_get_instance();
	kz->popup_tablist      = NULL;

	priv->merge_id     = 0;

	priv->event        = NULL;

	priv->gesture      = kz_gesture_new();
	priv->start_x      = 0;
	priv->start_y      = 0;
	priv->is_gesture   = FALSE;

	priv->is_button3_pressed  = FALSE;
	priv->is_button3_scrolled = FALSE;
	
	priv->sidebar_was_shown = FALSE;

	kz->tabs            = KZ_BOOKMARK_FOLDER(kz_bookmark_folder_new("Window"));
	kz->closed_tabs     = KZ_BOOKMARK_FOLDER(kz_bookmark_folder_new("Closed tabs"));
	kz->history_search  = kz_bookmark_folder_new(NULL);
	
	kz_ext_setup(kz);

	kz_window_setup_action_groups(kz);

	/* gesture */
	{
		kz_window_update_gesture_items(kz);
		g_signal_connect(priv->gesture,
				 "stack_motion",
				 G_CALLBACK(cb_gesture_stack_motion), kz);
		g_signal_connect(KZ_GET_GLOBAL_PROFILE,
				 "changed::Gesture",
				 G_CALLBACK(cb_profile_gesture_changed), kz);
	}

	/* top level vbox */
	{
		gtk_container_add(GTK_CONTAINER(kz),
				  kz->top_vbox);
		gtk_widget_show(kz->top_vbox);
	}

	/* menu & toolbar */
	{
		GtkAccelGroup *accel_group;

		kz->menu_box = menu_box = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(kz->top_vbox), menu_box,
				   FALSE, FALSE, 0);   
		gtk_widget_show(menu_box);

		gtk_ui_manager_insert_action_group(kz->menu_merge,
						   kz->actions, 0);
		gtk_ui_manager_insert_action_group(kz->menu_merge,
						   kz->popup_actions, 0);
		gtk_ui_manager_insert_action_group(kz->menu_merge,
						   kz->tabpop_actions, 0);
		
		g_signal_connect(kz->menu_merge, "add_widget",
				 G_CALLBACK(cb_menu_merge_add_widget),
				 menu_box);
		accel_group = gtk_ui_manager_get_accel_group(kz->menu_merge);
		gtk_window_add_accel_group(GTK_WINDOW(kz), accel_group);

		kz_window_sync_ui_level(kz);
		gtk_ui_manager_ensure_update(kz->menu_merge);
		g_signal_connect(KZ_GET_GLOBAL_PROFILE,
				 "changed::Global",
				 G_CALLBACK(cb_profile_global_changed), kz);
		g_signal_connect(KZ_GET_PROXY,
				 "changed",
				 G_CALLBACK(cb_profile_proxy_changed), kz);


		/* FIXME */
		setting = gtk_settings_get_default();

		if (setting)
		{
			gtk_settings_set_long_property(setting,
							"gtk-toolbar-style",
							GTK_TOOLBAR_ICONS,
							"");
		}

	}

	/* smart bookmark */
	g_signal_connect_after(KZ_GET_SMART_BOOKMARK,
			       "insert-child",
			       G_CALLBACK(cb_smartbookmark_insert_child), kz);
	g_signal_connect_after(KZ_GET_SMART_BOOKMARK,
			       "remove-child",
			       G_CALLBACK(cb_smartbookmark_remove_child), kz);
	g_signal_connect_after(KZ_GET_SMART_BOOKMARK,
			       "children-reordered",
			       G_CALLBACK(cb_smartbookmark_reordered), kz);
	kz_actions_update_smartbookmarks(kz, KZ_GET_SMART_BOOKMARK);

	/* bookmark bar */
	gtk_box_pack_start(GTK_BOX(kz->top_vbox), 
			   kz->bookmark_bars_area, FALSE, FALSE, 0);
	/* gtk_widget_show(kz->bookmark_bars_area); */

	for (node = kz->bookmark_bars;
	     node;
	     node = g_list_next(node))
	{
		gtk_box_pack_start(GTK_BOX(kz->bookmark_bars_area), 
				   node->data, FALSE, FALSE, 0);    
		gtk_widget_show(node->data);
	}

	g_signal_connect_after(KZ_GET_BAR_BOOKMARK,
			       "insert-child",
			       G_CALLBACK(cb_bookmark_bars_insert_child), kz);
	g_signal_connect_after(KZ_GET_BAR_BOOKMARK,
			       "remove-child",
			       G_CALLBACK(cb_bookmark_bars_remove_child), kz);
	g_signal_connect_after(KZ_GET_BAR_BOOKMARK,
			       "children-reordered",
			       G_CALLBACK(cb_bookmark_bars_reordered), kz);

	/* paned widget to separate sidebar and main contents */
	{
		kz->pane = kz_paned_new();
		gtk_container_add(GTK_CONTAINER(kz->top_vbox), kz->pane);
		gtk_widget_show(kz->pane);
	}

	/* main notebook widget */
	{
		g_signal_connect(kz->notebook, "page-reordered",
				 G_CALLBACK(cb_notebook_reorder_page), kz);
		g_signal_connect(kz->notebook, "switch-page", 
				 G_CALLBACK(cb_notebook_switch_page), kz);
		g_signal_connect_after(kz->notebook, "switch-page",
				 G_CALLBACK(cb_notebook_switch_page_after), kz);

		gtk_paned_add2(GTK_PANED(kz->pane), kz->notebook);
		gtk_widget_show(kz->notebook);

		/* sidebar */
		kz->sidebar = kz_sidebar_new(kz);
		gtk_widget_set_size_request(kz->sidebar, 150, -1);
		gtk_paned_add1(GTK_PANED(kz->pane), kz->sidebar);
		{
			GtkAction *action;
			action = gtk_action_group_get_action(kz->actions,
							     "ShowHideSidebar");
			g_signal_connect(kz->sidebar, "map",
					 G_CALLBACK(cb_sidebar_map), action);
			g_signal_connect(kz->sidebar, "unmap",
					 G_CALLBACK(cb_sidebar_unmap), action);
		}
	}

	/* status bar */
	{
		gtk_box_pack_start(GTK_BOX(kz->top_vbox), kz->statusbar,
				   FALSE, FALSE, 0);   
		gtk_widget_show(kz->statusbar);
	}

	gtk_drag_dest_set(GTK_WIDGET(kz), 
			  GTK_DEST_DEFAULT_ALL, 
                          drag_targets, n_drag_targets,
			  GDK_ACTION_MOVE | GDK_ACTION_COPY);

	/* from Galeon-1.3.18 */
	/* initialize the listener for the key theme */
	settings = gtk_settings_get_default();
	g_signal_connect(settings,
			 "notify::gtk-key-theme-name",
			 G_CALLBACK (gtk_key_theme_changed_cb),
			 NULL);
	gtk_key_theme_changed_cb(settings, 0, 0);

	kz_window_restore_state(kz);
}


static void
kz_window_append_tab (KzWindow *kz, GtkWidget *widget, GtkWidget *parent)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_actions_set_tab_sensitive(kz, KZ_WINDOW_CURRENT_WEB(kz));
}


static void
kz_window_remove_tab (KzWindow *kz, GtkWidget *widget)
{
	KzWeb *web;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (!GTK_WIDGET_REALIZED(kz)) return;

	web = KZ_WINDOW_CURRENT_WEB(kz);
	kz_actions_set_sensitive(kz, web);
	kz_actions_set_tab_sensitive(kz, web);
}


GtkWidget *
kz_window_new(const gchar *url)
{
	KzWindow *kz = g_object_new(KZ_TYPE_WINDOW,
				   "type", GTK_WINDOW_TOPLEVEL,
				   "allow-shrink", TRUE,
				   "title", _("Kazehakase"),
				   "icon", kz_icon,
				    NULL);
	
	if (url)
		kz_window_open_new_tab(kz, url);

	return GTK_WIDGET(kz);
}


void
kz_window_sync_ui_level (KzWindow *kz)
{
	GtkAction *action;
	const gchar *action_str;
	GError *err = NULL;
	gchar *ui_file;
	GtkWidget *widget;
	KzWeb *kzweb;
	KzWindowPrivate *priv;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	priv = KZ_WINDOW_GET_PRIVATE(kz);

	switch (KZ_GET_UI_LEVEL) 
	{
	case KZ_UI_LEVEL_CUSTOM:
		action_str = "UILevelCustom";
		ui_file = g_build_filename(KZ_GET_USER_DIR,
					   "kz-ui.xml",
				           NULL);
		break;
	case KZ_UI_LEVEL_EXPERT:
		action_str = "UILevelExpert";
		ui_file = g_build_filename(KZ_GET_SYSTEM_CONFIG_DIR,
					   "kz-ui-expert.xml", NULL);
		break;
	case KZ_UI_LEVEL_MEDIUM:
		action_str = "UILevelMedium";
		ui_file = g_build_filename(KZ_GET_SYSTEM_CONFIG_DIR,
					   "kz-ui-medium.xml", NULL);
		break;
	case KZ_UI_LEVEL_BEGINNER:
	default:
		action_str = "UILevelBeginner";
		ui_file = g_build_filename(KZ_GET_SYSTEM_CONFIG_DIR,
					   "kz-ui-beginner.xml", NULL);
		break;
	}

	action = gtk_action_group_get_action(kz->actions, action_str);
	if (!gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))
		gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), TRUE);

	if (priv->merge_id != 0)
	{
		/* remove feed_info explicitly */
		GtkWidget *menu_bar;
		menu_bar = gtk_ui_manager_get_widget(kz->menu_merge,
						     "/menubar");
		gtk_container_remove(GTK_CONTAINER(menu_bar), kz->feed_info);

		gtk_ui_manager_remove_ui(kz->menu_merge, priv->merge_id);
		priv->merge_id = 0;
		kz_actions_remove_smartbookmarks(kz, KZ_GET_SMART_BOOKMARK);
        	gtk_ui_manager_ensure_update(kz->menu_merge);
	}

	priv->merge_id = gtk_ui_manager_add_ui_from_file(kz->menu_merge,
						         ui_file, &err);
	if (err)
	{
		g_warning("%s", err->message);
		g_error_free(err);
	}
        gtk_ui_manager_ensure_update(kz->menu_merge);

	/* update bookmarks */
	widget = gtk_ui_manager_get_widget(kz->menu_merge,
					   "/menubar/BookmarksMenu");
	if (GTK_IS_MENU_ITEM(widget))
		widget = gtk_menu_item_get_submenu(GTK_MENU_ITEM(widget));
	if (GTK_IS_MENU_SHELL(widget))
	{
		kz_bookmark_menu_remove_menuitems(GTK_MENU_SHELL(widget), kz);
		kz_bookmark_menu_append_menuitems(GTK_MENU_SHELL(widget), kz,
						  KZ_GET_MENU_BOOKMARK);
	}

	/* update clips */
	widget = gtk_ui_manager_get_widget(kz->menu_merge,
					   "/menubar/ToolsMenu/ClipMenu");
	if (GTK_IS_MENU_ITEM(widget))
		widget = gtk_menu_item_get_submenu(GTK_MENU_ITEM(widget));
	if (GTK_IS_MENU_SHELL(widget))
	{
		kz_bookmark_menu_remove_menuitems(GTK_MENU_SHELL(widget), kz);
		kz_bookmark_menu_append_menuitems(GTK_MENU_SHELL(widget), kz,
						  KZ_GET_CLIP_BOOKMARK);
	}

	/* append recent close tab menu */
	widget = gtk_ui_manager_get_widget(kz->menu_merge,
					   "/menubar/TabMenu/RecentCloseTabMenu");

	if (GTK_IS_MENU_ITEM(widget))
	{
		GtkWidget *submenu;
		submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(widget));
		if (!submenu)
		{
			submenu = gtk_menu_new();
			gtk_menu_item_set_submenu(GTK_MENU_ITEM(widget), submenu);
		}
		if (GTK_IS_MENU_SHELL(submenu))
		{
			kz_bookmark_menu_remove_tab_menuitems(GTK_MENU_SHELL(submenu), kz);
			kz_bookmark_menu_append_tab_menuitems(GTK_MENU_SHELL(submenu), kz,
							      KZ_BOOKMARK(kz->closed_tabs));
		}
	}

	/* feed info */
	{
		GtkWidget *menu_bar;
		menu_bar = gtk_ui_manager_get_widget(kz->menu_merge,
						     "/menubar");

		gtk_container_add(GTK_CONTAINER(menu_bar), kz->feed_info);
		gtk_menu_item_set_right_justified(GTK_MENU_ITEM(kz->feed_info),
						  TRUE);
	}

	/* update proxy */
	kz_window_sync_proxy(kz);

	/* for stop/reload button updating */
	kzweb = KZ_WINDOW_CURRENT_WEB(kz);
	if (kzweb)
		kz_actions_set_sensitive(kz, kzweb);

	/* smart bookmark */
	kz_actions_update_smartbookmarks(kz, KZ_GET_SMART_BOOKMARK);

	g_free(ui_file);
}

void
kz_window_sync_proxy (KzWindow *kz)
{
	GtkWidget *menuitem, *submenu = NULL;
	GtkAction *action;
	gboolean use_proxy = FALSE;

	KZ_CONF_GET("Global", "use_proxy", use_proxy, BOOL);
	
	action = gtk_action_group_get_action(kz->actions,
					     "ToggleProxyUse");
	if (action)
		gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action),
					     use_proxy);
	KZ_WINDOW_SET_VISIBLE(kz, "StockProxyMenu", use_proxy);
	if (!use_proxy) return;

	menuitem = gtk_ui_manager_get_widget(kz->menu_merge,
					     "/menubar/EditMenu/ProxyMenu");
	if (!GTK_IS_MENU_ITEM(menuitem))
		return;
	
	submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menuitem));

	if (!submenu)
		submenu = gtk_menu_new();

	kz_proxy_menu_remove_menuitems(GTK_MENU_SHELL(submenu), kz);
	kz_proxy_menu_append_menuitems(GTK_MENU_SHELL(submenu), kz);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
}

GtkWidget *
kz_window_open_new_tab (KzWindow *kz, const gchar *url)
{
	return kz_window_open_new_tab_with_parent(kz, url, NULL);
}

/* It's temporary implementation */
static KzWeb *
kz_window_create_web (KzWindow *kz)
{
	gchar *engine;
	KzWeb *web = NULL;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	engine = KZ_CONF_GET_STR("Browser", "layout_engine");
	if (engine)
	{
		if (engine[0])
			web = KZ_WEB(kz_web_new(engine, "gecko"));
		g_free(engine);
	}

	if (!web)
		web = KZ_WEB(kz_web_new("gecko", "gecko"));
	if (!web)
		web = KZ_WEB(kz_web_new("webkit_gtk", "webkit_gtk"));
	if (!web)
		web = KZ_WEB(kz_web_new("ie", "ie"));

	return web;
}


GtkWidget *
kz_window_open_new_tab_with_parent (KzWindow *kz, const gchar *url,
				    GtkWidget *parent)
{
	KzWeb *web;
	KzTabLabel *kztab, *sibtab;
	
	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	if (url) g_return_val_if_fail(g_utf8_validate(url, strlen(url), NULL), NULL);

	web = kz_window_create_web(kz);
	g_return_val_if_fail(web, NULL);
	kztab = KZ_TAB_LABEL(kz_tab_label_new(kz, web));

	kz_notebook_open_new_tab(KZ_NOTEBOOK(kz->notebook), web, kztab);
	sibtab = kz_notebook_get_sibling_tab_label(KZ_NOTEBOOK(kz->notebook), kztab);
	/* insert tab bookmark */
	kz_bookmark_folder_insert_before(kz->tabs,
					 KZ_BOOKMARK(kztab->history),
					 sibtab ? KZ_BOOKMARK(sibtab->history) : NULL);

	kz_window_set_web_callbacks(kz, web);

	g_signal_emit(kz, kz_window_signals[APPEND_TAB_SIGNAL],
		      0, web, parent);

	kz_web_load_uri(web, url);

	return GTK_WIDGET(web);
}


void
kz_window_restore_tabs (KzWindow *kz, KzBookmark *tabs)
{
	GList *childtabs, *tabnode;
	KzWindowPrivate *priv;
	KzWeb *curweb;
	gint pos;
	g_return_if_fail(KZ_IS_WINDOW(kz));

	priv = KZ_WINDOW_GET_PRIVATE(kz);

	/* remove old tab bookmark */
	g_object_unref(kz->tabs);
	kz->tabs = g_object_ref(tabs);

	childtabs = kz_bookmark_folder_get_children(KZ_BOOKMARK_FOLDER(tabs));
	/* tabs */
	for (tabnode = childtabs; tabnode; tabnode = g_list_next(tabnode))
	{
		KzWeb *kzweb;
		KzTabLabel *kztab;
		KzBookmarkFolder *child;
		kzweb = kz_window_create_web(kz);
		kztab = KZ_TAB_LABEL(kz_tab_label_new(kz, kzweb));
		child = KZ_BOOKMARK_FOLDER(tabnode->data);

		kz_notebook_prepend_new_tab(KZ_NOTEBOOK(kz->notebook), kzweb, kztab);
		kz_window_set_web_callbacks(kz, kzweb);
		kz_tab_label_set_history(kztab, child);
	}
	g_list_free(childtabs);

	pos = kz_bookmark_folder_get_current_position(KZ_BOOKMARK_FOLDER(tabs));
	kz_notebook_set_current_page(KZ_NOTEBOOK(kz->notebook), pos);
	curweb = KZ_WINDOW_NTH_WEB(kz, pos);
	if (curweb)
	{
		kz_window_set_cur_web_callbacks(kz, curweb);
		kz_actions_set_sensitive(kz, curweb);
		kz_actions_set_tab_sensitive(kz, curweb);
	}
}


void
kz_window_close_tab (KzWindow *kz, KzWeb *web)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_WEB(web));

	kz_window_unset_cur_web_callbacks(kz, web);
	kz_window_unset_web_callbacks(kz, web);

	kz_notebook_close_tab(KZ_NOTEBOOK(kz->notebook), web);
}

static void
unset_each_web_callbacks (gpointer data, gpointer user_data)
{
	kz_window_unset_web_callbacks(KZ_WINDOW(user_data),
					KZ_WEB(data));
}

static void
disconnect_all_web_signals (KzWindow *kz)
{
        kz_notebook_foreach_web(KZ_NOTEBOOK(kz->notebook),
				  unset_each_web_callbacks, kz);
}

void
kz_window_close_all_tab(KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

        disconnect_all_web_signals(kz);
	kz_notebook_close_all_tab(KZ_NOTEBOOK(kz->notebook));
}


/* FIXME!! It's a hacky way. */
void
kz_window_move_tab (KzWindow *src_kz, KzWindow *dest_kz, GtkWidget *widget)
{
	KzNotebook *src_notebook, *dest_notebook;
	KzTabLabel *label;
	KzWeb *kzweb;
	KzTabLabel *new_kztab;

	g_return_if_fail(KZ_IS_WINDOW(src_kz));
	g_return_if_fail(KZ_IS_WINDOW(dest_kz));
	g_return_if_fail(KZ_IS_WEB(widget));

	src_notebook = KZ_NOTEBOOK(src_kz->notebook);
	dest_notebook = KZ_NOTEBOOK(dest_kz->notebook);

	kzweb = KZ_WEB(widget);

	label = kz_notebook_get_tab_label(dest_notebook, kzweb);

	/* the kzweb is already the kz's child */
	if (label) return;

	label = kz_notebook_get_tab_label(src_notebook, kzweb);
	g_return_if_fail(label);

	/* create new tab label */
	new_kztab = KZ_TAB_LABEL(kz_tab_label_new(dest_kz, kzweb));

	/* move the page to this window */
	kz_window_unset_cur_web_callbacks(src_kz, kzweb);
	kz_window_unset_web_callbacks(src_kz, kzweb);

	kz_notebook_move_tab(src_notebook, dest_notebook, widget);

	kz_bookmark_folder_prepend(dest_kz->tabs, KZ_BOOKMARK(new_kztab->history));
	kz_window_set_web_callbacks(dest_kz, kzweb);

	g_signal_emit(src_kz,
		      kz_window_signals[REMOVE_TAB_SIGNAL],
		      0, kzweb);
	g_signal_emit(dest_kz,
		      kz_window_signals[APPEND_TAB_SIGNAL],
		      0, kzweb, NULL);
}


static void
drag_data_received (GtkWidget *widget,
                    GdkDragContext *context,
                    gint x, gint y,
                    GtkSelectionData *seldata,
                    guint info,
                    guint time)
{
	switch (info)
	{
	 case TARGET_NETSCAPE_URL:
	 case TARGET_TEXT_URI_LIST:
	 {
		KzWindow *kz = KZ_WINDOW(widget);
		gchar **strings;

		if (seldata->length < 0) return;

		strings = g_strsplit((const gchar*)seldata->data, "\n", 2);
		kz_window_open_new_tab(kz, strings[0]);
		g_strfreev(strings);
		gtk_drag_finish(context, TRUE, FALSE, time);
		break;
	 }
	}
}


static gboolean
delete_event (GtkWidget *widget, GdkEventAny *event)
{
	KzWindow *kz;

	kz = KZ_WINDOW(widget);

	kz_window_store_state(kz);

	KZ_FREEZE_SESSION;
	kz_window_close_all_tab(kz);
	KZ_THAW_SESSION;

	return FALSE;
}


/* from Galeon-1.3.18 */
static void
gtk_key_theme_changed_cb (GtkSettings  *settings,
                          GParamSpec   *pspec,
                          gpointer dummy)
{
	gchar *key_theme_name;

	g_object_get(settings,
		     "gtk-key-theme-name", &key_theme_name,
		     NULL);
	if (key_theme_name && g_ascii_strcasecmp(key_theme_name, "Emacs") == 0)
	{
		key_theme_is_emacs = TRUE;
	}
	else
	{
		key_theme_is_emacs = FALSE;
	}
	g_free(key_theme_name);
}

static gboolean
key_press_event(GtkWidget *widget, GdkEventKey *event)
{
	KzWindow *kz;

	gboolean shortcircuit;
	gboolean force_chain;
	gboolean handled;
	guint modifier;

	kz = KZ_WINDOW(widget);

	/* In an attempt to get the mozweb playing nice with things like emacs keybindings
	 * we are passing important events to the focused child widget before letting the window's
	 * base handler see them. This is *completely against* stated gtk2 policy but the 
	 * 'correct' behaviour is exceptionally useless. We need to keep an eye out for 
	 * unexpected consequences of this decision. IME's should be a high concern, but 
	 * considering that the IME folks complained about the upside-down event propagation
	 * rules, we might be doing them a favour.
	 */

	shortcircuit = FALSE;
	force_chain  = FALSE;
	handled = FALSE;
	modifier = event->state & gtk_accelerator_get_default_mod_mask();
	/* guint i; */

	if (event->keyval == GDK_Escape)
	{
		/* Always pass Escape to both the widget, and the parent */
		shortcircuit = TRUE;
		force_chain = TRUE;
	}
	else if (key_theme_is_emacs && 
		 (modifier == GDK_CONTROL_MASK) && event->length > 0 &&
		 /* But don't pass Ctrl+Enter twice */
		 event->keyval != GDK_Return)
	{
		/* Pass CTRL+letter characters to the widget */
		shortcircuit = TRUE;
	}

	if (shortcircuit)
	{
		GtkWidget *widget = gtk_window_get_focus(GTK_WINDOW(kz));

		if (GTK_IS_WIDGET(widget))
		{
			handled = gtk_widget_event(widget, (GdkEvent*)event);
		}

		if (handled && !force_chain)
		{
			return handled;
		}
	}

	return GTK_WIDGET_CLASS(kz_window_parent_class)->key_press_event(widget, event);
}

static void
destroy (GtkObject *object)
{
	KzWindow *kz = KZ_WINDOW(object);
	KzWindowPrivate *priv = KZ_WINDOW_GET_PRIVATE(kz);

	if (kz->notebook) {
		disconnect_all_web_signals(kz);
		kz->notebook = NULL;
	}

	if (GTK_OBJECT_CLASS(kz_window_parent_class)->destroy)
		GTK_OBJECT_CLASS(kz_window_parent_class)->destroy(object);

	if (priv->gesture)
		g_object_unref(priv->gesture);
	priv->gesture = NULL;

	if (priv->event)
		kz_embed_event_free((KzEmbedEvent *) priv->event);
	priv->event = NULL;

	if (kz->feed_info)
		g_object_unref(kz->feed_info);
	kz->feed_info = NULL;

	if (kz->menu_merge)
		g_object_unref(kz->menu_merge);
	kz->menu_merge = NULL;

	if (kz->actions)
		g_object_unref(kz->actions);
	kz->actions = NULL;

	if (kz->popup_actions)
		g_object_unref(kz->popup_actions);
	kz->popup_actions = NULL;

	if (kz->tabpop_actions)
		g_object_unref(kz->tabpop_actions);
	kz->tabpop_actions = NULL;

	if (kz->popup)
		g_object_unref(kz->popup);
	kz->popup = NULL;

	if (kz->popup_tablist)
		gtk_widget_destroy(kz->popup_tablist);
	kz->popup_tablist = NULL;

	if (kz->tabs) {
		/* CAUTION: make KzApp sure to remove tabs before freeing it. */
		g_object_unref(kz->tabs);
	}
	kz->tabs = NULL;

	if (kz->closed_tabs)
		g_object_unref(kz->closed_tabs);
	kz->closed_tabs = NULL;

	if (kz->history_search)
		g_object_unref(kz->history_search);
	kz->history_search = NULL;

	g_signal_handlers_disconnect_by_func
		(KZ_GET_BAR_BOOKMARK,
		 G_CALLBACK(cb_bookmark_bars_insert_child), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_BAR_BOOKMARK,
		 G_CALLBACK(cb_bookmark_bars_remove_child), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_BAR_BOOKMARK,
		 G_CALLBACK(cb_bookmark_bars_reordered), kz);

	g_signal_handlers_disconnect_by_func
		(KZ_GET_SMART_BOOKMARK,
		 G_CALLBACK(cb_smartbookmark_insert_child), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_SMART_BOOKMARK,
		 G_CALLBACK(cb_smartbookmark_remove_child), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_SMART_BOOKMARK,
		 G_CALLBACK(cb_smartbookmark_reordered), kz);

	g_signal_handlers_disconnect_by_func
		(KZ_GET_GLOBAL_PROFILE,
		 G_CALLBACK(cb_profile_global_changed), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_GLOBAL_PROFILE,
		 G_CALLBACK(cb_profile_gesture_changed), kz);
	g_signal_handlers_disconnect_by_func
		(KZ_GET_PROXY,
		 G_CALLBACK(cb_profile_proxy_changed), kz);
}


void
kz_window_update_gesture_items (KzWindow *kz)
{
	GtkAction *action;
	KzGestureItems *items;
	GList *list, *node;
	KzWindowPrivate *priv;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	priv = KZ_WINDOW_GET_PRIVATE(kz);

	items = kz_gesture_items_new();

	list = kz_profile_enum_key(KZ_GET_GLOBAL_PROFILE, "Gesture", FALSE);

	for (node = list; node; node = g_list_next(node))
	{
		const gchar *action_name = node->data;
		gchar *gesture;

		action = gtk_action_group_get_action(kz->actions,
						     action_name);
		if (!action) continue;

		gesture = KZ_CONF_GET_STR("Gesture", action_name);
		if (!gesture) continue;
		if (!*gesture)
		{
			g_free(gesture);
			continue;
		}

		kz_gesture_items_set_action(items, action,
					    0, gesture);

		g_free(gesture);
	}

	kz_gesture_set_items(priv->gesture, items);
	kz_gesture_items_unref(items);

	g_list_free(list);
}


void
kz_window_store_state (KzWindow *kz)
{
	GtkAction *action;
	gint i, client_x, client_y, width, height;
	const gchar *label;
	gboolean active, maximized;
	KzWindowPrivate *priv;
	GdkWindowState state;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	priv = KZ_WINDOW_GET_PRIVATE(kz);

	kz_profile_set_auto_save(KZ_GET_GLOBAL_PROFILE, FALSE);
	/*
	 * window size
	 */

	state = gdk_window_get_state(GTK_WIDGET(kz)->window);
	maximized = state & GDK_WINDOW_STATE_MAXIMIZED;
	KZ_CONF_SET("MainWindow", "maximized", maximized, BOOL);

	if (!maximized)
	{
		gdk_window_get_geometry(GTK_WIDGET(kz)->window,
					 &client_x, &client_y,
					 &width, &height, NULL);
		KZ_CONF_SET("MainWindow", "width",  width,  INT);
		KZ_CONF_SET("MainWindow", "height", height, INT);
	}

	/*
	 * sidebar
	 */
	for (i = 0; i < G_N_ELEMENTS(positems); i++)
	{
		action = gtk_action_group_get_action(kz->actions,
						     positems[i].sidebar_act);
		active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
		if (active)
		{
			KZ_CONF_SET_STR("MainWindow", "sidebar_pos",
					positems[i].pos);
			/* width */
			/*
			 * On startup time, allocation.width and allocation.height are
			 * not set correctly(maybe both 1), so we should use 
			 * sidebar or notebook width according to sidebar position.
			 */
			if (priv->sidebar_was_shown)
			{
				if (!strcmp(positems[i].pos, "top"))
					width = kz->sidebar->allocation.height;
				else if (!strcmp(positems[i].pos, "bottom"))
					width = kz->notebook->allocation.height;
				else if (!strcmp(positems[i].pos, "left"))
					width = kz->sidebar->allocation.width;
				else if (!strcmp(positems[i].pos, "right"))
					width = kz->notebook->allocation.width;
				KZ_CONF_SET("MainWindow", "sidebar_width", width, INT);
			}
			break;
		}
	}

	/* content */
	label = kz_sidebar_get_current(KZ_SIDEBAR(kz->sidebar));
	if (label && *label)
		KZ_CONF_SET_STR("MainWindow", "sidebar", label);

	/* visible */
	action = gtk_action_group_get_action(kz->actions,
					     "ShowHideSidebar");
	active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	KZ_CONF_SET("MainWindow", "show_sidebar", active, BOOL);

	/* bookmarkbar */
	action = gtk_action_group_get_action(kz->actions,
					     "ShowHideBookmarkbars");
	active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	KZ_CONF_SET("MainWindow", "show_bookmarkbars", active, BOOL);


	/*
	 * tab position
	 */

	for (i = 0; i < G_N_ELEMENTS(positems); i++)
	{
		action = gtk_action_group_get_action(kz->actions,
						     positems[i].tab_act);
		active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
		if (active)
		{
			KZ_CONF_SET_STR("MainWindow", "tab_pos",
					positems[i].pos);
			break;
		}
	}

	kz_profile_set_auto_save(KZ_GET_GLOBAL_PROFILE, TRUE);
	kz_profile_save(KZ_GET_GLOBAL_PROFILE);
}


void
kz_window_restore_state (KzWindow *kz)
{
	GtkAction *action = NULL;
	gint width = 640, height = 450, sidebar_width = 150;
	gboolean active = FALSE, bookmarkbar = TRUE, maximized = FALSE;
	gchar *tab_pos, *label, *sidebar_pos;
	gint i;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	/*
	 * window size
	 */

	KZ_CONF_GET("MainWindow", "width",  width,  INT);
	KZ_CONF_GET("MainWindow", "height", height, INT);

	gtk_window_set_default_size(GTK_WINDOW(kz), width, height);

	KZ_CONF_GET("MainWindow", "maximized", maximized, BOOL);
	if (maximized)
		gtk_window_maximize(GTK_WINDOW(kz));

	/*
	 * sidebar
	 */
	/* content */
	label = KZ_CONF_GET_STR("MainWindow", "sidebar");
	if (label && *label)
		kz_sidebar_set_current(KZ_SIDEBAR(kz->sidebar), label);
	g_free(label);

	/* visible */
	KZ_CONF_GET("MainWindow", "show_sidebar", active, BOOL);
	action = gtk_action_group_get_action(kz->actions,
					     "ShowHideSidebar");
	gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), active);
	if (active)
	{
		sidebar_pos = KZ_CONF_GET_STR("MainWindow", "sidebar_pos");
		for (i = 0; sidebar_pos && i < G_N_ELEMENTS(positems); i++)
		{
			if (strcasecmp(sidebar_pos, positems[i].pos)) continue;

			/* width */
			KZ_CONF_GET("MainWindow", "sidebar_width", sidebar_width, INT);
			action = gtk_action_group_get_action(kz->actions,
							     positems[i].sidebar_act);
			gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action),
						     TRUE);
			/*
			 * On startup time, allocation.width and allocation.height are not set
			 * correctly(maybe both 1), so we should use sidebar or notebook width
			 * according to sidebar position.
			 */
			gtk_paned_set_position(GTK_PANED(kz->pane), sidebar_width);
		}
		g_free(sidebar_pos);
	}
	KZ_CONF_GET("MainWindow", "show_bookmarkbars", bookmarkbar, BOOL);
	action = gtk_action_group_get_action(kz->actions,
					     "ShowHideBookmarkbars");
	gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), bookmarkbar);
	
	/*
	 * tab position
	 */

	tab_pos = KZ_CONF_GET_STR("MainWindow", "tab_pos");
	for (i = 0; tab_pos && i < G_N_ELEMENTS(positems); i++)
	{
		if (strcasecmp(tab_pos, positems[i].pos)) continue;

		action = gtk_action_group_get_action(kz->actions,
						     positems[i].tab_act);
		gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action),
					     TRUE);
	}
	g_free(tab_pos);
}


static void
kz_window_set_cur_web_callbacks(KzWindow *kz, KzWeb *kzweb)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_WEB(kzweb));

	/*
	 * mouse event signal
	 */
	g_signal_connect(kzweb,
			 "kz-dom-mouse-down",
			 G_CALLBACK(cb_web_dom_mouse_down), kz);
	g_signal_connect(kzweb,
			 "kz-dom-mouse-up",
			 G_CALLBACK(cb_web_dom_mouse_up), kz);
	g_signal_connect(kzweb,
			 "kz-dom-mouse-over",
			 G_CALLBACK(cb_web_dom_mouse_over), kz);
	g_signal_connect(kzweb, "kz-dom-mouse-click",
			 G_CALLBACK(cb_web_dom_mouse_click), kz);

	/*
	 * other callbacks
	 */
	g_signal_connect(kzweb, "kz-title",
			 G_CALLBACK(cb_web_title_changed), kz);
	g_signal_connect(kzweb, "kz-location",
			 G_CALLBACK(cb_web_location_changed), kz);
	g_signal_connect(kzweb, "kz-link-message",
			 G_CALLBACK(cb_web_link_message), kz);
}


static void
kz_window_unset_cur_web_callbacks(KzWindow *kz, KzWeb *kzweb)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_WEB(kzweb));

	/*
	 * mouse event signal
	 */
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_dom_mouse_down), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_dom_mouse_up), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_dom_mouse_over), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_dom_mouse_click), kz);

	/*
	 * other callbacks
	 */
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_title_changed), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_location_changed), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_link_message), kz);
}


static void
kz_window_set_web_callbacks(KzWindow *kz, KzWeb *kzweb)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_WEB(kzweb));

	/*
	 * other callbacks
	 */
	g_signal_connect(kzweb, "kz-net-start",
			 G_CALLBACK(cb_web_load_started), kz);
	g_signal_connect(kzweb, "kz-net-stop",
			 G_CALLBACK(cb_web_load_finished), kz);
	g_signal_connect(kzweb, "kz-new-window",
			 G_CALLBACK(cb_web_new_window), kz);
	g_signal_connect(kzweb, "destroy",
			 G_CALLBACK(cb_web_close_tab), kz);
}


static void
kz_window_unset_web_callbacks(KzWindow *kz, KzWeb *kzweb)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_WEB(kzweb));

	/*
	 * other callbacks
	 */
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_load_started), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_load_finished), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_new_window), kz);
	g_signal_handlers_disconnect_by_func
		(kzweb,
		 G_CALLBACK(cb_web_close_tab), kz);
}

void
kz_window_load_url (KzWindow *kz, const gchar *url)
{
        KzWeb *web;

	g_return_if_fail(KZ_IS_WINDOW(kz));

        web = KZ_WINDOW_CURRENT_WEB(kz);
	if (kz_notebook_get_n_pages(KZ_NOTEBOOK(kz->notebook)) < 1 ||
	    !web)
	{
		kz_window_open_new_tab(kz, url);
	}
	else
	{
		kz_web_load_uri(web, url);
	}
}


const gchar *
kz_window_get_title (KzWindow *kz)
{
        KzWeb *web;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

        web = KZ_WINDOW_CURRENT_WEB(kz);

	return web ? kz_web_get_title(web) : NULL;
}


const gchar *
kz_window_get_uri (KzWindow *kz)
{
        KzWeb *web;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

        web = KZ_WINDOW_CURRENT_WEB(kz);

	return web ? kz_web_get_location(web) : NULL;
}


KzTabLabel *
kz_window_get_tab_label (KzWindow *kz, KzWeb *web)
{
	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_WEB(web), NULL);

	return kz_notebook_get_tab_label(KZ_NOTEBOOK(kz->notebook), web);
}


GNode *
kz_window_get_tree (KzWindow *kz)
{
	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	return kz_notebook_get_tree(KZ_NOTEBOOK(kz->notebook));
}

void
kz_window_set_location_entry_text (KzWindow *kz, const gchar *text)
{
	GtkAction *action;
	action = gtk_action_group_get_action(kz->actions,
					     "LocationEntry");
	if (action && KZ_IS_ENTRY_ACTION(action))
		kz_entry_action_set_text(KZ_ENTRY_ACTION(action), text ? text : "");
}

gboolean
kz_window_activate_action(KzWindow *kz, const gchar *name)
{
	GtkAction *action;
	action = gtk_action_group_get_action(kz->actions, name);

	if (action)
		gtk_action_activate(action);
	return action != NULL;
}

gboolean
kz_window_activate_popup_action(KzWindow *kz, const gchar *name)
{
	GtkAction *action;
	action = gtk_action_group_get_action(kz->popup_actions, name);

	if (action)
		gtk_action_activate(action);
	return action != NULL;
}

gboolean
kz_window_activate_tabpop_action(KzWindow *kz, const gchar *name)
{
	GtkAction *action;
	action = gtk_action_group_get_action(kz->tabpop_actions, name);

	if (action)
		gtk_action_activate(action);
	return action != NULL;
}


const KzEmbedEventMouse *
kz_window_get_mouse_event_info (KzWindow *kz)
{
	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	return KZ_WINDOW_GET_PRIVATE(kz)->event;
}



/*****************************************************************************
 *                                                                           *
 *                                Callbacks                                  *
 *                                                                           *
 *****************************************************************************/
static void
cb_profile_global_changed (KzProfile *profile,
			   const gchar *section, const gchar *key,
			   const gchar *old_value,
			   KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	switch (key[0])
	{
	case 'u':/* ui_level */
		if (strcmp(key, "ui_level") == 0)
		{
			kz_window_sync_ui_level(kz);
		}
		if (strcmp(key, "use_proxy") == 0)
		{
			kz_window_sync_proxy(kz);
		}
		break;
	case 'p':
		if (strcmp(key, "proxy_name") == 0)
		{
			kz_window_sync_proxy(kz);
		}
		break;
	default:
		break;
	}
}

static void
cb_profile_proxy_changed (KzProfile *profile,
			  const gchar *section, const gchar *key,
			  const gchar *old_value,
			  KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_window_sync_proxy(kz);
}

static void
cb_profile_gesture_changed (KzProfile *profile,
			   const gchar *section, const gchar *key,
			   const gchar *old_value,
			   KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_window_update_gesture_items(kz);
}


static void
cb_bookmark_bars_insert_child (KzBookmark *bookmark,
			       KzBookmark *child, KzBookmark *sibling,
			       KzWindow *kz)
{
	GtkWidget *bar;
	gint pos;

	g_return_if_fail(KZ_IS_BOOKMARK(child));
	g_return_if_fail(KZ_IS_BOOKMARK(sibling));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	bar = kz_bookmark_bar_new(kz, child);
	gtk_box_pack_start(GTK_BOX(kz->bookmark_bars_area), 
			   bar, FALSE, FALSE, 0);    
	gtk_widget_show(bar);

	/* reorder */
	pos = kz_bookmark_folder_get_child_index(KZ_BOOKMARK_FOLDER(bookmark), child);
	gtk_box_reorder_child(GTK_BOX(kz->bookmark_bars_area),
			      bar, pos);
}


static void
cb_bookmark_bars_remove_child (KzBookmark *bookmark, KzBookmark *child,
			       KzWindow *kz)
{
	GList *node;

	g_return_if_fail(KZ_IS_BOOKMARK(child));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	for (node = kz->bookmark_bars;
	     node;
	     node = g_list_next(node))
	{
		if (KZ_BOOKMARK(KZ_BOOKMARK_BAR(node->data)->folder) == child)
		{
			gtk_widget_destroy(node->data);
			break;
		}
	}
}


static GtkWidget *
find_bookmark_bar (KzWindow *kz, KzBookmark *folder)
{
	GList *node;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_BOOKMARK(folder), NULL);

	for (node = kz->bookmark_bars;
	     node;
	     node = g_list_next(node))
	{
		KzBookmarkBar *bar;

		if (!KZ_IS_BOOKMARK_BAR(node->data))
		{
			g_warning("KzWindow: find_bookmark_bar: "
				  "Invalid bookmark bar!: %p", node->data);
			continue;
		}

		bar = KZ_BOOKMARK_BAR(node->data);
		if (folder == bar->folder)
			return GTK_WIDGET(bar);
	}

	return NULL;
}


static void
cb_bookmark_bars_reordered (KzBookmark *bookmark, KzWindow *kz)
{
	GList *children, *node;
	gint pos = 0;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	children = kz_bookmark_folder_get_children(KZ_BOOKMARK_FOLDER(bookmark));
	for (node = children; node; node = g_list_next(node))
	{
		GtkWidget *bar;

		bar = find_bookmark_bar(kz, node->data);
		if (!bar)
		{
			g_warning("KzWindow: reorder bookmark bars: "
				  "bookmark bar for %p is not exist!",
				  node->data);
			continue; /* create? */
		}

		gtk_box_reorder_child(GTK_BOX(kz->bookmark_bars_area),
				      bar, pos);
		pos++;
	}
	g_list_free(children);
}


static void
cb_smartbookmark_insert_child (KzBookmark *bookmark,
			       KzBookmark *child, KzBookmark *sibling,
			       KzWindow *kz)
{
	g_return_if_fail(KZ_IS_BOOKMARK(child));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_actions_insert_smartbookmark(kz, bookmark, child, sibling);
}


static void
cb_smartbookmark_remove_child (KzBookmark *bookmark, KzBookmark *child,
			       KzWindow *kz)
{
	g_return_if_fail(KZ_IS_BOOKMARK(child));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_actions_remove_smartbookmark(kz, bookmark, child);	
}


static void
cb_smartbookmark_reordered (KzBookmark *bookmark, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_actions_update_smartbookmarks(kz, bookmark);	
}



static void
cb_menu_merge_add_widget (GtkUIManager *merge, GtkWidget *widget, GtkBox *box)
{
	gtk_box_pack_start(box, widget, FALSE, FALSE, 0);
}


static void
cb_clipboard_get_text(GtkClipboard *clipboard, const gchar *text, gpointer data)
{
	char **received_text = data;
	*received_text = g_strdup(text);
}


static void
cb_notebook_reorder_page (GtkNotebook *notebook, GtkWidget *child,
			  guint page_num, KzWindow *kz)
{
	g_signal_emit(kz, kz_window_signals[REORDER_TAB_SIGNAL],
		      0, child, page_num);

}

static void
cb_notebook_switch_page (GtkNotebook *notebook, GtkNotebookPage *page,
			 guint page_num, KzWindow *kz)
{
	KzWeb *kzweb = KZ_WINDOW_NTH_WEB(kz, page_num);
	KzWeb *cur = KZ_WINDOW_CURRENT_WEB(kz);
	const gchar *location;
	GtkAction *action;
	KzTabLabel *tab;

	g_return_if_fail(KZ_IS_WEB(cur));
	g_return_if_fail(KZ_IS_WEB(kzweb));

	kz_window_unset_cur_web_callbacks(kz, cur);
	kz_window_set_cur_web_callbacks(kz, kzweb);

	location = kz_web_get_location(kzweb);

	action = gtk_action_group_get_action(kz->actions, "LocationEntry");
	if (KZ_IS_ENTRY_ACTION(action))
	{
		GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
		GObject *owner = gtk_clipboard_get_owner(clipboard);
		gchar *text = NULL;
		if (GTK_IS_ENTRY(owner)) 
                {
			gtk_clipboard_request_text(clipboard,
						   cb_clipboard_get_text,
						   (gpointer)&text);
		}
		kz_window_set_location_entry_text(kz, location);

		if (text) 
                {
			gtk_clipboard_set_text(clipboard, text, -1);
			g_free(text);
		}
	}

	action = gtk_action_group_get_action(kz->actions, "Zoom");
	if (KZ_IS_ZOOM_ACTION(action))
	{
		gint ratio;
		ratio = kz_web_get_zoom_ratio(kzweb);
		kz_zoom_action_set_ratio(KZ_ZOOM_ACTION(action), ratio);
	}

	tab = kz_window_get_tab_label(kz, kzweb);
	g_return_if_fail(tab);

	kz_actions_set_sensitive(kz, kzweb);
	/* kz_actions_set_tab_sensitive(kz, kzweb); */

	/* set current_position in history */
	/* FIXME: KzSession should block automatically */
	if (!kz_session_is_frozen(KZ_SESSION(KZ_GET_CURRENT_SESSION)))
		kz_bookmark_folder_set_current_position(kz->tabs, page_num);
}

static void
cb_notebook_switch_page_after (GtkNotebook *notebook, GtkNotebookPage *page,
			       guint page_num, KzWindow *kz)
{
	kz_feed_info_change_state(KZ_FEED_INFO(kz->feed_info));
}


static void
cb_gesture_stack_motion (KzGesture *gesture, KzGestureMotion motion,
			 KzWindow *kz)
{
	const gchar *label;
	gchar buf1[256], buf2[256];

	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_gesture_create_gesture_string(gesture, buf1, G_N_ELEMENTS(buf1));
	label = kz_gesture_get_matched_label(gesture);
	if (label)
		g_snprintf(buf2, G_N_ELEMENTS(buf2),
			   _("Gesture: %s(Action: %s)"), buf1, label);
	else
		g_snprintf(buf2, G_N_ELEMENTS(buf2),
			   _("Gesture: %s"), buf1);

	kz_statusbar_set_gesture_text(KZ_STATUSBAR(kz->statusbar), buf2);
}


static void
cb_sidebar_map (GtkWidget *widget, GtkToggleAction *action)
{
	KzWindowPrivate *priv;
	g_return_if_fail(GTK_IS_TOGGLE_ACTION(action));

	priv = KZ_WINDOW_GET_PRIVATE(KZ_SIDEBAR(widget)->kz);
	priv->sidebar_was_shown = TRUE;

	gtk_toggle_action_set_active(action, TRUE);
}


static void
cb_sidebar_unmap (GtkWidget *widget, GtkToggleAction *action)
{
	g_return_if_fail(GTK_IS_TOGGLE_ACTION(action));

	gtk_toggle_action_set_active(action, FALSE);
}


/*****************************************************************************
 *                                                                           *
 *                        Callbacks for Embed                                *
 *                                                                           *
 *****************************************************************************/
static void
cb_web_title_changed (KzWeb *web, const gchar *title, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (web == KZ_WINDOW_CURRENT_WEB(kz))
		gtk_window_set_title(GTK_WINDOW(kz), title);
}


static void
cb_web_location_changed (KzWeb *web, const gchar *location, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (location && web == KZ_WINDOW_CURRENT_WEB(kz))
		kz_window_set_location_entry_text(kz, location);

	kz_actions_set_sensitive(kz, web);
}


static void
cb_web_link_message (KzWeb *web, const gchar *message, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if(kz->statusbar) 
		kz_statusbar_set_link_text(KZ_STATUSBAR(kz->statusbar), message);
}


static void
cb_web_load_started (KzWeb *web, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_actions_set_sensitive(kz, KZ_WINDOW_CURRENT_WEB(kz));

	kz_feed_info_change_state(KZ_FEED_INFO(kz->feed_info));
}


static void
cb_web_load_finished (KzWeb *web, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_actions_set_sensitive(kz, KZ_WINDOW_CURRENT_WEB(kz));

	kz_feed_info_change_state(KZ_FEED_INFO(kz->feed_info));
}


static void
cb_web_new_window (KzWeb *web, KzWeb **newWeb, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	*newWeb = KZ_WEB(kz_window_open_new_tab(kz, NULL));
	gtk_widget_show(GTK_WIDGET(*newWeb));
}


static void
cb_web_close_tab (GtkObject *obj, KzWindow *kz)
{
	KzWeb *kzweb;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	kzweb = KZ_WEB(obj);

	kz_window_unset_cur_web_callbacks(kz, kzweb);
	kz_window_unset_web_callbacks(kz, kzweb);

	kz_notebook_close_tab(KZ_NOTEBOOK(kz->notebook), kzweb);
	kz_statusbar_set_link_text(KZ_STATUSBAR(kz->statusbar), NULL);

	g_signal_emit(kz, kz_window_signals[REMOVE_TAB_SIGNAL],
		      0, kzweb);
}


static gboolean
cb_web_dom_mouse_click (KzWeb *web, KzEmbedEventMouse *event,
			  KzWindow *kz)
{
	gint button;
     	glong type;
	KzTabLabel *kztab = NULL;
	GtkWidget *widget;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	type = event->cinfo.context;
	button = event->button;

	switch (button) {
	case 0:
		if ((type & KZ_CONTEXT_LINK) && event->cinfo.link)
		{
			const gchar *location;
			gint current_page;

			current_page = kz_notebook_get_current_page(KZ_NOTEBOOK(kz->notebook));
			kztab = kz_notebook_get_nth_tab_label(KZ_NOTEBOOK(kz->notebook),
							      current_page);

			/* with Shift key or in locked tab, open new tab */
			if ((event->modifier & KZ_SHIFT_KEY) ||
			    (kztab && kz_tab_label_get_lock(kztab)))
			{
				kz_window_open_new_tab_with_parent
					(kz, event->cinfo.link, GTK_WIDGET(web));
				return TRUE;
			}
			location = kz_web_get_location(KZ_WEB(web));
			if (!strncmp(location, "history-search:", 15))
			{
				kz_web_load_uri(KZ_WEB(web), event->cinfo.link);
			}
		}
		break;
	case 1:
		if ((type & KZ_CONTEXT_LINK) && event->cinfo.link)
		{
			gboolean focus_mid_click_link = FALSE;
			KZ_CONF_GET("Tab", "focus_mid_click_link", focus_mid_click_link, BOOL);

			widget = kz_window_open_new_tab_with_parent(kz, event->cinfo.link,
							   	    GTK_WIDGET(web));

			if(((event->modifier & KZ_SHIFT_KEY) != 0) ^ focus_mid_click_link)
			{
				kz_notebook_set_current_page(KZ_NOTEBOOK(kz->notebook),
							     kz_notebook_page_num(KZ_NOTEBOOK(kz->notebook), widget));
			}
			return TRUE;
		}
		else if (!(type & KZ_CONTEXT_INPUT))
		{
			kz_window_activate_action(kz, "OpenSelection");
			return TRUE;
		}
		break;
	default:
		break;
	}

	return FALSE;
}


static gboolean
cb_web_dom_mouse_down (KzWeb *kzweb, KzEmbedEventMouse *event,
			 KzWindow *kz)
{
	gint button;
     	glong type;
	KzWindowPrivate *priv;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	priv = KZ_WINDOW_GET_PRIVATE(kz);
	type = event->cinfo.context;
	button = event->button;
	if (priv->event)
		kz_embed_event_free((KzEmbedEvent *) priv->event);
	priv->event = (KzEmbedEventMouse *) kz_embed_event_copy((KzEmbedEvent *) event);

	switch (button)
	{
	case 0:
		break;
	case 1:
	{
		gchar *modifier_name = NULL;
		gchar *tmp_name = NULL;
		tmp_name = KZ_CONF_GET_STR("Global", "autoscroll_modifier");

		if(tmp_name)
		{
			modifier_name = g_ascii_strdown(tmp_name, -1);
			g_free(tmp_name);
		}

		if(modifier_name) 
		{
			gint i;
			gint modifier_id = 2;
			for (i = 0; i < G_N_ELEMENTS(modifier_map); i++)
			{
				if(!strcmp(modifier_name, modifier_map[i].name))
				{
					modifier_id = modifier_map[i].id;
					break;
				}
			}

			if((event->modifier & modifier_id) && 
			   (!((KzEmbedEvent*)event)->link))
			{
				kz_window_activate_action(kz, "AutoScrollMode");
			}
			g_free(modifier_name);
		}
		else
		{
			if (!((KzEmbedEvent*)event)->link)
				kz_window_activate_action(kz, "AutoScrollMode");
			break;
		}
		break;
	}
	case 2:
	{
		static GdkCursor *cursor = NULL;
		gboolean use_gesture = TRUE;
		gint x, y, win_x, win_y, win_x_pos, win_y_pos;

		priv->is_button3_pressed = TRUE;
		priv->is_button3_scrolled = FALSE;

		priv->is_gesture = FALSE;
		KZ_CONF_GET("Gesture", "use_gesture", use_gesture, BOOL);
		if (!use_gesture)
		{
			gdk_pointer_grab(GTK_WIDGET(kz)->window,
					  FALSE,
					  GDK_BUTTON_RELEASE_MASK |
					  GDK_BUTTON_PRESS_MASK,
					  NULL,	cursor, gtk_get_current_event_time());
			break;
		}
		/* start gesture */

		gtk_widget_get_pointer(GTK_WIDGET(kzweb), &x, &y);

		priv->start_x = x;
		priv->start_y = y;

		gdk_window_get_root_origin(GTK_WIDGET(kzweb)->window,
					    &win_x, &win_y);
		gdk_window_get_position(GTK_WIDGET(kzweb)->window,
					 &win_x_pos, &win_y_pos);

		gtk_widget_get_pointer(GTK_WIDGET(kzweb),
					&x, &y);
		kz_gesture_start(priv->gesture, 0, x, y);

		if (!cursor) cursor = gdk_cursor_new(GDK_HAND1);
		gdk_pointer_grab(GTK_WIDGET(kz)->window,
				  FALSE,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK |
				  GDK_BUTTON_PRESS_MASK,
				  NULL, cursor, gtk_get_current_event_time());
		kz_statusbar_set_gesture_text(KZ_STATUSBAR(kz->statusbar), _("Gesture:"));
		break;
	}
	default:
		break;
	}

	kz_actions_set_selection_sensitive(kz, kzweb);

	return FALSE;
}


static gboolean
cb_web_dom_mouse_up (KzWeb *web, KzEmbedEventMouse *event,
		       KzWindow *kz)
{
	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	kz_actions_set_selection_sensitive(kz, web);

	return FALSE;
}

static gboolean
cb_web_dom_mouse_over (KzWeb *web, KzEmbedEventMouse *event,
		         KzWindow *kz)
{
	static glong previous_event_context = KZ_CONTEXT_NONE;
	glong type;
	gboolean popup_thumbnail;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	KZ_CONF_GET("Global", "popup_thumbnail", popup_thumbnail, BOOL);
	if (!popup_thumbnail) return FALSE;

	type = event->cinfo.context;
	if ((type & KZ_CONTEXT_LINK) && event->cinfo.link)
	{
		KzWeb *web = KZ_WINDOW_CURRENT_WEB(kz);
		GdkDisplay *display = gtk_widget_get_display(GTK_WIDGET(web));
		gint x, y;

		gdk_display_get_pointer(display, NULL, &x, &y, NULL);

		kz_popup_preview_start(kz->popup, event->cinfo.link, event->cinfo.img, x, y);
	}
	else if(!(type & KZ_CONTEXT_LINK) && (previous_event_context & KZ_CONTEXT_LINK))
	{
		kz_popup_preview_reset(kz->popup);
	}

	previous_event_context = type;
	kz_actions_set_selection_sensitive(kz, web);

	return FALSE;
}

static gboolean
motion_notify_event (GtkWidget *widget,
		     GdkEventMotion *event)
{
	KzWindow *kz;
	gint x, y;
	/* GdkModifierType state; */
	KzWindowPrivate *priv;
	KzWeb *kzweb =KZ_WINDOW_CURRENT_WEB(KZ_WINDOW(widget));

	if (!kzweb)
		return FALSE;

	kz = KZ_WINDOW(widget);
	priv = KZ_WINDOW_GET_PRIVATE(kz);

	gtk_widget_get_pointer(GTK_WIDGET(kzweb), &x, &y);

	if (abs(x - priv->start_x) > 2 ||
	    abs(y - priv->start_y) > 2)
	{
		priv->is_gesture = TRUE;
	}

	if (kz_gesture_is_started(priv->gesture))
	{
		kz_gesture_update_position(priv->gesture, x, y);
	}

	kz_actions_set_selection_sensitive(kz, kzweb);

	return FALSE;
}


static gboolean
button_release_event (GtkWidget *widget,
		      GdkEventButton *event)
{
	KzWindow *kz;
	gboolean retval = FALSE;
	KzWindowPrivate *priv;

	kz = KZ_WINDOW(widget);
	priv = KZ_WINDOW_GET_PRIVATE(kz);

        if (gdk_pointer_is_grabbed())
        {
                gdk_pointer_ungrab(gtk_get_current_event_time());
        }

	if (kz_gesture_is_started(priv->gesture))
	{
		if (event->button == 3 && !priv->is_button3_scrolled)
			kz_gesture_perform(priv->gesture);
		else
			kz_gesture_cancel(priv->gesture);
		kz_statusbar_set_gesture_text(KZ_STATUSBAR(kz->statusbar), NULL);
	}

	if (!priv->is_gesture &&
	    !priv->is_button3_scrolled &&
	    event->button == 3)
	{
		kz_actions_popup_menu_modal(kz, event->button, event->time);
		retval = TRUE;
	}

	if (event->button == 3)
	{
		if (priv->is_button3_scrolled)
			retval = TRUE;
		priv->is_button3_pressed = FALSE;
		priv->is_button3_scrolled = FALSE;
	}
	
	priv->start_x = 0;
	priv->start_y = 0;
	priv->is_gesture = FALSE;

	if (priv->event)
		kz_embed_event_free((KzEmbedEvent *) priv->event);
	priv->event = NULL;

	return retval;
}

static gboolean
scroll_event (GtkWidget *widget,
	      GdkEventScroll *event)
{
	KzWindow *kz;
	gboolean retval = FALSE;
	KzWindowPrivate *priv;

	kz = KZ_WINDOW(widget);
	priv = KZ_WINDOW_GET_PRIVATE(kz);

	if (priv->is_button3_pressed)
	{
		retval = kz_notebook_scroll_tab(KZ_NOTEBOOK(kz->notebook), event->direction);
		priv->is_button3_scrolled = retval;
	}
	
	return retval;
}

static gboolean 
focus_out_event (GtkWidget *widget,
		 GdkEventFocus *event)
{
	if(!event->in) 
	{
		KzWindowPrivate *priv = KZ_WINDOW_GET_PRIVATE(widget);
		/* focus is out */
		if (kz_gesture_is_started(priv->gesture))
		{
			kz_gesture_cancel(priv->gesture);
			if (gdk_pointer_is_grabbed())
				gdk_pointer_ungrab(gtk_get_current_event_time());

			kz_statusbar_set_gesture_text(KZ_STATUSBAR(KZ_WINDOW(widget)->statusbar), NULL);
		}
	}
	return FALSE;
}

void
kz_window_append_closed_tab (KzWindow *kz, KzBookmarkFolder *bookmark)
{
	KzBookmark *last;
	kz_bookmark_folder_prepend(kz->closed_tabs, KZ_BOOKMARK(bookmark));

	/* removed the oldest one */
	last = kz_bookmark_folder_get_nth_child(kz->closed_tabs, MAX_CLOSED_TABS);
	if (last)
		kz_bookmark_folder_remove(kz->closed_tabs, last);
}

void
kz_window_switch_layout_engine (KzWindow *kz, const gchar *engine_name)
{
	KzWeb *web = KZ_WINDOW_CURRENT_WEB(kz);

	kz_web_change_engine(KZ_WEB(web), engine_name);
}

