/***************************************************************************
                          barcodecache.h  -  description
                             -------------------
    begin                : Sam Aug 16 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef BARCODECACHE_H
#define BARCODECACHE_H

#include <qobject.h>
#include <qstring.h>
#include <qvaluelist.h>

#include "mybarcode.h"
#include "xmlutils.h"

class QPixmap;
/** This singleton is a application wide cache for Barcodes.
  * The barcodes are saved to the harddisk as PNG files and can
  * be read from the cache as QPixmaps. Generating barcodes
  * is very slow. As a reason this cache greatly increases KBarcodes
  * speed.
  * Not only the PNG's are written to the harddisk, but also a
  * XML index file. 
  *
  * @author Dominik Seichter
  * @short a application wide cache for barcodes.
  */
class BarcodeCache : public QObject, public XMLUtils {
    Q_OBJECT

    typedef struct data {
        barcodeData settings;
        int resolution;
        
        QString path;

        bool operator==( const struct data d ) const {
            bool b = ( resolution == d.resolution && settings == d.settings );
        
            return b;
        }
    };
    
    public:
        /** get an instance of the BarcodeCache singleton as there is no public constructor.
          */
        static BarcodeCache* instance();

        /** read a barcode with the @p barcodeData & b, the resolution @p int resolution
          * and the value @p QString value from the cache. If no such barcode is in the cache
          * 0 is returned. @p value is a additional parameter to support barcode sequences
          *
          *  @return a QPixmap of the barcode which must be deleted or 0.
          */
        QPixmap* read( const barcodeData & b, int resolution, const QString & value );

        /** write a barcode created with @p barcodeData & b, the resolution @p int resolution
          * and the value @p QString value to the cache. @p QString value is used as parameter
          * to support barcode sequences. You have to give a pointer to the @p QPixmap* pix
          * so that the barcode can be saved as a PNG.
          */
        void write( const barcodeData & b, int resolution, QPixmap* pix, const QString & value );

        unsigned int count() const { return m_data.count(); }
        unsigned int size() const { return m_size; }
        void setSize( unsigned int s ) { m_size = s; }

        /** clear the cache in memory and on the harddisk
          */
        void clear();
        
    private slots:
        /** save the cache from memory to harddisk.
          * This slot is called when KBarcode quits, but not on crashes!
          */
        void flush();
        
    private: 
        BarcodeCache();
        ~BarcodeCache();
        data fillStruct( const barcodeData & b, int resolution );

        static BarcodeCache* self;
        
        QString m_xml;
        QString m_dir;
        QValueList<data> m_data;

        unsigned int m_size;
};

#endif
