/***************************************************************************
                          barcodecombo.cpp  -  description
                             -------------------
    begin                : Son Apr 13 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "barcodecombo.h"
#include "mybarcode.h"
#include "barcodedialogs.h"

// Qt includes
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qregexp.h>
#if QT_VERSION >= 0x030100
    #include <ktextedit.h>
#else
    #include <qtextedit.h>
#endif

// KDE includes
#include <knuminput.h>
#include <klineedit.h>
#include <klocale.h>
#include <kpushbutton.h>

extern QValueList<barcodeFormat> codes;

BarcodeCombo::BarcodeCombo(QWidget *parent, const char *name )
    : KComboBox( false, parent, name )

{
    for( unsigned int i = 0; i < codes.count(); i++ )
        this->insertItem( codes[i].text );
}

BarcodeCombo::~BarcodeCombo()
{
}

const QString BarcodeCombo::getEncodingType()
{
    for( unsigned int i = 0; i < codes.count(); i++ )
        if( codes[i].text == currentText() )
            return codes[i].name;

    return "code39";
}

void BarcodeCombo::setEncodingType( const QString & type )
{
    for( unsigned int i = 0; i < (unsigned int)codes.count(); i++ )
        if( codes[i].name ==  type ) {
            for( unsigned int z = 0; z < (unsigned int)count(); z++ )
                if( text( z ) == codes[i].text )
                    setCurrentItem( z );
        }
}

BarcodeWidget::BarcodeWidget(QWidget *parent, const char *name )
    : QWidget( parent, name )
{
    QGridLayout* grid = new QGridLayout( this, 6, 6 );

    labelStandard = new QLabel( i18n( "Encoding Type:" ), this );
    grid->addWidget( labelStandard, 1, 0 );

    comboStandard = new BarcodeCombo( this, "comboStandard" );
    connect( comboStandard, SIGNAL( activated(int) ), this, SLOT( encodingChanged() ) );
    connect( comboStandard, SIGNAL( activated(int) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( comboStandard, 1, 1, 1, 3 );

    labelData = new QLabel( i18n( "Value:" ), this );
    grid->addWidget( labelData, 2, 0 );

    data = new KLineEdit( this, "data" );
    connect( data, SIGNAL( textChanged( const QString & ) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( data, 2, 2, 1, 3 );

#if QT_VERSION >= 0x030100
        multi = new KTextEdit( this );
#else
        multi = new QTextEdit( this );
#endif
    multi->setTextFormat( QTextEdit::PlainText );
    multi->setWordWrap( QTextEdit::NoWrap );
    multi->setEnabled( false );
    multi->setVScrollBarMode( QScrollView::AlwaysOn );
    multi->setHScrollBarMode( QScrollView::AlwaysOn );
    multi->hide();
    connect( multi, SIGNAL( textChanged() ), this, SLOT( changed() ) );    
    grid->addMultiCellWidget( multi, 3, 3, 1, 3 );
    
    checkText = new QCheckBox( this, "checkText" );
    checkText->setText( i18n( "&Display text" ) );
    checkText->setChecked( true );
    connect( checkText, SIGNAL( clicked() ), this, SLOT( changed() ) );
    grid->addWidget( checkText, 4, 1 );

    buttonAdvanced = new KPushButton( i18n("&Advanced..."), this );
//    buttonAdvanced->setEnabled( false );
    grid->addWidget( buttonAdvanced, 4, 3 );
    
    spinMargin = new KIntNumInput( this, "spinMargin" );
    spinMargin->setLabel( i18n( "Margin:" ), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    spinMargin->setRange( 0, 10000, 1, false );
    spinMargin->setValue( 10 );
    connect( spinMargin, SIGNAL( valueChanged(int) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( spinMargin, 5, 5, 0, 1 );

    spinRotation = new KIntNumInput( this );
    spinRotation->setLabel( i18n( "Rotation:" ), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    spinRotation->setRange( 0, 360, 90, false );
    spinRotation->setValue( 0 );
    connect( spinRotation, SIGNAL( valueChanged(int) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( spinRotation, 5, 5, 2, 3 );

    spinScale = new KIntNumInput( spinMargin, 1000, this );
    spinScale->setLabel( i18n("Scale (in permille):"), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    spinScale->setRange( 100, 10000, 100, false );
    spinScale->setValue( 1000 );
    connect( spinScale, SIGNAL( valueChanged(int) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( spinScale, 6, 6, 0, 1 );

    spinCut = new KIntNumInput( spinRotation, 100, this );
    spinCut->setLabel( i18n("Crop:"), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    spinCut->setRange( 1, 100, 1, false );
    connect( spinCut, SIGNAL( valueChanged(int) ), this, SLOT( changed() ) );
    grid->addMultiCellWidget( spinCut, 6, 6, 2, 3 );

    connect( buttonAdvanced, SIGNAL( clicked() ), this, SLOT( advanced() ) );

    m_enabledata = true;
    m_multi = false;
    encodingChanged();

    BarCode::fillDefault( &barcode );
}

barcodeData BarcodeWidget::getData() const
{
    return barcode;
}

void BarcodeWidget::setData( const barcodeData b )
{
    comboStandard->setEncodingType( b.type );
    encodingChanged();
    if( !m_multi )
        data->setText( b.value );
    else
        multi->setText( b.value );

    checkText->setChecked( b.text );

    spinMargin->setValue( b.margin );
    spinRotation->setValue( b.rotation );
    spinCut->setValue( int(b.cut*100) );
    spinScale->setValue( int(b.scale*1000) );

    barcode = b;    
}

void BarcodeWidget::setStandardEnabled( bool b )
{
    labelStandard->setEnabled( b );
    comboStandard->setEnabled( b );
}

void BarcodeWidget::setDataEnabled( bool b )
{
    labelData->setEnabled( b );
    m_enabledata = b;
    encodingChanged();
}

void BarcodeWidget::defaults()
{
    barcodeData d;
    BarCode::fillDefault( &d );
    setData( d );
}

void BarcodeWidget::encodingChanged()
{
    spinCut->setEnabled( BarCode::hasFeature( comboStandard->getEncodingType(), NOCUT ) ? false : true );
    if( !spinCut->isEnabled() )
        spinCut->setValue( 100 ); // TODO: Don't hardcode
    
    spinScale->setEnabled( BarCode::hasFeature( comboStandard->getEncodingType(), NOSCALE ) ? false : true );
    if( !spinScale->isEnabled() )
        spinScale->setValue( 1000 ); // TODO: Don't hardcode
        
    checkText->setEnabled( BarCode::hasFeature( comboStandard->getEncodingType(), NOTEXT ) ? false : true );

    if( BarCode::hasFeature( comboStandard->getEncodingType(), MULTILINE ) ) {
        multi->setEnabled( m_enabledata );
        multi->show();
        multi->setFocus();
        data->setEnabled( false );
        m_multi = true;
    } else {
        data->setEnabled( m_enabledata );
        multi->hide();
        data->setFocus();
        multi->setEnabled( false );
        m_multi = false;
    }

//    if( BarCode::hasFeature( comboStandard->getEncodingType(), PDF417BARCODE ) ||
//        BarCode::hasFeature( comboStandard->getEncodingType(), TBARCODE ) ||
//        BarCode::hasFeature( comboStandard->getEncodingType(), DATAMATRIX ) )
//        buttonAdvanced->setEnabled( true );
//    else
//        buttonAdvanced->setEnabled( false );
}

void BarcodeWidget::advanced()
{
    AdvancedBarcodeDialog* abd = new AdvancedBarcodeDialog( comboStandard->getEncodingType(), this, "abd" );
    abd->setData( &barcode );
    if( abd->exec() == QDialog::Accepted )
        abd->getData( &barcode );
    /*
    if( BarCode::hasFeature( comboStandard->getEncodingType(), PDF417BARCODE ) ) {
        PDF417BarcodeDlg* tbd = new PDF417BarcodeDlg( this );
        tbd->setData( &barcode );
        if( tbd->exec() == QDialog::Accepted ) 
            tbd->getData( &barcode );
        
        return;
    } else if( BarCode::hasFeature( comboStandard->getEncodingType(), DATAMATRIX ) ) {
        DataMatrixDlg* tbd = new DataMatrixDlg( this );
        tbd->setData( &barcode );
        if( tbd->exec() == QDialog::Accepted ) 
            tbd->getData( &barcode );

        return;        
    } else if( BarCode::hasFeature( comboStandard->getEncodingType(), TBARCODEADV ) ) {
        TBarcodeDlg* tbd = new TBarcodeDlg( this );
        tbd->setData( &barcode );
        if( tbd->exec() == QDialog::Accepted )
            tbd->getData( &barcode );

        return;        
    } else {
        SequenceDlg * sd = new SequenceDlg( this );
        sd->setData( &barcode );
        if( sd->exec() == QDialog::Accepted )
            sd->getData( &barcode );
    } */
}

void BarcodeWidget::changed()
{
    if( !m_multi )
        barcode.value = data->text();
    else
        barcode.value = multi->text();

    barcode.type = comboStandard->getEncodingType();
    barcode.text = checkText->isChecked();

    barcode.margin = spinMargin->value();
    barcode.rotation = spinRotation->value();
    barcode.cut = (double)spinCut->value()/100.0;
    barcode.scale = (double)spinScale->value()/1000.0;
}
