/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qspinbox.h>
#include <qfile.h>

#include <kio/jobclasses.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <klineedit.h>
#include <kurl.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <settings.h>
#include <entryfield.h>
#include <file.h>
#include <fileimporterbibtex.h>
#include "webqueryieeexplore.h"

#define min(a,b) ((a)>(b)?(b):(a))

namespace KBibTeX
{
    WebQueryIEEExploreWidget::WebQueryIEEExploreWidget( QWidget *parent, const char *name )
            : WebQueryWidget( parent, name )
    {
        init();

        Settings *settings = Settings::self();
        QString value = settings->getWebQueryDefault( "IEEE" );
        value = value == QString::null ? "" : value;
        lineEditQuery->setText( value );
        slotTextChanged( value, true );
    }

    WebQueryIEEExplore::WebQueryIEEExplore( QWidget* parent ): WebQuery( parent ), m_numberOfMatches( "Your search matched <strong>(\\d+)</strong> of" ), m_findArNumber( "srchabstract.jsp\\?arnumber=(\\d+)" ), m_date( "(?:([A-Z][a-z]{2,3})(?:(?:/|-\\d+\\s+)([A-Z][a-z]{2,3}))?)?\\.?\\s*(\\d{4})$" ), m_risURL( "http://ieeexplore.ieee.org/xpls/citationAct" )
    {
        m_widget = new WebQueryIEEExploreWidget( parent );
        m_bibtexImporter = new BibTeX::FileImporterBibTeX( false );
    }

    WebQueryIEEExplore::~WebQueryIEEExplore()
    {
        delete m_bibtexImporter;
        delete m_widget;
    }

    QString WebQueryIEEExplore::title()
    {
        return i18n( "IEEExplore" );
    }

    QString WebQueryIEEExplore::disclaimer()
    {
        return i18n( "What is IEEE Xplore?" );
    }

    QString WebQueryIEEExplore::disclaimerURL()
    {
        return "http://ieeexplore.ieee.org/guide/g_oview_faq.jsp";
    }

    WebQueryWidget *WebQueryIEEExplore::widget()
    {
        return m_widget;
    }

    void WebQueryIEEExplore::query()
    {
        WebQuery::query();
        Settings *settings = Settings::self();
        settings->setWebQueryDefault( "IEEE", m_widget->lineEditQuery->text() );

        m_arnumList.clear();
        m_numMaxHits = min( 50, m_widget->spinBoxMaxHits->value() );
        setNumStages( m_numMaxHits + 1 );
        QString searchTerm = m_widget->lineEditQuery->text().stripWhiteSpace().replace( '$', "" );
        if ( searchTerm.isEmpty() )
        {
            setEndSearch( WebQuery::statusInvalidQuery );
            return;
        }

        QString category = "metadata";
        KURL url = KURL( QString( "http://ieeexplore.ieee.org/search/freesearchresult.jsp?queryText=" ).append( searchTerm.replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" ) ).append( "+%3Cin%3E+" ).append( category ).append( "&ResultCount=" ).append( QString::number( m_numMaxHits ) ).append( "&ResultStart=0" ) );

        QString completeText = downloadHTML( url );
        if ( completeText != QString::null && !m_aborted )
        {
            if ( completeText.find( "You have entered an invalid search" ) > -1 )
            {
                KMessageBox::information( m_widget, i18n( "You have entered an invalid search." ), i18n( "Search Error" ) );
                setEndSearch( WebQuery::statusInvalidQuery );
                return;
            }
            else if ( completeText.find( "No results" ) > -1 )
            {
                setEndSearch( WebQuery::statusSuccess );
                return;
            }

            m_numberOfMatches.search( completeText );
            bool ok;
            m_numMatches = m_numberOfMatches.cap( 1 ).toInt( &ok );
            if ( !ok ) m_numMatches = 0;

            if ( m_numMatches <= 0 )
            {
                setEndSearch( WebQuery::statusSuccess );
                return;
            }

            m_findArNumber.search( completeText );
            for ( int i = 0; i < m_numMatches; ++i )
            {
                if ( m_findArNumber.cap( 1 ).isEmpty() ) break;
                int arnum = m_findArNumber.cap( 1 ).toInt( &ok );
                if ( !ok || arnum <= 0 ) break;
                m_arnumList.append( arnum );

                m_findArNumber.search( completeText,  m_findArNumber.pos( 1 ) + 1 );
            }

            if ( !m_arnumList.isEmpty() )
            {
                m_hitCounter = 1;
                fetchNext();
            }
            else
            {
                setEndSearch( WebQuery::statusSuccess );
            }
        }
        else if ( !m_aborted )
        {
            QString message = KIO::NetAccess::lastErrorString();
            if ( message.isEmpty() )
                message.prepend( '\n' );
            message.prepend( QString( i18n( "Querying database '%1' failed." ) ).arg( title() ) );
            KMessageBox::error( m_parent, message );
            setEndSearch( WebQuery::statusError );
        }
        else
        {
            setEndSearch( WebQuery::statusSuccess );
        }
    }

    void WebQueryIEEExplore::fetchNext()
    {
        if ( m_arnumList.isEmpty() ) return;

        int arnum = m_arnumList.first();
        m_arnumList.remove( m_arnumList.begin() );

        m_incomingData = "";
        QString data = "dlSelect=cite&fileFormate=BibTex&arnumber=%3Carnumber%3E" + QString::number( arnum ) + "%3C%2Farnumber%3E&Submit=Download";

        KIO::TransferJob *job = KIO::http_post( m_risURL, data.utf8(), false );
        job->addMetaData( "content-type", "Content-Type: application/x-www-form-urlencoded" );
        connect( job, SIGNAL( data( KIO::Job *, const QByteArray & ) ), this, SLOT( slotData( KIO::Job *, const QByteArray & ) ) );
        connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotResult( KIO::Job * ) ) );
    }

    void WebQueryIEEExplore::slotData( KIO::Job *, const QByteArray &data )
    {
        if ( data.size() > 0 )
            m_incomingData.append( QCString( data, data.size() + 1 ) );
    }

    void WebQueryIEEExplore::slotResult( KIO::Job *job )
    {
        QRegExp m_date( "^(((\\d{1,2}(-\\d{1,2}))\\s+)?(([A-Z][a-z]{2,3})(/([A-Z][a-z]{2,3}))?)\\.?\\s+)?(\\d{4})$" );
        if ( job->error() )
        {
            job->showErrorDialog();
            return;
        }

        enterNextStage();
        m_incomingData.replace( "<br>", "" );

        BibTeX::File *bibtexFile = m_bibtexImporter->load( m_incomingData );
        if ( bibtexFile != NULL )
        {
            for ( BibTeX::File::ElementList::iterator it = bibtexFile->begin(); it != bibtexFile->end(); ++it )
            {
                BibTeX::Entry *entry = dynamic_cast<BibTeX::Entry*>( *it );
                if ( entry != NULL )
                {
                    fixDate( entry );
                    emit foundEntry( new BibTeX::Entry( entry ), false );
                }
            }
            delete bibtexFile;
        }

        if ( !m_arnumList.isEmpty() )
            fetchNext();
        else
        {
            setEndSearch( WebQuery::statusSuccess );
        }
    }

    void WebQueryIEEExplore::fixDate( BibTeX::Entry *entry )
    {
        m_date.search( entry->getField( BibTeX::EntryField::ftYear )->value()->text() );

        if ( !m_date.cap( 3 ).isNull() && !m_date.cap( 3 ).isEmpty() )
        {
            BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftYear );
            if ( field == NULL )
            {
                field = new BibTeX::EntryField( BibTeX::EntryField::ftJournal );
                entry->addField( field );
            }
            field->setValue( new BibTeX::Value( m_date.cap( 3 ), true ) );
        }

        QString newMonth = QString::null;
        if ( !m_date.cap( 1 ).isNull() && !m_date.cap( 1 ).isEmpty() && ( newMonth = parseMonth( m_date.cap( 1 ) ) ) != QString::null )
        {
            BibTeX::EntryField * field = entry->getField( BibTeX::EntryField::ftMonth );
            if ( field == NULL )
            {
                field = new BibTeX::EntryField( BibTeX::EntryField::ftJournal );
                entry->addField( field );
            }

            field->setValue( new BibTeX::Value( newMonth, true ) );

            newMonth = QString::null;
            if ( !m_date.cap( 2 ).isNull() && !m_date.cap( 2 ).isEmpty() && ( newMonth = parseMonth( m_date.cap( 2 ) ) ) != QString::null )
            {
                field->value()->items.append( new BibTeX::PlainText( "/" ) );
                field->value()->items.append( new BibTeX::MacroKey( newMonth ) );
            }
        }
    }

    QString WebQueryIEEExplore::parseMonth( const QString &month )
    {
        for ( unsigned int i = 0; i < sizeof( BibTeX::MonthsTriple ) / sizeof( BibTeX::MonthsTriple[0] ); ++i )
            if ( month.startsWith( BibTeX::MonthsTriple[i], false ) ) return BibTeX::MonthsTriple[i];
        return QString::null;
    }

}
#include "webqueryieeexplore.moc"
