/***************************************************************************

   Copyright (C) 2005-2007 by Christian Weilbach <christian_weilbach@web.de>
   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "profileconfigdialog.h"
#include "profileconfig.h"

#include <QTimer>

#include <kdebug.h>
#include <kwallet.h>
#include <kconfigdialog.h>
#include <kio/job.h>
#include <kmessagebox.h>

#include <kblog/gdata.h>
#include <kblog/blogger1.h>
#include <kblog/metaweblog.h>
#include <kblog/movabletype.h>
#include <kblog/wordpressbuggy.h>

#define TIMEOUT 45000

namespace KBlogger
{

ProfileConfigDialog::ProfileConfigDialog(KConfigDialog* parent): mWallet(0), mBlog(0),
                               mFetchProfileIdTimer(0),mFetchAPITimer(0)
{
    kDebug();

    setupUi(this);

    // manage the automatic ID fetching
    connect(kcfg_User, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(kcfg_Url, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(passwordLineEdit, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(kcfg_Type, SIGNAL( activated( int ) ), this, SLOT( enableFetchBlogId() ) );

    mWallet = KWallet::Wallet::openWallet( "kdewallet", this->winId() );
    if ( mWallet ) {
        mWallet->createFolder( "kblogger" );
        mWallet->setFolder( "kblogger" );
        kDebug() << "Wallet successfully opened.";
        readPasswordFromWallet();
    }

    connect(kcfg_Type, SIGNAL( activated( int ) ), this, SLOT( proposeUrl( int ) ) );

    connect(parent, SIGNAL(okClicked()),
            this, SLOT(savePasswordToWallet()));
    connect(parent, SIGNAL(applyClicked()),
            this, SLOT(savePasswordToWallet()));

    connect(mFetchIDButton, SIGNAL( clicked() ), this, SLOT( fetchBlogId() ) );
    connect(mAutoconfigureButton, SIGNAL( clicked() ), this, SLOT( autoconfigure() ) );

    enableFetchBlogId();
}

ProfileConfigDialog::~ProfileConfigDialog()
{
    kDebug();
}

void ProfileConfigDialog::readPasswordFromWallet()
{
    kDebug() << kcfg_Url->text();
    if ( !mWallet ) return;
    QString buffer;
    mWallet->readPassword( kcfg_Url->text() + '_' + kcfg_User->text(), buffer );
    kDebug() << "Password from Wallet: " << buffer;
    passwordLineEdit->setText( buffer );
}

void ProfileConfigDialog::savePasswordToWallet()
{
    kDebug();
    if ( !mWallet ) return;
    mWallet->writePassword( kcfg_Url->text() + '_' + kcfg_User->text(), passwordLineEdit->text() );
}

void ProfileConfigDialog::enableFetchBlogId()
{
    kDebug();
    if ( kcfg_User->text().isEmpty() ||
            kcfg_Url->text().isEmpty() ||
            passwordLineEdit->text().isEmpty() ) {
        mFetchIDButton->setEnabled( false );
        mAutoconfigureButton->setEnabled( false );
    }
    else {
        mFetchIDButton->setEnabled( true );
        mAutoconfigureButton->setEnabled( true );
    }
}

void ProfileConfigDialog::autoconfigure()
{
    if( kcfg_User->text().isEmpty() ||
         passwordLineEdit->text().isEmpty() ||
         kcfg_Url->text().isEmpty() ) {

         kDebug()<< "fill in all relevant information first";
         KMessageBox::sorry( 0,
              i18n("You need to set the username, password and url of your blog."));
         return;
    }
    mAutoconfigureButton->setEnabled( false );
    mFetchIDButton->setEnabled( false );

    // do the api guessing with the url
    if( kcfg_Url->text().indexOf( "xmlrpc.php" ) != -1 ){
        kcfg_Type->setCurrentIndex( 3 );
        fetchBlogId();
        return;
    }
    if( kcfg_Url->text().indexOf( "blogspot" ) != -1 ){
        kcfg_Type->setCurrentIndex( 4 );
        fetchBlogId();
        return;
    }
    if( kcfg_Url->text().indexOf( "wordpress" ) != -1 ){
        kcfg_Type->setCurrentIndex( 3 );
        kcfg_Url->setText( kcfg_Url->text() + "/xmlrpc.php" );
        fetchBlogId();
        return;
    }
    if( kcfg_Url->text().indexOf( "livejournal" ) != -1 ){
        kcfg_Type->setCurrentIndex( 0 );
        kcfg_Url->setText( "http://www.livejournal.com/interface/blogger/" );
        kcfg_BlogId->setText( kcfg_User->text() );
        mAutoconfigureButton->setEnabled( true );
        mFetchIDButton->setEnabled( true );
        return;
    }
    if( kcfg_Url->text().indexOf( "drupal" ) != -1 ){
        kcfg_Type->setCurrentIndex( 3 );
        kcfg_Url->setText( kcfg_Url->text() + "/xmlrpc.php" );
        fetchBlogId();
        return;
    }

    mFetchAPITimer = new QTimer( this );
    mFetchAPITimer->setSingleShot( true );
    connect( mFetchAPITimer, SIGNAL( timeout() ),
                    this, SLOT( handleFetchAPITimeout() ) );
    mFetchAPITimer->start( TIMEOUT );

    // try to get the api from the html otherwise
    httpGetJob = KIO::storedGet( kcfg_Url->text(), KIO::NoReload, KIO::HideProgressInfo );

    connect( httpGetJob, SIGNAL( result( KJob* ) ),
             this, SLOT( gotHtml( KJob* ) ) );
}

void ProfileConfigDialog::proposeUrl( int i )
{
    kDebug();
    if ( kcfg_Url->text().isEmpty() ) {
        switch ( i ) {
        case 0: // All Blogger1 derivates use xmlrpc.php most likely
        case 1:
        case 2:
        case 3:
            kcfg_Url->setText( "http://your-blog-address.com/xmlrpc.php" );
            break;
        case 4:
            kcfg_Url->setText( "http://your-blog.blogspot.com" );
            break;
        };
    }
}

void ProfileConfigDialog::fetchBlogId()
{
    kDebug();
    mFetchProfileIdTimer = new QTimer( this );
    mFetchProfileIdTimer->setSingleShot( true );
    connect( mFetchProfileIdTimer, SIGNAL( timeout() ),
             this, SLOT( handleFetchIDTimeout() ) );
    mFetchProfileIdTimer->start( TIMEOUT );
    delete mBlog;
    switch ( kcfg_Type->currentIndex() ) {
    case 0: // Blogger1
    case 1: // All others should support it as well, as they are extensions to Blogger1 we can call Blogger1
    case 2:
    case 3:
        mBlog = new KBlog::Blogger1( kcfg_Url->text() );
        dynamic_cast<KBlog::Blogger1*>(mBlog)->setUsername( kcfg_User->text() );
        dynamic_cast<KBlog::Blogger1*>(mBlog)->setPassword( passwordLineEdit->text() );
        connect( dynamic_cast<KBlog::Blogger1*>(mBlog), SIGNAL(fetchedProfileId( const QString& ) ),
                 this, SLOT( fetchedProfileId( const QString& ) ) );
        connect( dynamic_cast<KBlog::Blogger1*>(mBlog), SIGNAL(listedBlogs( const QList<QMap<QString, QString> >&)),
                 this, SLOT(fetchedBlogId( const QList<QMap<QString, QString> >&)));
        connect( dynamic_cast<KBlog::Blogger1*>(mBlog), SIGNAL(error( KBlog::Blog::ErrorType, const QString& ) ),
                 this, SLOT( handleFetchError( KBlog::Blog::ErrorType, const QString& ) ) );
        dynamic_cast<KBlog::Blogger1*>(mBlog)->listBlogs();
        break;

    case 4: // GData
        mBlog = new KBlog::GData( kcfg_Url->text() );
        dynamic_cast<KBlog::GData*>(mBlog)->setUsername( kcfg_User->text() );
        dynamic_cast<KBlog::GData*>(mBlog)->setPassword( passwordLineEdit->text() );
        connect( dynamic_cast<KBlog::GData*>(mBlog), SIGNAL(fetchedProfileId( const QString& ) ),
                 this, SLOT( fetchedProfileId( const QString& ) ) );
        dynamic_cast<KBlog::GData*>(mBlog)->fetchProfileId();

        break;
    };
    kcfg_BlogId->setText( i18n( "Please wait..." ) );
    kcfg_BlogId->setEnabled( false );
    mFetchIDButton->setEnabled( false );
}

void ProfileConfigDialog::fetchedBlogId( const QList<QMap<QString, QString> >& list )
{
    kDebug();
    // TODO readd when we have done multi id handling correctly
//     QList<QMap<QString,QString> >::ConstIterator it = list.begin();
//     QList<QMap<QString,QString> >::ConstIterator end = list.end();
//     for(; it != end; ++it ) {
//         kcfg_BlogId->setText( ( *it ).values().first() );
//     }
//     delete mFetchProfileIdTimer;
    kcfg_BlogId->setText( list.first().values().first() );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchError( KBlog::Blog::ErrorType, const QString& error )
{
    kDebug();
//     delete mFetchProfileIdTimer;
    KMessageBox::error( 0,
                        error );
    kcfg_BlogId->setText( QString() );
//     kcfg_BlogId->setEditable( true );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchIDTimeout()
{
    kDebug();
    KMessageBox::error( 0,
                        i18n( "Fetching the blog's id timed out. Are you online?" ));
    kcfg_BlogId->setText( QString() );
//     kcfg_BlogId->setEditable( true );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchAPITimeout()
{
    kDebug();
    KMessageBox::error( 0,
                        i18n( "Autoconfiguration timed out. Are you online?" ));
//     kcfg_BlogId->setEditable( true );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::fetchedProfileId( const QString &id )
{
    kDebug();
    delete mFetchProfileIdTimer;
    connect( dynamic_cast<KBlog::GData*>(mBlog), SIGNAL(listedBlogs( const QList<QMap<QString, QString> >&)),
             this, SLOT(fetchedBlogId( const QList<QMap<QString, QString> >&)));
    connect( dynamic_cast<KBlog::GData*>(mBlog), SIGNAL(error( KBlog::Blog::ErrorType, const QString& ) ),
             this, SLOT( handleFetchError( KBlog::Blog::ErrorType, const QString& ) ) );
    dynamic_cast<KBlog::GData*>(mBlog)->listBlogs();
}

void ProfileConfigDialog::gotHtml(KJob *job)
{
    kDebug();
    if(!job) return;
    if ( job->error() ) {
        KMessageBox::error( 0,
                            job->errorString());
        mAutoconfigureButton->setEnabled( true );
//         delete mFetchAPITimer;
        return;
    }
    QString httpData( dynamic_cast<KIO::StoredTransferJob*>(job)->data() );
    delete job;

    QRegExp rxGData( QString( "content='blogger' name='generator'" ) );
    if( rxGData.indexIn( httpData ) != -1 ){
        kDebug() << "content='blogger' name='generator' matched";
        kcfg_Type->setCurrentIndex( 4 );
        QRegExp rxBlogId( QString( "BlogID=(\\d+)" ) );
        kcfg_BlogId->setText( rxBlogId.cap(1) );
        mAutoconfigureButton->setEnabled( true );
        mFetchIDButton->setEnabled( true );
//         delete mFetchAPITimer;
        return;
    }

    QRegExp rxLiveJournal( QString( "rel=\"openid.server\" href=\"http://www.livejournal.com/openid/server.bml\"" ) );
    if( rxLiveJournal.indexIn( httpData ) != -1 ){
        kDebug() << " rel=\"openid.server\" href=\"http://www.livejournal.com/openid/server.bml\" matched";
        kcfg_Type->setCurrentIndex( 0 );
        kcfg_Url->setText( "http://www.liverjournal.com/interface/blogger/" );
        kcfg_BlogId->setText( kcfg_User->text() );
        mAutoconfigureButton->setEnabled( true );
        mFetchIDButton->setEnabled( true );
//         delete mFetchAPITimer;
        return;
    }

    QRegExp rxWordpress( QString( "name=\"generator\" content=\"WordPress" ) );
    if( rxWordpress.indexIn( httpData ) != -1 ){
        kDebug() << "name=\"generator\" content=\"WordPress matched";
        kcfg_Type->setCurrentIndex( 3 );
        kcfg_Url->setText( kcfg_Url->text() + "/xmlrpc.php" );
//         delete mFetchAPITimer;
        fetchBlogId();
        return;
    }

    // add MT for WordpressBuggy -> URL/xmlrpc.php exists
    testXmlRpcJob = KIO::storedGet( kcfg_Url->text() + "/xmlrpc.php", KIO::NoReload, KIO::HideProgressInfo );

    connect( testXmlRpcJob, SIGNAL( result( KJob* ) ),
             this, SLOT( gotXmlRpcTest( KJob* ) ) );
};

void ProfileConfigDialog::gotXmlRpcTest(KJob *job)
{
    kDebug();
    if( !job ) return;
    if ( job->error() ) {
        KMessageBox::error( 0,
                            i18n("Impossible to get the API: %1", job->errorString()));
        mAutoconfigureButton->setEnabled( true );
//         delete mFetchAPITimer;
        return;
    }
    delete job;

    kcfg_Type->setCurrentIndex( 3 );
    kcfg_Url->setText( kcfg_Url->text() + "/xmlrpc.php" );
//     delete mFetchAPITimer;
    fetchBlogId();
}

} //namespace

#include "profileconfigdialog.moc"

