/***************************************************************************

   Copyright (C) 2005-2007 by Christian Weilbach <christian_weilbach@web.de>
   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "uploadmediadialog.h"

#include <QFileInfo>
#include <QLabel>
#include <QLayout>
#include <QImage>
#include <QBuffer>


#include <kfiledialog.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kdebug.h>
#include <kurlrequester.h>
#include <kvbox.h>
#include <kio/job.h>

#include "kbloggerconfig.h"
#include "imagemanager.h"
#include "backend.h"
#include "kbloggermedia.h"
#include "blogchooser.h"
#include "itemsmanager.h"
#include "composereditor.h"

const int MAX_FILE_SIZE_WARNING = 1024; // 1Mb

namespace KBlogger
{

UploadMediaDialog::UploadMediaDialog( const QString& filename,
                                      const QString& blogname,
                                      QWidget *parent)
        : QDialog(parent),
        mMedia(0),
        mBlogname(blogname),
        fileJob(0),
        mBlogChooser(0),
        mImageManager(0)
{
    kDebug();
    setupUi(this);
    setAttribute( Qt::WA_DeleteOnClose ); //Destroy on close

    //Add Blog Chooser
    if (blogname.isEmpty()) {
        mBlogChooser = new BlogChooser(blogGroupBox);
        blogGroupBox->layout()->addWidget(mBlogChooser);
    } else {
        QLabel *blognameLabel = new QLabel(mBlogname, blogGroupBox);
        blogGroupBox->layout()->addWidget(blognameLabel);
    }

    connect ( FilekURLRequester, SIGNAL(  textChanged(const QString&)),
              this, SLOT( updateServersideName(const QString&)) );
    connect ( FilekURLRequester, SIGNAL(  textChanged(const QString&)),
              this, SLOT( updateFilesize(const QString&)) );
    connect ( enqueueButton, SIGNAL ( clicked() ),
              this, SLOT( enqueueSlot() ));

    // initialize if called with filename
    if ( !filename.isEmpty() ) {
        FilekURLRequester->setUrl( filename );
    }
    exec();
}

UploadMediaDialog::~UploadMediaDialog()
{
    kDebug();
}

void UploadMediaDialog::enqueueSlot()
{
    kDebug();
    if ( mBlogChooser ) {
        if ( mBlogChooser->blogCombo->currentIndex() == 0 ) {
            KMessageBox::sorry ( this, i18n ( "Select a blog first."));
            return;
        }
    }

    enqueueButton->setEnabled(false);

    fileJob = KIO::storedGet( FilekURLRequester->url() );

    connect( fileJob, SIGNAL( result( KJob* ) ),
             this, SLOT( handleFileJobResult( KJob* ) ) );
}

void UploadMediaDialog::handleFileJobResult( KJob* job )
{
    kDebug();
    if ( job->error() ) {
        KMessageBox::error ( this,  job->errorString() );
        return;
    }

    // prepare the composer for media insertion
    ComposerEditor *compEditor = qobject_cast<ComposerEditor*> (parent());
    if (!compEditor){
        kDebug() << "casting error from parent() to ComposerEditor*.";
    }

    QByteArray mediaData( dynamic_cast<KIO::StoredTransferJob*>(job)->data() );
    QString mediaMimetype( dynamic_cast<KIO::TransferJob*>(job)->mimetype() );

    if (mBlogname.isEmpty())
        mBlogname = mBlogChooser->blogCombo->currentText();
    kDebug() << "media created with success" << endl;

    // TODO check for mediatypes, since most blogs only support image + a view else
    // create a thumbnail for images
    // TODO check the mimetype of images better
    if( mediaMimetype.indexOf( "image" ) != -1 ){
        kDebug() << "do QImage resizing";
        QImage imageToResize( mediaData );
        if( imageToResize.isNull() ) kDebug() << "could not initialize the image"; 
        imageToResize = imageToResize.scaledToWidth( 240 ); // TODO add a dialog
        QByteArray thumbnailData;
        QBuffer buffer(&thumbnailData);
        buffer.open(QIODevice::WriteOnly);
        imageToResize.save( &buffer, "PNG" );
        mThumbnail = new KBloggerMedia( thumbnailData,
                                "thumbnail_"+NameLineEdit->text(),
                                mediaMimetype,
                                mBlogname
                              );

        mMedia = new KBloggerMedia( mediaData,
                                NameLineEdit->text(),
                                mediaMimetype,
                                mBlogname
                              );

        if ( mMedia->isCached() && mThumbnail->isCached() ) {
            ItemsManager::self()->addMedia(mMedia);
            ItemsManager::self()->addMedia(mThumbnail);
            if(compEditor) compEditor->insertThumbnail(mThumbnail,mMedia);
        }
        else {
            QString msg = i18n("Impossible to copy the file %1 in the cache",
                               FilekURLRequester->url().url() );
            KMessageBox::sorry ( this, msg );
        }
        close();
        return;
    }

     // upload the original file otherwise
    mMedia = new KBloggerMedia( mediaData,
                              NameLineEdit->text(),
                              mediaMimetype,
                              mBlogname
                           );
    if ( mMedia->isCached() ) {
         ItemsManager::self()->addMedia(mMedia);
         if(compEditor) compEditor->insertMedia(mMedia);
    }
    else {
        QString msg = i18n("Impossible to copy the file %1 in the cache",
                           FilekURLRequester->url().url() );
        KMessageBox::sorry ( this, msg );
    }
    close();
}

void UploadMediaDialog::updateServersideName(const QString& filename)
{
    kDebug();
    QFileInfo fileToUpload(filename);
    NameLineEdit->setText(fileToUpload.fileName());
}

void UploadMediaDialog::updateFilesize(const QString& filename)
{
    kDebug();
    double filesize;
    QFileInfo fileToUpload(filename);
    filesize = fileToUpload.size();
    if ( !filesize ) {
        filesizeLabel->setText(i18n("<font color='red'>This file is not valid</font>"));
    } else {
        filesize = filesize / 1024; //Convert byte in KByte
        if ( filesize > MAX_FILE_SIZE_WARNING ) {
            filesizeLabel->setText(i18n ("File size: ") + "<font color='red'>"+QString::number(filesize) +"</font>"+ i18n(" Kb"));
            KMessageBox::sorry ( this, i18n ( "Check if your blog supports that large files." ) );
        } else {
            filesizeLabel->setText(i18n ("File size: ") + "<font color='green'>" + QString::number(filesize) + "</font>" + i18n(" Kb"));
        }
        if (mImageManager)
            mImageManager->setImage(filename);
    }
    enqueueButton->setEnabled(true);
}

} //namespace

#include "uploadmediadialog.moc"
