/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "backend/backend.h"

#include <klocale.h>
#include <kmessagebox.h>

#include <kblog/blogpost.h>
#include <kblog/blogmedia.h>

#include "itemsmanager.h"
#include "post/post.h"
#include "media/media.h"
#include "backend/blogserver.h"
#include "mainwindow.h"
#include "profileconfig.h"
#include "backend/blogjob.h"
#include "backend/bloglist.h"
#include "backend/blogjobqueue.h"
#include "backend/waitdialog.h"

Q_DECLARE_METATYPE(KBlogger::Post*)
Q_DECLARE_METATYPE(KBlog::BlogPost*)
Q_DECLARE_METATYPE(KBlog::BlogMedia*)

namespace KBlogger
{

Backend *Backend::s_self = 0;

Backend *Backend::self(QObject* parent)
{
    kDebug();
    if ( !s_self ) {
        s_self = new Backend(parent);
        s_self->initialize();
    }
    return s_self;
}

Backend::Backend( QObject* parent): QObject(parent),
    mJobsQueue(0), mWaitDialog(0)
{
    kDebug();
    mMainWindow = qobject_cast<MainWindow*> (parent);
}

void Backend::initialize()
{
    kDebug();

    populateBlogsList();

    delete mJobsQueue;
    mJobsQueue = new BlogJobQueue(this);

    connect(mJobsQueue, SIGNAL(jobDone(KJob*)),
            this, SLOT( slotJobDone(KJob*) ));
    connect(mJobsQueue, SIGNAL(jobsStopped()),
            this, SLOT( slotJobsStopped() ));
    connect(mJobsQueue, SIGNAL(jobsDone()),
            this, SLOT( slotJobsDone() ));
    connect(mJobsQueue, SIGNAL(statusMessage(const QString&)),
            this, SLOT( slotStatusMessage(const QString&) ));
    connect(mJobsQueue, SIGNAL(error(const QString&)),
            this, SLOT( slotError(const QString&) ));
}

Backend::~Backend()
{
    kDebug();
    delete mWaitDialog;
    delete mJobsQueue;
    s_self = 0;
}

void Backend::runQueuedJobs()
{
    kDebug();

    // show the wait dialog with the progress bar
    delete mWaitDialog;
    // create mWaitDialog with max jobs size and mMainWindow as its parent
    mWaitDialog = new WaitDialog( mJobsQueue->queueSize(), mMainWindow );
    connect( mWaitDialog, SIGNAL( finished() ),
             this, SLOT( slotCancelClicked() ) );
    connect( mJobsQueue, SIGNAL(waitMessage(const QString&)),
            mWaitDialog, SLOT(setText(const QString&)));
    mWaitDialog->show();
    mJobsQueue->run();
}

void Backend::populateBlogsList()
{
    kDebug();

    mBlogsMap.clear();

    ProfileConfig *profileConfig = ProfileConfig::prefs("THIS_ACCOUNT_DOESNT_EXIST");
    QStringList groupList = profileConfig->config()->groupList();
    delete profileConfig;
    profileConfig = 0; // shouldn't be used again anyway

    kDebug () << "ProfileConfigDialog::populateAccountComboBox()" <<  groupList << endl;
    groupList = groupList.filter( QRegExp("^Account_"));

    QStringListIterator groupListIterator(groupList);
    while ( groupListIterator.hasNext() ) {
        const QString accountName = groupListIterator.next(); //Contains "Account_yourAccountName"

        QString blogname(accountName);
        blogname.remove(0, 8); // Contains "ACCOUNT_"

        BlogServer blog;

        blog.setBlogname( blogname );
        blog.setUsername( ProfileConfig::prefs(accountName)->user() );
        blog.setUrl( ProfileConfig::prefs(accountName)->url());
        blog.setType( ProfileConfig::prefs(accountName)->type() );
        blog.setBlogId( ProfileConfig::prefs(accountName)->blogId() );

        kDebug() << "Added blog to the mBlogsMap: " << blogname << endl
        << "Username=" << blog.username() << endl
        << "url=" << blog.url() << endl
        << "api=" << blog.type() << endl;
        kDebug() << "password=" << blog.password() << endl;

        mBlogsMap.insert(blogname, blog);
    }
}

void Backend::listPosts( const QString& blogname, int downloadCount, QWidget *caller)
{
    kDebug() << blogname;
    if (!caller) 
        caller = mMainWindow;
    QVariant n(downloadCount);
    BlogServer blog = mBlogsMap[blogname];
    kDebug() << "blogname of blog is" << blog.blogname() << endl;

    BlogJob *job = createBlogJob(blog,BlogJob::LIST_RECENT_POSTS,n);
    connect( job, SIGNAL(listedRecentPosts(const QList<KBlog::BlogPost>&)),
             this, SLOT(slotListedRecentPosts(const QList<KBlog::BlogPost>& )));

    mJobToWidget[job]=caller;
    mJobToBlogname[job]=blog.blogname();
    if (mJobsQueue->addJob(job)){
        runQueuedJobs();
    }
}

void Backend::listCategories(const QString& blogname, QWidget *caller)
{
    kDebug();
    BlogServer blog = mBlogsMap[blogname];
    if (!caller)
        caller = mMainWindow;
    QVariant n(0);
    BlogJob *job = createBlogJob(blog,BlogJob::LIST_CATEGORIES,n);
    connect( job, SIGNAL(listedCategories(const QList<QMap<QString, QString> >&)),
             this, SLOT(slotListedCategories(const QList<QMap<QString, QString> >& )));

    
    mJobToWidget[job]=caller;
    if ( mJobsQueue->addJob(job) ) {
        runQueuedJobs();
    }
}

void Backend::sendPost( Post* kbPost, QWidget *caller )
{
    kDebug();
    Q_ASSERT(kbPost);

    if ( kbPost->status() != KBlog::BlogPost::New ) {
        kError() << "This post is already uploaded, is this a bug?" << endl;
        return;
    }

    if (!caller)
       caller = mMainWindow;

    kDebug() << "\tBlog name:" << kbPost->getBlogName();
    kDebug() << "\tPost Title:" << kbPost->title() << endl;
    kDebug() << "\tContent:" << kbPost->content() << endl;
    kDebug() << "\tCategories:" << kbPost->categories().join(", ") << endl;
    kDebug() << "\tDateTime:" <<  kbPost->creationDateTime().dateTime().toString() << endl;

    if ( kbPost->hasPendingMedia() ) {
        KMessageBox::sorry( caller ,
                     i18n("This post cannot be uploaded. Upload its media first"));
        return;
    }

    QVariant n;
    n.setValue(static_cast<KBlog::BlogPost*>(kbPost));

    BlogServer blog = mBlogsMap[kbPost->getBlogName()];
    BlogJob *job;
    QString statusBarMsg;
    if ( QString(kbPost->postId()).isEmpty() ) {
        job = createBlogJob(blog,BlogJob::CREATE_POST,n);
        statusBarMsg = i18n("New post upload enqueued in the job list.");
    } else {
        job = createBlogJob(blog,BlogJob::MODIFY_POST,n);
        statusBarMsg = i18n("Modified post upload enqueued in the job list.");
    }

    mJobToWidget[job]=caller;
    mMoveDraftToSent[job]=kbPost;
    if ( mJobsQueue->addJob(job) )
        emit statusBarMessage( statusBarMsg );
    else
        emit statusBarMessage( i18n("It is Impossible to enqueue this new post.") );
}

void Backend::fetchPost(Post* kbPost, QWidget *caller)
{
    kDebug();
    Q_ASSERT(kbPost);
    if (!caller)
       caller = mMainWindow;

    QString id = kbPost->postId();
    if (id.isEmpty()){
        kDebug() << "The id is empty.";
        return;
    }

    QVariant n;
    n.setValue(static_cast<KBlog::BlogPost*>(kbPost));
    BlogServer blog = mBlogsMap[kbPost->getBlogName()];
    BlogJob *job = createBlogJob(blog,BlogJob::FETCH_POST,n);

    mJobToWidget[job]=caller;
    mJobsQueue->addJob(job);
}

void Backend::removePost( Post *kbPost, QWidget *caller )
{
    kDebug();
    Q_ASSERT(kbPost);
    if (!caller)
        caller = mMainWindow;

    if ( kbPost->hasPendingMedia() ) {
        KMessageBox::sorry( caller ,
                     i18n("This post cannot be uploaded. Upload its media first"));
        return;
    }

    QString id = kbPost->postId();
    if (id.isEmpty()){
        kDebug() << "The id is empty.";
        return;
    }

    QVariant n;
    n.setValue(static_cast<KBlog::BlogPost*>(kbPost));

    BlogServer blog = mBlogsMap[kbPost->getBlogName()];
    BlogJob *job = createBlogJob(blog,BlogJob::REMOVE_POST,n);
    mRemoveFromTrash[job]=kbPost;
    mJobToWidget[job]=caller;
    if ( mJobsQueue->addJob(job) ) {
        runQueuedJobs();
    }
}

void Backend::sendMedia( Media *kbMedia, QWidget *caller )
{
    kDebug();
    Q_ASSERT(kbMedia);
    if (!caller)
        caller = mMainWindow;

//     if ( kbMedia->hasPendingMedia() ) {
//         KMessageBox::sorry( caller ,
//                      i18n("This post cannot be uploaded. Upload its media first"));
//         return;
//     }

    KBlog::BlogMedia *media = static_cast<KBlog::BlogMedia*>(kbMedia);
    QVariant n;
    n.setValue(media);
    BlogServer blog = mBlogsMap[ kbMedia->getBlogname() ];
    BlogJob *job = createBlogJob(blog,BlogJob::CREATE_MEDIA,n);

    mJobToWidget[job]=caller;
    mAddMedia[job]=media;
    if ( mJobsQueue->addJob(job) )
        emit statusBarMessage( i18n("New Media upload enqueued in the job list.") );
    else
        emit statusBarMessage( i18n("It is impossible to enqueue this new media.") );
}

void Backend::sync()
{
    kDebug();
    mJobsQueue->clear();
    ItemsManager::self()->enqueueMediaToUpload();
    ItemsManager::self()->enqueuePostsToUpload();
    runQueuedJobs();
}

void Backend::slotStopQueue()
{
    kDebug();
    mJobsQueue->stop();
    mJobsQueue->clear();
    statusBarMessage( i18n("All jobs have been removed from the queue.") );
}

void Backend::slotStatusBarMessage( const QString& message )
{
    kDebug();
    emit statusBarMessage(message);
}

void Backend::slotListedCategories( const QList<QMap<QString, QString> >& categories )
{
    kDebug();
    emit listedCategories(categories);
}

void Backend::slotListedRecentPosts( const QList<KBlog::BlogPost>& posts )
{
    kDebug();
    QString blogname = mJobToBlogname[qobject_cast<KJob*>(sender())];
    mJobToBlogname.remove(qobject_cast<KJob*>(sender()));
    for (int i = 0; i < posts.size(); ++i) {
        KBlogger::Post *post = new KBlogger::Post(&posts.at(i));
        post->setBlogName( blogname );
        if ( post->title().isEmpty() ) {
            QString title = i18n("Untitled post of %1", KDateTime::currentUtcDateTime().toString()) ;
            post->setTitle(title);
        }
        kDebug() 
        << "blogname=" << blogname
        << " title=" << post->title()
        << " status=" << post->status();
        post->setPrivate( false ); //NOTE: Metaweblog dooesn't return published var! :( // TODO sure?
        ItemsManager::self()->addToSentList(static_cast<KBlogger::Post*>(post));
    }
    ItemsManager::self()->getUncachedMediaInThePosts( blogname );
}

void Backend::slotJobsDone()
{
    kDebug();
    if(mWaitDialog) mWaitDialog->close();
    mWaitDialog = 0;
    emit statusBarMessage(i18n("All Jobs Done."));
    emit jobsFinished();
}

void Backend::slotJobDone(KJob* job)
{
    kDebug();
    if(mMoveDraftToSent.contains(job)){
        ItemsManager::self()->moveDraftToSentList(mMoveDraftToSent[job]);
        mMoveDraftToSent.remove(job);
    }

    if(mRemoveFromTrash.contains(job)){
        ItemsManager::self()->removeTrashedPost(mRemoveFromTrash[job]);
        mRemoveFromTrash.remove(job);
    }

    if(mAddMedia.contains(job)){
        ItemsManager::self()->addMedia(static_cast<KBlogger::Media*>(mAddMedia[job]));
        mAddMedia.remove(job);
    }

//    delete job; //TODO handle job deletion, once they don't delete themselves anymore (4.1)
    mWaitDialog->jobDone();
}

void Backend::slotJobsStopped()
{
    kDebug();
    if(mWaitDialog) mWaitDialog->close();
    mWaitDialog = 0;
}

void Backend::slotCancelClicked()
{
    kDebug();
    mWaitDialog = 0;
    mJobsQueue->clear();
}

void Backend::slotError( const QString& message )
{
    kDebug();
    if(mWaitDialog) mWaitDialog->close();
    mWaitDialog = 0;
    mJobsQueue->clear();
    KMessageBox::error(mMainWindow , message); 
    emit errorMessage(message);
}

BlogServer Backend::blogServer(const QString& blogname)
{
    kDebug() << "BlogName:" << blogname 
             << "Blog stored in the Map" << mBlogsMap.count();

    return mBlogsMap[blogname];
}

QList<BlogServer> Backend::blogServers()
{
    kDebug();
    return mBlogsMap.values();
}

BlogJob* Backend::createBlogJob( BlogServer &blogServer, BlogJob::BlogAction action, QVariant& arg )
{
    kDebug();
    kDebug() << "BlogServer blogname=" << blogServer.blogname() << endl;
    QList<QHash<QString,QVariant> > blogList = BlogList();
    QHash<QString,QVariant> blogHash = blogList[blogServer.type()];
    QString api = blogHash["api"].toString();
    KUrl url = blogServer.url();
    BlogJob* job = 0;
    if( api == "Blogger1" )
        job = new BlogJob( BlogJob::BLOGGER1, url, action, arg, this );
    if( api == "MetaWeblog" )
        job = new BlogJob( BlogJob::METAWEBLOG, url, action, arg, this );
    if( api == "MovableType" )
        job = new BlogJob( BlogJob::MOVABLETYPE, url, action, arg, this );
    if( api == "WordpressBuggy" )
        job = new BlogJob( BlogJob::WORDPRESSBUGGY, url, action, arg, this );
    if( api == "GData" )
        job = new BlogJob( BlogJob::GDATA, url, action, arg, this );
    if( job == 0 ){
        kError() << "Could no match api: " << api;
        return 0;
    }
    job->setUsername( blogServer.username() );
    job->setPassword( blogServer.password() );
    job->setBlogId( blogServer.blogId() );
    return job;
}

} //namespace

#include "backend.moc"

