/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "composer/composereditor.h"

#include <kurl.h>
#include <ktoolbar.h>
#include <kaction.h>
#include <ktoggleaction.h>
#include <kactioncollection.h>
#include <kactionmenu.h>
#include <ktextbrowser.h>
#include <khtml_part.h>
#include <khtmlview.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <kcolorscheme.h>

#include <QLayout>

#include "composer/linkdialog.h"
#include "media/uploadmediadialog.h"
#include "composer/htmlexporter.h"
#include "composer/composer.h"
#include "media/media.h"
#include "composer/visualeditor.h"
#include "media/medialistview.h"
#include "composer/weblogstylegetter.h"

namespace KBlogger
{

ComposerEditor::ComposerEditor(Composer *composer, QWidget *parent) :
        QWidget(parent),
        mComposer(composer),
        mVisualTextEditor(0),
        mWeblogStyleGetter(0),
        mMediaEnabled(false),
        prevPage(0)
{
    setupUi( this );

    kDebug();

    //Add a VisualEditor ActionCollection and Toolbar.
    mActionCollection = new KActionCollection ( this );
    mToolBar = new KToolBar ( visualPageWidget );
    mToolBar->setIconDimensions(16);
    mToolBar->setToolButtonStyle(Qt::ToolButtonIconOnly);
    visualPageWidget->layout()->addWidget ( mToolBar );
    createActions();

    //Add VisualEditor
    mVisualTextEditor = new VisualEditor(visualPageWidget);
    visualPageWidget->layout()->addWidget ( mVisualTextEditor );

    connect( mVisualTextEditor, SIGNAL( currentFontChanged( const QFont & ) ),
             this, SLOT( updateCharFmt() ) );
    connect( mVisualTextEditor, SIGNAL( droppedMedia( const QString & ) ),
             this, SLOT( showUploadMediaDialog( const QString& ) ) );

    updateCharFmt();

    //KHTML, Posts Preview Widget
    mPreviewBrowser = new KHTMLPart(previewPageWidget);
    previewPageWidget->layout()->addWidget(mPreviewBrowser->view());
    mToolBarPreview = new KToolBar ( previewPageWidget );
    previewPageWidget->layout()->addWidget ( mToolBarPreview );

    mPreviewBrowser->view()->show();
    mPreviewBrowser->setJScriptEnabled(true);
    mPreviewBrowser->setJavaEnabled(false);
    mPreviewBrowser->setMetaRefreshEnabled(false);
    mPreviewBrowser->setPluginsEnabled(true);
    mVisualTextEditor->setCheckSpellingEnabled ( true );

    connect ( composerTabWidget, SIGNAL ( currentChanged ( int) ),
              this, SLOT ( syncEditors (int) ) );
    connect ( getStyleButton, SIGNAL( clicked() ),
              this, SLOT( getStyle() ) );

}

ComposerEditor::~ComposerEditor()
{
    kDebug();
}

void ComposerEditor::enableMedia( bool enabled )
{
    kDebug();
    mMediaEnabled = enabled;
}

void ComposerEditor::createActions()
{
    kDebug();
    //Add Bold Action
    KLocalizedString boldKLS = ki18nc("a button to make the text in the editor bold or make bold text non-bold.", "bold" );
    boldAction = new KToggleAction (KIcon("format-text-bold"), boldKLS.toString(), this);
    mActionCollection->addAction("bold", boldAction);
    connect(boldAction, SIGNAL(triggered()), this, SLOT(toggleBold()));
    mToolBar->addAction(boldAction);

    //Add Italic Action
    KLocalizedString italicKLS = ki18nc( "a button to make the text in the editor italic or make italic text non-italic.", "italic");
    italicAction = new KToggleAction (KIcon("format-text-italic"), italicKLS.toString(), this);
    mActionCollection->addAction("italic", italicAction);
    connect(italicAction, SIGNAL(triggered()), this, SLOT(toggleItalic()));
    mToolBar->addAction(italicAction);

    //Add Underline Action
    underlineAction = new KToggleAction (KIcon("format-text-underline"), i18n("underline"), this);
    mActionCollection->addAction("underline", underlineAction);
    connect(underlineAction, SIGNAL(triggered()), this, SLOT(toggleUnderline()));
    mToolBar->addAction(underlineAction);

    //Add Strikeout Tag Action
    strikeoutAction = new KToggleAction (KIcon("format-text-strikethrough"), i18n("strikeout"), this);
    mActionCollection->addAction("strikeout", strikeoutAction);
    connect(strikeoutAction, SIGNAL(triggered()), this, SLOT(toggleStrikeout()));
    mToolBar->addAction(strikeoutAction);

    //FontWeight Menu & Actions
    fontSizeMoreAction = new KAction (KIcon("format-font-size-more"),
                                      i18n("font size more"), this);
    mActionCollection->addAction("format-font-size-more", fontSizeMoreAction);
    connect(fontSizeMoreAction, SIGNAL( triggered()), this, SLOT( fontSizeMore() ));
    mToolBar->addAction(fontSizeMoreAction);

    fontSizeLessAction = new KAction (KIcon("format-font-size-less"),
                                      i18n("font size less"), this);
    mActionCollection->addAction("format-font-size-less", fontSizeLessAction);
    connect(fontSizeLessAction, SIGNAL(triggered()), this, SLOT( fontSizeLess() ));
    mToolBar->addAction(fontSizeLessAction);

    //Add HTML TAG <Code> Action
    codeAction = new KAction (KIcon("text-x-generic"), i18n("code"), this);
    mActionCollection->addAction("code", codeAction);
    connect(codeAction, SIGNAL(triggered()), this, SLOT(toggleCode()));
    mToolBar->addAction(codeAction);

    //Add Link Action
    addLinkAction = new KAction (KIcon("bookmark-new"), i18n("add link"), this);
    mActionCollection->addAction("add_link", addLinkAction);
    connect(addLinkAction, SIGNAL(triggered()), this, SLOT(showLinkDialog()));
    mToolBar->addAction(addLinkAction);

    //Add Media Actions
    insertMediaMenu = new KActionMenu (KIcon("insert-image"), i18n("insert media"), this);
    insertMediaMenu->setDelayed( false );
    mToolBar->addAction(insertMediaMenu);

    newMediaAction = new KAction (KIcon("document-new"), i18n("New media from file..."), this);
    mActionCollection->addAction("addMedia", newMediaAction);
    connect(newMediaAction, SIGNAL(triggered()), this, SLOT(showUploadMediaDialog()));
    insertMediaMenu->addAction(newMediaAction);

    addMediaAction = new KAction (KIcon("view-preview"), i18n("Media"), this);
    mActionCollection->addAction("addMedia", addMediaAction);
    connect(addMediaAction, SIGNAL(triggered()), this, SLOT( showMediaList()));
    insertMediaMenu->addAction(addMediaAction);

    //Add Html Tag Action
    addHtmlAction = new KAction (KIcon("text-html"), i18n("add HTML code"), this);
    mActionCollection->addAction("addHtmlCode", addHtmlAction);
    connect(addHtmlAction, SIGNAL(triggered()), this, SLOT(showHtmlDialog()));
    mToolBar->addAction(addHtmlAction);

    //Visual Editor Spell Checker Action
    spellCheckAction = new KAction (KIcon("tools-check-spelling"), i18n("check spell"), this);
    mActionCollection->addAction("checkSpell", addLinkAction);
    connect(spellCheckAction, SIGNAL(triggered()), this, SLOT(spellchecking()));
    mToolBar->addAction(spellCheckAction);
}

void ComposerEditor::updateCharFmt()
{
    kDebug();
    boldAction->setChecked( mVisualTextEditor->fontWeight() >= QFont::Bold );
    italicAction->setChecked( mVisualTextEditor->fontItalic() );
    underlineAction->setChecked( mVisualTextEditor->fontUnderline() );
    strikeoutAction->setChecked( mVisualTextEditor->currentFont().strikeOut() );
}

void ComposerEditor::spellchecking()
{
    mVisualTextEditor->checkSpelling();
}

void ComposerEditor::toggleItalic()
{
    mVisualTextEditor->setFontItalic ( !mVisualTextEditor->fontItalic() );
}

void ComposerEditor::toggleBold()
{
    kDebug();

    if ( mVisualTextEditor->fontWeight() >= QFont::Bold )
        mVisualTextEditor->setFontWeight( QFont::Normal );
    else
        mVisualTextEditor->setFontWeight( QFont::Bold );
}

void ComposerEditor::toggleUnderline()
{
    mVisualTextEditor->setFontUnderline ( !mVisualTextEditor->fontUnderline() );
}

void ComposerEditor::toggleStrikeout()
{
    QFont fontInVisual( mVisualTextEditor->currentFont() );
    fontInVisual.setStrikeOut(!fontInVisual.strikeOut());
    mVisualTextEditor->setCurrentFont(fontInVisual);
}

void ComposerEditor::toggleCode()
{
    static QString preFontFamily;
    if ( mVisualTextEditor->fontFamily() != "Courier New,courier" ) {
        preFontFamily = mVisualTextEditor->fontFamily();
        mVisualTextEditor->setFontFamily("Courier New,courier");
    } else {
        mVisualTextEditor->setFontFamily(preFontFamily);
    }
}

void ComposerEditor::fontSizeMore()
{
    kDebug();

    QTextCharFormat format = mVisualTextEditor->currentCharFormat ();

    int idx = format.intProperty(QTextFormat::FontSizeAdjustment);
    if ( idx < 3 ) {
        format.setProperty ( QTextFormat::FontSizeAdjustment, QVariant( ++idx ));
        mVisualTextEditor->setCurrentCharFormat (format);
    }
}

void ComposerEditor::fontSizeLess()
{
    kDebug();
    QTextCharFormat format = mVisualTextEditor->currentCharFormat ();

    int idx = format.intProperty(QTextFormat::FontSizeAdjustment);
    if ( idx > -1 ) {
        format.setProperty ( QTextFormat::FontSizeAdjustment, QVariant( --idx ));
        mVisualTextEditor->setCurrentCharFormat (format);
    }
}

void ComposerEditor::showUploadMediaDialog(const QString& filename )
{
    kDebug();
    if(!mMediaEnabled){
         kDebug() << "you can't add media with this api.";
         KMessageBox::sorry ( this, i18n( "File uploading is not possible on this blog." ) );
         return;
    }
    Q_ASSERT(mComposer );
    new UploadMediaDialog(filename, mComposer->blogname(), this);
}

void ComposerEditor::showMediaList()
{
    kDebug();
    if(!mMediaEnabled){
         kDebug() << "you can't add media with this api.";
         KMessageBox::sorry ( this, i18n( "File uploading is not possible on this blog." ) );
         return;
    }
    new MediaListView(mComposer->blogname(), this);
}

void ComposerEditor::insertMedia(Media* media)
{
    kDebug();
    if (!media) return;
    QString name = media->name();
    QString target;
    kDebug() << "media->url(): " << media->url();
    if ( media->url().isValid() ) {
        //it's an uploaded file
        target = media->url().url();
    } else {
        //it's a local file
        target = media->cachedFilename();
    }

    if ( !media->mimetype().contains("image") ) {
        addLink(target, name);
    } else {
        QTextCursor cursor = mVisualTextEditor->textCursor();
        cursor.insertImage(target);
    }
}

void ComposerEditor::insertThumbnail(Media* thumbnail, Media* image)
{
    kDebug();
    if(!thumbnail || !image) return;
    QString imageFilename, thumbnailFilename;
    if ( image->url().isValid() ) {
        //it's an uploaded image
        thumbnailFilename = thumbnail->url().url();
        imageFilename = image->url().url();
    } else {
        //it's a local image
        thumbnailFilename = thumbnail->cachedFilename();
        imageFilename = image->cachedFilename();
    }
    mVisualTextEditor->insertHtml( "<a href="+imageFilename+"><img src="
                                                   +thumbnailFilename+"/></a>" );
}

void ComposerEditor::showLinkDialog()
{
    QTextCharFormat currentCharFormat(mVisualTextEditor->textCursor().charFormat());
    QString selectedText, ancorHref;
    selectedText = mVisualTextEditor->textCursor().selectedText();
    ancorHref = currentCharFormat.anchorHref();
    kDebug() << "ComposerEditor::showLinkDialog()" << selectedText << endl;

    mLinkDialog = new LinkDialog( this );
    connect ( mLinkDialog, SIGNAL ( addLink ( const QString&, const QString& ) ),
              this, SLOT ( addLink ( const QString&, const QString& ) ) );    

    if ( selectedText.isEmpty() ) { //Add a link at the cursor position
        mLinkDialog->setLinkName (QString());
        mLinkDialog->show();
        return;
    } else {
        if ( currentCharFormat.isAnchor() ) {
            currentCharFormat.setAnchor(false);
            currentCharFormat.setUnderlineStyle(QTextCharFormat::NoUnderline);
            currentCharFormat.clearForeground();
            mVisualTextEditor->textCursor().setCharFormat(currentCharFormat);
        } else {
            mLinkDialog->setLinkName ( selectedText );
            mLinkDialog->show();
        }
    };
}

void ComposerEditor::addLink ( const QString& target, const QString& name ) //TODO add Edit Link
{
    kDebug() << "target" << target << "name" << name;
    //insertTag (target,name,0,link);
    QTextCharFormat currentCharFormat( mVisualTextEditor->textCursor().charFormat() );
    if (target.isEmpty()) { //remove link
        currentCharFormat.setAnchor(false);
        currentCharFormat.setUnderlineStyle(QTextCharFormat::NoUnderline);
        currentCharFormat.clearForeground();
    } else { //set link
        kDebug();
        currentCharFormat.setAnchor(true);
        currentCharFormat.setAnchorHref(target);
        currentCharFormat.setUnderlineStyle(QTextCharFormat::SingleUnderline);
        currentCharFormat.setForeground(KColorScheme(QPalette::Active, KColorScheme::Window).foreground(KColorScheme::LinkText).color());
        if ( name != mVisualTextEditor->textCursor().selectedText() )
            mVisualTextEditor->textCursor().insertText(name, currentCharFormat);
    }
    mVisualTextEditor->textCursor().setCharFormat(currentCharFormat);
}


void ComposerEditor::showHtmlDialog()
{
    QString code;
    QRegExp testRegExp("(<[^<]*>)([^<]*)(</[^<]*>)");
    QValidator* validator = new QRegExpValidator(testRegExp, this);
    code = KInputDialog::getText(i18n("KBlogger - Visual Editor"), i18n("Insert here your HTML code:"), "<tag>"+i18n("text")+"</tag>", 0, this, validator);
    mVisualTextEditor->insertHtml(code);
}

QString ComposerEditor::htmlToRichtext(const QString& html)
{
    kDebug() << "in" << html;
    QString richText = html;

    richText.remove(QChar('\n'));

    richText.replace(QRegExp("<del>(.*)</del>"), "<s>\\1</s>");

    //Note: Qt Error:
    //      QDocument converts <h1> in [ font-size: xx-large + bold ]
    //      and font-size: xx-large in <h1>
    richText.replace("<h1>", "<span style=\"font-size: xx-large\" >");
    richText.replace("<h2>", "<span style=\"font-size: x-large\" >");
    richText.replace("<h3>", "<span style=\"font-size: large\" >");
    //richText.replace("<h4>", "<span style=\"font-size: medium\" >");
    richText.replace(QRegExp("<h4>(.*)</h4>"), "\\1");
    richText.replace("<h5>", "<span style=\"font-size: small\" >");
    richText.replace(QRegExp("</h[1-5]>"), "</span>");

    //kDebug() << "out" << richText;
    //return richText;
    QString h;
    QString basePath = Media::cachePath();
    h = "<html><head><base href=\"" + basePath + "\" /></head>" +
        "<body><p>" + richText + "</p></body></html>";
    kDebug() << "out" << h;
    return h;
}

void ComposerEditor::syncEditors(int i)
{
    kDebug();

    QWidget* mTab = composerTabWidget->widget(i);
    // TODO remove if we know how to replace it, toHtml() creates a mess ( a lot of css )
    // toHtml() does not work with drupal
    // this is incompatible with licensing policy (it has gplv2 only)
    htmlExporter* htmlExp = new htmlExporter();
    if (!mTab) {
        if (visualPageWidget->isVisible())
            contentTextHtmlEditor->setPlainText( htmlExp->toHtml(mVisualTextEditor->document()) );
    }

    if (mTab == visualPageWidget && prevPage == 1) // HTML Editor =>VISUAL composer
        mVisualTextEditor->setHtml(htmlToRichtext(contentTextHtmlEditor->toPlainText()));
    else
        if (mTab == htmlPageWidget && prevPage == 0) // Visual Editor => HTML composer.
            contentTextHtmlEditor->setPlainText( htmlExp->toHtml(mVisualTextEditor->document()) );
        else
            if (mTab == previewPageWidget) { // HTML Editor =>Preview  & Visual Editor => Preview
                if ( prevPage == 0 )
                    contentTextHtmlEditor->setPlainText( htmlExp->toHtml(mVisualTextEditor->document()) );
                setHtmlPreview( i18n("Post Preview"),
                                contentTextHtmlEditor->toPlainText() );
            }
    prevPage = i;
    delete htmlExp;
}

void ComposerEditor::getStyle()
{
    kDebug();
    delete mWeblogStyleGetter;
    mWeblogStyleGetter = new WeblogStyleGetter(mComposer->blogname(), this);
    connect( mWeblogStyleGetter, SIGNAL( styleFetched() ), this, SLOT( reloadHtmlPreview()));
}

void ComposerEditor::reloadHtmlPreview()
{
    kDebug();
    setHtmlPreview( i18n("Post Preview"), contentTextHtmlEditor->toPlainText() );
}

void ComposerEditor::setHtmlPreview(const QString& title, const QString& content)
{
    kDebug();
    if (!mPreviewBrowser) return;
    mPreviewBrowser->begin();
    mPreviewBrowser->openStream( "text/html", KUrl() );
    QString html;
    if (!mComposer->blogname().isEmpty()) {
        html = WeblogStyleGetter::styledHtml(mComposer->blogname(), title, content);
    }
    mPreviewBrowser->writeStream(html.toAscii() );
    mPreviewBrowser->closeStream();

}

} //namespace

#include "composereditor.moc"

