/***************************************************************************
 *   Copyright (C) 2005 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PREDICTORDATA_H
#define PREDICTORDATA_H

#include <qdatetime.h>
#include <qstring.h>
#include <qstringlist.h>

enum PredictorAppType {MFOLD, CHARMM};

enum PredictorAminoAcid
  {GLY, ALA, SER, CYS, VAL, THR, ILE, PRO, MET, ASP,
   ASN, LEU, LYS, GLU, GLN, ARG, HIS, PHE, TYR, TRP,
   AminoAcids};

struct PredictorBurials
{
  double avg[AminoAcids+1][AminoAcids+1];
  double sdev[AminoAcids+1][AminoAcids+1];
  unsigned pairs[AminoAcids+1][AminoAcids+1];
  
  bool parse(const QStringList &lines);
};

struct PredictorECovers24
{
  double value[AminoAcids][25];
  
  bool parse(const QStringList &lines);
};

struct PredictorProfile3
{
  double value[AminoAcids][5][5][5];
  
  bool parse(const QStringList &lines);
};

struct PredictorQuasi3
{
  double par[2][AminoAcids][AminoAcids];
  double mid[2][AminoAcids][AminoAcids];
  double ant[2][AminoAcids][AminoAcids];
  
  bool parse(const QStringList &lines);
};

struct PredictorScale3B
{
  PredictorAminoAcid aa[3];
  unsigned count[3];
  double value;
  
  bool parse(const QString &line);
};

struct PredictorS1234
{
  double r1_2[AminoAcids][AminoAcids][3];
  double r1_3[AminoAcids][AminoAcids][4];
  double r1_4[AminoAcids][AminoAcids][14];
  double r1_5[AminoAcids][AminoAcids][7];
  
  bool parse(const QStringList &lines);
};

struct PredictorMonssterAtom
{
  unsigned x, y, z;
  
  bool parse(const QString &line);
};

struct PredictorMonssterInput
{
  unsigned random,        // random seed
           ncycle,        // number of cycles per temperature step
           icycle,        // Monte Carlo moves per cycle
           tsteps,        // temperature steps
           resmin,        // first fragment residue (fixed template)
           resmax;        // last fragment residue (fixed template)
  double temp[2],         // temperature range
         softcore,        // soft core radius for large residues
         central,         // centrosymmetric potential scaling
         stiff,           // general stiffness potential scaling
         pair,            // pair potential scaling,
         kdcore,          // hydrophobic core potential scaling
         hbond,           // H-bond cooperativity potential scaling
         shrt,            // short scale potential scaling
         burial,          // burial potential scaling
         multibody,       // multi body potential scaling
         threebody;       // three body potential scaling
  
  bool parse(const QStringList &lines);
};

struct PredictorMonssterResidue
{
  unsigned resSeq;
  PredictorAminoAcid resName;
  unsigned count[2];
  
  bool parse(const QString &line);
};

struct PredictorMonssterSeq
{
  QValueList<PredictorMonssterResidue> groups;
  unsigned atoms;
  
  bool parse(const QStringList &lines);
  
  QString toString() const;
};

struct PredictorMonssterRestraint
{
  unsigned num;
  double value;
  
  bool parse(const QString &line);
};

struct PredictorAtomPDB
{
  unsigned serial;              // Atom serial number
  struct {
    unsigned remoteness,        // Distance from the alpha carbon
             branch;            // Branch
    QString iupac;              // Atom name in IUPAC format
  } name;
  QChar altLoc;                 // Alternate location indicator
  PredictorAminoAcid resName;   // Residue name
  QChar chainID;                // Chain identifier
  unsigned resSeq;              // Residue sequence number
  QChar iCode;                  // Code for insertion of residues
  double x,                     // Orthogonal coordinates for X in Angstroms
         y,                     // Orthogonal coordinates for Y in Angstroms
         z,                     // Orthogonal coordinates for Z in Angstroms
         occupancy,             // Occupancy
         tempFactor;            // Temperature factor
  QString segID,                // Segment identifier
          element,              // Element symbol
          charge;               // Charge on the atom
  
  bool parse(const QString &line);
};

bool operator<(const PredictorAtomPDB &a1, const PredictorAtomPDB &a2);

enum PredictorHelixClass
{ RightHandedApha=1, RightHandedOmega, RightHandedPi, RightHandedGamma, RightHanded310,
  LeftHandedAlpha, LeftHandedOmega, LeftHandedGamma, Ribbon27, Polyproline };

struct PredictorHelixPDB
{
  unsigned serNum;                // Helix serial number
  QString helixID;                // Helix identifier
  struct {
    PredictorAminoAcid resName;   // Name of the residue
    QChar chainID;                // Chain identifier
    unsigned seqNum;              // Sequence number
    QChar iCode;                  // Insertion code
  } init, end;
  PredictorHelixClass helixClass; // Helix class
  QString comment;                // Comment about the helix
  unsigned length;                // Length of the helix
  
  bool parse(const QString &line);
};

bool operator<(const PredictorHelixPDB &h1, const PredictorHelixPDB &h2);

struct PredictorSheetPDB
{
  unsigned strand;                // Strand number
  QString sheetID;                // Sheet identifier
  unsigned numStrands;            // Number of strands in sheet
  struct {
    PredictorAminoAcid resName;   // Residue name
    QChar chainID;                // Chain identifier
    unsigned seqNum;              // Residue sequence number
    QChar iCode;                  // Insertion code
  } init, end;
  int sense;                      // Sense (0=first, 1=parallel, -1=anti-parallel
  struct {
    QString atom;                 // Atom
    PredictorAminoAcid resName;   // Residue name
    QChar chainId;                // Chain identifier
    unsigned resSeq;              // Residue sequence number
    QChar iCode;                  // Insertion code
  } curr, prev;                   // Registration
  
  bool parse(const QString &line);
};

bool operator<(const PredictorSheetPDB &s1, const PredictorSheetPDB &s2);

struct PredictorTurnPDB
{
  unsigned seq;                   // Turn number
  QString turnID;                 // Turn identifier
  struct {
    PredictorAminoAcid resName;   // Name of the residue
    QChar chainID;                // Chain identifier
    unsigned seqNum;              // Sequence number
    QChar iCode;                  // Insertion code
  } init, end;
  QString comment;                // Comment about the turn
  
  bool parse(const QString &line);
};

bool operator<(const PredictorTurnPDB &t1, const PredictorTurnPDB &t2);

struct PredictorProteinPDB
{
  unsigned groups;
  QValueList<PredictorAtomPDB> atoms;
  QValueList<PredictorHelixPDB> helix;
  QValueList<PredictorSheetPDB> sheet;
  QValueList<PredictorTurnPDB> turn;

  bool parse(const QStringList &lines);

  QString toString() const;
};

struct PredictorProteinNOE
{
  struct {
    unsigned index;
    QString name;
  } select[2];
  double kmin,
         rmin,
         kmax,
         rmax;
  
  bool parse(const QString &line);
};

struct PredictorMFold
{
  PredictorBurials burials;
  PredictorECovers24 ecovers_24;
  PredictorProfile3 profile3;
  PredictorQuasi3 quasi3;
  QValueList<PredictorScale3B> scale3b;
  PredictorS1234 s1234, s1234h, s1234e;
  struct {
    QValueList<PredictorMonssterAtom> init_chain;
    PredictorMonssterInput input;
    PredictorMonssterSeq seq;
    QValueList<PredictorMonssterRestraint> restraints;
    struct {
      QValueList<PredictorMonssterAtom> chain;
      PredictorProteinPDB pdb;
      QValueList<PredictorProteinNOE> noe;
    } final;
  } monsster;
};

struct PredictorSeed
{
  unsigned stream;
  
  bool parse(const QStringList &lines);
};

struct PredictorCharmmInp
{
  unsigned ntemps,
           nsteps;
  struct {
    unsigned low,
             high;
  } t;

  bool parse(const QStringList &lines);
};

struct PredictorCharmm
{
  PredictorCharmmInp inp;
  struct {
    PredictorProteinPDB pdb, final_pdb;
    QValueList<PredictorProteinNOE> noe;
    unsigned residuals;
  } protein;
  struct {
    unsigned stream;
  } seed;
};

struct PredictorResult
{
  PredictorAppType app_type;
  PredictorMFold mfold;
  PredictorCharmm charmm;
};

struct PredictorMonssterRestart
{
  unsigned line1a[2];
  double line1b[5];
  unsigned line2[3];
  double line3[2];
  QValueList<PredictorMonssterAtom> chain;
  
  bool parse(const QStringList &lines);
};
  
struct PredictorState
{
  PredictorAppType app_type;
  PredictorMonssterRestart monsster_restart;
};

#endif
