/***************************************************************************
                          kcoverpage.cpp  -  description
                             -------------------
    begin                : Sat Apr 15 2000
    copyright            : (C) 2000 by Pascal 'PeP' Panneels
    email                : pepouille@skynet.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kcoverpage.h>
#include <qcolor.h>
#include <qpainter.h>

KCoverPage::KCoverPage()
{
	NumberOfObjects=0;
	PageToPrint = false;
	PageColor = Qt::white;
	IsRoundPage = false;
}

KCoverPage::~KCoverPage()
{
	Clear();
}

KCoverObject * KCoverPage::GetSelectedObject(QPoint &p)
{
	// look in the multi strings list, scanning from top to beyond
	for (std::list<KCoverMultiString *>::reverse_iterator i=MultiStrings.rbegin(); i!=MultiStrings.rend();i++)
		if ( (*i)->IsPointInObject(p) )
			return *i;
	
	// look in the curved strings list, scanning from top to beyond
	for (std::list<KCoverCurvedString *>::reverse_iterator i=CurvedStrings.rbegin(); i!=CurvedStrings.rend();i++)
		if ( (*i)->IsPointInObject(p) )
			return *i;
	
	// look in the strings list, scanning from top to beyond
	for (std::list<KCoverString *>::reverse_iterator i=Strings.rbegin(); i!=Strings.rend();i++)
		if ( (*i)->IsPointInObject(p) )
			return *i;
	
	// look in the images list, scanning from top to beyond
	for (std::list<KCoverImage *>::reverse_iterator i=Images.rbegin(); i!=Images.rend();i++)
		//if ( (p.x()>=(*i).X) && (p.x()<=(*i).X+(*i).Image.width()) && (p.y()>=(*i).Y) && (p.y()<=(*i).Y+(*i).Image.height()) )
		if ( (*i)->IsPointInObject(p) )
			return *i;
		
	return NULL;
}

/** add an image to the list */
void KCoverPage::AddImage(KCoverImage * img)
{
	Images.push_back(img);
	NumberOfObjects++;
}

/** return the image #i of the list */
KCoverImage * KCoverPage::GetImage(int i)
{
	std::list<KCoverImage *>::iterator j=Images.begin();
	for (; i>0; i--)
		j++;
	return (*j);
}

/** Erase the selected object */
void KCoverPage::DeleteSelectedObject(KCoverObject *obj)
{
	switch ( obj->GetClassType() )
		{
		case TYPE_CLASS_IMAGE :
			{
			KCoverImage * img = dynamic_cast<KCoverImage *>(obj);
			std::list<KCoverImage *>::iterator j;
			j = std::find(Images.begin(), Images.end(), img );
			Images.erase(j);
			NumberOfObjects--;
			} break;
		case TYPE_CLASS_STRING :
			{
			KCoverString * str = dynamic_cast<KCoverString *>(obj);
			std::list<KCoverString *>::iterator j;
			j = std::find(Strings.begin(), Strings.end(), str );
			Strings.erase(j);
			NumberOfObjects--;
			} break;
		case TYPE_CLASS_CURVED_STRING :
			{
			KCoverCurvedString * str = dynamic_cast<KCoverCurvedString *>(obj);
			std::list<KCoverCurvedString *>::iterator j;
			j = std::find(CurvedStrings.begin(), CurvedStrings.end(), str );
			CurvedStrings.erase(j);
			NumberOfObjects--;
			} break;
		case TYPE_CLASS_MULTI_STRING :
			{
			KCoverMultiString * str = dynamic_cast<KCoverMultiString *>(obj);
			std::list<KCoverMultiString *>::iterator j;
			j = std::find(MultiStrings.begin(), MultiStrings.end(), str );
			MultiStrings.erase(j);
			NumberOfObjects--;
			} break;
		}
}

/**  sort the image list */
void KCoverPage::SortImages()
{
	Images.sort();
}

/** add a string to the page */
void KCoverPage::AddString(KCoverString *str)
{
    Strings.push_back( str );
	NumberOfObjects++;
}

KCoverString * KCoverPage::GetLastString()
{
	std::list<KCoverString *>::iterator j=Strings.end();
	return *(--j);
}


/** sort the string list */
void KCoverPage::SortStrings()
{
	Strings.sort();
}

/** return the string number i */
KCoverString * KCoverPage::GetString(int i)
{
	std::list<KCoverString *>::iterator j=Strings.begin();
	for (; i>0; i--)
		j++;
	return *j;
}

/** clear the page content */
void KCoverPage::Clear()
{
	Images.erase( Images.begin(), Images.end() );
	Strings.erase( Strings.begin(), Strings.end() );
	CurvedStrings.erase( CurvedStrings.begin(), CurvedStrings.end() );
	MultiStrings.erase( MultiStrings.begin(), MultiStrings.end() );
    Background.Clear();
	
	NumberOfObjects = 0;
	PageToPrint = false;
	PageColor = Qt::white;
}

/** add a curved string to the page */
void KCoverPage::AddCurvedString(KCoverCurvedString *str)
{
	CurvedStrings.push_back( str );
	NumberOfObjects++;
}

/** sort the curved string list */
void KCoverPage::SortCurvedStrings()
{
	CurvedStrings.sort();
}

/** return the string number i */
KCoverCurvedString * KCoverPage::GetCurvedString(int i)
{
	std::list<KCoverCurvedString *>::iterator j=CurvedStrings.begin();
	for (; i>0; i--)
		j++;
	return *j;
}

/** add a multi string to the page */
void KCoverPage::AddMultiString(KCoverMultiString *str)
{
	MultiStrings.push_back(str);
	NumberOfObjects++;
}

/** sort the multi string list */
void KCoverPage::SortMultiStrings()
{
	MultiStrings.sort();
}

/** return the string number i */
KCoverMultiString * KCoverPage::GetMultiString(int i)
{
	std::list<KCoverMultiString *>::iterator j=MultiStrings.begin();
	for (; i>0; i--)
		j++;
	return *j;
}

/** save a page */
bool KCoverPage::Save(QFile &file, int pagenumber)
{
	char tmp[20];
	unsigned int nb;
	
	if ( !NumberOfObjects && !Background.ToBeSaved() )
		return false;  // page is not saved
		
	tmp[0] = TAG_PAGE;
	file.writeBlock( tmp, 1 );
	file.writeBlock( (char *)&pagenumber, sizeof(int) );
	// save the strings
	nb = Strings.size();
	file.writeBlock( (char *)&nb, sizeof(int) );
	if ( nb>0 )
		for ( std::list<KCoverString *>::iterator i=Strings.begin(); i!=Strings.end(); i++ )
			(*i)->Save(file);
	// save the mutli line strings
	nb = MultiStrings.size();
	file.writeBlock( (char *)&nb, sizeof(int) );
	if ( nb>0 )
		for ( std::list<KCoverMultiString *>::iterator i=MultiStrings.begin(); i!=MultiStrings.end(); i++ )
			(*i)->Save(file);
	// save the curved strings
	nb = CurvedStrings.size();
	file.writeBlock( (char *)&nb, sizeof(int) );
	if ( nb>0 )
		for ( std::list<KCoverCurvedString *>::iterator i=CurvedStrings.begin(); i!=CurvedStrings.end(); i++ )
			(*i)->Save(file);
	// save the images
	nb = Images.size();
	file.writeBlock( (char *)&nb, sizeof(int) );
	if ( nb>0 )
		for ( std::list<KCoverImage *>::iterator i=Images.begin(); i!=Images.end(); i++ )
			(*i)->Save(file);
	
	Background.Save(file);
	
	return true; // page is saved
}

/** load a page */
void KCoverPage::Load(QFile &file, int version)
{
	unsigned int nb;
	int err;
	
	err = file.readBlock((char *)&nb, sizeof(int));
	for (unsigned int i=0; i<nb; i++ )
		{
		KCoverString *str = new KCoverString;
		str->Load(file, version);
		AddString( str );
	    }
	
	err = file.readBlock((char *)&nb, sizeof(int));
	for (unsigned int i=0; i<nb; i++ )
		{
		KCoverMultiString *str = new KCoverMultiString;
		str->Load(file, version);
		AddMultiString( str );
	    }

	err = file.readBlock((char *)&nb, sizeof(int));
	for (unsigned int i=0; i<nb; i++ )
		{
		KCoverCurvedString *str = new KCoverCurvedString;
		str->Load(file, version);
		AddCurvedString( str );
	    }
	err = file.readBlock((char *)&nb, sizeof(int));
	for (unsigned int i=0; i<nb; i++ )
		{
		KCoverImage *str = new KCoverImage;
		str->Load(file, version);
		AddImage( str );
	    }
	
	if ( version>2 )
		Background.Load(file,version);
}

/** paint the page */
void KCoverPage::Paint(QPainter *paint, QPoint orig)
{
	
	// PeP 290601:
	Background.Paint( paint, orig+QPoint(1,1), FRONT_H-1, FRONT_V-1 );
	
	if ( IsRoundPage )
		{
		// draw the layout
		DrawLayout( paint, orig );
	    }
	
	// draw the images
	DrawImages( paint, orig );

    // draw the texts
    DrawTexts( paint, orig );
    	
    // draw the curved texts
	DrawCurvedTexts( paint, orig );
	
    // draw the multi line texts
    DrawMultiStrings( paint, orig );

    if ( !IsRoundPage )
		{
		// draw the layout
		DrawLayout( paint, orig );
        }
}

/** draw the frame border (page bordrer) */
void KCoverPage::DrawLayout(QPainter *paint, QPoint orig)
{
	// draw the layout
    paint->setPen( Qt::black );

    paint->drawLine( orig.x(), orig.y(), orig.x()+FRONT_H, orig.y() );
    paint->drawLine( orig.x(), orig.y()+FRONT_V, orig.x()+FRONT_H, orig.y()+FRONT_V );
    paint->drawLine( orig.x(), orig.y(), orig.x(), orig.y()+FRONT_V );
    paint->drawLine( orig.x()+FRONT_H, orig.y(), orig.x()+FRONT_H, orig.y()+FRONT_V );

    // fill the inner rectangle with the page color
	//paint->fillRect( orig.x()+1, orig.y()+1, FRONT_H-1, FRONT_V-1, PageColor );
	
	// draw the background
	// PeP 290601: Background.Paint( *paint, orig+QPoint(1,1), FRONT_H-1, FRONT_V-1 );
}

/** draw the images */
void KCoverPage::DrawImages(QPainter *paint, QPoint orig)
{
    if ( Images.size() )
    	{
    	for (unsigned int i=0;i<Images.size();i++)
    		{
    		KCoverImage * img = GetImage(i);
				img->Paint(  paint, orig, FRONT_H, FRONT_V );
    		}
    	}
}

/** draw the texts */
void KCoverPage::DrawTexts(QPainter *paint, QPoint orig)
{
    if ( Strings.size() )
    	{
    	for (unsigned int i=0;i<Strings.size();i++)
    		{
    		KCoverString * str = GetString(i);
    		str->Paint( paint, orig );
    		}
    	}
}

/** draw the curved texts */
void KCoverPage::DrawCurvedTexts(QPainter *paint, QPoint orig)
{
    if ( CurvedStrings.size() )
    	{
    	for (unsigned int i=0;i<CurvedStrings.size();i++)
    		{
    		KCoverCurvedString * str = GetCurvedString(i);
    		str->Paint( paint, orig );
    		}
    	}
}

/** draw the multi line texts */
void KCoverPage::DrawMultiStrings(QPainter *paint, QPoint orig)
{
    if ( MultiStrings.size() )
    	{
    	for (unsigned int i=0;i<MultiStrings.size();i++)
    		{
    		KCoverMultiString * str = GetMultiString(i);
    		str->Paint( paint, orig );
    		}
    	}
}

/** check to see if a point is in the page or not */
bool KCoverPage::IsPointInPage(QPoint ClickedPos)
{
	return ( ClickedPos.x()>=0 && ClickedPos.x()<=FRONT_H && ClickedPos.y()>=0 && ClickedPos.y()<=FRONT_V );
}

/** compute the new coordinates for a specified object to be aligned */
void KCoverPage::AlignObject( KCoverObject * obj, const int halign, const int valign )
{
	switch ( obj->GetClassType() )
		{
		// curved string --> we align the center of the circle
		case TYPE_CLASS_CURVED_STRING :
			{
			KCoverCurvedString * curobj = dynamic_cast<KCoverCurvedString *>(obj);
			// align the object vertically
			switch ( valign )
				{
				case Top :
					switch ( curobj->Type )
						{
						case TYPE_RAINBOW_UP :
						case TYPE_CIRCLE :							
							obj->Y = obj->Width()/2;
							break;
						case TYPE_RAINBOW_DOWN :
							obj->Y = 0;
							break;
						}
					break;
				case Center :
					obj->Y = GetPageWidth()/2;
					break;
				case Bottom :
					switch ( curobj->Type )
						{
						case TYPE_RAINBOW_DOWN :
						case TYPE_CIRCLE :							
							obj->Y = GetPageHeight()-obj->Width()/2;
							break;
						case TYPE_RAINBOW_UP :
							obj->Y = GetPageHeight();
							break;
						}
				 	break;
				 // nothing to do ... just to avoid warning during compilation
				 case Left :
				 case Right :
				 case None :
				 	break;
				}
			// align the object horizontally
			switch ( halign )
				{
				case Left :
					obj->X = GetPageLeftMargin() + obj->Width()/2;
					break;
				case Center :
					obj->X = GetPageWidth()/2;
					break;
				case Right :
					obj->X = GetPageRightMargin() - obj->Width()/2;
					break;
				// nothing to do ... just to avoid warning during compilation				
				case Top :
				case Bottom :
				case None :
					break;
				}
				
			}
			break;
		default :
			{
			// pos X
			switch ( halign )
				{
				case Left :
					obj->X = GetPageLeftMargin();
					break;
				case Center :
					obj->X = (GetPageWidth()-obj->Width())/2;
					break;
				case Right :
					obj->X = GetPageRightMargin()-obj->Width();
					break;
				// nothing to do ... just to avoid warning during compilation				
				case Top :
				case Bottom :
				case None :
					break;
				}
			// pos Y
			switch ( valign )
				{
				case Top :
					obj->Y = obj->GetTopAlignDelta();
					break;
				case Center :
					obj->Y = (GetPageHeight()-obj->Height())/2 + obj->GetMiddleAlignDelta();
					break;
				case Bottom :
					obj->Y = GetPageHeight()-obj->Height()+obj->GetBottomAlignDelta();
					break;
				 // nothing to do ... just to avoid warning during compilation
				 case Left :
				 case Right :
				 case None :
				 	break;
				}
			}
			break;		
		}
}


/** return  the page width in pixels */
const unsigned int KCoverPage::GetPageWidth() const
{
	return FRONT_H;
}

/** return  the page height in pixels */
const unsigned int KCoverPage::GetPageHeight() const
{
	return FRONT_V;
}

/** return the page left margin position */
const unsigned int KCoverPage::GetPageLeftMargin() const
{
	return 0;
}

/** return the page right margin position */
const unsigned int KCoverPage::GetPageRightMargin() const
{
	return FRONT_H;
}
