/***************************************************************************
                         script.cpp  -  manage a script
                         -------------------
    begin                : Sun Jul 27 2003
    copyright            : (C) 2003 by Claudiu Costin
    email                : claudiuc@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qstringlist.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdom.h>

#include <kdebug.h>
#include <kstddirs.h>

#include "script.h"

// -------------------- GUIDATA ------------------------- //

GuiData::GuiData() {
  address=QString::null;
  port="21";
  pidfile="/var/run/pure-ftpd.pid";

  maxclients=50;
  clientsperip=50;

  pureftpd="/usr/local/sbin/pure-ftpd";
  pureftpwho="/usr/local/sbin/pure-ftpwho";
  puremrtginfo="/usr/local/sbin/pure-mrtginfo";
  pureauthd="/usr/local/sbin/pure-authd";
  purequotacheck="/usr/local/sbin/pure-quotacheck";
  pureuploadscript="/usr/local/bin/pure-uploadscript";
  purepw="/usr/local/bin/pure-pw";
  purepwconvert="/usr/local/bin/pure-pwconvert";
  purestatsdecode="/usr/local/bin/pure-statsdecode";

  authtype=0;
  anonlogin=false;
  allowmkdir=false;
  ftpuserdeny=true;
  denyupload=true;
  anonreaddotfiles=false;
  nonanonlogin=false;
  readdotfiles=false;
  writedotfiles=false;
  restrictedip=false;
  loginip=QString::null;

  disablechmod=false;
  usechrootall=false;
  usechroot=false;
  useuidlogin=false;
  chrootgid=100;
  uidlogin=100;
  idletime=30;
  maxfiles=2000;
  maxdepth=5;
  filesumask=0133;
  dirsumask=0022;
  debug=0;

  usesyslog=true;
  logpid=false;
  usefilelog=false;
  resolve=false;

  facility=4;
  logformat=0;
  pathlog="/var/log/pureftpd.log";

  useportrange=false;
  usequota=false;
  usedisklimit=false;
  useloadlimit=false;

  portlow=1024;
  porthigh=65535;
  quotamaxfiles=1024;
  quotamaxsize=2;
  diskpercent=90;
  grpbandwidth=0;
  bandupload=100;
  banddownload=100;
  grpratio=0;
  ratioupload=1;
  ratiodownload=1;
  systemload=2.0;

  onlyipv4=false;
  activeftp=false;
  usepassiveip=false;
  useuploadscript=false;
  allowresume=false;
  createhomedir=false;
  usefortunes=false;
  neveroverwrite=false;
  norename=false;
  customerproof=false;
  enableworkarounds=true;
  fxpeveryone=false;
  fxpnonanonym=false;

  passiveip=QString::null;
  fortunesfile=QString::null;
}

// -------------------- SCRIPTITEM ---------------------- //

QString ScriptItem::id() {
  return m_id;
}

QString ScriptItem::title() {
  return m_title;
}

QString ScriptItem::description() {
  return m_description;
}

QString ScriptItem::outputfile() {
  return m_outputfile;
}

int ScriptItem::outputfilemode() {
  return m_outputfilemode;
}

QString ScriptItem::scriptfile() {
  return m_scriptfile;
}

QString ScriptItem::templateid() {
  return m_templateid;
}

bool ScriptItem::isChanged() {
  return m_changed;
}

GuiData ScriptItem::guidata() {
  return m_guidata;
}

KListViewItem* ScriptItem::li() {
  return m_li;
}

void ScriptItem::setId(const QString &t) {
  m_id = t;
}

void ScriptItem::setTitle(const QString &t) {
  m_title = t;
}

void ScriptItem::setDescription(const QString &t) {
  m_description = t;
}

void ScriptItem::setOutputFile(const QString &t) {
  m_outputfile = t;
}

void ScriptItem::setOutputFileMode(int t) {
  m_outputfilemode = t;
}

void ScriptItem::setScriptFile(const QString &t) {
  m_scriptfile = t;
}

void ScriptItem::setTemplateId(const QString &t) {
  m_templateid = t;
}

void ScriptItem::setChanged(bool t) {
  m_changed = t;
}

void ScriptItem::setGuiData(const GuiData t) {
  m_guidata = t;
}

void ScriptItem::setLi(KListViewItem *t) {
  m_li = t;
}

// -------------------- SCRIPT -------------------------- //

Script::Script()
{
  m_authmethods["unix"] = Script::NoPath;
  m_authmethods["pam"] = Script::NoPath;
  m_authmethods["ldap"] = Script::WithPath;
  m_authmethods["mysql"] = Script::WithPath;
  m_authmethods["pgsql"] = Script::WithPath;
  m_authmethods["puredb"] = Script::WithPath;
  m_authmethods["extauth"] = Script::WithPath;
}

ScriptItemMap& Script::map()
{
  return m_map;
}

Script::IOStatus Script::load(ScriptItem &i, const QString &scriptfile, QString &readid)
{
  QString id, title, description, templateid, outputfile, mode;
  QString errMsg;
  int errLine, errColumn;
  KStandardDirs *dirs = KGlobal::dirs();
  QFile f(scriptfile);
  IOStatus ret;

  QDomDocument doc(dirs->findResource("data","kcmpureftpdscript/kcmpureftpdscr.dtd"));
  if (f.open(IO_ReadOnly)) {
    if (doc.setContent(&f,false,&errMsg,&errLine,&errColumn)) {
      GuiData g;
      id = title = description = QString::null;
      mode = templateid = outputfile = QString::null;

      QDomElement docElem = doc.documentElement();
      QDomNode n = docElem.firstChild();
      while (!n.isNull()) {
        QDomElement e = n.toElement();
        if (!e.isNull()) {
          if (!e.tagName().compare("id")) id=e.text().stripWhiteSpace();
          if (!e.tagName().compare("title")) title=e.text().stripWhiteSpace();
          if (!e.tagName().compare("description")) description=e.text().stripWhiteSpace();
          if (!e.tagName().compare("outputfile")) {
            outputfile=e.text().stripWhiteSpace();
            mode = e.attribute("mode","755").stripWhiteSpace();
          }
          if (!e.tagName().compare("templateid")) templateid=e.text().stripWhiteSpace();
          if (!e.tagName().compare("guidata")) {
            QDomNode ng = e.firstChild();
            while (!ng.isNull()) {
              QDomElement eg = ng.toElement();
              if (!eg.isNull()) {
                // "Scripts" tab

                // "Run" tab
                if (!eg.tagName().compare("address")) g.address = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("port")) g.port = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("pidfile")) g.pidfile = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("maxclients")) g.maxclients = eg.text().stripWhiteSpace().toUInt();
                if (!eg.tagName().compare("clientsperip")) g.clientsperip = eg.text().stripWhiteSpace().toUInt();
                if (!eg.tagName().compare("pureftpd")) g.pureftpd = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("pureftpwho")) g.pureftpwho = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("puremrtginfo")) g.puremrtginfo = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("pureauthd")) g.pureauthd = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("purequotacheck")) g.purequotacheck = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("pureuploadscript")) g.pureuploadscript = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("purepw")) g.purepw = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("purepwconvert")) g.purepwconvert = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("purestatsdecode")) g.purestatsdecode = eg.text().stripWhiteSpace();

                // "Authentication" tab
                if (!eg.tagName().compare("auth")) {
                  g.authtype = eg.attribute("type","0").toUInt();
                  QDomNode na = eg.firstChild();
                  while (!na.isNull()) {
                    QDomElement ea = na.toElement();
                    if (!ea.isNull()) {
                      if (!ea.tagName().compare("authitem")) {
                        QString method, path;
                        method = ea.attribute("method","");
                        path = ea.attribute("path","");
                        if (m_authmethods.keys().contains(method)) {
                          bool found=false;
                          // search for duplicate methods
                          QValueList<AuthMethod>::ConstIterator ai;
                          for (ai=g.authlist.begin(); ai!=g.authlist.end(); ++ai) {
                            if ((*ai).method == method) {
                              found=true;
                              break;
                            }
                          }
                          // if not found, then it's ok to add the method-path pair
                          if (!found) {
                            AuthMethod am;
                            am.method=method;
                            if (m_authmethods[method] == Script::NoPath) {
                              am.path=QString::null;
                            } else {
                              am.path=path;
                            }
                            g.authlist.append(am);
                          }
                        }
                      }
                    }
                    na = na.nextSibling();
                  }
                }

                // "Users" tab
                if (!eg.tagName().compare("anonlogin")) g.anonlogin = (!eg.text().compare("1"));
                if (!eg.tagName().compare("allowmkdir")) g.allowmkdir = (!eg.text().compare("1"));
                if (!eg.tagName().compare("ftpuserdeny")) g.ftpuserdeny = (!eg.text().compare("1"));
                if (!eg.tagName().compare("denyupload")) g.denyupload = (!eg.text().compare("1"));
                if (!eg.tagName().compare("anonreaddotfiles")) g.anonreaddotfiles = (!eg.text().compare("1"));
                if (!eg.tagName().compare("nonanonlogin")) g.nonanonlogin = (!eg.text().compare("1"));
                if (!eg.tagName().compare("readdotfiles")) g.readdotfiles = (!eg.text().compare("1"));
                if (!eg.tagName().compare("writedotfiles")) g.writedotfiles = (!eg.text().compare("1"));
                if (!eg.tagName().compare("restrictedip")) g.restrictedip = (!eg.text().compare("1"));
                if (!eg.tagName().compare("loginip")) g.loginip = eg.text().stripWhiteSpace();

                // "Security & Logging" tab
                if (!eg.tagName().compare("disablechmod")) g.disablechmod = (!eg.text().compare("1"));
                if (!eg.tagName().compare("usechrootall")) g.usechrootall = (!eg.text().compare("1"));
                if (!eg.tagName().compare("usechroot")) g.usechroot = (!eg.text().compare("1"));
                if (!eg.tagName().compare("chrootgid")) g.chrootgid = eg.text().toUInt();
                if (!eg.tagName().compare("useuidlogin")) g.useuidlogin = (!eg.text().compare("1"));
                if (!eg.tagName().compare("uidlogin")) g.uidlogin = eg.text().toUInt();
                if (!eg.tagName().compare("idletime")) g.idletime = eg.text().toUInt();
                if (!eg.tagName().compare("maxfiles")) g.maxfiles = eg.text().toUInt();
                if (!eg.tagName().compare("maxdepth")) g.maxdepth = eg.text().toUInt();
                if (!eg.tagName().compare("filesumask")) g.filesumask = eg.text().toUInt(0,8);
                if (!eg.tagName().compare("dirsumask")) g.dirsumask = eg.text().toUInt(0,8);
                if (!eg.tagName().compare("debug")) g.debug = eg.text().toUInt();
                if (!eg.tagName().compare("usesyslog")) g.usesyslog = (!eg.text().compare("1"));
                if (!eg.tagName().compare("facility")) g.facility = eg.text().toUInt();
                if (!eg.tagName().compare("logpid")) g.logpid = (!eg.text().compare("1"));
                if (!eg.tagName().compare("usefilelog")) g.usefilelog = (!eg.text().compare("1"));
                if (!eg.tagName().compare("logformat")) g.logformat = eg.text().toUInt();
                if (!eg.tagName().compare("pathlog")) g.pathlog = eg.text();
                if (!eg.tagName().compare("resolve")) g.resolve = (!eg.text().compare("1"));

                // "Limits" tab
                if (!eg.tagName().compare("useportrange")) g.useportrange = (!eg.text().compare("1"));
                if (!eg.tagName().compare("portlow")) g.portlow = eg.text().toUInt();
                if (!eg.tagName().compare("porthigh")) g.porthigh = eg.text().toUInt();
                if (!eg.tagName().compare("usequota")) g.usequota = (!eg.text().compare("1"));
                if (!eg.tagName().compare("quotamaxfiles")) g.quotamaxfiles = eg.text().toUInt();
                if (!eg.tagName().compare("quotamaxsize")) g.quotamaxsize = eg.text().toUInt();
                if (!eg.tagName().compare("usedisklimit")) g.usedisklimit = (!eg.text().compare("1"));
                if (!eg.tagName().compare("diskpercent")) g.diskpercent = eg.text().toUInt();
                if (!eg.tagName().compare("useloadlimit")) g.useloadlimit = (!eg.text().compare("1"));
                if (!eg.tagName().compare("systemload")) g.systemload = QABS(eg.text().toDouble());
                if (!eg.tagName().compare("grpbandwidth")) g.grpbandwidth = eg.text().toUInt();
                if (!eg.tagName().compare("bandupload")) g.bandupload = eg.text().toUInt();
                if (!eg.tagName().compare("banddownload")) g.banddownload = eg.text().toUInt();
                if (!eg.tagName().compare("grpratio")) g.grpratio = eg.text().toUInt();
                if (!eg.tagName().compare("ratioupload")) g.ratioupload = eg.text().toUInt();
                if (!eg.tagName().compare("ratiodownload")) g.ratiodownload = eg.text().toUInt();

                // "Miscellaneous" tab
                if (!eg.tagName().compare("onlyipv4")) g.onlyipv4 = (!eg.text().compare("1"));
                if (!eg.tagName().compare("activeftp")) g.activeftp = (!eg.text().compare("1"));
                if (!eg.tagName().compare("usepassiveip")) g.usepassiveip = (!eg.text().compare("1"));
                if (!eg.tagName().compare("passiveip")) g.passiveip = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("useuploadscript")) g.useuploadscript = (!eg.text().compare("1"));
                if (!eg.tagName().compare("allowresume")) g.allowresume = (!eg.text().compare("1"));
                if (!eg.tagName().compare("createhomedir")) g.createhomedir = (!eg.text().compare("1"));
                if (!eg.tagName().compare("usefortunes")) g.usefortunes = (!eg.text().compare("1"));
                if (!eg.tagName().compare("fortunesfile")) g.fortunesfile = eg.text().stripWhiteSpace();
                if (!eg.tagName().compare("neveroverwrite")) g.neveroverwrite = (!eg.text().compare("1"));
                if (!eg.tagName().compare("norename")) g.norename = (!eg.text().compare("1"));
                if (!eg.tagName().compare("customerproof")) g.customerproof = (!eg.text().compare("1"));
                if (!eg.tagName().compare("enableworkarounds")) g.enableworkarounds = (!eg.text().compare("1"));
                if (!eg.tagName().compare("fxpeveryone")) g.fxpeveryone = (!eg.text().compare("1"));
                if (!eg.tagName().compare("fxpnonanonym")) g.fxpnonanonym = (!eg.text().compare("1"));
              }
              ng = ng.nextSibling();
            }
          }
        }
        n = n.nextSibling();
      }
      if (!id.isEmpty() && -1==id.find("/") &&
          !templateid.isEmpty() && -1==templateid.find("/")) {
        bool ok;
        int outputfilemode=mode.toUInt(&ok,8);
        if (!ok) outputfilemode=0644;

        i.setId(id);
        i.setTitle(title);
        i.setDescription(description);
        i.setOutputFile(outputfile);
        i.setOutputFileMode(outputfilemode);
        i.setScriptFile(scriptfile);
        i.setTemplateId(templateid);
        i.setGuiData(g);
        readid=id;
        ret=Script::LoadOk;
      } else {
        ret=Script::DataError;
      }
    } else {
      kdWarning() << "Parsing error for \"" << f.name() << "\" file!" << endl;
      kdWarning() << "message=" << errMsg << endl
                  << "line=" << errLine << " column=" << errColumn;
      ret=Script::ParseError;
    }
    f.close();
  } else {
    ret = Script::OpenError;
  }
  return ret;
}

void Script::loadAll()
{
  QStringList list = KGlobal::dirs()->findAllResources("data","kcmpureftpdscript/*.purescr",false,true);
  QString id;
  IOStatus ret;

  for (QStringList::Iterator it=list.begin(); it!=list.end(); ++it) {
    ScriptItem s;

    ret=load(s,*it,id);
    if (ret == Script::LoadOk)
      m_map[id] = s;
  }
}

Script::IOStatus Script::save(ScriptItem &item)
{
  IOStatus ret;
  QDomElement e, g, a;
  QDomText t;

  QDomImplementation impl;
  QDomDocumentType doctype = impl.createDocumentType("kcmpureftpdscr", QString::null, "kcmpureftpdscr.dtd");
  QDomDocument doc(doctype);
  doc.appendChild(doc.createProcessingInstruction("xml", "version=\"1.0\" encoding=\"UTF-8\""));

  QDomElement root = doc.createElement("kcmpureftpdscr");
  doc.appendChild(root);

  // ----- administrative data -----
  e = doc.createElement("id");
  e.appendChild(doc.createTextNode(item.id()));
  root.appendChild(e);
  e = doc.createElement("title");
  e.appendChild(doc.createTextNode(item.title()));
  root.appendChild(e);
  e = doc.createElement("description");
  e.appendChild(doc.createTextNode(item.description()));
  root.appendChild(e);
  e = doc.createElement("outputfile");
  e.setAttribute("mode",QString::number(item.outputfilemode(),8));
  e.appendChild(doc.createTextNode(item.outputfile()));
  root.appendChild(e);
  e = doc.createElement("templateid");
  e.appendChild(doc.createTextNode(item.templateid()));
  root.appendChild(e);

  // ----- GUI data -----
  g = doc.createElement("guidata");
  root.appendChild(g);

  // "Run" tab
  e = doc.createElement("address");
  e.appendChild(doc.createTextNode(item.guidata().address));
  g.appendChild(e);
  e = doc.createElement("port");
  e.appendChild(doc.createTextNode(item.guidata().port));
  g.appendChild(e);
  e = doc.createElement("pidfile");
  e.appendChild(doc.createTextNode(item.guidata().pidfile));
  g.appendChild(e);
  e = doc.createElement("maxclients");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().maxclients)));
  g.appendChild(e);
  e = doc.createElement("clientsperip");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().clientsperip)));
  g.appendChild(e);
  e = doc.createElement("pureftpd");
  e.appendChild(doc.createTextNode(item.guidata().pureftpd));
  g.appendChild(e);
  e = doc.createElement("pureftpwho");
  e.appendChild(doc.createTextNode(item.guidata().pureftpwho));
  g.appendChild(e);
  e = doc.createElement("puremrtginfo");
  e.appendChild(doc.createTextNode(item.guidata().puremrtginfo));
  g.appendChild(e);
  e = doc.createElement("pureauthd");
  e.appendChild(doc.createTextNode(item.guidata().pureauthd));
  g.appendChild(e);
  e = doc.createElement("purequotacheck");
  e.appendChild(doc.createTextNode(item.guidata().purequotacheck));
  g.appendChild(e);
  e = doc.createElement("pureuploadscript");
  e.appendChild(doc.createTextNode(item.guidata().pureuploadscript));
  g.appendChild(e);
  e = doc.createElement("purepw");
  e.appendChild(doc.createTextNode(item.guidata().purepw));
  g.appendChild(e);
  e = doc.createElement("purepwconvert");
  e.appendChild(doc.createTextNode(item.guidata().purepwconvert));
  g.appendChild(e);
  e = doc.createElement("purestatsdecode");
  e.appendChild(doc.createTextNode(item.guidata().purestatsdecode));
  g.appendChild(e);

  // "Authentication" tab
  a = doc.createElement("auth");
  a.setAttribute("type",QString::number(item.guidata().authtype));
  g.appendChild(a);

  QValueList<AuthMethod> authlist = item.guidata().authlist;
  QValueList<AuthMethod>::Iterator it;
  for (it=authlist.begin(); it!=authlist.end(); ++it) {
    e = doc.createElement("authitem");
    e.setAttribute("method",(*it).method);
    e.setAttribute("path",(*it).path);
    a.appendChild(e);
  }

  // "Users" tab
  e = doc.createElement("anonlogin");
  e.appendChild(doc.createTextNode((item.guidata().anonlogin)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("allowmkdir");
  e.appendChild(doc.createTextNode((item.guidata().allowmkdir)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("ftpuserdeny");
  e.appendChild(doc.createTextNode((item.guidata().ftpuserdeny)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("denyupload");
  e.appendChild(doc.createTextNode((item.guidata().denyupload)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("anonreaddotfiles");
  e.appendChild(doc.createTextNode((item.guidata().anonreaddotfiles)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("nonanonlogin");
  e.appendChild(doc.createTextNode((item.guidata().nonanonlogin)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("readdotfiles");
  e.appendChild(doc.createTextNode((item.guidata().readdotfiles)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("writedotfiles");
  e.appendChild(doc.createTextNode((item.guidata().writedotfiles)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("restrictedip");
  e.appendChild(doc.createTextNode((item.guidata().restrictedip)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("loginip");
  e.appendChild(doc.createTextNode(item.guidata().loginip));
  g.appendChild(e);

  // "Security & Logging" tab
  e = doc.createElement("disablechmod");
  e.appendChild(doc.createTextNode((item.guidata().disablechmod)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("usechrootall");
  e.appendChild(doc.createTextNode((item.guidata().usechrootall)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("usechroot");
  e.appendChild(doc.createTextNode((item.guidata().usechroot)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("chrootgid");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().chrootgid,10)));
  g.appendChild(e);
  e = doc.createElement("useuidlogin");
  e.appendChild(doc.createTextNode((item.guidata().useuidlogin)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("uidlogin");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().uidlogin,10)));
  g.appendChild(e);
  e = doc.createElement("idletime");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().idletime,10)));
  g.appendChild(e);
  e = doc.createElement("maxfiles");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().maxfiles,10)));
  g.appendChild(e);
  e = doc.createElement("maxdepth");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().maxdepth,10)));
  g.appendChild(e);
  e = doc.createElement("filesumask");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().filesumask,8)));
  g.appendChild(e);
  e = doc.createElement("dirsumask");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().dirsumask,8)));
  g.appendChild(e);
  e = doc.createElement("debug");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().debug,10)));
  g.appendChild(e);
  e = doc.createElement("usesyslog");
  e.appendChild(doc.createTextNode((item.guidata().usesyslog)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("facility");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().facility,10)));
  g.appendChild(e);
  e = doc.createElement("logpid");
  e.appendChild(doc.createTextNode((item.guidata().logpid)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("usefilelog");
  e.appendChild(doc.createTextNode((item.guidata().usefilelog)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("logformat");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().logformat,10)));
  g.appendChild(e);
  e = doc.createElement("pathlog");
  e.appendChild(doc.createTextNode(item.guidata().pathlog));
  g.appendChild(e);
  e = doc.createElement("resolve");
  e.appendChild(doc.createTextNode((item.guidata().resolve)? "1" : "0"));
  g.appendChild(e);

  // "Limits" tab
  e = doc.createElement("useportrange");
  e.appendChild(doc.createTextNode((item.guidata().useportrange)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("portlow");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().portlow,10)));
  g.appendChild(e);
  e = doc.createElement("porthigh");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().porthigh,10)));
  g.appendChild(e);
  e = doc.createElement("usequota");
  e.appendChild(doc.createTextNode((item.guidata().usequota)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("quotamaxfiles");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().quotamaxfiles,10)));
  g.appendChild(e);
  e = doc.createElement("quotamaxsize");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().quotamaxsize,10)));
  g.appendChild(e);
  e = doc.createElement("usedisklimit");
  e.appendChild(doc.createTextNode((item.guidata().usedisklimit)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("diskpercent");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().diskpercent,10)));
  g.appendChild(e);
  e = doc.createElement("useloadlimit");
  e.appendChild(doc.createTextNode((item.guidata().useloadlimit)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("systemload");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().systemload,'f',2)));
  g.appendChild(e);
  e = doc.createElement("grpbandwidth");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().grpbandwidth,10)));
  g.appendChild(e);
  e = doc.createElement("bandupload");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().bandupload,10)));
  g.appendChild(e);
  e = doc.createElement("banddownload");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().banddownload,10)));
  g.appendChild(e);
  e = doc.createElement("grpratio");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().grpratio,10)));
  g.appendChild(e);
  e = doc.createElement("ratioupload");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().ratioupload,10)));
  g.appendChild(e);
  e = doc.createElement("ratiodownload");
  e.appendChild(doc.createTextNode(QString::number(item.guidata().ratiodownload,10)));
  g.appendChild(e);

  // "Miscellaneous" tab
  e = doc.createElement("onlyipv4");
  e.appendChild(doc.createTextNode((item.guidata().onlyipv4)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("activeftp");
  e.appendChild(doc.createTextNode((item.guidata().activeftp)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("usepassiveip");
  e.appendChild(doc.createTextNode((item.guidata().usepassiveip)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("passiveip");
  e.appendChild(doc.createTextNode(item.guidata().passiveip));
  g.appendChild(e);
  e = doc.createElement("useuploadscript");
  e.appendChild(doc.createTextNode((item.guidata().useuploadscript)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("allowresume");
  e.appendChild(doc.createTextNode((item.guidata().allowresume)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("createhomedir");
  e.appendChild(doc.createTextNode((item.guidata().createhomedir)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("usefortunes");
  e.appendChild(doc.createTextNode((item.guidata().usefortunes)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("fortunesfile");
  e.appendChild(doc.createTextNode(item.guidata().fortunesfile));
  g.appendChild(e);
  e = doc.createElement("neveroverwrite");
  e.appendChild(doc.createTextNode((item.guidata().neveroverwrite)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("norename");
  e.appendChild(doc.createTextNode((item.guidata().norename)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("customerproof");
  e.appendChild(doc.createTextNode((item.guidata().customerproof)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("enableworkarounds");
  e.appendChild(doc.createTextNode((item.guidata().enableworkarounds)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("fxpeveryone");
  e.appendChild(doc.createTextNode((item.guidata().fxpeveryone)? "1" : "0"));
  g.appendChild(e);
  e = doc.createElement("fxpnonanonym");
  e.appendChild(doc.createTextNode((item.guidata().fxpnonanonym)? "1" : "0"));
  g.appendChild(e);

  // ----- save the XML data to file -----
  QFile f(item.scriptfile());
  if (f.open(IO_WriteOnly)) {
    QTextStream s(&f);
    s << doc.toString();
    f.close();
    ret = Script::SaveOk;
  } else {
    ret = Script::OpenError;
  }
  return ret;
}

void Script::saveAll()
{
}

void Script::clear()
{
  m_map.clear();
}

ScriptItemIterator Script::begin()
{
  return m_map.begin();
}

ScriptItemIterator Script::end()
{
  return m_map.end();
}

QMap<QString,Script::AuthMethodOption> Script::authMethods() const
{
  return m_authmethods;
}

