/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                        *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "kdarConfig.h"

#include "kdar.h"
#include "kdarEnums.h"
#include "KDarInteraction.h"
#include "archiveHandle.h"

#include <kapp.h>
#include <kcombobox.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kfontrequester.h>
#include <kglobal.h>
#include <klineedit.h>
#include <klistbox.h>
#include <klocale.h>
#include <knuminput.h>
#include <kpassdlg.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kurlrequester.h>

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qdir.h>
#include <qfont.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qspinbox.h>
#include <qvalidator.h>
#include <qvaluelist.h>
#include <qwidgetstack.h>

#include <dar/libdar.hpp>

KDarConfig::KDarConfig (const KDarConfig &_kdc)
  : kdarConfigureDlg(0,0)
  , m_nodialog(true)
  , m_kc(0)
  , m_ZCompression( true )
  , m_BZ2Compression( true )
  , m_strongCrypto ( true )
{
    //set up an integer validator on the slicesize lineEdits
    kdDebug() << "kdarView::kdarView(): integer validator for lineEdits" << endl;
    QRegExp rx("[1-9][0-9]*");
    QRegExpValidator * restrictToWholeNumbers = new QRegExpValidator(rx, this);
    sliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    firstSliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    //Default media types:
    mediaNames << i18n( "media name for custom", "Custom" ) << "DVD (4.7GB)" << "DVD (2GB)" << "CDR (99 min.)"       << "CDR (89 min.)" << "CDR (80 min.)" << "CDR (74 min.)" << "CDR (63 min.)" << "CDR (50 min.)" << "CDR (34 min.)" << "CDR (30 min.)" << "CDR (27 min.)" << "CDR (23 min.)" << "CDR (21 min.)" << "CDR (20 min.)" << "Floppy (1440 kB)" << "Zip (100 MB)";
    statusBarMediaNames << i18n( "statusbar name for custom", "custom" ) << "DVD (4.7GB)" << "DVD (2GB)" << "CDR (870 MB)"       << "CDR (782 MB)" << "CDR (700 MB)" << "CDR (650 MB)" << "CDR (553 MB)" << "CDR (439 MB)" << "CDR (298 MB)" << "CDR (263 MB)" << "CDR (237 MB)" << "CDR (202 MB)" << "CDR (184 MB)" << "CDR (175 MB)" << "Floppy (1440 kB)" << "Zip (100 MB)";
    //These sizes are from xcdroast.
    mediaSizes << "700" << "4400" << "2" << "870" << "782" << "700" << "650" << "553" << "439" << "298" << "263" << "237" << "202" << "184" << "175" << "1440" << "100";
    mediaFirstSizes = mediaSizes;
    mediaSuffices << KDar::MB << KDar::MB << KDar::GB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::KB << KDar::MB;
    mediaFirstSuffices = mediaSuffices;

    //Copy the values:
    setArchiveName( _kdc.archiveName() );
    setArchiveStorageDirectory( _kdc.archiveStorageDirectory() );
    setNewArchiveName( _kdc.newArchiveName() );
    setDirectoryToBackup( _kdc.directoryToBackup() );
    setDifferentialBackup( _kdc.differentialBackup() );
    setDifferentialBackupArchiveName( _kdc.differentialBackupArchiveName() );
    setExtractArchiveDirectory( _kdc.extractArchiveDirectory() );
    setInputPipe( _kdc.inputPipe() );
    setOutputPipe( _kdc.outputPipe() );
    setExecuteCommand( _kdc.executeCommand() );
    setCryptoAlgorithm( _kdc.cryptoAlgorithm() );
    setCryptoBlockSize( _kdc.cryptoBlockSize() );
    setStorePassword( _kdc.storePassword() );
    setExecuteRef( _kdc.executeRef() );
    setPassRef( _kdc.passRef() );

    setVerbose( _kdc.verbose() );
    setAllowOverwrite( _kdc.allowOverwrite() );
    setWarnOnOverwrite( _kdc.warnOnOverwrite() );
    setUserEA( _kdc.userEA() );
    setSystemEA( _kdc.systemEA() );
    setIgnoreID( _kdc.ignoreID() );
    setNoDelete( _kdc.noDelete() );
    setRestoreRecent( _kdc.restoreRecent() );
    setFlatRestore( _kdc.flatRestore() );

    setUseCreateArchiveWizard( _kdc.useCreateArchiveWizard() );
    setSplitArchive( _kdc.splitArchive() );
    setPauseBetweenSlices( _kdc.pauseBetweenSlices() );
    setKeepPruned( _kdc.keepPruned() );
    setIgnoreDump( _kdc.ignoreDump() );
    setMedia( _kdc.media() );
    setSliceSizeString( _kdc.sliceSizeString() );
    setSliceSizeSuffix( _kdc.sliceSizeSuffix() );
    setFirstSliceSizeStrings( _kdc.firstSliceSizeStrings() );
    setFirstSliceSizeSuffices( _kdc.firstSliceSizeSuffices() );
    setCompressionAlgorithm( _kdc.compressionAlgorithm() );
    setCompressionLevel( _kdc.compressionLevel() );
    setMinimumCompressionSizeInt( _kdc.minimumCompressionSizeInt() );
    setHourShiftInt( _kdc.hourShiftInt() );

    setCompressionMaskList(_kdc.compressionMaskList() );
    setDirectoryMaskList( _kdc.directoryMaskList() );
    setFileMaskList( _kdc.fileMaskList() );
    setCompressionMaskType( _kdc.compressionMaskType() );
    setDirectoryMaskTypes( _kdc.directoryMaskTypes() );
    setFileMaskType( _kdc.fileMaskType() );

    setGeneralFont( _kdc.generalFont() );
    setFileBrowserFont( _kdc.fileBrowserFont() );
    setMessageWindowFont( _kdc.messageWindowFont() );
    setStatusBarFont( _kdc.statusBarFont() );

    setShowKDarSplashScreen( _kdc.showKDarSplashScreen() );
    setDryRun( _kdc.dryRun() );
    setIsolationArchiveName( _kdc.isolationArchiveName() );
    setLogFile( _kdc.logFile() );
    setLogLevel( _kdc.logLevel() );

    m_ZCompression = _kdc.ZCompression();
    m_BZ2Compression = _kdc.BZ2Compression();
    m_strongCrypto = _kdc.strongCrypto();
}


KDarConfig::KDarConfig( QWidget *parent, const char *name,
                            KDarConfig *_kdc, bool addHelpButton )
    : kdarConfigureDlg (parent, name), m_nodialog(false)
    , m_kc(0)
    , m_compressionDlg( new kdarCompressionFilterDialog )
    , m_directoryDlg( new kdarDirectoryFilterDialog )
    , m_fileDlg( new kdarFileFilterDialog )
    , m_ZCompression( true )
    , m_BZ2Compression( true )
    , m_strongCrypto ( true )
{
    //set up an integer validator on the slicesize lineEdits
    kdDebug() << "kdarView::kdarView(): integer validator for lineEdits" << endl;
    QRegExp rx("[1-9][0-9]*");
    QRegExpValidator * restrictToWholeNumbers = new QRegExpValidator(rx, this);
    sliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    firstSliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    //Default media types:
    mediaNames << i18n( "media name for custom", "Custom" ) << "DVD (4.7GB)" << "DVD (2GB)" << "CDR (99 min.)"       << "CDR (89 min.)" << "CDR (80 min.)" << "CDR (74 min.)" << "CDR (63 min.)" << "CDR (50 min.)" << "CDR (34 min.)" << "CDR (30 min.)" << "CDR (27 min.)" << "CDR (23 min.)" << "CDR (21 min.)" << "CDR (20 min.)" << "Floppy (1440 kB)" << "Zip (100 MB)";
    statusBarMediaNames << i18n( "statusbar name for custom", "custom" ) << "DVD (4.7GB)" << "DVD (2GB)" << "CDR (870 MB)"       << "CDR (782 MB)" << "CDR (700 MB)" << "CDR (650 MB)" << "CDR (553 MB)" << "CDR (439 MB)" << "CDR (298 MB)" << "CDR (263 MB)" << "CDR (237 MB)" << "CDR (202 MB)" << "CDR (184 MB)" << "CDR (175 MB)" << "Floppy (1440 kB)" << "Zip (100 MB)";
    mediaSizes << "700" << "4400" << "2" << "870" << "782" << "700" << "650" << "553" << "439" << "298" << "263" << "237" << "202" << "184" << "175" << "1440" << "100";
    mediaFirstSizes = mediaSizes;

    mediaSuffices << KDar::MB << KDar::MB << KDar::GB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::MB << KDar::KB << KDar::MB;
    mediaFirstSuffices = mediaSuffices;

    //Clear the default mediaType list
    mediaTypeKComboBox->clear();
    //Load the mediaTypeKComboBox with the enumerated types
    mediaTypeKComboBox->insertStringList( mediaNames );
    //Set the default media to "custom"
    mediaTypeKComboBox->setCurrentItem( KDar::CDR80 );

    //Set the archiveStorageDir KURL Requester mode and caption:
    archiveStorageDirKURLRequester->setMode(18);
    archiveStorageDirKURLRequester->setCaption( i18n( "Please choose (or create) a storage directory" ) );

    m_kc = KGlobal::config();

    connect( this, SIGNAL( compressionMaskListChanged() ), this, SLOT( slotUpdateCompressionMaskList() ) );
    connect( this, SIGNAL( fileMaskListChanged() ), this, SLOT( slotUpdateFileMaskList() ) );
    connect ( this, SIGNAL( directoryMaskListChanged() ), this, SLOT( slotUpdateDirectoryMaskList() ) );

    if( _kdc == 0 )
    {
        readGlobalSettings();
    }
    else
    {
        setArchiveName( _kdc->archiveName() );
        setArchiveStorageDirectory( _kdc->archiveStorageDirectory() );
        setNewArchiveName( _kdc->newArchiveName() );
        setDirectoryToBackup( _kdc->directoryToBackup() );
        setDifferentialBackup( _kdc->differentialBackup() );
        setDifferentialBackupArchiveName( _kdc->differentialBackupArchiveName() );
        setExtractArchiveDirectory( _kdc->extractArchiveDirectory() );
        setInputPipe( _kdc->inputPipe() );
        setOutputPipe( _kdc->outputPipe() );
        setExecuteCommand( _kdc->executeCommand() );
        setCryptoAlgorithm( _kdc->cryptoAlgorithm() );
        setCryptoBlockSize( _kdc->cryptoBlockSize() );
        setStorePassword( _kdc->storePassword() );
        setExecuteRef( _kdc->executeRef() );
        setPassRef( _kdc->passRef() );

        setVerbose( _kdc->verbose() );
        setAllowOverwrite( _kdc->allowOverwrite() );
        setWarnOnOverwrite( _kdc->warnOnOverwrite() );
        setUserEA( _kdc->userEA() );
        setSystemEA( _kdc->systemEA() );
        setIgnoreID( _kdc->ignoreID() );
        setNoDelete( _kdc->noDelete() );
        setRestoreRecent( _kdc->restoreRecent() );
        setFlatRestore( _kdc->flatRestore() );

        setUseCreateArchiveWizard( _kdc->useCreateArchiveWizard() );
        setSplitArchive( _kdc->splitArchive() );
        setPauseBetweenSlices( _kdc->pauseBetweenSlices() );
        setKeepPruned( _kdc->keepPruned() );
        setIgnoreDump( _kdc->ignoreDump() );
        setMedia( _kdc->media() );
        setSliceSizeString( _kdc->sliceSizeString() );
        setSliceSizeSuffix( _kdc->sliceSizeSuffix() );
        setFirstSliceSizeStrings( _kdc->firstSliceSizeStrings() );
        setFirstSliceSizeSuffices( _kdc->firstSliceSizeSuffices() );
        setCompressionAlgorithm( _kdc->compressionAlgorithm() );
        setCompressionLevel( _kdc->compressionLevel() );
        setMinimumCompressionSizeInt( _kdc->minimumCompressionSizeInt() );
        setHourShiftInt( _kdc->hourShiftInt() );

        setCompressionMaskList(_kdc->compressionMaskList() );
        setDirectoryMaskList( _kdc->directoryMaskList() );
        setFileMaskList( _kdc->fileMaskList() );
        setCompressionMaskType( _kdc->compressionMaskType() );
        setDirectoryMaskTypes( _kdc->directoryMaskTypes() );
        setFileMaskType( _kdc->fileMaskType() );

        setGeneralFont( _kdc->generalFont() );
        setFileBrowserFont( _kdc->fileBrowserFont() );
        setMessageWindowFont( _kdc->messageWindowFont() );
        setStatusBarFont( _kdc->statusBarFont() );

        setShowKDarSplashScreen( _kdc->showKDarSplashScreen() );
        setDryRun( _kdc->dryRun() );
        setIsolationArchiveName( _kdc->isolationArchiveName() );
        setLogFile( _kdc->logFile() );
        setLogLevel( _kdc->logLevel() );
        m_ZCompression = _kdc->ZCompression();
        m_BZ2Compression = _kdc->BZ2Compression();
        m_strongCrypto = _kdc->strongCrypto();
    }
    if( addHelpButton == true )
    {
//  QPushButton *pushButton = new KPushButton( KStdGuiItem::help(), this );
//  connect( pushButton, SIGNAL(clicked()), this, SLOT(sHelp()) );
//  glay->addWidget(pushButton, 5, 2);
    }

    fillInDialog();
}

KDarConfig::~KDarConfig()
{
}

bool KDarConfig::readGlobalSettings()
{
    m_kc->setGroup("General");
        setUseCreateArchiveWizard( m_kc->readBoolEntry( "useCreateArchiveWizard", true) );
        setVerbose( m_kc->readBoolEntry( "verbose", true) );
        QString file = m_kc->readEntry( "logFile", "" );
        if ( file.isEmpty() )
        {
            file = KGlobal::dirs()->saveLocation( "data", "kdar/" ) + "kdar.log";
        }
        setLogFile( file );
        setLogLevel( m_kc->readNumEntry( "logLevel", KDar::LOG_BASIC ) );
        kdDebug() << "logFile is " << logFile() << endl;
        setDryRun( m_kc->readBoolEntry( "dryRun", false) );
        setShowKDarSplashScreen( m_kc->readBoolEntry( "showKDarSplashScreen", true ) );
        QString lastArchive = m_kc->readEntry( "lastArchive", "select archive");
        if (lastArchive[0] != '/')
        {
            lastArchive = "select archive";
        }
        setArchiveName( lastArchive );
        setArchiveStorageDirectory( m_kc->readEntry( "archiveStorageDir", "" ) );
        setNewArchiveName( m_kc->readEntry( "newArchiveName", "" ) );
        setDirectoryToBackup( m_kc->readEntry( "directoryToBackup", "" ) );
        setDifferentialBackup( m_kc->readBoolEntry( "differentialBackup", false ) );
        setDifferentialBackupArchiveName( m_kc->readEntry( "differentialBackupArchiveName", "" ) );
    m_kc->setGroup("Filesystem");
        setUserEA( m_kc->readBoolEntry( "userEA", true ) );
        setSystemEA( m_kc->readBoolEntry( "systemEA", true ) );
        setIgnoreID( m_kc->readBoolEntry( "ignoreID", false ) );
    m_kc->setGroup("Create / Isolate");
        setInputPipe( m_kc->readEntry( "inputPipe", "" ) );
        setOutputPipe( m_kc->readEntry( "outputPipe", "" ) );
        setExecuteCommand( m_kc->readEntry( "executeCommand", "" ) );
//        setPasswordString( m_kc->readEntry( "passwordString", "" ) );
        //the password string is a temporary holding string.
        setPasswordString( "" );
        setExecuteRef( m_kc->readEntry( "executeRef", "" ) );
        setPassRef( m_kc->readEntry( "passRef", "" ) );
        setSplitArchive( m_kc->readBoolEntry( "splitArchive", true ) );
        setPauseBetweenSlices( m_kc->readBoolEntry( "pauseBetweenSlices", true ) );
        setKeepPruned( m_kc->readBoolEntry( "keepPruned", true ) );
        setIgnoreDump( m_kc->readBoolEntry( "ignoreDump", false ) );
        setSliceSizeString( m_kc->readEntry( "sliceSizeString", "700" ) );
        setSliceSizeSuffix( m_kc->readNumEntry( "sliceSizeSuffix", KDar::MB ) );
        setFirstSliceSizeStrings( m_kc->readListEntry( "firstSliceSizeStrings" ) );
        setFirstSliceSizeSuffices( m_kc->readIntListEntry( "firstSliceSizeSuffices" ) );
        //Now that the various slice sizes are known, we can set the media type.
        setMedia( m_kc->readNumEntry( "media", KDar::CDR80 ) );
        setCompressionAlgorithm( m_kc->readNumEntry( "compressionAlgorithm", libdar::bzip2 ) );
        setCompressionLevel( m_kc->readNumEntry( "compressionLevel", 9 ) );
        setMinimumCompressionSizeInt( m_kc->readNumEntry( "minimumCompressionSizeInt", 150 ) );
        setCompressionMaskList(m_kc->readListEntry( "compressionMaskList" ) );
        setDirectoryMaskList( m_kc->readListEntry( "directoryMaskList" ) );
        setFileMaskList( m_kc->readListEntry( "fileMaskList" ) );
        setCompressionMaskType( m_kc->readNumEntry( "compressionMaskType", KDar::DO_NOT_INCLUDE ) );
        setDirectoryMaskTypes( m_kc->readListEntry( "directoryMaskTypes" ) );
        setFileMaskType( m_kc->readNumEntry( "fileMaskType", KDar::DO_NOT_INCLUDE ) );
    m_kc->setGroup("Extract");
        setExtractArchiveDirectory( m_kc->readEntry( "extractArchiveDirectory", "" ) );
        setNoDelete( m_kc->readBoolEntry( "noDelete", true ) );
        setRestoreRecent( m_kc->readBoolEntry( "restoreRecent", true ) );
        setFlatRestore( m_kc->readBoolEntry( "flatRestore", false ) );
        setAllowOverwrite( m_kc->readBoolEntry( "allowOverwrite", true ) );
        setWarnOnOverwrite( m_kc->readBoolEntry( "warnOnOverwrite", false ) );
    m_kc->setGroup( "Cryptography" );
        setCryptoAlgorithm( m_kc->readNumEntry( "cryptoAlgorithm", KDar::CRYPTO_NONE ) );
        setCryptoBlockSize( m_kc->readNumEntry( "cryptoBlockSize", 10240 ) );
        setStorePassword( m_kc->readBoolEntry( "storePassword", true ) );
    m_kc->setGroup("Diff");
        setHourShiftInt( m_kc->readNumEntry( "hourShiftInt", 0 ) );
    m_kc->setGroup("Fonts");
        QFont defaultFont = kapp->font();
        setGeneralFont( m_kc->readFontEntry( "generalFont", &defaultFont ) );
        setFileBrowserFont( m_kc->readFontEntry( "fileBrowserFont", &defaultFont ) );
        setMessageWindowFont( m_kc->readFontEntry( "messageWindowFont", &defaultFont ) );
        setStatusBarFont( m_kc->readFontEntry( "statusBarFont", &defaultFont ) );

    return true;
}

bool KDarConfig::writeGlobalSettings ()
{

    m_kc->setGroup("General");
        m_kc->writeEntry( "useCreateArchiveWizard", useCreateArchiveWizard()  );
        m_kc->writeEntry( "verbose", verbose() );
        m_kc->writeEntry( "logFile", logFile() );
        m_kc->writeEntry( "logLevel", logLevel() );
        m_kc->writeEntry( "dryRun", dryRun() );
        m_kc->writeEntry( "showKDarSplashScreen", showKDarSplashScreen() );
        QString lastArchive( archiveName() );
        if (lastArchive[0] != '/' )
        {
            lastArchive = "select archive";
        }
        m_kc->writeEntry( "lastArchive", lastArchive );
        m_kc->writeEntry( "archiveStorageDir", archiveStorageDirectory() );
        m_kc->writeEntry( "newArchiveName", newArchiveName() );
        m_kc->writeEntry( "directoryToBackup", directoryToBackup() );
        m_kc->writeEntry( "differentialBackup", differentialBackup() );
        m_kc->writeEntry( "differentialBackupArchiveName", differentialBackupArchiveName() );
    m_kc->setGroup("Filesystem");
        m_kc->writeEntry( "userEA", userEA() );
        m_kc->writeEntry( "systemEA", systemEA() );
        m_kc->writeEntry( "ignoreID", ignoreID() );
    m_kc->setGroup("Create / Isolate");
        m_kc->writeEntry( "inputPipe", inputPipe() );
        m_kc->writeEntry( "outputPipe", outputPipe() );
        m_kc->writeEntry( "executeCommand", executeCommand() );
        m_kc->writeEntry( "executeRef", executeRef() );
        m_kc->writeEntry( "passRef", passRef() );
        m_kc->writeEntry( "splitArchive", splitArchive() );
        m_kc->writeEntry( "pauseBetweenSlices", pauseBetweenSlices() );
        m_kc->writeEntry( "keepPruned", keepPruned() );
        m_kc->writeEntry( "ignoreDump", ignoreDump() );
        m_kc->writeEntry( "media", media() );
        m_kc->writeEntry( "sliceSizeString", sliceSizeString() );
        m_kc->writeEntry( "sliceSizeSuffix", sliceSizeSuffix() );
        m_kc->writeEntry( "firstSliceSizeStrings", firstSliceSizeStrings() );
        m_kc->writeEntry( "firstSliceSizeSuffices", firstSliceSizeSuffices() );
        m_kc->writeEntry( "compressionAlgorithm", compressionAlgorithm() );
        m_kc->writeEntry( "compressionLevel", compressionLevel() );
        m_kc->writeEntry( "minimumCompressionSizeInt", minimumCompressionSizeInt() );
        m_kc->writeEntry( "compressionMaskList", compressionMaskList() );
        m_kc->writeEntry( "directoryMaskList", directoryMaskList() );
        m_kc->writeEntry( "fileMaskList", fileMaskList() );
        m_kc->writeEntry( "compressionMaskType", compressionMaskType() );
        m_kc->writeEntry( "directoryMaskTypes", directoryMaskTypes() );
        m_kc->writeEntry( "fileMaskType", fileMaskType() );
    m_kc->setGroup("Extract");
        m_kc->writeEntry( "extractArchiveDirectory", extractArchiveDirectory() );
        m_kc->writeEntry( "noDelete", noDelete() );
        m_kc->writeEntry( "restoreRecent", restoreRecent() );
        m_kc->writeEntry( "flatRestore", flatRestore() );
        m_kc->writeEntry( "allowOverwrite", allowOverwrite() );
        m_kc->writeEntry( "warnOnOverwrite", warnOnOverwrite() );
    m_kc->setGroup( "Cryptography" );
        m_kc->writeEntry( "cryptoAlgorithm", cryptoAlgorithm() );
        m_kc->writeEntry( "cryptoBlockSize", cryptoBlockSize() );
        m_kc->writeEntry( "storePassword", storePassword() );
        kdDebug() << "KDarConfig::writeGlobalSettings(): storePassword() is " << storePassword() << endl;

    m_kc->setGroup("Diff");
        m_kc->writeEntry( "hourShiftInt", hourShiftInt() );
    m_kc->setGroup("Fonts");
        m_kc->writeEntry( "generalFont", generalFont() );
        m_kc->writeEntry( "fileBrowserFont", fileBrowserFont() );
        m_kc->writeEntry( "messageWindowFont", messageWindowFont() );
        m_kc->writeEntry( "statusBarFont", statusBarFont() );

    m_kc->sync();
    return true;
}

void KDarConfig::updateGUI()
{
    fillInDialog();
}

QString KDarConfig::cleanPath( const QString & path )
{
    //Leave trailing slashes as they are?
    //This removes trailing slashes...
    bool emptyString = path.isEmpty();
    if ( emptyString )
    {
        //Return the empty path.
        return path;
    }
    else
    {
        return QDir::cleanDirPath( path );
    }
}

QStringList KDarConfig::cleanPath( const QStringList & paths )
{
    //Iterate over the QStringList, cleaning paths as we go:
    QStringList cleanedPaths( paths );
    QStringList::Iterator cleaned = cleanedPaths.begin();
    for ( QStringList::ConstIterator it = paths.begin(); it != paths.end(); ++it )
    {
        ( *cleaned ) = cleanPath( *it );
        ++cleaned;
    }
    return cleanedPaths;
}

void KDarConfig::fillInDialog()
{
    if ( m_nodialog )
    {
        return;
    }
    kdDebug() << "KDarConfig::fillinDialog" << endl;
    //Set all dialog settings here.
    useCreateArchiveWizardCheckBox->setChecked( useCreateArchiveWizard() );
    verboseOutputCheckBox->setChecked( verbose() );
    logFileKURLRequester->lineEdit()->setText( logFile() );
    logNoneRadioButton->setChecked( logLevel() == KDar::LOG_NONE );
    logBasicRadioButton->setChecked( logLevel() == KDar::LOG_BASIC );
    logFullRadioButton->setChecked( logLevel() == KDar::LOG_FULL );
    showKDarSplashCheckBox->setChecked( showKDarSplashScreen() );
    archiveStorageDirKURLRequester->lineEdit()->setText( archiveStorageDirectory() );
    userEACheckBox->setChecked( !userEA() );
    systemEACheckBox->setChecked( !systemEA() );
    ignoreIDCheckBox->setChecked( ignoreID() );
    compressionLevelSpinBox->setValue( compressionLevel() );
    noneRadioButton->setChecked( compressionAlgorithm() == libdar::none );
    gzipRadioButton->setChecked( compressionAlgorithm() == libdar::gzip );
    bzip2RadioButton->setChecked( compressionAlgorithm() == libdar::bzip2 );
    minCompressionSizeSpinBox->setValue( minimumCompressionSizeInt() );
    keepPrunedCheckBox->setChecked( keepPruned() );
    ignoreDumpCheckBox->setChecked( ignoreDump() );
    extractNoDeleteCheckBox->setChecked( noDelete() );
    extractMostRecentCheckBox->setChecked( restoreRecent() );
    extractFlatCheckBox->setChecked( flatRestore() );
    extractAllowOverwriteCheckBox->setChecked( allowOverwrite() );
    extractWarnOverwriteCheckBox->setChecked( warnOnOverwrite() );
    hourShiftBox->setValue( hourShiftInt() );
    generalKFontRequester->setFont( generalFont() );
    fileBrowserKFontRequester->setFont( fileBrowserFont() );
    messageWindowKFontRequester->setFont( messageWindowFont() );
    statusBarKFontRequester->setFont( statusBarFont() );

    //Slice sizes
    splitArchiveCheckBox->setChecked( splitArchive() );
    pauseCheckBox->setChecked( pauseBetweenSlices() );
    mediaTypeKComboBox->setCurrentItem( media() );
    setMediaSize( media() );
    commandBetweenSliceCreationKLineEdit->setText( executeCommand() );
    //Encryption
    int algo = cryptoAlgorithm();
    //Disable scrambling:
    if ( algo == KDar::CRYPTO_BLOWFISH ) algo = 1;
    cryptoAlgorithmKComboBox->setCurrentItem( algo );
    cryptoBlockSizeKIntNumInput->setValue( cryptoBlockSize() );
    cryptoStorePasswordCheckBox->setChecked( storePassword() );
    //Compression filters
    fillListView ( compressionFiltersListView, compressionMaskList() );
    compressionTypeComboBox->setCurrentItem( compressionMaskType() );
    //File filters
    fillListView ( fileFiltersListView, fileMaskList() );
    fileTypeComboBox->setCurrentItem( fileMaskType() );
    //Directories
    loadDirectoryFilters();
}

void KDarConfig::setGlobalConfig( KConfig * kconfig )
{
    m_kc = kconfig;
}

void KDarConfig::setArchiveName( const QString & archiveName )
{
    m_archiveName = cleanPath( archiveName );
}

void KDarConfig::setArchiveStorageDirectory( const QString & archiveStorageDirectory )
{
    m_archiveStorageDirectory = cleanPath( archiveStorageDirectory );
}

void KDarConfig::setNewArchiveName( const QString & newArchiveName )
{
    m_newArchiveName = cleanPath( newArchiveName );
}

void KDarConfig::setDirectoryToBackup( const QString & directoryToBackup )
{
    m_directoryToBackup = cleanPath( directoryToBackup );
}

void KDarConfig::setDifferentialBackup( const bool differentialBackup )
{
    m_differentialBackup = differentialBackup;
}

void KDarConfig::setDifferentialBackupArchiveName( const QString & differentialBackupArchiveName )
{
    m_differentialBackupArchiveName = cleanPath( differentialBackupArchiveName );
}

void KDarConfig::setInputPipe( const QString & inputPipe )
{
    m_inputPipe = cleanPath( inputPipe );
}

void KDarConfig::setOutputPipe( const QString & outputPipe )
{
    m_outputPipe = cleanPath( outputPipe );
}

void KDarConfig::setExecuteCommand( const QString & executeCommand )
{
    m_executeCommand = executeCommand;
}

void KDarConfig::setCryptoAlgorithm( const int cryptoAlgorithm )
{
    m_cryptoAlgorithm = cryptoAlgorithm;
    //This ensures that the crypto_scrambling is disabled until
    //libdar supports automatic encryption detection.
    if ( cryptoAlgorithm == KDar::CRYPTO_SCRAMBLE ) m_cryptoAlgorithm = KDar::CRYPTO_BLOWFISH;
}

void KDarConfig::setCryptoBlockSize( const int cryptoBlockSize )
{
    m_cryptoBlockSize = cryptoBlockSize;
}

void KDarConfig::setStorePassword( bool storePassword )
{
    m_storePassword = storePassword;
    kdDebug() << "KDarConfig::setStorePassword(): m_storePassword is " << m_storePassword << endl;
}

void KDarConfig::setPasswordString( const QString & passwordString )
{
    m_passwordString = passwordString;
}

void KDarConfig::setExecuteRef( const QString & executeRef )
{
    m_executeRef = executeRef;
}

void KDarConfig::setPassRef( const QString & passRef )
{
    m_passRef = passRef;
}

void KDarConfig::setVerbose( const bool verbose )
{
    m_verbose = verbose;
}

void KDarConfig::setAllowOverwrite( const bool allowOverwrite )
{
    m_allowOverwrite = allowOverwrite;
}

void KDarConfig::setWarnOnOverwrite( const bool warnOnOverwrite )
{
    m_warnOnOverwrite = warnOnOverwrite;
}

void KDarConfig::setUserEA( const bool userEA )
{
    m_userEA = userEA;
}

void KDarConfig::setSystemEA( const bool systemEA )
{
    m_systemEA = systemEA;
}

void KDarConfig::setIgnoreID( const bool ignoreID )
{
    m_ignoreID = ignoreID;
}

void KDarConfig::setNoDelete( const bool noDelete )
{
    m_noDelete = noDelete;
}

void KDarConfig::setRestoreRecent( const bool restoreRecent )
{
    m_restoreRecent = restoreRecent;
}

void KDarConfig::setFlatRestore( const bool flatRestore )
{
    m_flatRestore = flatRestore;
}

void KDarConfig::setUseCreateArchiveWizard( const bool useCreateArchiveWizard )
{
    m_useCreateArchiveWizard = useCreateArchiveWizard;
}

void KDarConfig::setSplitArchive( const bool splitArchive )
{
    m_splitArchive = splitArchive;
}

void KDarConfig::setPauseBetweenSlices( const bool pauseBetweenSlices )
{
    m_pauseBetweenSlices = pauseBetweenSlices;
}

void KDarConfig::setKeepPruned( const bool keepPruned )
{
    m_keepPruned = keepPruned;
}

void KDarConfig::setIgnoreDump( const bool ignoreDump )
{
    m_ignoreDump = ignoreDump;
}

void KDarConfig::setMedia( const int media )
{
    m_media = media;
}

void KDarConfig::setSliceSizeString( const QString & sliceSizeString )
{
    //Make sure we have a valid slice size: whole numbers only.
    //i.e. suppose the config file is modified...
    QRegExpValidator *validator = ( QRegExpValidator * ) sliceSizeLineEdit->validator();
    if ( validator )
    {
        QString input = sliceSizeString;
        int pos = input.length() - 1;
        if ( validator->validate( input, pos ) == QValidator::Acceptable )
        {
            m_sliceSizeString = sliceSizeString;
            mediaSizes[ KDar::CUSTOM ] = sliceSizeString;
        }
    }
    else
    {
        kdDebug() << "KDarConfig::setSliceSizeString() error: QValidator is not set." << endl;
    }
}

void KDarConfig::setSliceSizeSuffix( const int sliceSizeSuffix )
{
    m_sliceSizeSuffix = sliceSizeSuffix;
    mediaSuffices[ KDar::CUSTOM ] = sliceSizeSuffix;
}

void KDarConfig::setFirstSliceSizeStrings( const QStringList & firstSliceSizeStrings)
{
    if ( !firstSliceSizeStrings.empty() )
    {
        mediaFirstSizes = firstSliceSizeStrings;
    }
}

void KDarConfig::setFirstSliceSizeString( int medium, const QString & firstSliceSizeString )
{
    //Make sure we have a valid slice size: whole numbers only.
    //i.e. suppose the config file is modified...
    QRegExpValidator *validator = ( QRegExpValidator * ) firstSliceSizeLineEdit->validator();
    if ( validator )
    {
        QString input = firstSliceSizeString;
        int pos = input.length() - 1;
        if ( validator->validate( input, pos ) == QValidator::Acceptable )
        {
            mediaFirstSizes[ medium ] = firstSliceSizeString;
        }
    }
    else
    {
        kdDebug() << "KDarConfig::setFirstSliceSizeString() error: QValidator is not set." << endl;
    }
}

void KDarConfig::setFirstSliceSizeSuffices( const QValueList<int> & firstSliceSizeSuffices )
{
    if ( !firstSliceSizeSuffices.empty() )
    {
        mediaFirstSuffices = firstSliceSizeSuffices;
    }
}

void KDarConfig::setFirstSliceSizeSuffix( int medium, const int firstSliceSizeSuffix )
{
    mediaFirstSuffices[ medium ] = firstSliceSizeSuffix;
}

void KDarConfig::setCompressionAlgorithm( const int compressionAlgorithm )
{
    m_compressionAlgorithm = compressionAlgorithm;
}

void KDarConfig::setCompressionLevel( const int compressionLevel )
{
    m_compressionLevel = compressionLevel;
}

void KDarConfig::setMinimumCompressionSizeInt( const int minimumCompressionSizeInt )
{
    m_minimumCompressionSizeInt = minimumCompressionSizeInt;
}

void KDarConfig::setHourShiftInt( const int hourShiftInt )
{
    m_hourShiftInt = hourShiftInt;
}

void KDarConfig::setCompressionMaskList( const QStringList & compressionMaskList )
{
    //Do we need to clean this path?
    //i.e. are these regexp filters? Does QDir::cleanDirPath affect them?
    m_compressionMaskList = cleanPath( compressionMaskList );
}

void KDarConfig::setDirectoryMaskList( const QStringList & directoryMaskList )
{
    //Do we need to clean this path?
    m_directoryMaskList = cleanPath( directoryMaskList );
}

void KDarConfig::setFileMaskList( const QStringList & fileMaskList )
{
    //Do we need to clean this path?
    //i.e. are these regexp filters? Does QDir::cleanDirPath affect them?
    m_fileMaskList = cleanPath( fileMaskList );
}

void KDarConfig::setCompressionMaskType( const int compressionMaskType )
{
    m_compressionMaskType = compressionMaskType;
}

void KDarConfig::setDirectoryMaskTypes( const QStringList & directoryMaskTypes )
{
    m_directoryMaskTypes = directoryMaskTypes;
}

void KDarConfig::setFileMaskType( const int fileMaskType )
{
    m_fileMaskType = fileMaskType;
}

void KDarConfig::setGeneralFont( const QFont & generalFont )
{
    m_generalFont = generalFont;
}

void KDarConfig::setFileBrowserFont( const QFont & fileBrowserFont )
{
    m_fileBrowserFont = fileBrowserFont;
}

void KDarConfig::setMessageWindowFont( const QFont & messageWindowFont )
{
    m_messageWindowFont = messageWindowFont;
}

void KDarConfig::setStatusBarFont( const QFont & statusBarFont )
{
    m_statusBarFont = statusBarFont;
}

void KDarConfig::setShowKDarSplashScreen( const bool showKDarSplashScreen )
{
    m_showKDarSplashScreen = showKDarSplashScreen;
}

void KDarConfig::setDryRun( const bool dryRun )
{
    m_dryRun = dryRun;
}

void KDarConfig::setIsolationArchiveName( const QString & isolationArchiveName )
{
    m_isolationArchiveName = cleanPath( isolationArchiveName );
}

void KDarConfig::setDiffArchiveDirectory( const QString & diffArchiveDirectory )
{
    m_diffArchiveDirectory = cleanPath( diffArchiveDirectory );
}

void KDarConfig::setExtractArchiveDirectory( const QString & extractArchiveDirectory )
{
    m_extractArchiveDirectory = cleanPath( extractArchiveDirectory );
}

void KDarConfig::setNewArchiveBasename( const QString & newArchiveBaseName )
{
    m_newArchiveBasename = cleanPath( newArchiveBaseName );
}

void KDarConfig::setLogFile( const QString & logFile )
{
    m_logFile = cleanPath( logFile );
}

void KDarConfig::setLogLevel( const int logLevel )
{
    m_logLevel = logLevel;
}

//member varible reading functions:

KConfig * KDarConfig::globalConfig() const
{
    return m_kc;
}

QString KDarConfig::archiveName() const
{
    return m_archiveName;
}

QString KDarConfig::archiveStorageDirectory() const
{
    //If we have an empty string, leave it alone
    //This allows KDar to prompt the user for a default directory
    //when starting for the first time.
    if ( m_archiveStorageDirectory.isEmpty() ) return QString( "" );
    //Otherwise, make sure we have a trailing slash:
    if ( m_archiveStorageDirectory.endsWith( "/" ) )
    {
        return m_archiveStorageDirectory;
    }
    else
    {
        return m_archiveStorageDirectory + "/";
    }
}

QString KDarConfig::newArchiveName( ) const
{
    return m_newArchiveName;
}

QString KDarConfig::directoryToBackup() const
{
    return m_directoryToBackup;
}

bool KDarConfig::differentialBackup() const
{
    return m_differentialBackup;
}

QString KDarConfig::differentialBackupArchiveName() const
{
    return m_differentialBackupArchiveName;
}

QString KDarConfig::inputPipe() const
{
    return m_inputPipe;
}

QString KDarConfig::outputPipe() const
{
    return m_outputPipe;
}

QString KDarConfig::executeCommand() const
{
    return m_executeCommand;
}

int KDarConfig::cryptoAlgorithm() const
{
    return m_cryptoAlgorithm;
}

int KDarConfig::cryptoBlockSize() const
{
    return m_cryptoBlockSize;
}

bool KDarConfig::storePassword() const
{
    return m_storePassword;
}

QString KDarConfig::passwordString() const
{
    return m_passwordString;
}

QString KDarConfig::executeRef() const
{
    return m_executeRef;
}

QString KDarConfig::passRef() const
{
    return m_passRef;
}

bool KDarConfig::verbose() const
{
    return m_verbose;
}

bool KDarConfig::allowOverwrite() const
{
    return m_allowOverwrite;
}

bool KDarConfig::warnOnOverwrite() const
{
    return m_warnOnOverwrite;
}

bool KDarConfig::userEA() const
{
    return m_userEA;
}

bool KDarConfig::systemEA() const
{
    return m_systemEA;
}

bool KDarConfig::ignoreID() const
{
    return m_ignoreID;
}

bool KDarConfig::noDelete() const
{
    return m_noDelete;
}

bool KDarConfig::restoreRecent() const
{
    return m_restoreRecent;
}

bool KDarConfig::flatRestore() const
{
    return m_flatRestore;
}

bool KDarConfig::useCreateArchiveWizard() const
{
    return m_useCreateArchiveWizard;
}

bool KDarConfig::splitArchive() const
{
    return m_splitArchive;
}

bool KDarConfig::pauseBetweenSlices() const
{
    return m_pauseBetweenSlices;
}

bool KDarConfig::keepPruned() const
{
    return m_keepPruned;
}

bool KDarConfig::ignoreDump() const
{
    return m_ignoreDump;
}

int KDarConfig::media() const
{
    return m_media;
}

QString KDarConfig::mediaText() const
{
    return statusBarMediaNames[ media() ];
}

QString KDarConfig::sliceSizeString() const
{
    return m_sliceSizeString;
}

int KDarConfig::sliceSizeSuffix() const
{
    return m_sliceSizeSuffix;
}

QString KDarConfig::firstSliceSizeString() const
{
    return mediaFirstSizes[ media() ];
}

int KDarConfig::firstSliceSizeSuffix() const
{
    return mediaFirstSuffices[ media() ];
}

QStringList KDarConfig::firstSliceSizeStrings() const
{
    return mediaFirstSizes;
}

QValueList<int> KDarConfig::firstSliceSizeSuffices() const
{
    return mediaFirstSuffices;
}

int KDarConfig::compressionAlgorithm() const
{
    return m_compressionAlgorithm;
}

int KDarConfig::compressionLevel() const
{
    return m_compressionLevel;
}

int KDarConfig::minimumCompressionSizeInt() const
{
    return m_minimumCompressionSizeInt;
}

int KDarConfig::hourShiftInt() const
{
    return m_hourShiftInt;
}

QStringList KDarConfig::compressionMaskList() const
{
    return m_compressionMaskList;
}

QStringList KDarConfig::directoryMaskList() const
{
    return m_directoryMaskList;
}

QStringList KDarConfig::fileMaskList() const
{
    return m_fileMaskList;
}

int KDarConfig::compressionMaskType() const
{
    return m_compressionMaskType;
}

QStringList KDarConfig::directoryMaskTypes() const
{
    return m_directoryMaskTypes;
}

int KDarConfig::fileMaskType() const
{
    return m_fileMaskType;
}

QFont KDarConfig::generalFont() const
{
    return m_generalFont;
}

QFont KDarConfig::fileBrowserFont() const
{
    return m_fileBrowserFont;
}

QFont KDarConfig::messageWindowFont() const
{
    return m_messageWindowFont;
}

QFont KDarConfig::statusBarFont() const
{
    return m_statusBarFont;
}

bool KDarConfig::showKDarSplashScreen() const
{
    return m_showKDarSplashScreen;
}

bool KDarConfig::dryRun() const
{
    return m_dryRun;
}

QString KDarConfig::isolationArchiveName() const
{
    return m_isolationArchiveName;
}

QString KDarConfig::diffArchiveDirectory() const
{
    return m_diffArchiveDirectory;
}

QString KDarConfig::extractArchiveDirectory() const
{
    return m_extractArchiveDirectory;
}

QString KDarConfig::newArchiveBasename() const
{
    return m_newArchiveBasename;
}

QString KDarConfig::logFile() const
{
    return m_logFile;
}

int KDarConfig::logLevel() const
{
    return m_logLevel;
}

libdar::infinint KDarConfig::libdarSliceSize()
{
    //code for calculating the slice size from strings and suffices.
    //defaults to 0 when not splitting archives:
    libdar::infinint libdarSliceSize( (libdar::U_I) 0 );
    if ( m_splitArchive )
    {
        //Catch possible exception from libdar::infinint being too small (i.e. 32-bit or 64-bit)
        try
        {
            libdarSliceSize = libdar::deci( kdar::toStdString( mediaSizes[ media() ] ) ).computer()*libdar::infinint( (libdar::U_I) 1024 ).power( (libdar::U_I) mediaSuffices[ media() ] );
            kdDebug() << "The slice size is " << libdar::deci(libdarSliceSize).human().c_str() << endl;
        }
        catch(libdar::Egeneric &e)
        {
// TODO    What is the best way to display info to the user here?
//             KDarInteraction ki( ;
//             libdar::user_interaction_warning(e.get_message());
        }
    }
    return libdarSliceSize;
}

libdar::infinint KDarConfig::libdarFirstSliceSize()
{
    //code for calculating the slice size from strings and suffices.
    libdar::infinint libdarFirstSliceSize( (libdar::U_I) 0 );
    if ( m_splitArchive )
    {
        //Catch possible exception from libdar::infinint being too small (i.e. 32-bit or 64-bit)
        try
        {
            libdarFirstSliceSize = libdar::deci( kdar::toStdString( mediaFirstSizes[ media() ] ) ).computer()*libdar::infinint( (libdar::U_I) 1024 ).power( (libdar::U_I) mediaFirstSuffices[ media() ] );
            kdDebug() << "The slice size is " << libdar::deci(libdarFirstSliceSize).human().c_str() << endl;
        }
        catch(libdar::Egeneric &e)
        {
// TODO What is the best way to display this info to the user?
//             libdar::user_interaction_warning(e.get_message());
        }
    }
    return libdarFirstSliceSize;
}

void KDarConfig::activateHelp( void )
{
    slotHelp();
}

void KDarConfig::slotHelp( void )
{
    kapp->invokeHelp( "configuration", "kdar" );
}

void KDarConfig::operator=( const KDarConfig &_kdc )
{
    //set up an integer validator on the slicesize lineEdits
    kdDebug() << "kdarView::kdarView(): integer validator for lineEdits" << endl;
    QRegExp rx("[1-9][0-9]*");
    QRegExpValidator * restrictToWholeNumbers = new QRegExpValidator(rx, this);
    sliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    firstSliceSizeLineEdit->setValidator( restrictToWholeNumbers );
    //Default media types:
    setArchiveName( _kdc.archiveName() );
    setArchiveStorageDirectory( _kdc.archiveStorageDirectory() );
    setExtractArchiveDirectory( _kdc.extractArchiveDirectory() );
    setInputPipe( _kdc.inputPipe() );
    setOutputPipe( _kdc.outputPipe() );
    setExecuteCommand( _kdc.executeCommand() );
    setCryptoAlgorithm( _kdc.cryptoAlgorithm() );
    setCryptoBlockSize( _kdc.cryptoBlockSize() );
    setStorePassword( _kdc.storePassword() );
    setExecuteRef( _kdc.executeRef() );
    setPassRef( _kdc.passRef() );

    setVerbose( _kdc.verbose() );
    setLogFile( _kdc.logFile() );
    setLogLevel( _kdc.logLevel() );
    setAllowOverwrite( _kdc.allowOverwrite() );
    setWarnOnOverwrite( _kdc.warnOnOverwrite() );
    setUserEA( _kdc.userEA() );
    setSystemEA( _kdc.systemEA() );
    setIgnoreID( _kdc.ignoreID() );
    setNoDelete( _kdc.noDelete() );
    setRestoreRecent( _kdc.restoreRecent() );
    setFlatRestore( _kdc.flatRestore() );

    setUseCreateArchiveWizard( _kdc.useCreateArchiveWizard() );
    setSplitArchive( _kdc.splitArchive() );
    setPauseBetweenSlices( _kdc.pauseBetweenSlices() );
    setKeepPruned( _kdc.keepPruned() );
    setIgnoreDump( _kdc.ignoreDump() );
    setMedia( _kdc.media() );
    setSliceSizeString( _kdc.sliceSizeString() );
    setSliceSizeSuffix( _kdc.sliceSizeSuffix() );
    setFirstSliceSizeStrings( _kdc.firstSliceSizeStrings() );
    setFirstSliceSizeSuffices( _kdc.firstSliceSizeSuffices() );
    setCompressionAlgorithm( _kdc.compressionAlgorithm() );
    setCompressionLevel( _kdc.compressionLevel() );
    setMinimumCompressionSizeInt( _kdc.minimumCompressionSizeInt() );
    setHourShiftInt( _kdc.hourShiftInt() );

    setCompressionMaskList(_kdc.compressionMaskList() );
    setDirectoryMaskList( _kdc.directoryMaskList() );
    setFileMaskList( _kdc.fileMaskList() );
    setCompressionMaskType( _kdc.compressionMaskType() );
    setDirectoryMaskTypes( _kdc.directoryMaskTypes() );
    setFileMaskType( _kdc.fileMaskType() );

    setGeneralFont( _kdc.generalFont() );
    setFileBrowserFont( _kdc.fileBrowserFont() );
    setMessageWindowFont( _kdc.messageWindowFont() );
    setStatusBarFont( _kdc.statusBarFont() );

    setShowKDarSplashScreen( _kdc.showKDarSplashScreen() );
    setDryRun( _kdc.dryRun() );

    fillInDialog();
}

void KDarConfig::fillListView(QListView *listView, QStringList stringList)
{
    QStringList tmpList;
    //clear the listView, so that we can fill it from scratch.
    if ( listView ) listView->clear();
    if ( listView && !stringList.empty() )
    {
        for (QStringList::Iterator it = stringList.begin(); it != stringList.end(); ++it)
        {
            tmpList = makeQStringList( listView );
            if ( !tmpList.contains( *it ) )
            {
                (void) new QListViewItem(listView, *it);
            }
        }
    }
}

void KDarConfig::loadDirectoryFilters()
{
    QStringList directoryList = directoryMaskList();
    QStringList maskTypes = directoryMaskTypes();
    //clear the listview, so we can fill it from scratch.
    directoryFiltersListView->clear();
    kdDebug() << "KDarConfig::loadDirectoryFilters(): maskTypes.count() = " << maskTypes.count() << endl;
    kdDebug() << "KDarConfig::loadDirectoryFilters(): directoryList.count() = " << directoryList.count() << endl;
    if ( !directoryList.empty() && !maskTypes.empty() && ( maskTypes.count() == directoryList.count() ) )
    {
        kdDebug() << "KDarConfig::loadDirectoryFilters(): Loading directory filters..." << endl;
        QStringList::Iterator typeIterator = maskTypes.begin();
        for ( QStringList::Iterator directoryIterator = directoryList.begin(); directoryIterator != directoryList.end(); ++directoryIterator )
        {
            QStringList tmpList = makeQStringList( directoryFiltersListView, 1 );
            if ( !tmpList.contains( *directoryIterator ) || ( *typeIterator == i18n( "automatic filter type", "Auto" ) ) )
            {
                QListViewItem* item = new QListViewItem( directoryFiltersListView, *typeIterator, *directoryIterator );
                if ( *typeIterator == i18n( "automatic filter type", "Auto" ) )
                {
                    item->setEnabled( false );
                }
            }
            ++typeIterator;
        }
    }
}

//Slots

void KDarConfig::slotArchiveName( const QString & archiveName )
{
    setArchiveName( archiveName );
    emit configChanged();
}

void KDarConfig::slotArchiveStorageDirectory( const QString & archiveStorageDirectory )
{
    if ( archiveStorageDirectory.isNull() )
    {
        kdDebug() << "KDarConfig::slotArchiveStorageDirectory(): archiveStorageDirectory is NULL!" << endl;
        return;
    }
    setArchiveStorageDirectory( archiveStorageDirectory );
    //update the auto directory filter
    QStringList directoryList = directoryMaskList();
    QStringList maskTypes = directoryMaskTypes();
    QStringList::Iterator directoryIterator = directoryList.begin();
    for ( QStringList::Iterator typeIterator = maskTypes.begin(); typeIterator != maskTypes.end(); ++typeIterator )
    {
        //Assumes we can only have 1 auto directory filter
        if ( *typeIterator == i18n( "automatic filter type", "Auto" ) )
        {
            *directoryIterator = archiveStorageDirectory;
        }
        ++directoryIterator;
    }
    setDirectoryMaskList( directoryList );
    loadDirectoryFilters();

    kdDebug() << "KDarConfig::slotArchiveStorageDirectory(): archiveStorageDirectory() is " << archiveStorageDirectory << endl;
    emit configChanged();
}

void KDarConfig::slotInputPipe( const QString & inputPipe )
{
    setInputPipe( inputPipe );
    emit configChanged();
}

void KDarConfig::slotOutputPipe( const QString & outputPipe )
{
    setOutputPipe( outputPipe );
    emit configChanged();

}

void KDarConfig::slotExecuteCommand( const QString & executeCommand )
{
    setExecuteCommand( executeCommand );
    emit configChanged();
}

void KDarConfig::slotCryptoAlgorithm( int algorithm )
{
    setCryptoAlgorithm( algorithm );
    emit configChanged();
}

void KDarConfig::slotCryptoBlockSize( int blockSize )
{
    setCryptoBlockSize( blockSize );
    emit configChanged();
}

void KDarConfig::slotExecuteRef( const QString & executeRef )
{
    setExecuteRef( executeRef );
    emit configChanged();
}

void KDarConfig::slotStorePassword( bool storePassword )
{
    setStorePassword( storePassword );
    kdDebug() << "KDarConfig::slotStorePassword(): storePassword is " << storePassword << endl;
    emit configChanged();
}

void KDarConfig::slotChangePassword()
{
    emit changeWalletPassword();
}

void KDarConfig::slotPassRef( const QString & passRef )
{
    setPassRef( passRef );
    emit configChanged();
}

void KDarConfig::slotVerbose( bool verbose )
{
    setVerbose( verbose );
    emit configChanged();
}

void KDarConfig::slotAllowOverwrite( bool allowOverwrite )
{
    setAllowOverwrite( allowOverwrite );
    emit configChanged();
}

void KDarConfig::slotWarnOnOverwrite( bool warnOnOverwrite )
{
    setWarnOnOverwrite( warnOnOverwrite );
    emit configChanged();
}

void KDarConfig::slotUserEA( bool disableUserEA )
{
    setUserEA( !disableUserEA );
    emit configChanged();
}

void KDarConfig::slotSystemEA( bool disableSystemEA )
{
    setSystemEA( !disableSystemEA );
    emit configChanged();
}

void KDarConfig::slotIgnoreID( bool ignoreID )
{
    setIgnoreID( ignoreID );
    emit configChanged();
}

void KDarConfig::slotNoDelete( bool noDelete )
{
    setNoDelete( noDelete );
    emit configChanged();
}

void KDarConfig::slotRestoreRecent( bool restoreRecent )
{
    setRestoreRecent( restoreRecent );
    emit configChanged();
}

void KDarConfig::slotFlatRestore( bool flatRestore )
{
    setFlatRestore( flatRestore );
    emit configChanged();
}

void KDarConfig::slotUseCreateArchiveWizard( bool useCreateArchiveWizard )
{
    setUseCreateArchiveWizard( useCreateArchiveWizard );
    emit configChanged();
}

void KDarConfig::slotSplitArchive( bool splitArchive )
{
    setSplitArchive( splitArchive );
    emit configChanged();
}

void KDarConfig::slotPauseBetweenSlices( bool pauseBetweenSlices )
{
    setPauseBetweenSlices( pauseBetweenSlices );
    emit configChanged();
}

void KDarConfig::slotKeepPruned( bool keepPruned )
{
    setKeepPruned( keepPruned );
    emit configChanged();
}

void KDarConfig::slotIgnoreDump( bool ignoreDump )
{
    setIgnoreDump( ignoreDump );
    emit configChanged();
}

void KDarConfig::slotMedia( int media )
{
    kdDebug() << "KDarConfig::slotMedia(): media is " << media << endl;
    setMedia( media );
    setMediaSize( media );
    emit configChanged();
}

void KDarConfig::slotSliceSizeString( const QString & sliceSizeString )
{
    if ( media() != KDar::CUSTOM )
    {
        setMedia( KDar::CUSTOM );
        mediaTypeKComboBox->blockSignals( true );
        mediaTypeKComboBox->setCurrentItem( KDar::CUSTOM );
        mediaTypeKComboBox->blockSignals( false );
    }
    setSliceSizeString( sliceSizeString );
    emit configChanged();
}

void KDarConfig::slotSliceSizeSuffix( int sliceSizeSuffix )
{
    if ( media() != KDar::CUSTOM )
    {
        setMedia( KDar::CUSTOM );
        mediaTypeKComboBox->blockSignals( true );
        mediaTypeKComboBox->setCurrentItem( KDar::CUSTOM );
        mediaTypeKComboBox->blockSignals( false );
    }
    setSliceSizeSuffix( sliceSizeSuffix );
    emit configChanged();
}

void KDarConfig::slotFirstSliceSizeString( const QString & firstSliceSizeString )
{
    //All media can have a custom first slice size string
    setFirstSliceSizeString( media(), firstSliceSizeString );
    emit configChanged();
}

void KDarConfig::slotFirstSliceSizeSuffix( int firstSliceSizeSuffix )
{
    //All media cna have a custom first slice size suffix
    setFirstSliceSizeSuffix( media(), firstSliceSizeSuffix );
    emit configChanged();
}

void KDarConfig::slotCompressionAlgorithmBZip2( bool toggled )
{
    if (toggled)
    {
        setCompressionAlgorithm( libdar::bzip2 );
    }
    emit configChanged();
}

void KDarConfig::slotCompressionAlgorithmGZip( bool toggled )
{
    if (toggled)
    {
        setCompressionAlgorithm( libdar::gzip );
    }
    emit configChanged();
}

void KDarConfig::slotCompressionAlgorithmNone( bool toggled )
{
    if (toggled)
    {
        setCompressionAlgorithm( libdar::none );
    }
    emit configChanged();
}

void KDarConfig::slotCompressionLevel( int compressionLevel )
{
    setCompressionLevel( compressionLevel );
    emit configChanged();
}

void KDarConfig::slotMinimumCompressionSizeInt( int minimumCompressionSizeInt )
{
    setMinimumCompressionSizeInt( minimumCompressionSizeInt );
    emit configChanged();
}

void KDarConfig::slotHourShiftInt( int hourShiftInt )
{
    setHourShiftInt( hourShiftInt );
    emit configChanged();
}

void KDarConfig::slotRemoveAllFileFilter()
{
    fileFiltersListView->clear();
    emit fileMaskListChanged();
}

void KDarConfig::slotRemoveFileFilter()
{
    QListViewItem * currItem = fileFiltersListView->selectedItem();
    if (!currItem) return;
    delete currItem;
    emit fileMaskListChanged();
}

void KDarConfig::slotEditFileFilter()
{
    QListViewItem* filterItem = fileFiltersListView->selectedItem();
    if ( filterItem )
    {
        m_fileDlg->fileFilterValueComboBox->setEditText( filterItem->text( 0 ) );
        if ( m_fileDlg->exec() )
        {
            if ( filterItem ) delete filterItem;
            newFileFilter();
        }
    }
}

void KDarConfig::slotNewFileFilter()
{
    if ( m_fileDlg->exec() )
    {
        newFileFilter();
    }
}

void KDarConfig::newFileFilter()
{
    QString fileMask = m_fileDlg->fileFilterValueComboBox->currentText();
    if ( !fileMask.isEmpty() )
    {
        if (fileMask != QString( i18n( "select all filters from list", "All of the above" ) ) )
        {
            //Add the filter if it is not already in the list
            QStringList tmpList = makeQStringList( fileFiltersListView );
            if ( !tmpList.contains( fileMask ) )
            {
                (void) new QListViewItem( fileFiltersListView, fileMask);
            }
        }
        else
        {
            for (int i = 0; i<(m_fileDlg->fileFilterValueComboBox->count()-1);++i)
            {
                //Add the filter if it is not already in the list
                QStringList tmpList = makeQStringList( fileFiltersListView );
                fileMask = m_fileDlg->fileFilterValueComboBox->text(i);
                if ( !tmpList.contains( fileMask ) )
                {
                    (void) new QListViewItem(fileFiltersListView, fileMask );
                }
            }
        }
        emit fileMaskListChanged();
    }
}

void KDarConfig::slotRemoveAllDirectoryFilter()
{
    directoryFiltersListView->clear();
    emit directoryMaskListChanged();
}

void KDarConfig::slotRemoveDirectoryFilter()
{
    QListViewItem * currItem = directoryFiltersListView->selectedItem();
    if (!currItem) return;
    delete currItem;
    emit directoryMaskListChanged();
}

void KDarConfig::slotEditDirectoryFilter()
{
    //Set the filesystem radiobutton, since if the user wants to edit the
    //filter, they can use the keyboard from the lineEdit --- you can't
    //edit the combobox anyway. Otherwise, we'd have to come up with a more
    //sophisticated filter system that keeps track of whether a given
    //filter came from the "list" or the "filesystem" part of the dialog.
    //The user can still select "list" and choose a combobox item.
    QListViewItem* filterItem = directoryFiltersListView->selectedItem();
    if ( filterItem )
    {
        m_directoryDlg->filesystemRadioButton->setChecked( true );
        QString type = filterItem->text( 0 );
        if ( type == i18n( "filter type for exclusion", "Exclude" ) )
        {
            m_directoryDlg->directoryFilesystemTypeComboBox->setCurrentItem( KDar::DO_NOT_INCLUDE );
        }
        else if ( type == i18n( "filter type for inclusion", "Include" ) )
        {
            m_directoryDlg->directoryFilesystemTypeComboBox->setCurrentItem( KDar::INCLUDE );
        }
        m_directoryDlg->directoryFilterValueKURLRequester->lineEdit()->setText( filterItem->text( 1 ) );
        if ( m_directoryDlg->exec() )
        {
            if ( filterItem ) delete filterItem;
            newDirectoryFilter();
        }
    }
}

void KDarConfig::slotNewDirectoryFilter()
{
    if ( m_directoryDlg->exec() )
    {
        newDirectoryFilter();
    }
}

void KDarConfig::newDirectoryFilter()
{
    if ( m_directoryDlg->listRadioButton->isChecked() )
    {
        QString subDir;
        subDir = m_directoryDlg->directoryListComboBox->currentText();
        if ( !subDir.isEmpty() )
        {
            if ( subDir != i18n( "select all filters from list", "All of the above" ) )
            {
                //Add the filter if it is not already in the list
                QStringList tmpList = makeQStringList( directoryFiltersListView, 1 );
                if ( !tmpList.contains( subDir ) )
                {
                    QString type = m_directoryDlg->directoryListTypeComboBox->currentText();
                    (void) new QListViewItem( directoryFiltersListView, type, subDir );
                }
            }
            else
            {
                QString type = m_directoryDlg->directoryListTypeComboBox->currentText();
                for (int i = 0; i<( m_directoryDlg->directoryListComboBox->count()-1 );++i)
                {
                    //Add the filter if it is not already in the list
                    QStringList tmpList = makeQStringList( directoryFiltersListView, 1 );
                    subDir = m_directoryDlg->directoryListComboBox->text(i);
                    if ( !tmpList.contains( subDir ) )
                    {
                        (void) new QListViewItem(directoryFiltersListView, type, subDir);
                    }
                }
            }
            emit directoryMaskListChanged();
        }
    }
    else
    {
        QString subDir = m_directoryDlg->directoryFilterValueKURLRequester->lineEdit()->text();
        if ( !subDir.isEmpty() )
        {
            //Add the filter if it is not already in the list
            QStringList tmpList = makeQStringList( directoryFiltersListView, 1 );
            if ( !tmpList.contains( subDir ) )
            {
                QString type = m_directoryDlg->directoryFilesystemTypeComboBox->currentText();
                (void) new QListViewItem(directoryFiltersListView, type, subDir);
            }
            emit directoryMaskListChanged();
        }
    }
}

void KDarConfig::slotRemoveAllCompressionFilter()
{
    compressionFiltersListView->clear();
    emit compressionMaskListChanged();
}

void KDarConfig::slotRemoveCompressionFilter()
{
    QListViewItem * currItem = compressionFiltersListView->selectedItem();
    if (!currItem) return;
    delete currItem;
    emit compressionMaskListChanged();
}

void KDarConfig::slotEditCompressionFilter()
{
    //Turns out that this is pretty easy to set up:
    QListViewItem* filterItem = compressionFiltersListView->selectedItem();
    if ( filterItem )
    {
        m_compressionDlg->compressionFilterValueComboBox->setEditText( filterItem->text( 0 ) );
        if ( m_compressionDlg->exec() )
        {
            if ( filterItem ) delete filterItem;
            newCompressionFilter();
        }
    }
}

void KDarConfig::slotNewCompressionFilter()
{
    if (m_compressionDlg->exec())
    {
        newCompressionFilter();
    }
}
void KDarConfig::newCompressionFilter()
{
    QString compressionMask = m_compressionDlg->compressionFilterValueComboBox->currentText();
    if ( !compressionMask.isEmpty() )
    {
        if ( compressionMask != i18n("select all filters from list", "All of the above" ) )
        {
            //Add the filter if it is not already in the list
            QStringList tmpList = makeQStringList( compressionFiltersListView );
            if ( !tmpList.contains( compressionMask ) )
            {
                (void) new QListViewItem(compressionFiltersListView, compressionMask);
            }
        }
        else
        {
            for (int i = 0; i<(m_compressionDlg->compressionFilterValueComboBox->count()-1);++i)
            {
                //Add the filter if it is not already in the list
                QStringList tmpList = makeQStringList( compressionFiltersListView );
                compressionMask = m_compressionDlg->compressionFilterValueComboBox->text(i);
                if ( !tmpList.contains( compressionMask ) )
                {
                    (void) new QListViewItem(compressionFiltersListView, compressionMask );
                }
            }
        }
        emit compressionMaskListChanged();
    }
}

void KDarConfig::slotUpdateCompressionMaskList()
{
    //Get a QStringList of compression filters and update the config object
    slotCompressionMaskList( makeQStringList( compressionFiltersListView ) );
}

void KDarConfig::slotUpdateDirectoryMaskList()
{
    //Get a QStringList of directory filters and update the config object
    QStringList directories;
    QStringList types;
    makeDirectoryQStringLists( directoryFiltersListView, directories, types );
    slotDirectoryMaskList( directories );
    slotDirectoryMaskTypes( types );
}

void KDarConfig::slotUpdateFileMaskList()
{
    //Get a QStringList of file filters and update the config object
    slotFileMaskList( makeQStringList( fileFiltersListView ) );
}

void KDarConfig::slotKDarSplashScreen( bool showKDarSplashScreen )
{
    setShowKDarSplashScreen( showKDarSplashScreen );
    emit configChanged();
}

void KDarConfig::slotRaiseWidgetStack( const QString & module )
{
    if ( module == i18n( "config module", "General" ) )
    {
        configureWidgetStack->raiseWidget( KDar::GENERAL );
    }
    else if ( module == i18n( "config module", "Fonts" ) )
    {
        configureWidgetStack->raiseWidget( KDar::FONTS );
    }
    else if ( module == i18n( "config module", "Slicing" ) )
    {
        configureWidgetStack->raiseWidget( KDar::SLICING );
    }
    else if ( module == i18n( "config module", "Compression" ) )
    {
        configureWidgetStack->raiseWidget( KDar::COMPRESSION );
    }
    else if ( module == i18n( "config module", "Directories" ) )
    {
        configureWidgetStack->raiseWidget( KDar::DIRECTORIES );
    }
    else if ( module == i18n( "config module", "Files" ) )
    {
        configureWidgetStack->raiseWidget( KDar::FILES );
    }
    else if ( module == i18n( "config module", "Filesystem" ) )
    {
        configureWidgetStack->raiseWidget( KDar::FILESYSTEM );
    }
    else if ( module == i18n( "config module", "Comparison" ) )
    {
        configureWidgetStack->raiseWidget( KDar::COMPARISON );
    }
    else if ( module == i18n( "config module", "Restore" ) )
    {
        configureWidgetStack->raiseWidget( KDar::RESTORE );
    }
    else if ( module == i18n( "config module", "Cryptography" ) )
    {
        configureWidgetStack->raiseWidget( KDar::CRYPTO );
    }
    else
    {
        kdDebug() << "KDarConfig::slotRaiseWidgetStack(): " << module << ": no such config module." << endl;
    }
}

QStringList KDarConfig::makeQStringList( QListView * listView, const int column )
{
    QStringList list;
    QListViewItem * tmpItem = listView->firstChild();
    QString text;
    while( tmpItem )
    {
        text = tmpItem->text( column );
        if ( !list.contains( text ) )
        {
            list << text;
        }
        tmpItem = tmpItem->nextSibling();
    }
    return list;
}

void KDarConfig::makeDirectoryQStringLists( QListView * listView, QStringList & directories, QStringList & types )
{
    QListViewItem * tmpItem = listView->firstChild();
    QString directoryText;
    QString typeText;
    QStringList dirList;
    QStringList typeList;

    while( tmpItem )
    {
        directoryText = tmpItem->text( 1 );
        if ( !dirList.contains( directoryText ) )
        {
            dirList << directoryText;
            typeList << tmpItem->text( 0 );
        }
        tmpItem = tmpItem->nextSibling();
    }
    directories = dirList;
    types = typeList;
}

void KDarConfig::slotCompressionMaskList( const QStringList & compressionMaskList )
{
    setCompressionMaskList( compressionMaskList );
    emit configChanged();
}

void KDarConfig::slotDirectoryMaskList( const QStringList & directoryMaskList )
{
    setDirectoryMaskList( directoryMaskList );
    emit configChanged();
}

void KDarConfig::slotFileMaskList( const QStringList & fileMaskList )
{
    setFileMaskList( fileMaskList );
    emit configChanged();
}

void KDarConfig::slotCompressionMaskType( int compressionMaskType )
{
    setCompressionMaskType( compressionMaskType );
    emit configChanged();
}

void KDarConfig::slotDirectoryMaskTypes( const QStringList & maskTypes )
{
    setDirectoryMaskTypes( maskTypes );
    emit configChanged();
}

void KDarConfig::slotFileMaskType( int fileMaskType )
{
    setFileMaskType( fileMaskType );
    emit configChanged();
}

void KDarConfig::slotGeneralFont( const QFont & generalFont )
{
    setGeneralFont (generalFont );
    emit configChanged();
}

void KDarConfig::slotFileBrowserFont( const QFont & fileBrowserFont )
{
    setFileBrowserFont( fileBrowserFont );
    emit configChanged();
}

void KDarConfig::slotMessageWindowFont( const QFont & messageWindowFont )
{
    setMessageWindowFont( messageWindowFont );
    emit configChanged();
}

void KDarConfig::slotStatusBarFont( const QFont & statusBarFont )
{
    setStatusBarFont( statusBarFont );
    emit configChanged();
}

void KDarConfig::slotLogFile( const QString & logFile )
{
    setLogFile( logFile );
    emit configChanged();
}

void KDarConfig::slotLogLevelNone( bool toggled )
{
    if ( toggled )
    {
        setLogLevel( KDar::LOG_NONE );
    }
    emit configChanged();
}

void KDarConfig::slotLogLevelBasic( bool toggled )
{
    if ( toggled )
    {
        setLogLevel( KDar::LOG_BASIC );
    }
    emit configChanged();
}

void KDarConfig::slotLogLevelFull( bool toggled )
{
    if ( toggled )
    {
        setLogLevel( KDar::LOG_FULL );
    }
    emit configChanged();
}

void KDarConfig::setMediaSize( int currentItem )
{
    kdDebug() << "KDarConfig::setMediaSize() reached. CurrentItem is " << currentItem << endl;
    //Block the textChanged() and activated() signals for the lineEdit and Combobox
    sliceSizeLineEdit->blockSignals( true );
    sliceSizeComboBox->blockSignals( true );
    sliceSizeLineEdit->setText( mediaSizes[ currentItem ] );
    sliceSizeComboBox->setCurrentItem( mediaSuffices[ currentItem ] );
    firstSliceSizeLineEdit->setText( mediaFirstSizes[ currentItem ] );
    firstSliceSizeComboBox->setCurrentItem( mediaFirstSuffices[ currentItem ] );
    //unblock the signals
    sliceSizeLineEdit->blockSignals( false );
    sliceSizeComboBox->blockSignals( false );

    kdDebug() << "KDarConfig::setMediaSize() reached. sliceSize is " << libdar::deci(libdarSliceSize()).human().c_str() << endl;
}

libdar::et_mask KDarConfig::libdarCreateCompressionMask()
{
    libdar::et_mask compr_mask;
    libdar::ou_mask compr_list;
    QStringList compressionList = compressionMaskList();
    //Iterate over the QStringList, adding the masks to the libdar::ou_mask
    for( QStringList::Iterator it = compressionList.begin(); it != compressionList.end(); ++it )
    {
// TODO store case sensitive filter info and give user option to toggle
        bool caseSensitive = false;
        compr_list.add_mask( libdar::simple_mask( kdar::toStdString( *it ), caseSensitive ) );
    }
    //If the compression list contains items, add them to the compression mask
    if ( !compressionList.empty() )
    {
        //compress
        if ( compressionMaskType() == KDar::INCLUDE )
        {
            compr_mask.add_mask( compr_list );
        }
        //do not compress
        else
        {
            compr_mask.add_mask( libdar::not_mask( compr_list ) );
        }
    }
    //otherwise, we compress everything
    else
    {
        compr_mask.add_mask( libdar::bool_mask( true ) );
    }

    return compr_mask;

}

libdar::et_mask KDarConfig::libdarCreateDirectoryMask()
{
//    kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): reached." << endl;
    //Directories to include ["list of paths"],
    //or exclude ["-P" option]
    libdar::et_mask subtree;
    libdar::ou_mask directoryIncludes;
    libdar::ou_mask directoryExcludes;

    QStringList directoryList = directoryMaskList();
    QStringList maskTypes = directoryMaskTypes();
//    kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): mask type count is " << maskTypes.count() << endl;
    kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): mask directory count is " << directoryList.count() << endl;

    //Iterate over the QStringList, adding the masks to the appropriate libdar::ou_mask
    if ( maskTypes.count() == directoryList.count() )
    {
        QStringList::Iterator typeIterator = maskTypes.begin();
        for( QStringList::Iterator directoryIterator = directoryList.begin(); directoryIterator != directoryList.end(); ++directoryIterator )
        {
// TODO handle caseSensitive:
            bool caseSensitive = false;
            //Add the directory to the appropriate include or exclude mask
            if ( *typeIterator == i18n( "filter type for inclusion", "Include" ) )
            {
                directoryIncludes.add_mask( libdar::simple_path_mask( libdar::path( kdar::toStdString( *directoryIterator ) ).display(), caseSensitive ) );
//                 kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): added include mask " << endl;
            }
            else //exclude "Exclude" and "Auto" masks
            {
                directoryExcludes.add_mask( libdar::simple_mask( libdar::path( kdar::toStdString( *directoryIterator ) ).display(), caseSensitive ) );
//                 kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): added exclude mask " << endl;
            }
            ++typeIterator;
        }
    }
    //If the include or exclude masks contain items, add them to the subtree mask
//     kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): directoryIncludes.size() = "  << directoryIncludes.size() << endl;
//     kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): directoryExcludes.size() = "  << directoryExcludes.size() << endl;

    if ( directoryIncludes.size() != 0 )
    {
        subtree.add_mask( directoryIncludes );
//         kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): subtree got include mask" << endl;

    }
    else
    {
        subtree.add_mask( libdar::bool_mask( true ) );
    }
    if ( directoryExcludes.size() != 0 )
    {
        subtree.add_mask( libdar::not_mask( directoryExcludes ) );
//         kdDebug() << "KDarConfig::libdarCreateDirectoryMask(): subtree got exclude mask" << endl;
    }

    return subtree;
}

libdar::et_mask KDarConfig::libdarCreateFileMask()
{
    libdar::et_mask file_mask;
    libdar::ou_mask file_list;
    QStringList fileList = fileMaskList();
    //Iterate over the QStringList, adding the masks to the libdar::ou_mask
    for( QStringList::Iterator it = fileList.begin(); it != fileList.end(); ++it )
    {
// TODO handle caseSensitive:
        bool caseSensitive = false;
        file_list.add_mask( libdar::simple_mask( kdar::toStdString(*it), caseSensitive ) );
    }
    //If the file list contains items, add them to the file mask
    if ( !fileList.empty() )
    {
        //include
        if ( fileMaskType() == KDar::INCLUDE )
        {
            file_mask.add_mask( file_list );
        }
        //ignore
        else
        {
            file_mask.add_mask( libdar::bool_mask( true ) );
            file_mask.add_mask( libdar::not_mask( file_list ) );
        }
    }
    //otherwise, we compress everything
    else
    {
        file_mask.add_mask( libdar::bool_mask( true ) );
    }

    return file_mask;

}

void KDarConfig::disableZCompression()
{
    //remove the Z compression option
    if ( gzipRadioButton->isChecked() )
    {
        gzipRadioButton->setChecked( false );
        noneRadioButton->setChecked( true );
        setCompressionAlgorithm( libdar::none );
    }
    if ( gzipQHBox1 )
    {
        gzipQHBox1->remove( gzipRadioButton );
        delete gzipQHBox1;
        gzipQHBox1 = 0;
        gzipRadioButton->hide();
    }

    if ( gzipQHBox2 )
    {
        gzipQHBox2->remove( gzipTextLabel );
        gzipQHBox2->remove( compressionLevelSpinBox );
        delete gzipQHBox2;
        gzipQHBox2 = 0;
        gzipTextLabel->hide();
        compressionLevelSpinBox->hide();
    }
    m_ZCompression = false;
}

void KDarConfig::disableBZ2Compression()
{
    bzip2RadioButton->setEnabled( false );
    if ( bzip2RadioButton->isChecked() )
    {
        bzip2RadioButton->setChecked( false );
        noneRadioButton->setChecked( true );
        setCompressionAlgorithm( libdar::none );
    }

    if ( bzip2QHBox )
    {
        bzip2QHBox->remove( bzip2RadioButton );
        delete bzip2QHBox;
        bzip2QHBox = 0;
        bzip2RadioButton->hide();
    }

    m_BZ2Compression = false;
}

void KDarConfig::disableCompressionModule()
{
    disableZCompression();
    disableBZ2Compression();
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Compression" ), Qt::CaseSensitive ) );
}

void KDarConfig::disableStrongCrypto()
{
    //remove the encryption item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Cryptography" ), Qt::CaseSensitive ) );
    m_strongCrypto = false;
}

void KDarConfig::setupCreateConfig()
{
    //remove the font item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Fonts" ), Qt::CaseSensitive ) );
    //remove the comparison item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Comparison" ), Qt::CaseSensitive ) );
    //remove the restore item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Restore" ), Qt::CaseSensitive ) );

    //Hide the useCreateArchiveWizard checkbox in the General module
    hideCreateArchiveWizardCheckBox();
    //Hide the showKDarSplashCheckBox
    hideKDarSplashCheckBox();
    //Hide the storePassword checkbox, since this is set once and for
    //all in the main config dialog.
    //It is confusing if the create/restore configs can have a different
    //setting from the main config, because the "open" method looks to the
    //main config to determine whether to look in a wallet for the password.
    passwordGroupBox->hide();
}

void KDarConfig::hideCreateArchiveWizardCheckBox()
{
    wizardQHBox->remove( useCreateArchiveWizardCheckBox );
    delete wizardQHBox;
    wizardQHBox = 0;
    useCreateArchiveWizardCheckBox->hide();
}

void KDarConfig::hideKDarSplashCheckBox()
{
    splashQHBox->remove( showKDarSplashCheckBox );
    delete splashQHBox;
    splashQHBox = 0;
    showKDarSplashCheckBox->hide();
}

void KDarConfig::setupRestoreConfig()
{
    //remove the font item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Fonts" ), Qt::CaseSensitive ) );
    //remove the slicing item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Slicing" ), Qt::CaseSensitive ) );
    //remove the compresion item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Compression" ), Qt::CaseSensitive ) );
    //remove the comparison item
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Comparison" ), Qt::CaseSensitive ) );
    //remove the cryptography item, since KDar now probes the archive to
    //to determine the type of encryption algorithm that was used to create
    //the archive.
    configModulesKListBox->takeItem( configModulesKListBox->findItem( i18n( "config module", "Cryptography" ), Qt::CaseSensitive ) );
    //Hide the useCreateArchiveWizard checkbox in the General module
    hideCreateArchiveWizardCheckBox();
    //Hide the showKDarSplashCheckBox
    hideKDarSplashCheckBox();
}

QString KDarConfig::buildDarCommandLine( KDar::libdarCommand command )
{
    QString darCommand( "dar" );
    QString newName = archiveHandle::parseStringSubstitutions( newArchiveName(), this );
    if ( newName[0] != '/' )
    {
        newName = cleanPath( archiveStorageDirectory() + "/" + newName );
    }
    QString archive = archiveName();
    if ( archive[0] != '/' )
    {
        archive = cleanPath( archiveStorageDirectory() + "/" + archive );
    }
    QString isolationName = isolationArchiveName();
    if ( isolationName[0] != '/' )
    {
        isolationName = cleanPath( archiveStorageDirectory() + "/" + isolationName );
    }
    QString refArchiveName = differentialBackupArchiveName();
    if ( refArchiveName[0] != '/' )
    {
        refArchiveName = cleanPath( archiveStorageDirectory() + "/" + refArchiveName );
    }

    //Other options...
    if ( dryRun() ) darCommand += " -e";
    if ( verbose() ) darCommand += " -v";

    //Compression masks
    QString compressionInclude( " -Y " );
    QString compressionExclude( " -Z " );
    QString compressionString( "" );
    QStringList compressionList( compressionMaskList() );
    for ( QStringList::Iterator it = compressionList.begin(); it != compressionList.end(); ++it )
    {
        if ( compressionMaskType() == KDar::INCLUDE )
        {
            compressionString += compressionInclude + "\"" + *it + "\"";
        }
        else
        {
            compressionString += compressionExclude + "\"" + *it + "\"";
        }
    }

    //File masks
    QString fileInclude( " -I " );
    QString fileExclude( " -X " );
    QString fileString( "" );
    QStringList fileList( fileMaskList() );
    for ( QStringList::Iterator it = fileList.begin(); it != fileList.end(); ++it )
    {
        if ( fileMaskType() == KDar::INCLUDE )
        {
            fileString += fileInclude + "\"" + *it + "\"";
        }
        else
        {
            fileString += fileExclude + "\"" + *it + "\"";
        }
    }

    //Directory filters
    QString directoryInclude( " " );
    QString directoryExclude( " -P " );
    QString directoryIncludeString( "" );
    QString directoryExcludeString( "" );
    QStringList directoryList( directoryMaskList() );
    QStringList maskTypes( directoryMaskTypes() );

    //Make sure the root directory has a trailing slash
    QString root( directoryToBackup() );
    if ( !root.endsWith( "/" ) ) root += "/";

    if ( maskTypes.count() == directoryList.count() )
    {
        QStringList::Iterator typeIterator = maskTypes.begin();
        for( QStringList::Iterator directoryIterator = directoryList.begin(); directoryIterator != directoryList.end(); ++directoryIterator )
        {
            QString filter( *directoryIterator );
            if ( filter.find( root ) == 0 )
            {
                filter.remove( 0, root.length() );
            }
            if ( *typeIterator == i18n( "filter type for inclusion", "Include" ) )
            {
                directoryIncludeString += directoryInclude + "\"" + filter + "\"";
            }
            else //exclude "Exclude" and "Auto" masks
            {
                directoryExcludeString += directoryExclude + "\"" + filter + "\"";
            }
            ++typeIterator;
        }
    }

    switch ( command )
    {
        case KDar::LIBDAR_CREATE:
            darCommand += " -c \"" + newName + "\"";
            darCommand += " -R \"" + root + "\"";
            if ( differentialBackup() ) darCommand += " -A \"" + refArchiveName + "\"";
            if ( !allowOverwrite() ) darCommand += " -n";
            if ( !warnOnOverwrite() ) darCommand += " -w";
            if ( libdarSliceSize() != 0 )
            {
                darCommand += " -s " + QString( libdar::deci( libdarSliceSize() ).human().c_str() );
                if ( libdarFirstSliceSize() != libdarSliceSize() && libdarFirstSliceSize() > 0 )
                {
                    darCommand += " -S " + QString( libdar::deci( libdarFirstSliceSize() ).human().c_str() );
                }
                if ( pauseBetweenSlices() ) darCommand += " -p";
            }
            if ( keepPruned() ) darCommand += " -D";
            if ( compressionAlgorithm() != libdar::none )
            {
                if ( compressionAlgorithm() == libdar::gzip ) darCommand += " -z" + QString::number( compressionLevel() );
                if ( compressionAlgorithm() == libdar::bzip2 ) darCommand += " -y";
                darCommand += " -m " + QString::number( minimumCompressionSizeInt() );
                //Add compression filters here.
                darCommand += compressionString;
            }
            if ( !userEA() ) darCommand += " -u";
            if ( !systemEA() ) darCommand += " -U";
            if ( !executeCommand().isEmpty() ) darCommand += " -E \"" + executeCommand() + "\"";
            if ( !passwordString().isEmpty() ) darCommand += " -K \"" + passwordString() + "\"";
            if ( ignoreDump() ) darCommand += " --nodump";
            if ( ignoreID() ) darCommand += " -O";
            if ( hourShiftInt() != 0 ) darCommand += " -H" + QString::number( hourShiftInt() );
            //File inclusion/exclusion filters:
            darCommand += fileString;
            //Directory filters
            darCommand += directoryExcludeString;
            darCommand += directoryIncludeString;
            break;
        case KDar::LIBDAR_DIFF:
            darCommand += " -d \"" + archive + "\"";
            darCommand += " -R \"" + diffArchiveDirectory() + "\"";
            if ( !userEA() ) darCommand += " -u";
            if ( !systemEA() ) darCommand += " -U";
            if ( ignoreID() ) darCommand += " -O";
            //File inclusion/exclusion filters:
            darCommand += fileString;
            //Directory filters
            darCommand += directoryExcludeString;
            darCommand += directoryIncludeString;
            break;
        case KDar::LIBDAR_ISOLATE:
            darCommand += " -C \"" + isolationName + "\"";
            darCommand += " -A \"" + archive + "\"";
            if ( !allowOverwrite() ) darCommand += " -n";
            if ( !warnOnOverwrite() ) darCommand += " -w";
            if ( libdarSliceSize() != 0 )
            {
                darCommand += " -s " + QString( libdar::deci( libdarSliceSize() ).human().c_str() );
                if ( libdarFirstSliceSize() != libdarSliceSize() && libdarFirstSliceSize() > 0 )
                {
                    darCommand += " -S " + QString( libdar::deci( libdarFirstSliceSize() ).human().c_str() );
                }
                if ( pauseBetweenSlices() ) darCommand += " -p";
            }
            if ( compressionAlgorithm() != libdar::none )
            {
                if ( compressionAlgorithm() == libdar::gzip ) darCommand += " -z" + QString::number( compressionLevel() );
                if ( compressionAlgorithm() == libdar::bzip2 ) darCommand += " -y";
                darCommand += " -m " + QString::number( minimumCompressionSizeInt() );
            }
            if ( !executeCommand().isEmpty() ) darCommand += " -E \"" + executeCommand() + "\"";
            if ( !passwordString().isEmpty() ) darCommand += " -K \"" + passwordString() + "\"";
            break;
        case KDar::LIBDAR_RESTORE:
            darCommand += " -x \"" + archive + "\"";
            darCommand += " -R \"" + extractArchiveDirectory()+ "\"";
            if ( !allowOverwrite() ) darCommand += " -n";
            if ( !warnOnOverwrite() ) darCommand += " -w";
            if ( noDelete() ) darCommand += " -k";
            if ( restoreRecent() ) darCommand += " -r";
            if ( flatRestore() ) darCommand += " -f";
            if ( hourShiftInt() != 0 ) darCommand += " -H" + QString::number( hourShiftInt() );
            if ( !userEA() ) darCommand += " -u";
            if ( !systemEA() ) darCommand += " -U";
            if ( ignoreID() ) darCommand += " -O";
             //File filters:
            darCommand += fileString;
            //directory filters
            darCommand += directoryExcludeString;
            darCommand += directoryIncludeString;
           break;
        case KDar::LIBDAR_TEST:
            darCommand += " -t \"" + archive + "\"";
            //File filters:
            darCommand += fileString;
            //directory filters
            break;
        default:
            ;
    }

    darCommand += "\n";
    return darCommand;
}

bool KDarConfig::ZCompression() const
{
    return m_ZCompression;
}

bool KDarConfig::BZ2Compression() const
{
    return m_BZ2Compression;
}

bool KDarConfig::strongCrypto() const
{
    return m_strongCrypto;
}

#include "kdarConfig.moc"
