/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   KWallet DCOP Interface is                                             *
 *    Copyright (c) 2002-2003 George Staikos <staikos@kde.org>             *
 *   KWallet is part of the KDE libraries                                  *
 *   Some code in this class is derived from the original KWallet          *
 *   code. This code is  copyright by George Staikos.                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "kwalletemu.h"
#include "pwmexception.h"
#include "pwmdoc.h"
#include "randomizer.h"
#include "pwminit.h"
#include "pwm.h"

#include <dcopclient.h>
#include <dcopref.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdatastream.h>

#include <qcstring.h>

#include <stdint.h>

using std::vector;


void KWalletEmu::ConnectionList::add(const QCString &client, PwMDoc *doc, int handle)
{
#ifdef PWM_DEBUG
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->handle == handle) {
			BUG();
			return;
		}
		++i;
	}
#endif // PWM_DEBUG
	connectionElement newItem;
	newItem.clientName = client;
	newItem.document = doc;
	newItem.handle = handle;
	connList.push_back(newItem);
}

void KWalletEmu::ConnectionList::del(int handle)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->handle == handle) {
			kwalletEmu->sig_applicationDisconnected(i->document->getTitle(),
								i->clientName);
			connList.erase(i);
			return;
		}
		++i;
	}
	printDebug("KWalletEmu::ConnectionList::del(int handle):  element not found.");
}

void KWalletEmu::ConnectionList::del(const QString &wallet)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->document->getTitle() == wallet) {
			kwalletEmu->sig_applicationDisconnected(i->document->getTitle(),
								i->clientName);
			connList.erase(i);
		}
		++i;
	}
}

void KWalletEmu::ConnectionList::delClient(const QCString &client)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->clientName == client) {
			kwalletEmu->sig_applicationDisconnected(i->document->getTitle(),
								i->clientName);
			connList.erase(i);
		}
		++i;
	}
}

bool KWalletEmu::ConnectionList::get(int handle, connectionElement *ret)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->handle == handle) {
			if (ret)
				*ret = *i;
			return true;
		}
		++i;
	}
	return false;
}

bool KWalletEmu::ConnectionList::find(const QCString &client, const QString &wallet,
				      connectionElement *ret)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->clientName == client) {
			if (wallet == i->document->getTitle()) {
				if (ret)
					*ret = *i;
				return true;
			}
		}
		++i;
	}
	return false;
}

bool KWalletEmu::ConnectionList::find(const QString wallet,
				      connectionElement *ret)
{
	vector<connectionElement>::iterator
			i = connList.begin(),
			end = connList.end();
	while (i != end) {
		if (i->document->getTitle() == wallet) {
			if (ret)
				*ret = *i;
			return true;
		}
		++i;
	}
	return false;
}


KWalletEmu::KWalletEmu(PwMInit *_init,
		       QObject *parent, const char *name)
 : QObject(parent, name)
 , DCOPObject(PWM_KWALLETEMU_DCOPOBJ)
{
	setEnabled(false);
	init = _init;
	if (!init->dcopClient()) {
		throw PwMException(PwMException::EX_GENERIC,
				   "dcop not available");
	}
	connList = new ConnectionList(this);
	if (!kdedUnloadKWallet()) {
		delete connList;
		throw PwMException(PwMException::EX_LOAD_MODULE,
				   "unload kwallet failed");
	}
	if (!kdedLoadEmu()) {
		kdedLoadKWallet();
		delete connList;
		throw PwMException(PwMException::EX_LOAD_MODULE,
				   "load emu failed");
	}
	connect(init->dcopClient(), SIGNAL(applicationRemoved(const QCString &)),
		this, SLOT(dcopAppDisconn(const QCString &)));
	setEnabled(true);
	sig_walletListDirty();
}

KWalletEmu::~KWalletEmu()
{
	int ret;
	sig_allWalletsClosed();
	setEnabled(false);
	if (!kdedUnloadEmu()) {
		printWarn("unloading KDED KWallet emulation "
			  "module failed!");
	}
	ret = kdedLoadKWallet();
	if (ret < 0) {
		printWarn("(re)loading KDED KWallet "
			  "module failed!");
	}
	if (unlikely(connList->getList()->size())) {
		printWarn("There are KWallet connections left "
			  "after shutting down the emulation!");
	}
	delete connList;
}

void KWalletEmu::connectDocSignals(PwMDoc *doc)
{
	connect(doc, SIGNAL(dataChanged(PwMDoc *)),
		this, SLOT(docChanged(PwMDoc *)));
	connect(doc, SIGNAL(docClosed(PwMDoc *)),
		this, SLOT(docClosed(PwMDoc *)));
	connect(doc, SIGNAL(docOpened(PwMDoc *)),
		this, SLOT(docOpened(PwMDoc *)));
	connect(doc, SIGNAL(docCreated(PwMDoc *)),
		this, SLOT(docCreated(PwMDoc *)));
	if (enabled) {
		sig_walletCreated(doc->getTitle());
		sig_walletListDirty();
	}
}

int KWalletEmu::kdedLoadKWallet()
{
	KConfig kwalletConf("kwalletrc");
	kwalletConf.setGroup("Wallet");
	bool loadKWallet = kwalletConf.readBoolEntry("Enabled", true);
	if (!loadKWallet) {
		printDebug("KWallet module not "
			   "loaded because it's disabled");
		return 1;
	}

	bool ok;
	QCString replyType;
	QByteArray replyData;
	QByteArray param;
	QDataStream paramDs(param, IO_WriteOnly);
	paramDs << QCString("kwalletd");
	ok = init->dcopClient()->call("kded", "kded",
				      "loadModule(QCString)",
				      param,
				      replyType,
				      replyData,
				      true);
	if (!ok) {
		printDebug("kded::loadModule(kwalletd) "
			   "dcop call failed");
		return -1;
	}
	if (replyType != "bool") {
		printWarn("kded::loadModule() return "
			  "type != bool");
		return -1;
	}
	bool boolRet;
	QDataStream replyStream(replyData, IO_ReadOnly);
	replyStream >> boolRet;
	if (!boolRet) {
		printDebug("loading kwalletd failed.");
		return -1;
	}
	return 0;
}

bool KWalletEmu::kdedLoadEmu()
{
	bool ok;
	QCString replyType;
	QByteArray replyData;
	QByteArray param;
	QDataStream paramDs(param, IO_WriteOnly);
	paramDs << QCString(PWM_KWALLETEMU_DCOPMODULE);
	ok = init->dcopClient()->call("kded", "kded",
				      "loadModule(QCString)",
				      param,
				      replyType,
				      replyData,
				      true);
	if (!ok) {
		printDebug("kded::loadModule(PWM_KWALLETEMU_DCOPMODULE) "
			   "dcop call failed");
		return false;
	}
	if (replyType != "bool") {
		printWarn("kded::loadModule() return "
			  "type != bool");
		return false;
	}
	bool boolRet;
	QDataStream replyStream(replyData, IO_ReadOnly);
	replyStream >> boolRet;
	if (!boolRet) {
		printDebug("loading PWM_KWALLETEMU_DCOPMODULE failed.");
		return false;
	}
	return true;
}

bool KWalletEmu::kdedUnloadKWallet()
{
	bool ok;
	QCString replyType;
	QByteArray replyData;
	ok = init->dcopClient()->call("kded", "kded",
				      "loadedModules()",
				      QByteArray(),
				      replyType,
				      replyData,
				      true);
	if (!ok) {
		printDebug("kded::loadedModules() dcop call failed");
		return false; // kded not running?
	}
	if (replyType != "QCStringList") {
		printWarn("kded::loadedModules() return "
			  "type != QCStringList");
		return false;
	}
	QCStringList modList;
	QDataStream replyStream(replyData, IO_ReadOnly);
	replyStream >> modList;
	if (modList.find("kwalletd") == modList.end()) {
		printDebug("KWalletd module was not loaded.");
		return true;
	}
	QByteArray param;
	QDataStream paramDs(param, IO_WriteOnly);
	paramDs << QCString("kwalletd");
	ok = init->dcopClient()->call("kded", "kded",
				      "unloadModule(QCString)",
				      param,
				      replyType,
				      replyData,
				      true);
	if (!ok) {
		printDebug("kded::unloadModule(QCString) dcop "
			   "call failed");
		return false;
	}
	if (replyType != "bool") {
		printWarn("kded::unloadModule(QCString) return "
			  "type != bool");
		return false;
	}
	bool boolRet;
	replyStream >> boolRet;
	if (!boolRet) {
		printDebug("unloading kwalletd module failed.");
		return false;
	}
	return true;
}

bool KWalletEmu::kdedUnloadEmu()
{
	bool ok;
	QCString replyType;
	QByteArray replyData;
	QByteArray param;
	QDataStream paramDs(param, IO_WriteOnly);
	paramDs << QCString(PWM_KWALLETEMU_DCOPMODULE);
	ok = init->dcopClient()->call("kded", "kded",
				      "unloadModule(QCString)",
				      param,
				      replyType,
				      replyData,
				      true);
	if (!ok) {
		printDebug("kded::unloadModule(PWM_KWALLETEMU_DCOPMODULE) "
			   "dcop call failed");
		return false;
	}
	if (replyType != "bool") {
		printWarn("kded::unloadModule() return "
			  "type != bool");
		return false;
	}
	bool boolRet;
	QDataStream replyStream(replyData, IO_ReadOnly);
	replyStream >> boolRet;
	if (!boolRet) {
		printDebug("unloading PWM_KWALLETEMU_DCOPMODULE failed.");
		return false;
	}
	return true;
}

void KWalletEmu::sig_walletClosed(const QString &wallet)
{
	printDebug("KWalletEmu::sig_walletClosed(const QString&): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletClosed(QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_walletClosed(int handle)
{
	printDebug("KWalletEmu::sig_walletClosed(int): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << handle;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletClosed(int)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_allWalletsClosed()
{
	printDebug("KWalletEmu::sig_allWalletsClosed(): emitting");
	PWM_ASSERT(enabled);
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_allWalletsClosed()",
				      QByteArray());
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_applicationDisconnected(const QString &wallet,
					     const QCString &application)
{
	printDebug("KWalletEmu::sig_applicationDisconnected(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	ds << application;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_applicationDisconnected(QString,QCString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_folderUpdated(const QString &wallet,
				   const QString &folder)
{
	printDebug("KWalletEmu::sig_folderUpdated(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	ds << folder;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_folderUpdated(QString,QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_folderListUpdated(const QString &wallet)
{
	printDebug("KWalletEmu::sig_folderListUpdated(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_folderListUpdated(QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_walletListDirty()
{
	printDebug("KWalletEmu::sig_walletListDirty(): emitting");
	PWM_ASSERT(enabled);
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletListDirty()",
				      QByteArray());
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_walletCreated(const QString &wallet)
{
	printDebug("KWalletEmu::sig_walletCreated(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletCreated(QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_walletOpened(const QString &wallet)
{
	printDebug("KWalletEmu::sig_walletOpened(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletOpened(QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::sig_walletDeleted(const QString &wallet)
{
	printDebug("KWalletEmu::sig_walletDeleted(): emitting");
	PWM_ASSERT(enabled);
	QByteArray data;
	QDataStream ds(data, IO_WriteOnly);
	ds << wallet;
	bool ok;
	ok = init->dcopClient()->send("kded", "kwalletd",
				      "sig_walletDeleted(QString)",
				      data);
	PWM_ASSERT(ok);
}

void KWalletEmu::docChanged(PwMDoc *doc)
{
	if (!enabled)
		return;
	sig_folderListUpdated(doc->getTitle());
}

void KWalletEmu::docClosed(PwMDoc *doc)
{
	if (!enabled)
		return;
	sig_walletDeleted(doc->getTitle());
	const std::vector<ConnectionList::connectionElement>
				*cl = connList->getList();
	std::vector<ConnectionList::connectionElement>::const_iterator
				i = cl->begin(),
				end = cl->end();
	while (i != end) {
		if (i->document == doc) {
			doClose(i->handle);
		}
		++i;
	}
	sig_walletListDirty();
}

void KWalletEmu::docOpened(PwMDoc *doc)
{
	if (!enabled)
		return;
	sig_walletCreated(doc->getTitle());
	sig_walletListDirty();
}

void KWalletEmu::docCreated(PwMDoc *doc)
{
	if (!enabled)
		return;
	sig_walletCreated(doc->getTitle());
	sig_walletListDirty();
}

void KWalletEmu::dcopAppDisconn(const QCString &appId)
{
	/* An application disconnected from the DCOP
	 * interface. Make sure there are no dead
	 * connections left.
	 */
	connList->delClient(appId);
}

QStringList KWalletEmu::getOpenDocList() const
{
	QStringList ret;
	const vector<PwMDocList::listItem> *docList = PwMDoc::getOpenDocList()->getList();
	vector<PwMDocList::listItem>::const_iterator dli = docList->begin(),
						     dlend = docList->end();
	while (dli != dlend) {
		ret.append((*dli).docId.c_str());
		++dli;
	}
	return ret;
}

int KWalletEmu::genHandle()
{
	int ret;
	BUG_ON((sizeof(int) != 4) && (sizeof(int) != 8));
	const int intMin = (sizeof(int) == 4) ? INT32_MIN : INT64_MIN;
	Randomizer rnd;
	do {
		rnd >> ret;
		if (ret < 0) {
			if (unlikely(ret == intMin))
				continue;
			ret = -ret;
		}
	} while (connList->get(ret, 0));
	return ret;
}

int KWalletEmu::connectPeer(const QCString &peer, PwMDoc *doc)
{
	int handle = genHandle();
	connList->add(peer, doc, handle);
	return handle;
}

PwMDoc * KWalletEmu::checkConn(const QCString &peer,
			       int handle)
{
	if (!enabled || !checkCaller())
		return 0;
	ConnectionList::connectionElement e;
	if (!connList->get(handle, &e)) {
		printDebug("KWalletEmu::checkConn(int "
			   "handle):  not found");
		return 0;
	}
	PWM_ASSERT(e.handle == handle);
	if (e.clientName != peer) {
		printDebug("KWalletEmu::checkConn(int "
			   "handle):  wrong peer");
		return 0;
	}
	PWM_ASSERT(e.document);
	return e.document;
}

bool KWalletEmu::checkCaller() const
{
	DCOPClient *dc = const_cast<KWalletEmu *>(this)->callingDcopClient();
	if (!dc)
		return false;
	if (dc->senderId() != "kded")
		return false;
	return true;
}

bool KWalletEmu::isEnabled(const QCString & /*peer*/) const
{
	printDebug("KWalletEmu::isEnabled()");
	if (!checkCaller())
		return false;
	return enabled;
}

QString KWalletEmu::localWallet() const
{
	KConfig cfg("kwalletrc");
	cfg.setGroup("Wallet");
	if (!cfg.readBoolEntry("Use One Wallet", true))
		return cfg.readEntry("Local Wallet", "localwallet");
	return cfg.readEntry("Default Wallet", "kdewallet");
}

QString KWalletEmu::networkWallet() const
{
	KConfig cfg("kwalletrc");
	cfg.setGroup("Wallet");
	return cfg.readEntry("Default Wallet", "kdewallet");
}

PwMDoc * KWalletEmu::getDocByWalletName(const QString &wallet) const
{
	QStringList docList(getOpenDocList());
	int foundPos = docList.findIndex(wallet);
	if (foundPos == -1) {
		if (wallet == localWallet() ||
		    wallet == networkWallet()) {
			/* Default wallet got selected, but there's
			 * no open doc by that name.
			 * We simply select the first open doc (if
			 * there is one)
			 */
			if (docList.isEmpty()) {
				printDebug("KWalletEmu: Request for default "
					   "wallet, but no document open. "
					   "Creating new document.");
			} else {
				foundPos = 0;
				printDebug("KWalletEmu: Request for default "
					   "wallet. Selecting first document.");
			}
		}
	}
	if (foundPos == -1)
		return 0; // not found.
	// document exists. Get the pointer.
	const vector<PwMDocList::listItem> *fullDocList = PwMDoc::getOpenDocList()->getList();
	PwMDocList::listItem selectedItem = (*fullDocList)[foundPos];
	PwMDoc *doc = selectedItem.doc;
	if (doc->isDeepLocked()) {
		doc->deepLock(false);
		// a deep-unlock failure is non-fatal, here.
	}
	return doc;
}

/* open() acts as some kind of "connect to doc" in
 * PwManager. The document has to be opened before this
 * function is called.
 */
int KWalletEmu::open(const QCString &peer,
		     const QString &wallet,
		     uint /*wId*/)
{
	printDebug("KWalletEmu::open(const QString&, uint)");
	if (!enabled || !checkCaller())
		return -1;
	if (wallet.isEmpty())
		return -1;

	bool isNewDoc = false;
	PwMDoc *doc = getDocByWalletName(wallet);
	if (!doc) {
		// document does not exist, yet. We create a new one
		if (init->tray()) {
			// tray icon exists, so we don't need to open a window.
			doc = init->createDoc();
		} else {
			// we must open a window, as otherwise there's no UI for the doc.
			PwM *newInstance = init->createMainWnd();
			doc = newInstance->curDoc();
		}
		PWM_ASSERT(doc);
		isNewDoc = true;
	}
	int handle = connectPeer(peer, doc);
	if (isNewDoc)
		sig_walletCreated(wallet);
	else
		sig_walletOpened(wallet);
	return handle;
}

int KWalletEmu::openPath(const QCString &peer,
			 const QString &path,
			 uint /*wId*/)
{
	printDebug("KWalletEmu::openPath(const QString&, uint)");
	if (!enabled || !checkCaller())
		return -1;
	if (path.isEmpty())
		return -1;
	// open the document in a new window.
	PwM *newInstance = init->createMainWnd(); //FIXME: If there is a tray, don't open a window.
	PwMDoc *doc = newInstance->openDoc(path); //FIXME: if it does not exist, create a new one.
	if (!doc) {
		newInstance->setForceQuit(true);
		delete_and_null(newInstance);
		return -1;
	}
	if (doc->isDeepLocked()) {
		doc->deepLock(false);
		// a deep-unlock failure is non-fatal, here.
	}
	int handle = connectPeer(peer, doc);
	sig_walletOpened(doc->getTitle()); //FIXME: opened or created.
	return handle;
}

void KWalletEmu::openAsynchronous(const QCString &peer,
				  const QString &wallet,
				  const QCString &returnObject,
				  uint wId)
{
	printDebug("KWalletEmu::openAsynchronous(const QString&, "
		   "const QCString&, const QCString&, uint");
	if (wallet.isEmpty() ||
	    returnObject.isEmpty()) {
		return;
	}
	int handle = open(peer, wallet, wId);
	DCOPRef(peer, returnObject).send("walletOpenResult", handle);
}

/* close the connection. _Not_ the document itself */
int KWalletEmu::close(const QCString &peer,
		      const QString &wallet,
		      bool force)
{
	printDebug("KWalletEmu::close(const QString&, bool)");
	if (!enabled || !checkCaller())
		return -1;
	if (wallet.isEmpty())
		return -1;
	ConnectionList::connectionElement e;
	if (!connList->find(peer, wallet, &e))
		return -1;
	return close(peer, e.handle, force);
}

/* close the connection. _Not_ the document itself */
int KWalletEmu::close(const QCString &peer,
		      int handle,
		      bool force)
{
	printDebug("KWalletEmu::close(int, bool)");
	if (!checkConn(peer, handle))
		return -1;
	return doClose(handle, force);
}

int KWalletEmu::doClose(int handle,
			bool force)
{
	PWM_ASSERT(enabled);
	PWM_ASSERT(checkCaller());
	ConnectionList::connectionElement e;
	if (!connList->get(handle, &e))
		return -1;
	QString docTitle(e.document->getTitle());
	sig_walletClosed(handle);
	sig_walletClosed(docTitle);
	if (force) {
		// close this wallet (document) for all users
		connList->del(docTitle);
	} else {
		// delete only the supplied handle
		connList->del(handle);
	}
	return 0;
}

ASYNC KWalletEmu::sync(const QCString & /*peer*/,
		       int /*handle*/)
{
	printDebug("KWalletEmu::sync(int)");
	/* We do nothing here, because PwManager has
	 * it's own mechanisms to keep the document in sync
	 * with the disk.
	 * This is not true if PwManager crashes, but I
	 * simply assume it doesn't crash. :)
	 */
}

int KWalletEmu::deleteWallet(const QCString & /*peer*/,
			     const QString &wallet)
{
	printDebug("KWalletEmu::deleteWallet(const QString&)");
	if (!enabled || !checkCaller())
		return -1;
	if (wallet.isEmpty())
		return -1;
	PwMDoc *doc = getDocByWalletName(wallet);
	if (!doc)
		return -1; // does not exist.
	doc->tryDelete();
	// We do not check tryDelete() return value here. I think
	// KWallet emulation should not care too much about it.
	return 0;
}

bool KWalletEmu::isOpen(const QCString & /*peer*/,
			const QString &wallet)
{
	printDebug("KWalletEmu::isOpen(const QString&)");
	if (!enabled || !checkCaller())
		return false;
	if (wallet.isEmpty())
		return false;
	return connList->find(wallet, 0);
}

bool KWalletEmu::isOpen(const QCString &peer,
			int handle)
{
	printDebug("KWalletEmu::isOpen(int)");
	return checkConn(peer, handle);
}

QStringList KWalletEmu::users(const QCString & /*peer*/,
			      const QString &wallet) const
{
	printDebug("KWalletEmu::users(const QString&)");
	QStringList ret;
	if (!enabled || !checkCaller())
		return ret;
	if (wallet.isEmpty())
		return ret;
	const std::vector<ConnectionList::connectionElement> * conn
				= connList->getList();
	std::vector<ConnectionList::connectionElement>::const_iterator
				i = conn->begin(),
				end = conn->end();
	while (i != end) {
		if (wallet == (*i).document->getTitle()) {
			ret.push_back((*i).clientName);
		}
		++i;
	}
	return ret;
}

void KWalletEmu::changePassword(const QCString & /*peer*/,
				const QString &wallet,
				uint /*wId*/)
{
	printDebug("KWalletEmu::changePassword(const QString&, uint)");
	if (!enabled || !checkCaller())
		return;
	if (wallet.isEmpty())
		return;
	PwMDoc *doc = getDocByWalletName(wallet);
	if (!doc)
		return; // does not exist.
	if (doc->isDeepLocked()) {
		// getDocByWalletName() could not unlock the document.
		return;
	}
	doc->changeMasterKey();
}

QStringList KWalletEmu::wallets(const QCString & /*peer*/) const
{
	printDebug("KWalletEmu::wallets()");
	if (!enabled || !checkCaller())
		return QStringList();
	return getOpenDocList();
}

QStringList KWalletEmu::folderList(const QCString &peer,
				   int handle)
{
	printDebug("KWalletEmu::folderList(int)");
	QStringList ret;
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return ret;
	doc->getCategoryList(&ret);
	return ret;
}

bool KWalletEmu::hasFolder(const QCString &peer,
			   int handle,
			   const QString &folder)
{
	printDebug("KWalletEmu::hasFolder(int, const QString&)");
	if (folder.isEmpty()) {
		// we never have an empty category name.
		return false;
	}
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	const char *_folder = folder.latin1();
	vector<string> catList;
	doc->getCategoryList(&catList);
	vector<string>::iterator i = catList.begin(),
				 end = catList.end();
	while (i != end) {
		if (*i == _folder)
			return true;
		++i;
	}
	return false;
}

bool KWalletEmu::createFolder(const QCString &peer,
			      int handle,
			      const QString &folder)
{
	printDebug("KWalletEmu::createFolder(int, const QString&)");
	if (folder.isEmpty())
		return false;
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	PwMerror err = doc->addCategory(folder, 0, true);
	if (err != e_success)
		return false;
	doc->flagDirty(); // we have to do this manually
	return true;
}

bool KWalletEmu::removeFolder(const QCString &peer,
			      int handle,
			      const QString &folder)
{
	printDebug("KWalletEmu::removeFolder(int, const QString&)");
	if (folder.isEmpty())
		return false;
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	return doc->delCategory(folder);
}

QStringList KWalletEmu::entryList(const QCString &peer,
				  int handle,
				  const QString &folder)
{
	printDebug("KWalletEmu::entryList(int, const QString&)");
	QStringList ret;
	if (folder.isEmpty())
		return ret;
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return ret;
	doc->getEntryList(folder, &ret);
	return ret;
}

QByteArray KWalletEmu::readEntry(const QCString &peer,
				 int handle,
				 const QString &folder,
				 const QString &key)
{
	printDebug("QByteArray KWalletEmu::readEntry(int, const QString&, const QString&)");
	return readEntryBinary(peer,
			       handle,
			       folder,
			       key,
			       BinEntryGen::KWalletStream);
}

QByteArray KWalletEmu::readMap(const QCString &peer,
			       int handle,
			       const QString &folder,
			       const QString &key)
{
	printDebug("QByteArray KWalletEmu::readMap(int, const QString&, const QString&)");
	return readEntryBinary(peer,
			       handle,
			       folder,
			       key,
			       BinEntryGen::KWalletMap);
}

QByteArray KWalletEmu::readEntryBinary(const QCString &peer,
				       int handle,
				       const QString& folder,
				       const QString& key,
				       BinEntryGen::DataType requestedType)
{
	PwMDataItem d;
	if (!doReadEntry(peer, handle, folder, key, &d))
		return QByteArray();
	BinEntryGen binGen;
	BinEntryGen::DataType type;
	type = binGen.binType(d);
	if (type != requestedType)
		return QByteArray();
	QByteArray ret;
	binGen.decode(d, &ret, &type);
	PWM_ASSERT(type == requestedType);
	return ret;
}

QString KWalletEmu::readPassword(const QCString &peer,
				 int handle,
				 const QString &folder,
				 const QString &key)
{
	printDebug("KWalletEmu::readPassword(int, const QString&, const QString&)");
	PwMDataItem ret;
	if (!doReadEntry(peer, handle, folder, key, &ret))
		return QString::null;
	return ret.pw.c_str();
}

bool KWalletEmu::doReadEntry(const QCString &peer,
			     int handle,
			     const QString &folder,
			     const QString &key,
			     PwMDataItem *ret)
{
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	if (doc->isDeepLocked()) {
		PwMerror err = doc->deepLock(false);
		if (err != e_success)
			return false;
	}
	if (folder.isEmpty() || key.isEmpty())
		return false;
	vector<unsigned int> foundPos;
	ret->desc = key.latin1();
	doc->findEntry(folder, *ret, SEARCH_IN_DESC,
		       &foundPos, true);
	if (!foundPos.size())
		return false;
	if (!doc->getEntry(folder, foundPos[0], ret, true))
		return false;
	return true;
}

int KWalletEmu::renameEntry(const QCString &peer,
			    int handle,
			    const QString &folder,
			    const QString &oldName,
			    const QString &newName)
{
	printDebug("KWalletEmu::renameEntry(int, const QString&, const QString&, const QString&)");
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return -1;
	if (folder.isEmpty() ||
	    oldName.isEmpty() ||
	    newName.isEmpty()) {
		return -1;
	}
	vector<unsigned int> foundPos;
	PwMDataItem d;
	d.desc = oldName.latin1();
	doc->findEntry(folder, d, SEARCH_IN_DESC,
		       &foundPos, true);
	if (!foundPos.size())
		return -1;
	if (!doc->getEntry(folder, foundPos[0], &d, true))
		return -1;
	d.desc = newName.latin1();
	if (!doc->editEntry(folder, folder, foundPos[0], &d))
		return -1;
	return 0;
}

int KWalletEmu::writeEntry(const QCString &peer,
			   int handle,
			   const QString &folder,
			   const QString &key,
			   const QByteArray &value,
			   int entryType)
{
	printDebug("KWalletEmu::writeEntry(int, const QString&, const QString&, const QByteArray&, int)");
	if (folder.isEmpty() ||
	    key.isEmpty() ||
	    value.isEmpty()) {
		return -1;
	}
	switch (entryType) {
		case Password: {
			QString strval;
			QDataStream ds(value, IO_ReadOnly);
			ds >> strval;
			return writePassword(peer, handle, folder, key, strval);
		} case Stream: {
			return writeEntry(peer, handle, folder, key, value);
		} case Map: {
			return writeMap(peer, handle, folder, key, value);
		}
	}
	return -1;
}

int KWalletEmu::writeEntry(const QCString &peer,
			   int handle,
			   const QString &folder,
			   const QString &key,
			   const QByteArray &value)
{
	printDebug("KWalletEmu::writeEntry(int, const QString&, const QString&, const QByteArray&)");
	if (folder.isEmpty() ||
	    key.isEmpty() ||
	    value.isEmpty()) {
		return -1;
	}
	return writeEntryBinary(peer,
				handle,
				folder,
				key,
				value,
				BinEntryGen::KWalletStream);
}

int KWalletEmu::writeMap(const QCString &peer,
			 int handle,
			 const QString &folder,
			 const QString &key,
			 const QByteArray &value)
{
	printDebug("KWalletEmu::writeMap(int, const QString&, const QString&, const QByteArray&)");
	if (folder.isEmpty() ||
	    key.isEmpty() ||
	    value.isEmpty()) {
		return -1;
	}
	return writeEntryBinary(peer,
				handle,
				folder,
				key,
				value,
				BinEntryGen::KWalletMap);
}

int KWalletEmu::writeEntryBinary(const QCString &peer,
				 int handle,
				 const QString& folder,
				 const QString& key,
				 const QByteArray& value,
				 BinEntryGen::DataType requestedType)
{
	PwMDataItem d;
	BinEntryGen binGen;
	binGen.encode(value, &d, requestedType);
	d.desc = key.latin1();
	return doWriteEntry(peer, handle, folder, d);
}

int KWalletEmu::writePassword(const QCString &peer,
			      int handle,
			      const QString &folder,
			      const QString &key,
			      QString value)
{
	printDebug("KWalletEmu::writePassword(int, const QString&, const QString&, const QString&)");
	if (folder.isEmpty() ||
	    key.isEmpty()) {
		return -1;
	}
	if (value.isEmpty())
		value = "";
	PwMDataItem d;
	d.clear();
	d.desc = key.latin1();
	d.pw = value.latin1();
	return doWriteEntry(peer, handle, folder, d);
}

int KWalletEmu::doWriteEntry(const QCString &peer,
			     int handle,
			     const QString& folder,
			     const PwMDataItem &value)
{
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return -1;
	bool ok;
	vector<unsigned int> foundPos;
	doc->findEntry(folder, value, SEARCH_IN_DESC,
		       &foundPos, true);
	if (foundPos.size()) {
		/* the entry does already exist.
		 * edit it.
		 */
		printDebug("KWalletEmu::doWriteEntry(): edit existing");
		PwMDataItem d;
		ok = doc->getEntry(folder, foundPos[0], &d, true);
		if (!ok)
			return -1;
		d.pw = value.pw;
		if (value.name != "")
			d.name = value.name; // This is only for binary entries.
		ok = doc->editEntry(folder, folder,
				    foundPos[0], &d);
		if (!ok)
			return -1;
		return 0;
	}
	/* the entry does not exist.
	 * create a new one.
	 */
	printDebug("KWalletEmu::doWriteEntry(): create new");
	PwMDataItem d = value;
	PwMerror err;
	err = doc->addEntry(folder, &d);
	if (err != e_success)
		return -1;
	return 0;
}

bool KWalletEmu::hasEntry(const QCString &peer,
			  int handle,
			  const QString &folder,
			  const QString &key)
{
	printDebug("KWalletEmu::hasEntry(int, const QString&, const QString&)");
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	if (folder.isEmpty() ||
	    key.isEmpty()) {
		return false;
	}
	vector<unsigned int> foundPos;
	PwMDataItem d;
	d.desc = key.latin1();
	doc->findEntry(folder, d, SEARCH_IN_DESC,
		       &foundPos, true);
	if (!foundPos.size())
		return false;
	return true;
}

int KWalletEmu::entryType(const QCString &peer,
			  int handle,
			  const QString &folder,
			  const QString &key)
{
	printDebug("KWalletEmu::entryType(int, const QString&, const QString&)");
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return Unknown;
	if (folder.isEmpty() ||
	    key.isEmpty()) {
		return Unknown;
	}
	bool ok;
	vector<unsigned int> foundPos;
	PwMDataItem d;
	d.desc = key.latin1();
	doc->findEntry(folder, d, SEARCH_IN_DESC,
		       &foundPos, true);
	if (!foundPos.size())
		return Unknown;
	ok = doc->getEntry(folder, foundPos[0], &d, true);
	if (!ok)
		return Unknown;
	BinEntryGen binGen;
	BinEntryGen::DataType type;
	type = binGen.binType(d);
	switch (type) {
	case BinEntryGen::None:
		return Password;
	case BinEntryGen::KWalletStream:
		return Stream;
	case BinEntryGen::KWalletMap:
		return Map;
	}
	return Unknown;
}

int KWalletEmu::removeEntry(const QCString &peer,
			    int handle,
			    const QString &folder,
			    const QString &key)
{
	printDebug("KWalletEmu::removeEntry(int, const QString&, const QString&)");
	PwMDoc *doc = checkConn(peer, handle);
	if (!doc)
		return false;
	if (folder.isEmpty() ||
	    key.isEmpty()) {
		return false;
	}
	vector<unsigned int> foundPos;
	PwMDataItem d;
	d.desc = key.latin1();
	doc->findEntry(folder, d, SEARCH_IN_DESC,
		       &foundPos, true);
	if (!foundPos.size())
		return -1;
	if (!doc->delEntry(folder, foundPos[0]))
		return -1;
	return 0;
}

bool KWalletEmu::disconnectApplication(const QCString & /*peer*/,
				       const QString &wallet,
				       const QCString &application)
{
	printDebug("KWalletEmu::disconnectApplication(const QString&, const QCString&)");
	if (!enabled || !checkCaller())
		return false;
	if (wallet.isEmpty() ||
	    application.isEmpty()) {
		return false;
	}
	ConnectionList::connectionElement element;
	if (!connList->find(application, wallet, &element))
		return false;
	connList->del(element.handle);
	sig_applicationDisconnected(wallet, application);
	return true;
}

void KWalletEmu::reconfigure(const QCString & /*peer*/)
{
	printDebug("KWalletEmu::reconfigure()");
	// we don't do anything here, yet.
}

bool KWalletEmu::folderDoesNotExist(const QCString & /*peer*/,
				    const QString &wallet,
				    const QString &folder)
{
	printDebug("KWalletEmu::folderDoesNotExist(const QString&, const QString&)");
	if (!enabled || !checkCaller())
		return true;
	if (wallet.isEmpty() ||
	    folder.isEmpty()) {
		return true;
	}
	PwMDoc *doc = getDocByWalletName(wallet);
	if (!doc)
		return true; // document does not exist.
	if (doc->findCategory(folder, 0))
		return false;
	return true;
}

bool KWalletEmu::keyDoesNotExist(const QCString & /*peer*/,
				 const QString &wallet,
				 const QString &folder,
				 const QString &key)
{
	printDebug("KWalletEmu::keyDoesNotExist(const QString&, const QString&, const QString&)");
	if (!enabled || !checkCaller())
		return true;
	if (wallet.isEmpty() ||
	    folder.isEmpty() ||
	    key.isEmpty()) {
		return true;
	}
	PwMDoc *doc = getDocByWalletName(wallet);
	if (!doc)
		return true; // document does not exist.
	vector<unsigned int> found;
	PwMDataItem d;
	d.desc = key.latin1();
	doc->findEntry(folder, d, SEARCH_IN_DESC,
		       &found, true);
	if (found.size())
		return false;
	return true;
}

void KWalletEmu::closeAllWallets(const QCString & /*peer*/)
{
	printDebug("KWalletEmu::closeAllWallets()");
	if (!enabled || !checkCaller())
		return;
	const vector<ConnectionList::connectionElement> *cl = connList->getList();
	vector<ConnectionList::connectionElement>::const_iterator i = cl->begin(),
								  end = cl->end();
	bool cleanClose = true;
	while (i != end) {
		if (doClose(i->handle, true))
			cleanClose = false;
		++i;
	}
	if (cleanClose)
		sig_allWalletsClosed();
}

#include "kwalletemu_skel.cpp"
#include "kwalletemu.moc"
