/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004, 2005  by Michael Buesch                           *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "masterkey/smartkey/libchipcard1/chipcard1interface.h"
#include "pwmexception.h"

#include <chipcard.h>


Chipcard1Interface::Chipcard1Interface()
 : ctTrader (0)
 , ctCard (0)
{
}

SmartKey_backend::Status Chipcard1Interface::write(const QByteArray &data)
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("ct_open() failed");
		goto out;
	}
	ret = ct_write(data);
	if (ret != SmartKey_backend::status_ok)
		printDebug("ct_write() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard1Interface::read(QByteArray *data)
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("ct_open() failed");
		goto out;
	}
	ret = ct_read(data);
	if (ret != SmartKey_backend::status_ok)
		printDebug("ct_read() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard1Interface::erase()
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("ct_open() failed");
		goto out;
	}
	ret = ct_erase();
	if (ret != SmartKey_backend::status_ok)
		printDebug("ct_erase() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard1Interface::ct_open()
{
	CTError err;
	int i;
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;

	showOpenWindow(true);

	ctTrader = new CTCardTrader(false,
				    0, 0,
				    CHIPCARD_STATUS_INSERTED,
				    CHIPCARD_STATUS_INSERTED |
				    CHIPCARD_STATUS_LOCKED_BY_OTHER,
				    CHIPCARD_STATUS_INSERTED);
	err = ctTrader->start();
	if (!err.isOk()) {
		printDebug("ctTrader->start() failed");
		ret = SmartKey_backend::status_errOpen;
		goto err_delete;
	}
	for (i = 0; i < CARD_OPEN_TIMEOUT; ++i) {
		if (isUserCancel()) {
			ret = SmartKey_backend::status_cancel;
			goto err_stop;
		}
		err = ctTrader->getNext(ctCard, 1);
		if (err.isOk())
			break;
		openWindowProgress();
	}
	if (!err.isOk()) {
		printDebug("card insert timeout");
		ret = SmartKey_backend::status_errTimeout;
		goto err_stop;
	}
	err = ctCard->openCard();
	if (!err.isOk()) {
		printDebug("ctCard->openCard() failed");
		ret = SmartKey_backend::status_errOpen;
		goto err_stop;
	}
out:
	showOpenWindow(false);
	return ret;

err_stop:
	ctTrader->stop();
	delete_ifnot_null(ctCard);
err_delete:
	delete_and_null(ctTrader);
	goto out;
}

void Chipcard1Interface::ct_close()
{
	ctCard->closeCard();
	ctTrader->stop();
	delete_and_null(ctCard);
	delete_and_null(ctTrader);
}

SmartKey_backend::Status Chipcard1Interface::ct_write(const QByteArray &data)
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	CTError err;
	std::string buf;
	showAccessWindow(true);
	unsigned int cardSize = ctCard->memorySize();
	if (cardSize < data.size() + CARD_BEGIN_OFFSET) {
		printDebug("card too small");
		ret = SmartKey_backend::status_err2small;
		goto out;
	}
	buf.assign(data.data(), data.size());
	err = ctCard->updateBinaryRaw(buf, CARD_BEGIN_OFFSET);
	if (!err.isOk()) {
		printDebug("ctCard->updateBinaryRaw() failed");
		ret = SmartKey_backend::status_errWrite;
		goto out;
	}
out:
	showAccessWindow(false);
	return ret;
}

SmartKey_backend::Status Chipcard1Interface::ct_read(QByteArray *data)
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	CTError err;
	showAccessWindow(true);
	unsigned int cardSize = ctCard->memorySize();
	std::string buf;
	err = ctCard->readBinaryRaw(buf, CARD_BEGIN_OFFSET, cardSize - CARD_BEGIN_OFFSET);
	if (!err.isOk()) {
		printDebug("ctCard->readBinaryRaw() failed");
		ret = SmartKey_backend::status_errRead;
		goto out;
	}
	data->duplicate(buf.c_str(), buf.size());
out:
	showAccessWindow(false);
	return ret;
}

SmartKey_backend::Status Chipcard1Interface::ct_erase()
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	CTError err;
	showAccessWindow(true);
	unsigned int cardSize = ctCard->memorySize();
	unsigned int i = 0;
	const unsigned int byteCntEachLoop = 255;
	std::string buf(byteCntEachLoop, static_cast<char>(0));
	while (i < cardSize) {
		if ((cardSize - i) < byteCntEachLoop)
			buf.assign(cardSize - i, static_cast<char>(0));
		err = ctCard->updateBinaryRaw(buf, i);
		if (!err.isOk()) {
			printDebug("ctCard->updateBinaryRaw() failed");
			ret = SmartKey_backend::status_errWrite;
			goto out;
		}
		i += byteCntEachLoop;
	}
out:
	showAccessWindow(false);
	return ret;
}
