/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004, 2005  by Michael Buesch                           *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "masterkey/smartkey/smartkey_backend.h"
#include "masterkey/smartkey/smartkey.h"
#include "ipc.h"
#include "waitwnd.h"
#include "pwmexception.h"

#include <time.h>
#include <unistd.h>

#include <string>

#include <kapplication.h>
#include <klocale.h>

#include <qeventloop.h>

#if defined(CONFIG_PWMANAGER_CHIPCARD1) && defined(CONFIG_PWMANAGER_CHIPCARD2)
# error "CONFIG_PWMANAGER_CHIPCARD1 and CONFIG_PWMANAGER_CHIPCARD2 defined. :-? confused."
#endif
#if !defined(CONFIG_PWMANAGER_CHIPCARD1) && !defined(CONFIG_PWMANAGER_CHIPCARD2)
# error "neither CONFIG_PWMANAGER_CHIPCARD1 nor CONFIG_PWMANAGER_CHIPCARD2 defined. :-? confused."
#endif
#ifdef CONFIG_PWMANAGER_CHIPCARD1
# include "masterkey/smartkey/libchipcard1/chipcard1interface.h"
#endif
#ifdef CONFIG_PWMANAGER_CHIPCARD2
# include "masterkey/smartkey/libchipcard2/chipcard2interface.h"
#endif


/** show the openWaitWnd. */
#define IPC_MSG_OWW_SHOW	"openWaitWnd_show"
/** hide the openWaitWnd. */
#define IPC_MSG_OWW_HIDE	"openWaitWnd_hide"
/** do timeout progress for openWaitWnd. */
#define IPC_MSG_OWW_PROGRESS	"openWaitWnd_progress"
/** show the accessWaitWnd. */
#define IPC_MSG_AWW_SHOW	"accessWaitWnd_show"
/** hide the accessWaitWnd. */
#define IPC_MSG_AWW_HIDE	"accessWaitWnd_hide"


/* variables for communication with the thread. */
static volatile SmartKey_backend::Task task = SmartKey_backend::task_none;
static volatile SmartKey_backend::Status status = SmartKey_backend::status_none;
static QByteArray *dataBuffer;
static ThreadIpcClient *threadIpc;


SmartKey_backend::SmartKey_backend(SmartKey *_sk)
 : sk (_sk)
 , openWaitWnd (0)
 , accessWaitWnd (0)
 , hostIpc (0)
{
}

SmartKey_backend::~SmartKey_backend()
{
	PWM_ASSERT(openWaitWnd == 0);
	PWM_ASSERT(accessWaitWnd == 0);
	PWM_ASSERT(hostIpc == 0);
}

SmartKey_backend::Status SmartKey_backend::writeCard(const QByteArray &data)
{
	PWM_ASSERT(status == status_none);
	PWM_ASSERT(task == task_none);
	task = task_write;
	// We cast it back to (const QByteArray &) in run() before using it
	dataBuffer = const_cast<QByteArray *>(&data);
	return fireUp();
}

SmartKey_backend::Status SmartKey_backend::readCard(QByteArray *data)
{
	PWM_ASSERT(status == status_none);
	PWM_ASSERT(task == task_none);
	task = task_read;
	dataBuffer = data;
	return fireUp();
}

SmartKey_backend::Status SmartKey_backend::eraseCard()
{
	PWM_ASSERT(status == status_none);
	PWM_ASSERT(task == task_none);
	task = task_erase;
	return fireUp();
}

void SmartKey_backend::cancel()
{
	if (!running())
		return;
	if (task == task_cancel)
		return;
	task = task_cancel;
	/* We don't need to lock openWaitWnd, as we can't
	 * race against fireUp(). cancel() is called synchronously
	 * via signal/slots in processEvents(), called by relax(),
	 * called by fireUp().
	 */
	if (openWaitWnd)
		openWaitWnd->setGenericText(i18n("Cancelling..."));
	while (!wait(0))
		relax();
}

SmartKey_backend::Status SmartKey_backend::fireUp()
{
	// initialize the IPC
	hostIpc = new ThreadIpcHost;
	connect(hostIpc, SIGNAL(receivedMessage(const QByteArray &)),
		this, SLOT(threadMessage(const QByteArray &)));
	threadIpc = new ThreadIpcClient(*hostIpc);

	// create the notification windows.
	openWaitWnd = new WaitWnd(i18n("Accessing SmartCard"),
				  i18n("Please insert the SmartCard into "
				       "your SmartCard Terminal."),
				  true, true, sk->parent);
	openWaitWndCnt = CARD_OPEN_TIMEOUT;
	connect(openWaitWnd, SIGNAL(closing()),
		this, SLOT(cancel()));
	accessWaitWnd = new WaitWnd(i18n("Accessing SmartCard"),
				    i18n("Accessing SmartCard.\n"
					 "Please do not remove the SmartCard."),
				    false, false, sk->parent);

	hostIpc->startLookup();
	start(); // now start the thread
	while (running())
		relax();
	hostIpc->stopLookup();

	delete_and_null(threadIpc);
	delete_and_null(hostIpc);
	delete_and_null(accessWaitWnd);
	delete_and_null(openWaitWnd);

	PWM_ASSERT(status != status_running);
	PWM_ASSERT(status != status_none);
	Status retStatus = status;
	status = status_none;
	task = task_none;
	return retStatus;
}

void SmartKey_backend::relax()
{
	QThread::msleep(10);
	bool inEventLoop = (kapp->eventLoop()->loopLevel() >= 1);
	if (inEventLoop)
		kapp->processEvents();
}

void SmartKey_backend::start(Priority priority)
{
	PWM_ASSERT(task != task_none);
	QThread::start(priority);
}

void SmartKey_backend::threadMessage(const QByteArray &msg)
{
	if (QCString(msg) == QCString(IPC_MSG_OWW_SHOW)) {
		openWaitWnd->show();
		if (sk->parent)
			sk->parent->setEnabled(false);
		openWaitWnd->setEnabled(true);
		goto oww_set_timeout;
	} else if (QCString(msg) == QCString(IPC_MSG_OWW_HIDE)) {
		if (sk->parent)
			sk->parent->setEnabled(true);
		openWaitWnd->hide();
	} else if (QCString(msg) == QCString(IPC_MSG_OWW_PROGRESS)) {
		PWM_ASSERT(openWaitWndCnt > 0);
		if (task == task_cancel)
			return;
		openWaitWndCnt--;
		goto oww_set_timeout;
	} else if (QCString(msg) == QCString(IPC_MSG_AWW_SHOW)) {
		accessWaitWnd->show();
		if (sk->parent)
			sk->parent->setEnabled(false);
		accessWaitWnd->setEnabled(true);
	} else if (QCString(msg) == QCString(IPC_MSG_AWW_HIDE)) {
		if (sk->parent)
			sk->parent->setEnabled(true);
		accessWaitWnd->hide();
	} else
		BUG();
	return;
oww_set_timeout:
	openWaitWnd->setGenericText(i18n("Timeout: %1")
			.arg(QString::number(openWaitWndCnt)));
}

void SmartKey_backend::run()
{
	status = status_running;
	Status operationStatus = status_none;
	ChipcardInterface *chipcard = getChipcardInterface();
	switch (task) {
	case task_read:
		operationStatus = chipcard->read(dataBuffer);
		break;
	case task_write:
		operationStatus = chipcard->write(*dataBuffer);
		break;
	case task_erase:
		operationStatus = chipcard->erase();
		break;
	case task_cancel:
		break;
	case task_none:
		BUG();
	}
	delete chipcard;
	PWM_ASSERT(operationStatus != status_none);
	PWM_ASSERT(operationStatus != status_running);
	status = operationStatus;
}

ChipcardInterface * SmartKey_backend::getChipcardInterface() const
{
	ChipcardInterface *ret;
#ifdef CONFIG_PWMANAGER_CHIPCARD1
	ret = new Chipcard1Interface;
#endif
#ifdef CONFIG_PWMANAGER_CHIPCARD2
	ret = new Chipcard2Interface;
#endif
	return ret;
}

bool ChipcardInterface::isUserCancel() const
{
	return (task == SmartKey_backend::task_cancel);
}

void ChipcardInterface::showOpenWindow(bool show) const
{
	if (show)
		threadIpc->sendMessage(QCString(IPC_MSG_OWW_SHOW));
	else
		threadIpc->sendMessage(QCString(IPC_MSG_OWW_HIDE));
}

void ChipcardInterface::openWindowProgress() const
{
	threadIpc->sendMessage(QCString(IPC_MSG_OWW_PROGRESS));
}

void ChipcardInterface::showAccessWindow(bool show) const
{
	if (show)
		threadIpc->sendMessage(QCString(IPC_MSG_AWW_SHOW));
	else
		threadIpc->sendMessage(QCString(IPC_MSG_AWW_HIDE));
}

#include "smartkey_backend.moc"
