/* This file is part of KolourPicker
   Copyright (c) 2001 Malte Starostik <malte@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.

$Id: kolourpicker.cpp 682260 2007-07-02 08:52:55Z ilic $
*/

#include <qfile.h>
#include <qtextstream.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qimage.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <qbitmap.h>
#include <qpainter.h>
#include <qtooltip.h>
#include <qcursor.h>
#include <QDesktopWidget>
//Added by qt3to4:
#include <QPixmap>
#include <Q3PtrList>
#include <QKeyEvent>
#include <QResizeEvent>
#include <QMouseEvent>
#include <QApplication>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kcomponentdata.h>
#include <kconfig.h>
#include <kaboutdata.h>
#include <kaboutapplicationdialog.h>
#include <kmessagebox.h>
#include <kmenu.h>

#include "kolourpicker.h"
#include "kolourpicker.moc"

#include <X11/Xlib.h>

// Applet initialization function
extern "C"
{
	KDE_EXPORT KPanelApplet* init(QWidget *parent, const QString& configFile)
	{
		KGlobal::locale()->insertCatalog("kolourpicker");
		return new KolourPicker(configFile, Plasma::Normal,
			Plasma::About | Plasma::Help,
			parent, "kolourpicker");
	}
}

KolourPicker::KolourPicker(const QString& configFile, Plasma::Type type,
	int actions, QWidget *parent, const char *name)
	: KPanelApplet(configFile, type, actions, parent),
	  m_picking(0)
{
	KAboutData *about = new KAboutData("kolourpicker", 0,
									   ki18n("Color Picker"),
									   "v0.1",
									   ki18n("An applet to pick color values from anywhere on the screen"),
									   KAboutData::License_GPL_V2,
									   ki18n("(c) 2001 Malte Starostik"));
	about->addAuthor(ki18n("Malte Starostik"), ki18n("Original Author"), "malte@kde.org");
	m_componentData = KComponentData(about);

	KConfigGroup conf(config(), "General");
	QStringList history = conf.readEntry("History",QStringList());
	for (QStringList::ConstIterator it = history.begin(); it != history.end(); ++it)
		m_history.append(QColor(*it));

	m_colourButton = new QPushButton(this);
	m_colourButton->setPixmap(SmallIcon("color-picker"));
	m_colourButton->setFixedSize(20, 20);
	QToolTip::add(m_colourButton, i18n("Pick a color"));
	connect(m_colourButton, SIGNAL(clicked()), SLOT(slotPick()));

	m_historyButton = new QPushButton(this);
	m_historyButton->setFixedSize(20, 20);
	if (m_history.count())
		m_historyButton->setPixmap(colorPixmap(m_history.last()));
	else
	{
		m_historyButton->setPixmap(colorPixmap(QColor()));
		m_historyButton->setEnabled(false);
	}
	
	QToolTip::add(m_historyButton, i18n("History"));
	connect(m_historyButton, SIGNAL(clicked()), SLOT(slotHistory()));
}

KolourPicker::~KolourPicker()
{
    KGlobal::locale()->removeCatalog("kolourpicker");
}


int KolourPicker::heightForWidth(int width) const
{
	return (width > 40) ? 22 : 44;
}

int KolourPicker::widthForHeight(int height) const
{
	return (height > 40) ? 22 : 44;
}

void KolourPicker::about()
{
	KAboutApplicationDialog dlg(m_componentData.aboutData());
	dlg.exec();
}

void KolourPicker::help()
{
	KMessageBox::information(0, i18n("Unimplemented help system."));
}

void KolourPicker::slotPick()
{
	m_picking = true;
	grabMouse(Qt::CrossCursor);
	grabKeyboard();
}

void KolourPicker::slotHistory()
{
	KMenu popup;
	popup.addTitle(SmallIcon("colorize"), i18n("History"));
	Q3PtrList<QMenu> subMenus;
	subMenus.setAutoDelete(true);
	QList<QColor>::ConstIterator it = m_history.end();
	while( it != m_history.begin()) {
		--it;
		QMenu *sub = copyPopup(*it, false);
		subMenus.append(sub);
		popup.insertItem(colorPixmap(*it),
			QString("%1, %2, %3").arg((*it).red()).arg((*it).green()).arg((*it).blue()),
			sub);
	}
	popup.addSeparator();
	QAction *clear = popup.addAction(SmallIcon("history-clear"), i18n("&Clear History"));
	QAction* id = popup.exec(QCursor::pos());
	if (id == clear)
	{
		m_history.clear();
		m_historyButton->setEnabled(false);
		arrangeButtons();
		KConfigGroup conf = config()->group("General");
		conf.writeEntry("History", QStringList());
		conf.sync();
	}
	else if (id != 0)
		setClipboard(id->text());
}

void KolourPicker::mouseReleaseEvent(QMouseEvent *e)
{
	if (m_picking)
	{
		m_picking = false;
		releaseMouse();
		releaseKeyboard();
		QDesktopWidget *desktop = QApplication::desktop();
		QPixmap pm = QPixmap::grabWindow(desktop->winId(),
			e->globalPos().x(), e->globalPos().y(), 1, 1);
		QImage img = pm.convertToImage();
		QColor color(img.pixel(0, 0));

		// eventually remove a dupe
		QList<QColor>::Iterator dupe = m_history.find(color);
		if (dupe != m_history.end())
			m_history.remove(dupe);

		m_history.append(color);
		while (m_history.count() >= 10)
			m_history.remove(m_history.begin());
		m_historyButton->setEnabled(true);
		arrangeButtons();
		QStringList history;
		for (QList<QColor>::ConstIterator it = m_history.begin();
			it != m_history.end();
			++it)
		{
			history.append((*it).name());
		}
		KConfigGroup conf = config()->group("General");
		conf.writeEntry("History", history);
		conf.sync();
		m_historyButton->setPixmap(colorPixmap(color));
		QMenu *popup = copyPopup(color, true);
		QAction *id = popup->exec(e->globalPos());
		if (id != 0)
			setClipboard( id->text() );
		delete popup;
	}
	else
		KPanelApplet::mouseReleaseEvent(e);
}

// set both clipboard and selection
void KolourPicker::setClipboard(const QString& text)
{
	QClipboard *clip = QApplication::clipboard();
	clip->setText(text,QClipboard::Clipboard);
	clip->setText(text,QClipboard::Clipboard);
}

void KolourPicker::keyPressEvent(QKeyEvent *e)
{
	if (m_picking)
	{
		if (e->key() == Qt::Key_Escape)
		{
			m_picking = false;
			releaseMouse();
			releaseKeyboard();
		}
		e->accept();
		return;
	}
	KPanelApplet::keyPressEvent(e);
}

void KolourPicker::resizeEvent(QResizeEvent *)
{
	arrangeButtons();
}

void KolourPicker::arrangeButtons()
{
	int h, w, p;

	if (orientation() == Qt::Horizontal)
	{
		h = height();
		if (h > 40)
		{
			// vertical layout
			p = (h - 40)/3;
			m_colourButton->setGeometry(2, p, 20, 20);
			m_historyButton->setGeometry(2, 2*p+20, 20, 20);
		}
		else
		{
			// horizontal layout
			p = (h - 20)/2;
			m_colourButton->setGeometry(2, p, 20, 20);
			m_historyButton->setGeometry(24, p, 20, 20);
		}
	}
	else
	{
		w = width();
		if (w > 40)
		{
			// horizontal layout
			p = (w - 40)/3;
			m_colourButton->setGeometry(p, 2, 20, 20);
			m_historyButton->setGeometry(2*p+20, 2, 20, 20);
		}
		else
		{
			// vertical layout
			p = (w - 20)/2;
			m_colourButton->setGeometry(p, 2, 20, 20);
			m_historyButton->setGeometry(p, 24, 20, 20);
		}
	}

	updateGeometry();
}

QMenu *KolourPicker::copyPopup(const QColor &c, bool title) const
{
	KMenu *popup = new KMenu;
	if (title)
		popup->addTitle(colorPixmap(c), i18n("Copy Color Value"));
	QString value;
	// r, g, b
	value.sprintf("%u, %u, %u", c.red(), c.green(), c.blue());
	popup->insertItem(SmallIcon("text"), value);
	// HTML, lower case hex chars
	value.sprintf("#%.2x%.2x%.2x", c.red(), c.green(), c.blue());
	popup->insertItem(SmallIcon("html"), value);
	if (value.find(QRegExp("[a-f]")) >= 0)
	{
		// HTML, upper case hex chars
		value.sprintf("#%.2X%.2X%.2X", c.red(), c.green(), c.blue());
    popup->insertItem(SmallIcon("html"), value);
  }
  // lower case hex chars 
  value.sprintf( "%.2x%.2x%.2x", c.red(), c.green(), c.blue() ); 
  popup->insertItem( SmallIcon( "html" ), value ); 
  if ( value.find( QRegExp( "[a-f]" ) ) >= 0 ) 
  { 
    //  upper case hex chars 
    value.sprintf( "%.2X%.2X%.2X", c.red(), c.green(), c.blue() ); 
    popup->insertItem( SmallIcon( "html" ), value ); 
  } 
	// Color name
	QStringList names = colorNames(c.red(), c.green(), c.blue());
	for (QStringList::ConstIterator it = names.begin(); it != names.end(); ++it)
		popup->insertItem(SmallIcon("text"), *it);
	return popup;
}

QPixmap KolourPicker::colorPixmap(const QColor &c) const
{
	QPixmap pm(16, 16);
	pm.fill(c);
	QBitmap mask(16, 16);
	mask.fill(Qt::color0);
	QPainter p(&mask);
	p.setPen(Qt::color1);
	p.setBrush(Qt::color1);
	p.drawEllipse(0, 0, 15, 15);
	p.end();
	pm.setMask(mask);
	return pm;
}

const QStringList &KolourPicker::colorNames(int r, int g, int b) const
{
	static QStringList NullList;
	if (m_colorNames.isEmpty())
	{
		QFile f("/usr/lib/X11/rgb.txt");
		if (!f.open(QIODevice::ReadOnly))
			return NullList;
		QTextStream str(&f);
		QString red, green, blue;
		while (!str.atEnd())
		{
			str >> red;
			if (red.simplified()[0].latin1() == '!')
			{
				str.readLine();
				continue;
			}
			str >> green >> blue;
			const_cast<KolourPicker *>(this)->m_colorNames[(red.toInt() << 16) + (green.toInt() << 8) + blue.toInt()]
					.append(str.readLine().simplified());
		}
	}
	return m_colorNames[(r << 16) + (g << 8) + b];
}

// vim: ts=4 sw=4 noet
