/*
    This file is part of the Konsole Terminal.
    
    Copyright (C) 2006 Robert Knight <robertknight@gmail.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301  USA.
*/

#ifndef VIEWCONTAINER_H
#define VIEWCONTAINER_H

#include <QObject>
#include <QList>

class QTabWidget;
class QWidget;

/**
 * An interface for container widgets which can hold one or more views.
 *
 * The container widget typically displays a list of the views which
 * it has and provides a means of switching between them.  
 *
 * Subclasses should reimplement the viewAdded() and viewRemoved() functions
 * to actually add or remove view widgets from the container widget, as well
 * as updating any navigation aids.
 * 
 */
class ViewContainer : public QObject
{
Q_OBJECT
    
public:
    virtual ~ViewContainer() { emit destroyed(this); }

    /** Returns the widget which contains the view widgets */
    virtual QWidget* containerWidget() const = 0;

    /** Adds a new view to the container widget */
    void addView(QWidget* view);
    /** Removes a view from the container */
    void removeView(QWidget* view);

    /** Returns a list of the contained views */
    const QList<QWidget*> views();
   
    /** 
     * Returns the view which currently has the focus or 0 if none
     * of the child views have the focus.
     */ 
    virtual QWidget* activeView() const = 0;

signals:
    /** Emitted when the container is deleted */
    void destroyed(ViewContainer* container);

protected:
    /** 
     * Performs the task of adding the view widget
     * to the container widget.
     */
    virtual void viewAdded(QWidget* view) = 0;
    /**
     * Performs the task of removing the view widget
     * from the container widget.
     */
    virtual void viewRemoved(QWidget* view) = 0;
    

private:
    QList<QWidget*> _views;
};

/** 
 * A view container which uses a QTabWidget to display the views and 
 * allow the user to switch between them.
 */
class TabbedViewContainer : public ViewContainer  
{
public:
    TabbedViewContainer();
    virtual ~TabbedViewContainer();
    
    virtual QWidget* containerWidget() const;
    virtual QWidget* activeView() const;

protected:
    virtual void viewAdded( QWidget* view );
    virtual void viewRemoved( QWidget* view ); 

private:
    QTabWidget* _tabWidget;  

    static int debug; 

};

#endif //VIEWCONTAINER_H
