/***************************************************************************
 *   Copyright 2007      Johannes Bergmeier <johannes.bergmeier@gmx.net>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#ifndef _KSUDOKUGAMEVARIANTS_H_
#define _KSUDOKUGAMEVARIANTS_H_

#include <QObject>
#include <QList>
#include <KUrl>
#include <QAbstractListModel>

class SKSolver;
namespace ksudoku {

class KsView;
class Game;
class GameVariantCollection;
class GameVariant {
public:
	explicit GameVariant(const QString& name, GameVariantCollection* collection=0);
	virtual ~GameVariant() {}

public:
	QString name() const { return m_name; }
	
	/// This method returs whether the variant has an configure option
	virtual bool canConfigure() const = 0;
	
	/// Shows a configure dialog and changes the settings
	virtual bool configure() = 0;
	
	/// Whether this variant can be started without any values in the grid
	virtual bool canStartEmpty() const = 0;
	
	/// Creates a game without a puzzle but with an empty grid
	virtual Game startEmpty() const = 0;
	
	/// Creates a instance of this game variant
	virtual Game createGame(int difficulty) const = 0;
	
	/// Creates the correct view for the game.
	/// Game needs to be compatible with this GameVariant
	virtual KsView* createView(const Game& game) const = 0;

private:
	QString m_name;
};

class GameVariantCollection : public QAbstractListModel {
friend class GameVariant;
Q_OBJECT
public:
	GameVariantCollection(QObject* parent, bool autoDel);
	~GameVariantCollection();
		
public:
	void addVariant(GameVariant* variant);
	
public:
	QVariant data(const QModelIndex &index, int role) const;
	int rowCount(const QModelIndex&) const;
	QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const;
	GameVariant* variant(const QModelIndex&) const;

signals:
	void newVariant(GameVariant* variant);
	
public:
	QList<GameVariant*> m_variants;
	bool m_autoDelete;
};

class SudokuGame : public GameVariant {
public:
	SudokuGame(const QString& name, uint order, GameVariantCollection* collection=0);
	
public:
	bool canConfigure() const;
	bool configure();
	bool canStartEmpty() const;
	Game startEmpty() const;
	Game createGame(int difficulty) const;
	KsView* createView(const Game& game) const;
	
private:
	uint m_order;
	uint m_symmetry;
	
	mutable SKSolver* m_solver;
};

class RoxdokuGame : public GameVariant {
public:
	RoxdokuGame(const QString& name, uint order, GameVariantCollection* collection=0);
	
public:
	bool canConfigure() const;
	bool configure();
	bool canStartEmpty() const;
	Game startEmpty() const;
	Game createGame(int difficulty) const;
	KsView* createView(const Game& game) const;
	
private:
	uint m_order;
	uint m_symmetry;
	
	mutable SKSolver* m_solver;
};

class CustomGame : public GameVariant {
public:
	CustomGame(const QString& name, const KUrl& url, GameVariantCollection* collection=0);
	
public:
	bool canConfigure() const;
	bool configure();
	bool canStartEmpty() const;
	Game startEmpty() const;
	Game createGame(int difficulty) const;
	KsView* createView(const Game& game) const;
	
private:
	KUrl m_url;
	mutable SKSolver* m_solver;
};

}

#endif
