// -*- C++ -*-

/***************************************************************************
 *   Copyright (C) 2005--2006 by Stefan Kebekus                            *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#ifndef _PDFRENDERER_H_
#define _PDFRENDERER_H_

#define UNSTABLE_POPPLER_QT4

#include <poppler-qt4.h>

#include <qmap.h>

#include "length.h"
#include "documentRenderer.h"
#include "ui_documentInfoWidget_base.h"

class DocumentInfoWidget_base : public QWidget, public Ui::DocumentInfoWidget_base
{
public:
  DocumentInfoWidget_base( QWidget *parent ) : QWidget( parent ) {
    setupUi( this );
  }
};


class KShellProcess;

class documentPage;
class PropertiesDialog;

class PdfSyncTarget
{
public:
  PdfSyncTarget()
    : record(0) {};

  PdfSyncTarget(Length x_, Length y_, int r)
    : x(x_), y(y_), record(r) {};

  Length x;
  Length y;
  int record;
};

class PdfSyncLine
{
public:
  PdfSyncLine()
    : linenumber(0), filename(QString::null) {};

  PdfSyncLine(int _linenumber, QString _filename)
    : linenumber(_linenumber), filename(_filename) {};

  int linenumber;
  QString filename;
};

/*! \brief Well-documented minimal implementation of a documentRenderer
  
This class provides a well-documented reference implementation of a
documentRenderer, suitable as a starting point for a real-world
implementation. This class is responsible for document loading and rendering. Apart from the constructor and the descructor, it
implements only the necessary methods setFile() and drawPage(). The
method setFile() ignores the file content and simulates a document
with 10 empty pages of A4 size and a few anchors and bookmarks.
*/

class PdfRenderer : public DocumentRenderer
{
  Q_OBJECT

public:
   /** Default constructor

       This constructor simply prints a message and calls the default constructor.
   */
   PdfRenderer(ligaturePluginGUI*);

   /** Destructor

       The destructor simpley prints a message. It uses the mutex to ensure that this class is not destructed while another thread is currently using it.
   */
  ~PdfRenderer();

  /** Opening a file

      This implementation does the necessary consistency checks and
      complains, e.g. if the file does not exist, but otherwise completely
      disregards the file content. It simulates a document of 10 empty pages of
      A4 size, with a few sample bookmarks and  anchors "p1", "p2", "p3"
      for page 1, 2 and 3, respectively.

      @param fname the name of the file that should be opened.
  */
  virtual bool setFile(const QString& fname, const KUrl&);
  virtual void clear();

  /** Check if the give file is a valid PDF Document. */
  virtual bool isValidFile(const QString& fileName) const;

  /** Rendering a page

      This implementation does almost nothing. First, it is checked if
      the arguments are in a reasonable range, and error messages are
      printed if this is not so. Secondly, the page is cleared and the
      background set to white. Nothing else is drawn.

      @param id The JobId describing the requested page.
  */
  virtual RenderedDocumentPagePixmap* drawPage(const JobId& id);
  virtual RenderedDocumentPagePixmap* getText(const JobId& id);

  virtual bool supportsTextSearch() const { return true; };

  /** PDF to PostScript conversion
      
  This method converts the document to a PostScript file.
  
  @param filename name of the PostScript file to generate
  
  @param pageList list of pages that are to be converted, with the
  usual convention that "1" means "first page"
  
  @returns 'true' if the conversion was successful, 'false' if it
  wasn't. The conversion can fail, for example, when the user aborts
  the operation.
  */
  bool convertToPSFile(QString filename, QList<int> &pageList );
  
  /** Saves the file */
  bool save(const QString &new_filename);

  void inverseSearch(PageNumber pageNumber, Length x, Length y, bool call = false);

  /** This method will try to parse the reference part of the PDF
      file's URL, (either a number, which is supposed to be a page
      number, or src:<line><filename>) and see if a corresponding
      section of the DVI file can be found. If so, it returns an
      anchor to that section. If not, it returns an invalid anchor. */
  virtual Anchor parseReference(const QString& reference);

 /** Sets the background_file as the current background. If an error
      occurs, a message will be shown to the user. */
  void setBackground(QString background_file);

  /** Deletes pages from the document */
  void deletePages(quint16 from, quint16 to);

  /** Inserts another PDF file into the current document.

  @param file PDF-file to be inserted

  @param page the number of the page where 'file' should start in the
  new document. For instance, if page == 1, the document 'file' will
  be prepended to the current document. If page == 2, the document
  will be inserted AFTER page 1 of the current document. The value
  page MUST ALWAYS SATISFY 0 < page < numPages()+1.

  If the insertion fails, e.g. because 'file' does not contain a valid
  PDF file, an error dialog is displayed.
  */
  void insertPages(QString file, PageNumber page);

public slots:
  void handleSRCLink(const QString& filename, int linenumber);

  /** Returns 'true' is the document was modified */
  bool showInfo();

private slots:
  void showThatSourceInformationIsPresent();

private:
  void fillInText(RenderedDocumentPagePixmap* page, Poppler::Page* pdfPage, double res);

  void getOutline();
  void getOutlineSub(QDomNode* tocParent, Bookmark* bookmarkParent);
	  
  bool parsePdfSync(QString basename, QTextStream& stream);

  KShellProcess* proc;
  QString export_errorString;

  QString path;
  QString filename;

  // To allow file editing, the renderer uses a "working copy" of the
  // PDF file. When a file is loaded, the workingCopyFileName is the
  // same as the fileName. Whenever the file is modified, a working
  // copy is created, whose file name is specified here. That file is
  // then used instead of the original file.
  //
  // When the file is closed, the working copy (if different from the
  // original file) will be deleted.
  QString workingCopyFileName;

  // pdfsync data structures
  QVector<QVector<PdfSyncTarget> > pdfSyncTargets;

  QMap<int, PdfSyncLine> pdfSyncLines;

  Poppler::Document* document;
};

#endif
