/***************************************************************************
 *   Copyright (C) 2005-2006 by Stefan Kebekus                             *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <config.h>

#include <kglobal.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kicontheme.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kio/job.h>
#include <kprocio.h>
#include <ktemporaryfile.h>

#include <QDateTime>
#include <QFileInfo>
#include <QLabel>
#include <QLineEdit>
#include <QTextEdit>
#include <QTextStream>

#include "documentWidget.h"
#include "ligaturePluginGUI.h"
#include "length.h"
#include "psRenderer.h"
#include "psMultipage.h"

#define KF_DEBUG

//#define PDFSYNC_DEBUG

PsRenderer::PsRenderer(ligaturePluginGUI* _multiPage)
  : DocumentRenderer(_multiPage), currentRenderingRequest(0), internalDoc(0), eps(false)
{
}



PsRenderer::~PsRenderer()
{
#ifdef KF_DEBUG
  kDebug() << "~PsRenderer" << endl;
#endif

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);
}


RenderedDocumentPagePixmap* PsRenderer::drawPage(const JobId& id)
{
#ifdef KF_DEBUG
  kDebug() << "PsRenderer::drawPage(JobId) called, page number " << id.pageNumber << endl;
#endif

  // Paranoid safety checks
  if (!id.pageNumber.isValid()) {
    kDebug() << "PsRenderer::drawPage(JobId) called  with an invalid page number" << endl;
    return 0;
  }

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // more paranoid safety checks
  if (id.pageNumber > numPages) {
    kError() << "PsRenderer::drawPage(JobId) called for a documentPage with page number " << id.pageNumber
	      << " but the current file has only " << numPages << " pages." << endl;
    return 0;
  }

  double res = id.resolution;

  currentPageNumber = id.pageNumber;

  int pageNumber = id.pageNumber - 1;

  bool isThumbnail = id.isThumbnail;

  // Generate a PNG-file
  KTemporaryFile tempPNGfile;
  tempPNGfile.setSuffix(".png");
  tempPNGfile.open();

  // Step 1: Write the PostScriptString to a File
  KTemporaryFile tempPSfile;
  tempPSfile.setSuffix(".ps");
  tempPSfile.open();

  if (!eps) {
    QList<int> pages;
    pages.append(currentPageNumber);
    generatePSFileForPage(tempPSfile.fileName(), pages);
  }

  QFileInfo tempPSfileInfo(tempPSfile.fileName());

  // Step 2: Call GS with the File
  KProcIO proc;
  QStringList argus;
  argus << "gs";
  argus << "-dSAFER" << "-dPARANOIDSAFER" << "-dDELAYSAFER" << "-dNOPAUSE" << "-dBATCH";
  argus << "-sDEVICE=png16m";
  argus << QString("-sOutputFile=%1").arg(tempPNGfile.fileName());

  if (eps)
  {
    argus << QString("-sExtraIncludePath=%1").arg(filename);
  }
  else
  {
    argus << QString("-sExtraIncludePath=%1").arg(tempPSfileInfo.dirPath(true));
  }

  SimplePageSize ps = sizeOfPage(id.pageNumber);
  int pageHeight = ps.sizeInPixel(res).height();
  int pageWidth = ps.sizeInPixel(res).width();

  if (isThumbnail) {
    // Because the rendering quality of ghostscript is quite bad at low resolutions, we
    // render thumbnails two times as big as needed, and then downscale the final image.
    argus << QString("-r%1").arg(2 * res);                       // resolution in dpi
    argus << QString("-g%1x%2").arg(2 * pageWidth).arg(2 * pageHeight); // page size in pixels
  } else {
    argus << QString("-r%1").arg(res);                       // resolution in dpi
    argus << QString("-g%1x%2").arg(pageWidth).arg(pageHeight); // page size in pixels
  }
  argus << "-dTextAlphaBits=4 -dGraphicsAlphaBits=2"; // Antialiasing
  argus << "-c" << KProcess::quote("<< /PermitFileReading [ ExtraIncludePath ] /PermitFileWriting [] /PermitFileControl [] >> setuserparams .locksafe");
  if (eps) {
    argus << "-c" << KProcess::quote("%1 %2 translate").arg(-internalDoc->boundingBox().llx()).arg(-internalDoc->boundingBox().lly());
    argus << "-f" << filename;
  } else
    argus << "-f" << tempPSfile.name();

  proc << argus;
  kDebug() << "start ghostscript for page " << currentPageNumber << "." << endl;

#ifdef __GNUC__
#warning CHECK THIS CALL
#endif  
  //  if (proc.start(KProcess::Block) == false)
  if (system(argus.join(" ").toLatin1()) != 0)

  {
    // Starting ghostscript did not work.
    // TODO: Issue error message, switch PS support off.
    kError() << "ghostscript could not be started" << endl;
  }

/*
  QString line;
  QString error;
  while (proc.readln(line) != -1)
  {
    error += line;
  }
  kDebug() << error << endl;
*/

  QImage img;
  bool ok = img.load(tempPNGfile.fileName(), "PNG");

  if (!ok)
  {
    kError() << "Loading temporary image file failed." << endl;
    return 0;
  }

  RenderedDocumentPagePixmap* page = multiPage->createDocumentPagePixmap(id);
  page->resize(pageWidth, pageHeight);

  if (isThumbnail)
  {
    // Downscale thumbnail image to the correct size.
    page->setImage(img.scaled(page->width(), page->height(), Qt::IgnoreAspectRatio, Qt::SmoothTransformation));
  }
  else
  {
    page->setImage(img);
  }

  page->isEmpty = false;
  return page;
}


bool PsRenderer::save(const QString &new_filename)
{
  kError() << "PsRenderer::save(const QString &new_filename)" << endl;
  
  if (new_filename.isEmpty())
    return false;
  
  QFileInfo src(workingCopyFileName);
  QFileInfo dest(new_filename);
  
  if (src.absFilePath() == dest.absFilePath())
    return true;

  kError() << "SAVING " << src.absFilePath() << endl;

  KProcIO proc;
  proc << "cp" << src.absFilePath() << dest.absFilePath();
  if (proc.start(KProcess::Block) == false) {
    KMessageBox::error(parentWidget, i18n("<qt>Could not start the <strong>cp</strong> copy command. The document is <strong>not</strong> saved.</qt>"),
                       i18n("Error saving document"));
    return false;
  }
  
  if (!proc.normalExit() || (proc.exitStatus() != 0)) {
    KMessageBox::error(parentWidget, i18n("<qt>Error saving the document. The document is <strong>not</strong> saved.</qt>"),
                       i18n("Error saving document"));
    return false;
  }
  
  _isModified = false;
  return true;
}


bool PsRenderer::showInfo()
{
  KDialog dialog(parentWidget);
  dialog.setCaption( i18n("Document Properties") );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setModal( true );

  // Setup the main properties widget
  DocumentInfoWidget_base infoWdg( &dialog );
  {
    KMimeType::Ptr mimetype = KMimeType::findByPath( filename );
    infoWdg.mimeTypePix->setPixmap(KIconLoader::global()->loadMimeTypeIcon(mimetype->iconName(),K3Icon::NoGroup, 48));

    infoWdg.m_mime->setText( mimetype->comment() );
    QFileInfo fi(filename);
    infoWdg.m_filename->setText( "<qt><strong>"+fi.fileName()+"</strong></qt>" );

    infoWdg.m_title->setText( internalDoc->title() );
    infoWdg.m_title->setCursorPosition(0);
    infoWdg.m_for->setText( internalDoc->author() );
    infoWdg.m_for->setCursorPosition(0);
    if (!internalDoc->creator().isEmpty())
      infoWdg.m_creator->setText( internalDoc->creator() );
    if (!internalDoc->date().isEmpty())
      infoWdg.m_date->setText(  internalDoc->date() );
    if (!internalDoc->copyright().isEmpty())
      infoWdg.m_copyright->setText( internalDoc->copyright() );

    infoWdg.m_path->setText( fi.dirPath(true) );
    infoWdg.m_size->setText( KIO::convertSize(fi.size()) );
    infoWdg.m_modified->setText( KGlobal::locale()->formatDateTime(fi.lastModified()) );
    infoWdg.m_pages->setText( QString("%1").arg(numPages) );
  }

  // Show the dialog
  dialog.setMainWidget(&infoWdg);
  dialog.exec();

  if (dialog.result() != QDialog::Accepted)
    return false;

  // Open the input file
  QFile psFile(workingCopyFileName);
  if (!psFile.open(IO_ReadOnly))
    return false;

  // Open the output file
  KTemporaryFile new_wcopy;
  new_wcopy.setSuffix(".pdf");
  new_wcopy.setAutoRemove(false);
  new_wcopy.open();
  QFile outFile(new_wcopy.fileName());
  if (!outFile.open(IO_WriteOnly))
    return false;

  //
  // Read, modify and write the PreProlog
  //

  // First read the PreProlog into a QByteArray.
  Position prolog = internalDoc->prolog();
  unsigned long length = prolog.first;
  QByteArray data(length);
  if (!psFile.at(0)) {
    kError() << "seeking to prolog failed" << endl;
    outFile.remove();
    return false;
  }
  psFile.readBlock(data.data(), length);

  // Read the QByteArray into QStrings, replace the text, if
  // necessary, and write them out
  QTextIStream i_stream(&data);
  QTextStream o_stream(&outFile);
  i_stream.setEncoding(QTextStream::Latin1);
  o_stream.setEncoding(QTextStream::Latin1);

  QStringList DSCComments;
  while(!i_stream.atEnd()) {
    QString line = i_stream.readLine();
    if (line.startsWith("%%Title:"))
      continue;
    if (line.startsWith("%%For:"))
      continue;
    DSCComments += line;
  }

  if (DSCComments.count() < 2) {
    DSCComments += "%% Ligature could not find any DSC comments at the beginning of the document. Expect problems.";
    DSCComments += "%%";
  }

  DSCComments.insert(1, "%%For: " + infoWdg.m_for->text());
  DSCComments.insert(1, "%%Title: " + infoWdg.m_title->text());

  for (int i = 0; i < DSCComments.size(); ++i)
    o_stream << DSCComments.at(i) << endl;
  
  //
  // Read and write the rest of the file
  //
  length = psFile.size() - prolog.first;
  data.resize(0);
  data.resize(length);

  psFile.readBlock(data.data(), length);
  outFile.writeBlock(data);


  // Now load the new working copy
#ifdef __GNUC__
#warning DUPLICATED CODE. OPTIMIZE THIS
#endif  
  clear();
  workingCopyFileName = new_wcopy.name();

  internalDoc = new GSInternalDocument(workingCopyFileName, GSInternalDocument::PS);

  // Set the number of pages page sizes
  kDebug() << "get number of pages." << endl;

  numPages = internalDoc->dsc()->page_count();

  // For some eps files dsc()->page_count() returns 0.
  // Work around this issue.
  if (numPages == 0)
    numPages = 1;

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);

  eps = false;
  KMimeType::Ptr mimetype = KMimeType::findByPath(workingCopyFileName);
  if (mimetype->name() == "image/x-eps")
  {
    kDebug() << "File is an eps" << endl;
    eps = true;

    Length w,h;
    QSize pageSize;
    pageSize = internalDoc->computePageSize(internalDoc->pageMedia(0));

    w.setLength_in_bigPoints(pageSize.width());
    h.setLength_in_bigPoints(pageSize.height());
    pageSizes[0].setPageSize(w, h);
    kDebug() << "eps file setup finished" << endl;
  }
  else if (internalDoc->dsc() && internalDoc->dsc()->isStructured())
  {
    kDebug() << "structured postscript file." << endl;
    internalDoc->setProlog(qMakePair (internalDoc->dsc()->beginprolog(), internalDoc->dsc()->endprolog()));
    internalDoc->setSetup(qMakePair (internalDoc->dsc()->beginsetup(), internalDoc->dsc()->endsetup()));

    CDSCPAGE* tmpPage;

    for (int i=0; i<numPages; i++)
    {
      tmpPage=(internalDoc->dsc()->page() + i);
      if (!tmpPage)
      {
        kDebug() << "no tmpPage for page number " << i << endl;
        continue;
      }
      QSize pageSize;
      pageSize = internalDoc->computePageSize(internalDoc->pageMedia(i));

      kDebug() << "size of Page " << i << ": (" << pageSize.width() << ", " << pageSize.height() << ")" << endl;
      Length w,h;
      w.setLength_in_bigPoints(pageSize.width());
      h.setLength_in_bigPoints(pageSize.height());
      pageSizes[i].setPageSize(w, h);

      internalDoc->insertPageData(i, qMakePair(tmpPage->begin, tmpPage->end));
    }
  }

  // Store path to current file
  QFileInfo fi(workingCopyFileName);
  path = fi.dirPath();

  return true;
}


bool PsRenderer::setFile(const QString& fname, const KUrl&)
{
#ifdef KF_DEBUG
  kDebug() << "PsRenderer::setFile(" << fname << ") called" << endl;
#endif

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // Check if we are reloading the file.
  bool reload = (fname == filename);

  // If fname is the empty string, then this means: "close".
  if (fname.isEmpty()) {
    kDebug() << "PdfRenderer::setFile( ... ) called with empty filename. Closing the file." << endl;

    return true;
  }

  // Paranoid saftey checks: make sure the file actually exists, and
  // that it is a file, not a directory. Otherwise, show an error
  // message and exit..
  QFileInfo fi(fname);
  workingCopyFileName = filename = fi.absFilePath();

  if (!fi.exists() || fi.isDir())
  {
    KMessageBox::error(parentWidget,
      i18n("<qt><strong>File error.</strong> The specified file '%1' does not exist.</qt>", filename),
      i18n("File Error!")
    );
    // the return value 'false' indicates that this operation was not successful.
    return false;
  }

  internalDoc = new GSInternalDocument(filename, GSInternalDocument::PS);

  // Set the number of pages page sizes
  kDebug() << "get number of pages." << endl;

  numPages = internalDoc->dsc()->page_count();

  // For some eps files dsc()->page_count() returns 0.
  // Work around this issue.
  if (numPages == 0)
    numPages = 1;

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);

  eps = false;
  KMimeType::Ptr mimetype = KMimeType::findByPath(fname);
  if (mimetype->name() == "image/x-eps")
  {
    kDebug() << "File is an eps" << endl;
    eps = true;

    Length w,h;
    QSize pageSize;
    pageSize = internalDoc->computePageSize(internalDoc->pageMedia(0));

    w.setLength_in_bigPoints(pageSize.width());
    h.setLength_in_bigPoints(pageSize.height());
    pageSizes[0].setPageSize(w, h);
    kDebug() << "eps file setup finished" << endl;
  }
  else if (internalDoc->dsc() && internalDoc->dsc()->isStructured())
  {
    kDebug() << "structured postscript file." << endl;
    internalDoc->setProlog(qMakePair (internalDoc->dsc()->beginprolog(), internalDoc->dsc()->endprolog()));
    internalDoc->setSetup(qMakePair (internalDoc->dsc()->beginsetup(), internalDoc->dsc()->endsetup()));

    CDSCPAGE* tmpPage;

    for (int i=0; i<numPages; i++)
    {
      tmpPage=(internalDoc->dsc()->page() + i);
      if (!tmpPage)
      {
        kDebug() << "no tmpPage for page number " << i << endl;
        continue;
      }
      QSize pageSize;
      pageSize = internalDoc->computePageSize(internalDoc->pageMedia(i));

      kDebug() << "size of Page " << i << ": (" << pageSize.width() << ", " << pageSize.height() << ")" << endl;

      Length w,h;
      w.setLength_in_bigPoints(pageSize.width());
      h.setLength_in_bigPoints(pageSize.height());
      pageSizes[i].setPageSize(w, h);

      internalDoc->insertPageData(i, qMakePair(tmpPage->begin, tmpPage->end));
    }
  }

  // Store path to current file
  path = fi.dirPath();

  // the return value 'true' indicates that this operation was not successful.
  return true;
}

bool PsRenderer::isValidFile(const QString fileName)
{
  return true;
  /*Poppler::Document* doc = Poppler::Document::load(fileName.ascii());
  if (doc)
  {
    delete doc;
    return true;
  }
  return false;
  */
}


void PsRenderer::clear(void)
{
  DocumentRenderer::clear();

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // Delete temporary files, if any
  QFileInfo wc(workingCopyFileName);
  QFileInfo oc(filename);
  if (wc.absFilePath() != oc.absFilePath())
    QFile::remove(workingCopyFileName);

  delete internalDoc;
  internalDoc = 0;
}


void PsRenderer::deletePages(quint16 from, quint16 to)
{
  if (internalDoc == 0)
    return;

  KTemporaryFile new_wcopy;
  new_wcopy.setSuffix(".ps");
  new_wcopy.open();

  QList<int> pages;
  for(int i=1; i<from; i++)
    pages << i;
  for(int i=to+1; i<numPages; i++)
    pages << i;

  if (generatePSFileForPage(new_wcopy.name(), pages) == false)
    return;

  _isModified = true;
#ifdef __GNUC__
#warning DUPLICATED CODE
#endif  
  clear();
  workingCopyFileName = new_wcopy.fileName();


  internalDoc = new GSInternalDocument(workingCopyFileName, GSInternalDocument::PS);

  // Set the number of pages page sizes
  kDebug() << "get number of pages." << endl;

  numPages = internalDoc->dsc()->page_count();

  // For some eps files dsc()->page_count() returns 0.
  // Work around this issue.
  if (numPages == 0)
    numPages = 1;

  // Set the page sizes in the pageSizes array
  pageSizes.resize(numPages);

  eps = false;
  KMimeType::Ptr mimetype = KMimeType::findByPath(workingCopyFileName);
  if (mimetype->name() == "image/x-eps")
  {
    kDebug() << "File is an eps" << endl;
    eps = true;

    Length w,h;
    QSize pageSize;
    pageSize = internalDoc->computePageSize(internalDoc->pageMedia(0));

    w.setLength_in_bigPoints(pageSize.width());
    h.setLength_in_bigPoints(pageSize.height());
    pageSizes[0].setPageSize(w, h);
    kDebug() << "eps file setup finished" << endl;
  }
  else if (internalDoc->dsc() && internalDoc->dsc()->isStructured())
  {
    kDebug() << "structured postscript file." << endl;
    internalDoc->setProlog(qMakePair (internalDoc->dsc()->beginprolog(), internalDoc->dsc()->endprolog()));
    internalDoc->setSetup(qMakePair (internalDoc->dsc()->beginsetup(), internalDoc->dsc()->endsetup()));

    CDSCPAGE* tmpPage;

    for (int i=0; i<numPages; i++)
    {
      tmpPage=(internalDoc->dsc()->page() + i);
      if (!tmpPage)
      {
        kDebug() << "no tmpPage for page number " << i << endl;
        continue;
      }
      QSize pageSize;
      pageSize = internalDoc->computePageSize(internalDoc->pageMedia(i));

      kDebug() << "size of Page " << i << ": (" << pageSize.width() << ", " << pageSize.height() << ")" << endl;
      Length w,h;
      w.setLength_in_bigPoints(pageSize.width());
      h.setLength_in_bigPoints(pageSize.height());
      pageSizes[i].setPageSize(w, h);

      internalDoc->insertPageData(i, qMakePair(tmpPage->begin, tmpPage->end));
    }
  }

  // Store path to current file
  QFileInfo fi(workingCopyFileName);
  path = fi.dirPath();

  // the return value 'true' indicates that this operation was not successful.
  return;
}


#include "psRenderer.moc"
