// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2006 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   Copyright (C) 2006 by Stefan Kebekus                                  *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <config.h>

#include "dataModel.h"
#include "dataView.h"
#include "kvsprefs.h"
#include "kvs_debug.h"

#include <klocale.h>

#include <QDomDocument>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>

//#define debug_DataModel


DataModel::DataModel(QObject* parent)
  : QObject(parent),
    _numberOfPages(0),
    resolutionInDPI(0.0)
{
  prefs = new KVSPrefs();
  prefs->readConfig();
}


DataModel::~DataModel()
{
  prefs->writeConfig();
  delete prefs;
}


PageNumber DataModel::currentPageNumber()
{
  return _currentPageNumber;
}

void DataModel::setCurrentPageNumber(const Anchor& anc, bool weak)
{
  if (anc.page > numberOfPages()) {
    kError(kvs::shell) << "DataModel::setCurrentPageNumber called with pageNumber higher than the number of pages in the document, anchor is " << anc << endl;
    return;
  }

  if (anc.page != _currentPageNumber) {
    _currentPageNumber = anc.page;
    emit currentPageNumberChanged();
  }
  if (!weak)
    emit GUIGotoAnchor(anc);
}


unsigned int DataModel::numberOfPages()
{
  return _numberOfPages;
}


void DataModel::setNumberOfPages(unsigned int pages)
{
  if (pages != _numberOfPages)
  {
    if (_currentPageNumber > _numberOfPages)
    {
      if (_numberOfPages == 0)
        setCurrentPageNumber(Anchor());
      else
        setCurrentPageNumber(Anchor(_numberOfPages));
    }
    _numberOfPages = pages;

    deselectAllPages();

    emit numberOfPagesChanged();
  }
}


void DataModel::setResolution(double resolution)
{
  if (resolution != resolutionInDPI)
  {
    resolutionInDPI = resolution;
    emit layoutChanged();
  }
}


void DataModel::setViewMode(int viewmode)
{
  if (viewmode != preferences()->viewMode())
  {
    preferences()->setViewMode(viewmode);
  }
  emit viewModeChanged();
}


void DataModel::deletePages(const PageNumber& from, const PageNumber& to)
{
  // Paranoid safety checks
  if ((from < 1) || (from > _numberOfPages) || (from > to) || (to > _numberOfPages)) {
    kError(kvs::shell) << "DataModel::deletePages(" << from << ", " << to << " ) called with numberOfPages=" << _numberOfPages << ". Aborting." << endl;
    return;
  }

  // Adjust number of pages
  setNumberOfPages(_numberOfPages + from - to -1);

  // Adjust bookmarks
  QList<PageNumber> keys = userBookmarks.keys();
  for(int i = 0; i < keys.count(); i++) {
    // Remove all bookmarks that point to pages that no longer exist
    if ((from <= keys[i]) && (keys[i] <= to))
      removeBookmark(keys[i]);
    
    // Correct all bookmarks that point after the pages that were
    // deleted
    if (keys[i] > to) {
      // Add new bookmark
      addBookmark(keys[i] + from - to -1, userBookmarks[keys[i]]);
      
      // Remove the old bookmark
      removeBookmark(keys[i]);
    }
  } // of for (all keys)
}


void DataModel::insertPages(const PageNumber& before, quint16 number_of_pages_inserted)
{
  // Paranoid safety checks
  if ((before == 0) || (before > _numberOfPages+1) || (number_of_pages_inserted == 0)) {
    kError(kvs::shell) << "DataModel::insertPages(" << before << ", " << number_of_pages_inserted << " ) called. Aborting." << endl;
    return;
  }

  // Adjust number of pages
  setNumberOfPages(_numberOfPages + number_of_pages_inserted);

  // Adjust bookmarks
  QList<PageNumber> keys = userBookmarks.keys();
  if (!keys.isEmpty())
  {
    for (int i = keys.count()-1; i >= 0; i--) {
      // Correct all bookmarks that point after the pages that were
      // inserted
      if (keys[i] >= before) {
        // Add new bookmark
        addBookmark(keys[i] + number_of_pages_inserted, userBookmarks[keys[i]]);
        
        // Remove the old bookmark
        removeBookmark(keys[i]);
      }
    } // of for (all keys)
  }
}


bool DataModel::isPageBookmarked(const PageNumber& page)
{
  return userBookmarks.contains(page);
}


QList<PageNumber> DataModel::bookmarks()
{
  return userBookmarks.keys();
}


QString DataModel::bookmarkLabel(const PageNumber& page)
{
  if (!isPageBookmarked(page))
    return QString::null;

  QString label = userBookmarks[page];

  if (label.isNull())
    label = i18n("Page %1", page);

  return label;
}


void DataModel::addBookmark(const PageNumber& page, const QString& text)
{
  if (!isPageBookmarked(page))
  {
    if (text != "")
    {
      userBookmarks.insert(page, text);
      kDebug() << "BookmarksDataModel::addBookmark(" << page << ", " << text << ")" << endl;

      emit bookmarkAdded(page, text);
    }
    else
    {
      userBookmarks.insert(page, QString::null);
      kDebug() << "BookmarksDataModel::addBookmark(" << page << ", )" << endl;

      emit bookmarkAdded(page, QString::null);
    }
  }
}


void DataModel::renameBookmark(const PageNumber& page, const QString& text)
{
  if (isPageBookmarked(page))
  {
    kDebug() << "BookmarksDataModel::renameBookmark(" << page << ", " << text << ")" << endl;
    userBookmarks.remove(page);
    userBookmarks.insert(page, text);

    emit bookmarkRenamed(page, text);
  }
}


void DataModel::removeBookmark(const PageNumber& page)
{
  userBookmarks.remove(page);

  emit bookmarkRemoved(page);;
}


void DataModel::removeAllBookmarks()
{
  userBookmarks.clear();

  emit allBookmarksRemoved();
}


void DataModel::loadDocumentInfo(const QString& filename)
{
  QFile infoFile(filename);
  if (!infoFile.exists() || !infoFile.open(QIODevice::ReadOnly))
    return;

  // Load DOM from XML file
  QDomDocument doc("documentInfo");
  if (!doc.setContent(&infoFile))
  {
    kDebug() << "Could not set content" << endl;
    infoFile.close();
    return;
  }
  infoFile.close();

  QDomElement root = doc.documentElement();
  if ( root.tagName() != "documentInfo" )
    return;

  // Parse the DOM tree
  QDomNode topLevelNode = root.firstChild();
  while (topLevelNode.isElement())
  {
    QString catName = topLevelNode.toElement().tagName();

    // Get bookmarks list from DOM
    if (catName == "bookmarkList")
    {
      QDomNode n = topLevelNode.firstChild();
      QDomElement e;
      PageNumber pageNumber;
      bool ok;
      while (n.isElement())
      {
        e = n.toElement();
        if (e.tagName() == "bookmark")
        {
          pageNumber = e.attribute("page", "-1").toInt(&ok);
          if (ok && pageNumber.isValid())// && pageNumber <= renderer->totalPages())
          {
            addBookmark(pageNumber, e.text());
          }
        }
        n = n.nextSibling();
      }
    } // </bookmarkList>
    topLevelNode = topLevelNode.nextSibling();
  } // </documentInfo>
}


void DataModel::saveDocumentInfo(const QString& filename) const
{
  kDebug() << "saveDocumentInfo(" << filename << ")" << endl;

  QFile infoFile(filename);

  if (userBookmarks.isEmpty())
  {
    // If there is no metadata remove the Document Info file.
    infoFile.remove();
    return;
  }

  if (infoFile.open(QIODevice::WriteOnly | QIODevice::Truncate))
  {
    // Create DOM
    QDomDocument doc("documentInfo");
    QDomElement root = doc.createElement("documentInfo");
    doc.appendChild(root);

    // Add bookmark list to DOM
    QDomElement bookmarkList = doc.createElement("bookmarkList");
    root.appendChild(bookmarkList);

    QList<PageNumber> keys = userBookmarks.keys();
    for (int i = 0; i < keys.count(); i++)
    {
      QDomElement bookmark = doc.createElement("bookmark");
      bookmark.setAttribute("page", keys[i]);

      QString label = userBookmarks[keys[i]];
      if (label != QString::null)
      {
        kDebug() << "bookmark label = " << label << endl;
        bookmark.appendChild(doc.createTextNode(label));
      }
      bookmarkList.appendChild(bookmark);
    }

    // Save DOM to XML file
    QString xml = doc.toString();
    QTextStream os(&infoFile);
    os << xml;
  }
  infoFile.close();
}


QList<PageNumber> DataModel::selectedPages()
{
  return _selectedPages;
}


bool DataModel::isSelected(const PageNumber& pageNumber) const
{
  return _selectedPages.contains(pageNumber);
}


QString DataModel::selectedPageRange()
{
  // We try to be smart and optimize the list by using ranges
  // ("5-11") wherever possible. The user will be thankful for
  // that. Complicated? Yeah, but that's life.
  if (_selectedPages.isEmpty() == true)
  {
    return "";
  }
  else
  {
    int commaflag = 0;
    QString range;
    QList<PageNumber>::ConstIterator it = _selectedPages.begin();
    do
    {
      int val = *it;
      if (commaflag == 1)
        range +=  QString(", ");
      else
        commaflag = 1;
      int endval = val;

      if (it != _selectedPages.end())
      {
        QList<PageNumber>::ConstIterator jt = it;
        jt++;
        do
        {
          int val2 = *jt;
          if (val2 == endval+1)
            endval++;
          else
            break;
          jt++;
        } while (jt != _selectedPages.end());
        it = jt;
      }
      else
      {
        it++;
      }
      if (endval == val)
      {
        range +=  QString("%1").arg(val);
      }
      else
      {
        range +=  QString("%1-%2").arg(val).arg(endval);
      }
    } while (it != _selectedPages.end());

    return range;
  }
}


void DataModel::selectPage(const PageNumber& page)
{
  if (page == PageNumber::invalidPage || page > numberOfPages())
    return;

  // Insertion sort
  if (_selectedPages.isEmpty())
  {
    _selectedPages.append(page);
  }
  else if (page < _selectedPages.first())
  {
    _selectedPages.prepend(page);
  }
  else if (page > _selectedPages.last())
  {
    _selectedPages.append(page);
  }
  else
  {
    QList<PageNumber>::Iterator i = _selectedPages.begin();
    while (i != _selectedPages.end())
    {
      if (page == *i)
      {
        return;
      }
      if (page > *i)
      {
        i++;
      }
      else
      {
        _selectedPages.insert(i, page);
        break;
      }
    }
  }

  emit pageSelected(page);
}


void DataModel::deselectPage(const PageNumber& page)
{
  _selectedPages.removeAll(page);
  emit pageDeselected(page);
}


void DataModel::deselectAllPages()
{
  for (int i = 0; i < _selectedPages.count(); i++)
  {
    emit pageDeselected(_selectedPages[i]);
  }
  _selectedPages.clear();
}


void DataModel::deselectText()
{
  selection.clear();
  emit selectionChanged(false);
}


void DataModel::selectText(const TextSelection& _selection)
{
  selection = _selection;
  emit selectionChanged(!selection.isEmpty());
}


void DataModel::setUserPreferredPageSize(SimplePageSize s)
{
#ifdef DataModel_DEBUG
  kdDebug(kvs::shell) << "DataModel::setUserPreferredPageSize(...)" << endl;
#endif

  bool iNE = _defaultPageSize.isNearlyEqual(s);
  _defaultPageSize = s;
  if (!iNE)
    emit numberOfPagesChanged();
}


QColor DataModel::paperColor() const
{
  QColor paperColor = Qt::white;
  if (prefs->changeColors())
  {
    if (prefs->renderMode() == KVSPrefs::EnumRenderMode::Inverted)
      paperColor = Qt::black;
    else if (prefs->renderMode() == KVSPrefs::EnumRenderMode::Paper)
      paperColor = prefs->paperColor();
    else if (prefs->renderMode() == KVSPrefs::EnumRenderMode::Recolor)
      paperColor = prefs->recolorBackground();
  }

  return paperColor;
}

#include "dataModel.moc"
