// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: documentPageCache
//
// Cache that holds a number of pages in a document.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004 Stefan Kebekus.
// (C) 2005-2006 Wilfried Huss.
//
// Distributed under the GPL.

#include <config.h>

#include "documentPageCache.h"
#include "documentRenderer.h"
#include "kvs_debug.h"
#include "kvsprefs.h"
#include "renderQueue.h"
#include "renderThread.h"

#include <QApplication>
#include <QPixmap>


//#define documentPageCache_DEBUG


DocumentPageCache::DocumentPageCache()
  : maxMemory(2*16777216), LRUCache(maxMemory),
    thumbnailCache(maxMemory/16), currentThumbnailWidth(0)
{
  renderQueue = new RenderQueue(this);
  renderThread = new RenderThread(renderQueue);
}


DocumentPageCache::~DocumentPageCache()
{
  renderQueue->clear();
  renderThread->stop();

  delete renderThread;
  renderThread = 0;
  delete renderQueue;
  renderQueue = 0;
}


void DocumentPageCache::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);
  connect(dataModel, SIGNAL(currentPageNumberChanged()), this, SLOT(currentPageChanged()));
}


void DocumentPageCache::setRenderer(DocumentRenderer* _renderer)
{
  // When we set a new renderer the old cache contents
  // will definitly not be needed anymore.
  clear();

  renderer = _renderer;
  renderThread->setRenderer(renderer);
}


SimplePageSize DocumentPageCache::sizeOfPage(const PageNumber& page) const
{
  // Paranoid safety checks
  if (!page.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::sizeOfPage( " <<  page << ") called with invalid page number." << endl;
    return SimplePageSize();
  }
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::sizeOfPage( " <<  page << ") called when no renderer was set." << endl;
    return SimplePageSize();
  }

  SimplePageSize s = renderer->sizeOfPage(page);
  if (!s.isValid())
  {
    // If the size is invalid use the size of the first Page in the document
    // as an estimate.
    s = renderer->sizeOfPage(1);
    if (!s.isValid())
      s = dataModel->defaultPageSize();
  }

  return s;
}


QSize DocumentPageCache::sizeOfPageInPixel(const PageNumber& pg) const
{
  // Paranoid safety checks
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::sizeOfPageInPixel( " << pg << " ) called but no renderer was set" << endl;
    return QSize();
  }
  if (!pg.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::sizeOfPageInPixel( " << pg << " ) called with invalid argument" << endl;
    return QSize();
  }

  SimplePageSize ps = sizeOfPage(pg);
  if (ps.isValid())
    return ps.sizeInPixel(dataModel->resolution());
  return dataModel->defaultPageSize().sizeInPixel(dataModel->resolution());
}


void DocumentPageCache::currentPageChanged()
{
  PageNumber currentPage = dataModel->currentPageNumber();

  // In continuous facing viewmode prerender the previous 3 and following 3 pages.
  // In the other viewmodes prerender only the previous page and the following page.
  int k = 1;
  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::ContinuousFacing)
  {
    k = 3;
  }

  for (int i = k; i >= 1; i--)
  {
    // Prerender previous page
    prerender(currentPage - i);
    // Prerender next page
    prerender(currentPage + i);
  }
  // We add the current page at last, so that it will be rendered first.
  prerender(currentPage);
}


void DocumentPageCache::prerenderThumbnail(const PageNumber& pageNumber, int width)
{
  // Paranoid checks
  if (renderer.isNull())
  {
    kError(kvs::shell) << "DocumentPageCache::isPageCached(..) called but no renderer was set" << endl;
    return;
  }
  if (!pageNumber.isValid())
  {
    return;
  }
  if (dataModel->numberOfPages() < pageNumber)
  {
    return;
  }

  JobId id(pageNumber, (double)(width)/sizeOfPage(pageNumber).width().getLength_in_inch(), dataModel->preferences()->rotation(), true);
  if (!thumbnailCache[id.key()])
  {
    // Add to render queue
    renderQueue->addRenderJob(id);
  }
}


void DocumentPageCache::prerender(const PageNumber& pageNumber)
{
  // Paranoid checks
  if (renderer.isNull())
  {
    kError(kvs::shell) << "DocumentPageCache::isPageCached(..) called but no renderer was set" << endl;
    return;
  }
  if (!pageNumber.isValid())
  {
    return;
  }
  if (dataModel->numberOfPages() < pageNumber)
  {
    return;
  }

  JobId id(pageNumber, dataModel->resolution(), dataModel->preferences()->rotation(), false);
  QString key = id.key();
  if (!LRUCache[key])
  {
    renderQueue->addRenderJob(id);
  }
}


bool DocumentPageCache::isThumbnailCached(const PageNumber& pageNumber, int width)
{
  // Paranoid checks
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::isThumbnailCached(..) called but no renderer was set" << endl;
    return false;
  }
  if (!pageNumber.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::isThumbnailCached( " << pageNumber << " ) called, with invalid argument." << endl;
    return false;
  }
  if (dataModel->numberOfPages() < pageNumber) {
    kError(kvs::shell) << "DocumentPageCache::isThumbnailCached( " << pageNumber
                  << " ) called but document contains only " << dataModel->numberOfPages() << " pages." << endl;
    return false;
  }

  currentThumbnailWidth = width;

  JobId id(pageNumber, (double)(width)/sizeOfPage(pageNumber).width().getLength_in_inch(), dataModel->preferences()->rotation(), true);
  // Check if the thumbnail that we are looking for is in the cache.
  ThumbnailPixmap* thumbnail = thumbnailCache[id.key()];

  if (thumbnail)
  {
    return true;
  }
  else
    return false;
}


bool DocumentPageCache::isPageCached(const PageNumber& pageNumber)
{
  // Paranoid checks
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::isPageCached(..) called but no renderer was set" << endl;
    return false;
  }
  if (!pageNumber.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::isPageCached( " << pageNumber << " ) called, with invalid argument." << endl;
    return false;
  }
  if (dataModel->numberOfPages() < pageNumber) {
    kError(kvs::shell) << "DocumentPageCache::isPageCached( " << pageNumber
                  << " ) called but document contains only " << dataModel->numberOfPages() << " pages." << endl;
    return false;
  }

  JobId id(pageNumber, dataModel->resolution(), dataModel->preferences()->rotation(), false);

  // Check if the page that we are looking for is in the cache.
  RenderedDocumentPagePixmap* page = LRUCache[id.key()];

  if (page)
  {
    return true;
  }
  else
    return false;
}

RenderedDocumentPagePixmap* DocumentPageCache::getPage(const PageNumber& pageNr, bool async)
{
#ifdef DocumentPageCache_DEBUG
  kDebug(kvs::shell) << "DocumentPageCache::getPage( pageNr=" << pageNr << " )" << endl;
#endif

  // Paranoid checks
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::getPage(..) called but no renderer was set" << endl;
    return 0;
  }
  if (!pageNr.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::getPage( " << pageNr << " ) called, with invalid argument." << endl;
    return 0;
  }
  if (dataModel->numberOfPages() < pageNr) {
    kError(kvs::shell) << "DocumentPageCache::getPage( " << pageNr << " ) called but document contains only " << dataModel->numberOfPages() << " pages." << endl;
    return 0;
  }

  JobId id(pageNr, dataModel->resolution(), dataModel->preferences()->rotation(), false);
  
  // First check if the page that we are looking for is in the cache
  RenderedDocumentPagePixmap* page = LRUCache[id.key()];

  if (page)
    return page;

  // The page was not found in the cache, so we have render the page
  if (dataModel->resolution() > 0.0)
  {
    if (async)
    {
      // Add to render queue
      renderQueue->addRenderJob(id);
      return 0;
    }
    else
    {
      QApplication::setOverrideCursor(Qt::waitCursor);
      page = renderer->drawPage(id);
      // Paint image to the page
      page->paintImage();
      QApplication::restoreOverrideCursor();
      addToCache(page, id);
    }
  }
  else
  {
    kError(kvs::shell) << "DocumentPageCache::getPage() called, but no resolution or negative resolution was set" << endl;
    return 0;
  }

  return page;
}


void DocumentPageCache::clear()
{
  renderQueue->clear();
  LRUCache.clear();
  thumbnailCache.clear();
}


QPixmap* DocumentPageCache::getThumbnail(const PageNumber& pageNr, int width)
{
  // Paranoid checks
  if (renderer.isNull()) {
    kError(kvs::shell) << "DocumentPageCache::createThumbnail(..) called but no renderer was set" << endl;
    return 0;
  }
  if (dataModel->numberOfPages() < pageNr) {
    kError(kvs::shell) << "DocumentPageCache::createThumbnail( " << pageNr << ", width ) called but document contains only " << dataModel->numberOfPages() << " pages." << endl;
    return 0;
  }
  if (!pageNr.isValid()) {
    kError(kvs::shell) << "DocumentPageCache::createThumbnail(..) called for page with invalid page specification" << endl;
    return 0;
  }
  if (!sizeOfPage().isValid()) {
    kError(kvs::shell) << "DocumentPageCache::createThumbnail(..) called for page with invalid size" << endl;
    return 0;
  }

  JobId id(pageNr, (double)(width)/sizeOfPage(pageNr).width().getLength_in_inch(), dataModel->preferences()->rotation(), true);
  ThumbnailPixmap* thumbnail = thumbnailCache[id.key()];
  if (thumbnail)
  {
    return thumbnail;
  }

  if (!renderer.isNull())
  {
    // Add to render queue
    renderQueue->addRenderJob(id);
  }
  return 0;
}


void DocumentPageCache::customEvent(QEvent* ce)
{
#ifdef DocumentPageCache_DEBUG
  kDebug(kvs::shell) << "custom event received" << endl;
#endif

  if (ce->type() == QEvent::User)
  {
    // It must be a RenderingFinishedEvent
    RenderingFinishedEvent* e = (RenderingFinishedEvent*)ce;

    //QueuedDocumentPage* renderedPage = renderQueue->takeFinishedPage();
    RenderedDocumentPagePixmap* page = e->getPage();
    // Safety check
    if (page)
    {
#ifdef DocumentPageCache_DEBUG
      kDebug(kvs::shell) << "add rendered page to cache." << endl;
#endif
    }
    else
    {
      kError(kvs::shell) << "finished queue was empty." << endl;
      return;
    }

    // Paint image to the page
    page->paintImage();

    if (page->getId().isThumbnail)
    {
      JobId id(page->getPageNumber(), (double)(page->width())/sizeOfPage(page->getPageNumber()).width().getLength_in_inch(), dataModel->preferences()->rotation(), true);
      ThumbnailPixmap* thumbnail = new ThumbnailPixmap(page->getPageNumber(), page->pixmap());
      thumbnailCache.insert(id.key(), thumbnail, thumbnail->width() * thumbnail->height());
      emit setThumbnail(page->getPageNumber());
      delete page;
    }
    else
    {
      addToCache(page, page->getId());
      emit updateWidget(page->getPageNumber());
    }
  }
}


void DocumentPageCache::addToCache(RenderedDocumentPagePixmap* page, JobId id)
{
  // We always set the cache capacity to be at least n times the cost of the page we want to insert.
  // Where n is the number of pages that can be visible at the same time at very high zoomlevels.
  // n depends on the layout mode.
  // If these pages are not all in the cache, scrolling the view becomes very slow, because for each
  // paint event the pages need to be rerendered.
  // We set n for each viewmode differently so that the user is able to reduce memory consuption by
  // switching to a simpler viewmode like Single Page.
  int n = 4;
  switch (dataModel->preferences()->viewMode())
  {
    case KVSPrefs::EnumViewMode::SinglePage:
      n = 1;
      break;
    case KVSPrefs::EnumViewMode::Continuous:
      n = 2;
      break;
    default:
      n = 4;
  }
  LRUCache.setMaxCost(qMax(page->memory() * n, maxMemory));

  if (!LRUCache.insert(id.key(), page, page->memory()))
  {
    kError(kvs::shell) << "DocumentPageCache::getPage(): inserting pagestructure into the cache failed.\n  This should never happen. If you see this message, something is very wrong." << endl;
  }
#ifdef documentPageCache_DEBUG
  kDebug(kvs::shell) << "current cache size = " << LRUCache.totalCost() << endl
                      << "maximal cache size = " << LRUCache.maxCost() << endl
                      << "items in the cache = " << LRUCache.count() << endl;
#endif
}


ThumbnailPixmap::ThumbnailPixmap(const PageNumber& _pageNumber, const QPixmap& thumbnail)
  : QPixmap(thumbnail), pageNumber(_pageNumber)
{}


#include "documentPageCache.moc"
