// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: documentPageCache
//
// Cache that holds a number of pages in a document.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004 Stefan Kebekus.
// (C) 2005-2006 Wilfried Huss.
//
// Distributed under the GPL.

#ifndef _documentpagecache_h_
#define _documentpagecache_h_

#include "ligature_export.h"

#include "dataView.h"
#include "pageNumber.h"
#include "pageSize.h"
#include "renderedDocumentPagePixmap.h"
#include "renderQueue.h"

#include <QCache>
#include <QEvent>
#include <QObject>
#include <QPointer>

class DocumentRenderer;
class ligaturePluginGUI;
class QPixmap;
class RenderedDocumentPage;
class RenderQueue;
class RenderThread;

class ThumbnailPixmap : public QPixmap
{
 public:
  ThumbnailPixmap(const PageNumber&, const QPixmap&);
 
  PageNumber pageNumber;
};


class LIGATURECORE_EXPORT DocumentPageCache: public QObject, public DataView
{
 Q_OBJECT

 public:
  DocumentPageCache();
  virtual ~DocumentPageCache();

  virtual void setupObservers(DataModel*);

  /** This method is used to name the DocumentRenderer, that the
      documentPageCache uses to render the page. The renderer can be
      used any time (e.g., triggered by an internal timer event), and
      must not be deleted before either the DocumentRenderer is
      deleted, or another renderer has been set. */
  void setRenderer(DocumentRenderer*);

  /** Returns the size of page 'page'. If the document does not
      specify a size (which happens, e.g., for some DVI-files), then
      the userPreferredSize is returned. */
  SimplePageSize sizeOfPage(const PageNumber& page = 1) const;

  /** Returns the size of page 'page', in pixels, using the resolution
      set with setResolution(). If the document does not specify a
      size (which happens, e.g., for some DVI-files), the
      userPreferredSize is used. */
  QSize sizeOfPageInPixel(const PageNumber& page) const;

  /** Returns a pointer to a documentPage structure, or 0, if the
      documentPage could not be generated for some reason (e.g.,
      because no renderer has been set). */
  RenderedDocumentPagePixmap* getPage(const PageNumber& pageNr, bool async = true);

  /** Checks if the given page is already in the cache. */
  bool isPageCached(const PageNumber& pageNumber);

  /** Checks if the requested thumbnail is already in the cache. */
  bool isThumbnailCached(const PageNumber& pageNumber, int width);

  /** Puts a request for a thumbnail into the rendering queue.

      When the renderer has finished processing the thumbnail the
      signal @ref setThumbnail() is emmited. */
  QPixmap* getThumbnail(const PageNumber& pageNr, int width);

 signals:
  void updateWidget(PageNumber);
  void setThumbnail(PageNumber);

 public slots:
  /** Clears the contents of the cache. */
  void clear();

 private:
  QPointer<DocumentRenderer> renderer;

  void customEvent(QEvent*);

  /** Put a request for the given page on the renderQueue.
      It is safe to call this function with page numbers that are out of range. */
  void prerender(const PageNumber& pageNumber);
  void prerenderThumbnail(const PageNumber& pageNumber, int width);

  void addToCache(RenderedDocumentPagePixmap*, JobId);

 private slots:
  void currentPageChanged();

 private:
  /** The maximum of memory used by the cache. (32MB)
      TODO: make this configurable, or detact an appropriate value at startup. */
  quint32 maxMemory;

  /** This list holds the cache. */
  QCache<QString, RenderedDocumentPagePixmap> LRUCache;

  QCache<QString, ThumbnailPixmap> thumbnailCache;

  RenderQueue* renderQueue;
  RenderThread* renderThread;

  //TODO: find a better way
  int currentThumbnailWidth;
};


#endif
