// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
#ifndef KMULTIPAGE_H
#define KMULTIPAGE_H

#include "ligature_export.h"

#include "dataView.h"
#include "documentRenderer.h"

#include <kparts/plugin.h>
#include <kurl.h>

class DocumentPageCache;
class DocumentWidget;
class KConfigDialog;
class KPrinter;
class PageView;


/** \brief This class provides plugin-specific GUI elements for ligature plugins

@author Wilfried Huss, Stefan Kebekus

 */

// TODO remove virtual inheritance for KDE 4. It's the reason for the strange DCOPObject construction
class LIGATUREPLUGINGUI_EXPORT ligaturePluginGUI : public KParts::Plugin, public DataView
{
  Q_OBJECT

public:
  ligaturePluginGUI(QObject *parent);
  virtual ~ligaturePluginGUI();

  /** @brief set parent widget

      Because a KParts::Plugin and therefore the ligaturePluginGUI is not a QWidget we need to
      set a widget to be used as the parent for dialogs that are opened by the ligaturePluginGUI
      or DocumentRenderer, i.e. to show error messages or progress information.

      @note This function needs to be called after the creation of a ligaturePluginGUI object.
  */
  void setParentWidget(QWidget *wdg) {parentWdg = wdg; if (!renderer.isNull()) renderer->setParentWidget(wdg);}

  /** Methods which are associated with the DCOP functionality of the
      ligaturePluginGUI.

      @returns the file name (not the URL) of the currently loaded file.
  */
  QString name_of_current_file();

  /** Methods which are associated with the DCOP functionality of the
      ligaturePluginGUI. This method checks if a given file is loaded.

      @returns true if the given filename is currently loaded.
  */
  bool is_file_loaded(const QString& filename);

  /** @brief Opens file and sets URL

      This method does the same as openFile, but sets the m_url of the
      ligaturePluginGUI. This can be important, for the following reason: assume
      that a DVI is or DJVU-file is located on a web server at
      baseURL=http://www.x.x/x.dvi The file may refer to external graphic
      files using relative links.

      The file is downloaded by the ligaturepart to a temporary file on the
      hard disk, say /tmp/t.dvi. The ligaturepart would then call this method
      with filename=/tmp/t.dvi and baseURL=http://www.x.x/x.dvi, so the
      DVI-renderer knows to interpret the link to t.jpg as
      http://www.x.x/t.jpg, and will download the file from there.

      @warning This method is virtual only for technical reasons. Do not
      re-implement this method.

      @returns true on success, false on failure.
  */
  virtual bool openUrl(const QString& filename, const KUrl& base_url);

  /** Returns a pointer to the renderer. */
  virtual QPointer<DocumentRenderer> getRenderer() const { return renderer; }

  /** @brief Prints a document

      This method prints a document. The default implementation offers
      fairly good printer support, but printing with the default
      implementation is usually quite slow as tremendous amounts of data
      needs to be transferred to the printer. To limit the data sent to
      the printer, this default implementation prints only at low
      resolution and produces mediocre quality. This method can (and
      should) therefore be re-implemented if you have good code to convert
      your document to PostScript.

      Example: If your document consists of a single A4 page that contains
      a DJVU image of 30KB, then the default implementation would render
      the image in 600dpi, i.e. in about 7000x5000 pixel, and then send
      this huge graphics uncompressed to the printer. A smart
      implementation, on the other hand, would send the DJVU-file directly
      to the printer, together with a DJVU decoder, which is implemented
      in PostScript and uses only a few KB of memory, making for less than
      40KB of data sent to the printer.

      If you decide to re-implement this method, you need to decide if
      your implementation will support the options offered by the "page
      size & placement" tab of the print dialog. If these options are set,
      pages that are smaller/larger than the printer's paper size will be
      shrunk/enlarged and optionally centered on the paper.

      If your implementation does not support the options, the following
      code should be used:
      @code
      // Obtain a fully initialized KPrinter structure, and disable all
      // entries in the "Page Size & Placement" tab of the printer dialog.
      KPrinter *printer = getPrinter(false);
      // Abort with an error message if no KPrinter could be initialized
      if (printer == 0) {
        kdError(kvs::shell) << "KPrinter not available" << endl;
        return;
      }

      // Show the printer options dialog. Return immediately if the user
      // aborts.
      if (!printer->setup(parentWdg, i18n("Print %1").arg(m_file.section('/', -1)) )) {
        delete printer;
        return;
      }

      ...  <Produce a PostScript file, with filename, say, tmpPSFile. You
      can use all the features that KPrinter has to offer. Note that
      printer->pageList() gives a list of pages to be printed, where "1"
      denotes the first page, "2" the second, etc.> ...

      printer->printFiles( QStringList(tmpPSFile), true );
      delete printer;
      @endcode

      If your implementation does support the options, code must be
      provided to support the KPrinter options
      "kde-ligature-shrinkpage", "kde-ligature-expandpage",
      "kde-ligature-centerpage" and "kde-ligature-rotatepage". It is
      important to note that "kde-ligature-rotatepage" and
      "kde-ligature-centerpage" should by default treated as "true",
      while the other options should be "false" by default. The following
      code sees to that:
      @code
      // Obtain a fully initialized KPrinter structure, and enable all
      // entries in the "Page Size & Placement" tab of the printer dialog.
      KPrinter *printer = getPrinter(true);
      // Abort with an error message if no KPrinter could be initialized
      if (printer == 0) {
        kdError(kvs::shell) << "KPrinter not available" << endl;
        return;
      }

      // Show the printer options dialog. Return immediately if the user
      // aborts.
      if (!printer->setup(parentWdg, i18n("Print %1").arg(m_file.section('/', -1)) )) {
        delete printer;
        return;
      }

      if (printer->option( "kde-ligature-shrinkpage" ) == "true")
        <Shrink some pages to paper size, where necessary>
      if (printer->option( "kde-ligature-expandpage" ) == "true")
        <Expand some pages to paper size, where necessary>
      if (printer->option( "kde-ligature-centerpage" ) != "false")
        <Center pages on paper>
      if (printer->option( "kde-ligature-rotatepage" ) != "false")
        <rotate pages by 90 deg. counterclockwise, if paper orientation is different from pages orientation>

      ...  <Produce a PostScript file, with filename, say, tmpPSFile. You
      can use all the features that KPrinter has to offer. Note that
      printer->pageList() gives a list of pages to be printed, where "1"
      denotes the first page, "2" the second, etc.> ...

      printer->printFiles( QStringList(tmpPSFile), true );
      delete printer;
      @endcode

      For more information, see the default implementation in the source
      file ligaturePluginGUI.cpp. You might also look at the documentation to
      getPrinter().
  */
  virtual void print();

  /** @brief Tests if the document specifies a paper size

      @returns true if the document specifies page sizes, and false
      otherwise.

      @warning The information returned by this method is not
      always 100% reliable. Although unlikely, it is theoretically
      possible that this method returns 'true', but still some of the
      sizes returned by sizeOfPage() are invalid.
  */
  virtual bool hasSpecifiedPageSizes() const {return renderer && renderer->hasSpecifiedPageSizes();}

  /** @brief closes a file

      @returns true on success.
  */
  virtual bool closeUrl();

  /** @brief reads settings

      Reimplement this function if a ligaturePluginGUI implementation has
      additional configuration settings.

      @note Reimplementations must call this.
  */
  virtual void readSettings();

  /** @brief writes settings

      Reimplement this function if a ligaturePluginGUI implementation has
      additional configuration settings.

      @note Reimplementations must call this.
  */
  virtual void writeSettings();

  /** @brief test if this implementation supports text operations

      @returns true if the fileformat supported by this implementation
      has support for text.
  */
  virtual bool supportsTextSearch() const { return getRenderer() && getRenderer()->supportsTextSearch(); }

  /** @brief Flag to indicate the document was modified since last saved

      ligaturePluginGUI implementations that offer functionality that
      modifies the document (e.g. remove or insert pages) must
      re-implement this method to return

      @returns 'true' if the document was modified since it was last
      saved
  */
  virtual bool isModified() const {return false;}

  /** @brief Adds pages to the Ligature's central preferences dialog

      This method can be re-implemented to add documenttype specific
      configuration pages to the central preferences dialog. The
      documentation to KConfigDialog explains how to do that.

      The default implementation does nothing.

      @param configDialog a pointer to the KConfigDialog the dialog to
      add pages to
  */
  virtual void addConfigDialogs(KConfigDialog* configDialog) { Q_UNUSED(configDialog); }

  /** @brief Creates new instances of DocumentWidget

      If you need special functionality and subclass the
      @ref DocumentWidget class, then you need to also reimplement
      this method to ensure that your new subclass of DocumentWidgets
      will be used.

      This function is also the right place to connect to signals emitted
      by @ref DocumentWidget.

      @note The caller has the responsibility to delete the created
      objects, when they are not needed anymore.

      @param parent the PageView that will show the widget.

      @param cache the DocumentPageCache that will be used by the widget
      to render the page.

      @returns a new object that is of type DocumentWidget or a subclass
      of DocumentWidget.
  */
  virtual DocumentWidget* createDocumentWidget(PageView *parent, DocumentPageCache *cache);

  /** @brief Creates new RenderedDocumentPagePixmap objects

      If a multipage implementation needs additional functionality overwrite
      this function to create objects of a suitable subclass of
      @ref RenderedDocumentPagePixmap.

      @note The caller has the responsibility to delete the created
      objects, when they are not needed anymore.

      @returns a new object of type RenderedDocumentPagePixmap or a subclass
      of it.
  */
  virtual RenderedDocumentPagePixmap* createDocumentPagePixmap(JobId id) const;

  /** @brief Updates the GUI after loading or closing of a file

      This method is called by openFile() when a new file was loaded,
      and by closeUrl() when a file is closed so that the ligaturePluginGUI
      implementation can update its own GUI, enable/disable actions,
      prepare info texts, etc. At the time the method is executed, the
      file has already been loaded into the renderer using
      renderer.setFile(), or closed using renderer.clear() and the
      standard ligaturePluginGUI GUI is set up. The filename can be accessed
      via m_file.

      The default implementation does nothing.

      @param success the return value of renderer.setFile() which
      indicates if the file could be successfully loaded by the
      renderer, or not. Note that setFile() is called even if the
      renderer returned 'false', so that the implemtation can then
      disable menu items, etc. */
  virtual void setFile(bool success);

protected:
  /** Used to enable/disable Actions of multiPage implementations.
      enableActions(true) should be called whenever a file is
      successfully loaded.  enableActions(false) is called when the
      file is closed.
  */
  virtual void enableActions(bool);

  /** Initializes all data structures that need to know the renderer.
      This function must be called in the constructor of multipage
      implementations.
  */
  void setRenderer(DocumentRenderer*);

public slots:
  /** @brief saves the opened document

      @param filename the filename of the destination

      @returns true on success.

      @note The default implementation starts a basic copy KIO-Job. A
      multipage implementation that wishes to offer saving in various
      formats must re-implement this slot. If saving to the file
      fileName fails, the implementation should present an error
      dialog to the user.
  */
  virtual bool slotSave(const QString &fileName);

  /** Is called if settings are changed in the configuration dialog.
      If this method is reimplemented in a child class, it needs to be
      called from there.
  */
  virtual void preferencesChanged();

signals:
  /** @brief emitted when the document has changed

      This signal can be emitted if the document or status of this class
      changed internally so that the number, size and position of widgets
      must be checked and all associated widgets should be repainted. This
      could be emitted, e.g., if pages are removed from a document.

      @warning When this signal is emitted, the whole GUI setup is
      re-computed, and all widgets are re-drawn. This can take
      considerable time. If the number of pages and the page sizes didn't
      change, the signal @ref renderModeChanged() is a much faster alternative.
  */
  void documentHasBeenModified();

  /**  @brief signal emitted to indicate that all widgets need to be repainted

      This signal can be emitted if the document or status of this class
      changed internally or of the renderer changed internally so that all
      associated widgets should be repainted. This signal is emitted,
      e.g., by the DJVU-plugin where if the user changes from "color"
      rendering mode to the faster "forground only".

      This signal is similar to @ref documentHasBeenModified(), but here
      the receiver assumes that that the number of pages and the pages sizes
      remain unchanged. This makes the repainting faster.
  */
  void renderModeChanged();

  /** @brief emit this signal to show a message in the status bar */
  void setStatusBarText(const QString&);

protected:
  /** @brief Allocates and initializes a KPrinter structure

      This method is used in implementations of the print() method. See
      the documentation of print() for more information and for example
      code.  This method allocates and initializes a KPrinter
      structure. The list of pages marked in the sidebar is already set in
      the "kde-range" option in the KPrinter structure. In this option "1"
      means: first page of the document, "2" the second, etc.

      @param enablePageSizeFeatures Enables or diables the entries in the
      "Page Size & Placement" tab of the print dialog.

      @returns a pointer to a KPrinter, or 0 if no KPrinter could be
      allocated.

      @warning The KPrinter allocated is owned by the caller must be
      deleted before the ligaturePluginGUI is deleted. Otherwise, a segfault will
      occur.
  */
  KPrinter *getPrinter(bool enablePageSizeFeatures=true);

  /** @brief Pointer to the parent widget

      This pointer is automatically set by the constructor.
  */
  QPointer<QWidget> parentWdg;


private:
  /* This method opens a file and sets up the GUI when the file is
     loaded. It calls setFile() so that implementations of ligaturePluginGUI
     can  update their own GUI. DO NOT REIMPLEMENT THIS METHOD. */
  bool openFile();

  /* Is set by setRenderer. */
  QPointer<DocumentRenderer> renderer;

protected:
  QString m_file;
  KUrl m_url;
};


#endif
