// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2004-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   Copyright (C) 2006 by Stefan Kebekus                                  *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <config.h>

#include "marklist.h"
#include "documentPageCache.h"
#include "kvs_debug.h"
#include "kvsprefs.h"

#include <kiconloader.h>
#include <klocale.h>
#include <kglobalsettings.h>

#include <QApplication>
#include <QCheckBox>
#include <QHBoxLayout>
#include <QLabel>
#include <QMenu>
#include <QMouseEvent>
#include <QTimer>
#include <QToolTip>
#include <QVBoxLayout>

#include "marklist.moc"

// Maximal number of MarkListWidgets we keep in the cache,
// to reuse for future use.
#define WIDGET_CACHE_SIZE 100

namespace {

/** Holds the icon used as a overlay on pages which are not drawn yet. */
QPixmap* waitIcon = 0;

QPixmap* bookmarkIcon = 0;

} // namespace anon


/****** ThumbnailWidget ******/

ThumbnailWidget::ThumbnailWidget(MarkListWidget* _parent, const PageNumber& _pageNumber, DocumentPageCache* _pageCache)
  : QWidget(_parent), pageNumber(_pageNumber), pageCache(_pageCache), parent(_parent)
{
  setAttribute(Qt::WA_NoSystemBackground, true);

  if (!waitIcon)
  {
    waitIcon = new QPixmap(KIconLoader::global()->loadIcon("gear", K3Icon::NoGroup, K3Icon::SizeSmall));
  }

  if (!bookmarkIcon)
  {
    bookmarkIcon = new QPixmap(KIconLoader::global()->loadIcon("attach", K3Icon::NoGroup, K3Icon::SizeSmall));
  }
}


void ThumbnailWidget::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);
}


void ThumbnailWidget::setPageNumber(const PageNumber& _pageNumber)
{
  pageNumber = _pageNumber;
  update();
}

void ThumbnailWidget::paintEvent(QPaintEvent* e)
{
  // Only repaint if the widget is really visible. We need to check this because Qt
  // sends paintEvents to all widgets that have ever been visible in the Scrollview
  // whenever the ScrollView is resized. This also increases the percieved performance
  // only thumbnails that are really needed are rendered.
  if (!parent->isVisible())
  {
    //kDebug(kvs::shell) << "Abort Thumbnail drawing for page " << pageNumber << endl;
    return;
  }

  QPainter p(this);
  p.setClipRect(e->rect());

  // Paint a black border around the widget
  //p.setRasterOp(Qt::CopyROP);
  p.setBrush(Qt::NoBrush);
  p.setPen(Qt::black);
  QRect outlineRect = rect();
  outlineRect.adjust(0, 0, -1, -1);
  p.drawRect(outlineRect);

  // Remove 1 pixel from all sides of the rectangle, to eliminate overdraw with
  // the black border.
  QRect thumbRect = rect();
  thumbRect.adjust(1, 1, -1, -1);

  // If the thumbnail is empty or has been marked for updating generate a new thumbnail.
  if (!pageCache->isThumbnailCached(pageNumber, thumbnailWidth()))
  {
    p.fillRect(thumbRect, dataModel->paperColor());

    // Draw busy indicator.
    // Im not really sure if this is a good idea.
    // While it is nice to see an indication that something is happening for pages which
    // take long to redraw, it gets quite annoing for fast redraws.
    // TODO: Disable or find something less distractiong.
    p.drawPixmap(10, 10, *waitIcon);

    if (parent->isPageBookmarked())
      p.drawPixmap(thumbRect.right()-26, 10, *bookmarkIcon);

      // Request thumbnail pixmap.
    pageCache->getThumbnail(pageNumber, thumbnailWidth());
    return;
  }

  QPixmap* thumbnail = pageCache->getThumbnail(pageNumber, thumbnailWidth());

  if (!thumbnail)
    return;

  QVector<QRect> damagedRects = e->region().rects();
  for (int i = 0; i < damagedRects.size(); i++)
  {
    // Paint the thumbnailpage where it intersects with the damaged area.
    QRect destRect = damagedRects[i].intersect(thumbRect);

    // The actual thumbnail starts at point (1,1) because of the outline
    // Therefore we need to shift the destination rectangle.
    QRect pixmapRect = destRect;
    pixmapRect.translate(-1, -1);

    // Paint the thumbnail
    if (!thumbnail->isNull())
    {
      p.drawPixmap(destRect.topLeft(), *thumbnail, pixmapRect);
    }
    else
    {
      // If the renderer has returned a null image for some reason. Fill
      // the widget with the background color.
      p.fillRect(destRect, dataModel->paperColor());
    }
  }

  if (parent->isPageBookmarked())
    p.drawPixmap(thumbRect.right()-26, 10, *bookmarkIcon);
}


int ThumbnailWidget::thumbnailWidth()
{
  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    return height() - 2;
  }
  else
  {
    return width() - 2;
  }
}

/****** MarkListWidget ******/


MarkListWidget::MarkListWidget(QWidget* _parent, MarkList* _markList, const PageNumber& _pageNumber,
                               DocumentPageCache* _pageCache, bool _showThumbnail)
  : QWidget(_parent), showThumbnail(_showThumbnail), pageNumber(_pageNumber),
    pageCache(_pageCache), markList(_markList)
{
  setAttribute(Qt::WA_NoSystemBackground, true);

  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setMargin(margin);

  thumbnailWidget = 0;
  _selected = false;
  if (showThumbnail)
  {
    thumbnailWidget = new ThumbnailWidget(this, pageNumber, pageCache);
    layout->addWidget(thumbnailWidget, 1, Qt::AlignTop);
  }

  QBoxLayout* bottomLayout = new QHBoxLayout();
  layout->addLayout(bottomLayout);

  checkBox = new QCheckBox(QString::null, this );
  checkBox->setFocusPolicy(Qt::NoFocus);
  checkBox->setToolTip(i18n("Select for printing"));
  bottomLayout->addWidget(checkBox, 0, Qt::AlignLeft);
  connect(checkBox, SIGNAL(toggled(bool)), this, SLOT(toggleSelection(bool)));

  pageLabel = new QLabel(QString("%1").arg(pageNumber), this);
  pageLabel->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
  bottomLayout->addWidget(pageLabel, 1);

  bookmarkLabel = new QLabel(this);
  bookmarkLabel->setPixmap(KIconLoader::global()->loadIcon("attach", K3Icon::NoGroup, K3Icon::SizeSmall));
  bookmarkLabel->setFixedSize(K3Icon::SizeSmall, K3Icon::SizeSmall);
  bottomLayout->addWidget(bookmarkLabel, 1);

  if (markList->isPageBookmarked(pageNumber) && !showThumbnail)
    bookmarkLabel->show();
  else
    bookmarkLabel->hide();

  _backgroundColor = KGlobalSettings::baseColor();

  // Alternate between colors.
  if ((pageNumber % 2 == 0) && KGlobalSettings::alternateBackgroundColor().isValid())
    _backgroundColor = KGlobalSettings::alternateBackgroundColor();

  show();
}


void MarkListWidget::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);

  if (showThumbnail)
    thumbnailWidget->setupObservers(dataModel);
}


void MarkListWidget::setPageNumber(const PageNumber& _pageNumber)
{
  pageNumber = _pageNumber;

  if (markList->isPageBookmarked(pageNumber) && !showThumbnail)
    bookmarkLabel->show();
  else
    bookmarkLabel->hide();

  _backgroundColor = KGlobalSettings::baseColor();

  // Alternate between colors.
  if ((pageNumber % 2 == 0) && KGlobalSettings::alternateBackgroundColor().isValid())
    _backgroundColor = KGlobalSettings::alternateBackgroundColor();

  setPaletteBackgroundColor( _backgroundColor );

  pageLabel->setText(QString("%1").arg(pageNumber));

  if (showThumbnail && thumbnailWidget)
  {
    thumbnailWidget->setPageNumber(pageNumber);
  }

  update();
}


bool MarkListWidget::isPageBookmarked() const
{
  return markList->isPageBookmarked(pageNumber);
}


bool MarkListWidget::isChecked() const
{
    return checkBox->isChecked();
}


void MarkListWidget::toggle()
{
    checkBox->toggle();
}


void MarkListWidget::toggleSelection(bool on)
{
  emit selectionToggled(pageNumber, on);
}


void MarkListWidget::addBookmark()
{
  if (showThumbnail)
  {
    thumbnailWidget->update();
    bookmarkLabel->hide();
  }
  else
  {
    bookmarkLabel->show();
  }
}


void MarkListWidget::setBookmarkLabel(const QString& label)
{
  QString temp = label;
  if (label.isNull())
    temp = i18n("Page %1", pageNumber);

  if (showThumbnail)
    thumbnailWidget->setToolTip(temp);
  bookmarkLabel->setToolTip(temp);
}


void MarkListWidget::removeBookmark()
{
  thumbnailWidget->setToolTip("");
  bookmarkLabel->setToolTip("");

  bookmarkLabel->hide();

  if (showThumbnail)
  {
    thumbnailWidget->update();
  }
}


void MarkListWidget::setChecked( bool checked )
{
    checkBox->setChecked(checked);
}


void MarkListWidget::setSelected( bool selected )
{
  if (_selected != selected)
  {
    _selected = selected;
    update();
  }
}


int MarkListWidget::setNewWidth(int width)
{
  int height = qMax(checkBox->height(), pageLabel->height()) + 2*margin;
  if (showThumbnail)
  {
    // Calculate size of Thumbnail
    int thumbnailWidth = qMin(width, dataModel->preferences()->maxThumbnailWidth());
    double aspectRatio = pageCache->sizeOfPage(pageNumber).aspectRatio();

    if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
    {
      aspectRatio = 1 / aspectRatio;
    }

    int thumbnailHeight = (int)((thumbnailWidth - 2*margin - 2) / aspectRatio + 0.5) + 2;

    // Resize Thumbnail if necessary
    if (thumbnailWidget->size() != QSize(thumbnailWidth, thumbnailHeight))
      thumbnailWidget->setFixedSize(thumbnailWidth - 2*margin, thumbnailHeight);

    height += thumbnailHeight + 2*margin;
  }

  setFixedSize(width, height);
  return height;
}


bool MarkListWidget::isVisible()
{
  QRect visibleRect(markList->contentsX(), markList->contentsY(),
                   markList->visibleWidth(), markList->visibleHeight());
  QRect widgetRect(markList->childX(this), markList->childY(this), width(), height());

  if (widgetRect.intersects(visibleRect))
    return true;

  return false;
}


bool MarkListWidget::isCompletelyVisible()
{
  QRect visibleRect(markList->contentsX(), markList->contentsY(),
                   markList->visibleWidth(), markList->visibleHeight());
  QRect widgetRect(markList->childX(this), markList->childY(this), width(), height());

  if (visibleRect.contains(widgetRect))
    return true;

  return false;
}

void MarkListWidget::mousePressEvent(QMouseEvent* e)
{
  // Select Page
  if (e->button() == Qt::LeftButton)
  {
    emit selected(pageNumber);
  }
  else if (e->button() == Qt::RightButton)
  {
    emit showPopupMenu(pageNumber, e->globalPos());
  }
}


void MarkListWidget::paintEvent(QPaintEvent* e)
{
  // Region from which rectangles occupied by child widgets will by substracted.
  QPainter p(this);
  if (_selected)
    p.fillRect(e->rect(), QApplication::palette().color( QPalette::Highlight ));
  else
    p.fillRect(e->rect(), _backgroundColor);
}


void MarkListWidget::setThumbnail()
{
  thumbnailWidget->update();
}


/****** MarkList ******/


MarkList::MarkList(QWidget* parent, const char* name)
  : SmoothScrollView(parent, name, Qt::WStaticContents | Qt::WNoAutoErase),
    testWidget(0), clickedThumbnail(0), contextMenu(0)
{
  currentPage = PageNumber::invalidPage;
  setFocusPolicy( Qt::StrongFocus );
  //viewport()->setFocusPolicy( QWidget::WheelFocus );
  setResizePolicy(Q3ScrollView::Manual);

  setVScrollBarMode(Q3ScrollView::AlwaysOn);
  setHScrollBarMode(Q3ScrollView::AlwaysOff);

  viewport()->setAttribute(Qt::WA_NoSystemBackground, true);

  setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);

  enableClipper(true);

  connect(this, SIGNAL(contentsMoving(int, int)), this, SLOT(slotCreateWidgets(int, int)));
  connect(this, SIGNAL(viewSizeChanged(const QSize&)), this, SLOT(slotStartFitTimer()));
  connect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToWidth()));
}


MarkList::~MarkList()
{
  clear();

  delete testWidget;
  delete contextMenu;
}


void MarkList::slotStartFitTimer()
{
  fitTimer.setSingleShot(true);
  fitTimer.start(100);
}


bool MarkList::isSmoothScrollDistance(double scrollDistance)
{
  double viewportDiagonal = sqrt((double)(visibleHeight()) * (double)(visibleHeight()) +
                                 (double)(visibleWidth()) * (double)(visibleWidth()));

  return scrollDistance < 1.5 * viewportDiagonal;
}


void MarkList::slotCreateWidgets()
{
  slotCreateWidgets(contentsX(), contentsY());
}

void MarkList::slotCreateWidgets(int x, int y)
{
  if (!isVisible())
    return;

  Q_UNUSED(x);

  unsigned int ytop = y;
  unsigned int ybottom = y + visibleHeight();

  int startIndex = 1;
  int stopIndex = dataModel->numberOfPages();

  //TODO: use binary search if the linear search turnes out to be a bottleneck.
  for (unsigned int i = 0; i < widgetPositionList.count(); i++)
  {
    if (widgetPositionList[i] >= ytop)
    {
      startIndex = i;
      break;
    }
  }
  if (startIndex == 0)
    startIndex = 1;

  //TODO: use binary search if the linear search turnes out to be a bottleneck.
  for (unsigned int i = startIndex; i < widgetPositionList.count(); i++)
  {
    if (widgetPositionList[i] >= ybottom)
    {
      stopIndex = i;
      break;
    }
  }

  QList<PageNumber> pages = widgetMap.keys();

  for (unsigned int i = 0; i < pages.count(); i++)
  {
    if (pages[i] < startIndex - 5 || pages[i] > startIndex + 5)
    {
      MarkListWidget* item = widgetMap[pages[i]];
      widgetMap.remove(pages[i]);

      // Since deleteing QWidgets is an expensive operation, we only do
      // it if we already have enough in the widget cache.
      if (widgetCache.size() <= WIDGET_CACHE_SIZE)
      {
        item->setSelected(false);
	item->setChecked(false);
        widgetCache.push_back(item);
        item->disconnect();
      }
      else
      {
        delete item;
      }
    }
  }

  // Prerender the pages right before and after the visible ones.
  if (dataModel->preferences()->showThumbnails())
  {
    //TODO: find a less ugly way to do this.
    if (startIndex - 1 > 0)
    {
      testWidget->setPageNumber(startIndex - 1);
      pageCache->getThumbnail(startIndex - 1, testWidget->thumbnailWidth());
    }
    if (stopIndex + 1 <= dataModel->numberOfPages())
    {
      testWidget->setPageNumber(stopIndex + 1);
      pageCache->getThumbnail(stopIndex + 1, testWidget->thumbnailWidth());
    }
  }

  for (int i = startIndex; i <= stopIndex; i++)
  {
    QWidget* widget = createWidget(i);
    widget->show();
  }

  viewport()->update();
}

void MarkList::viewportPaintEvent(QPaintEvent* e)
{
  // Region from which rectangles occupied by child widgets will be
  // subtracted.
  QRegion backgroundArea(e->rect());

  QMap<PageNumber, MarkListWidget*>::Iterator it;
  for (it = widgetMap.begin(); it != widgetMap.end(); ++it )
  {
    QWidget* item = *it;;

    QRect widgetGeometry = item->geometry();

    // Draw the widget.
    if (e->rect().intersects(widgetGeometry)) {
      QRect widgetRect = e->rect().intersect(widgetGeometry);
      widgetRect.moveBy(-widgetGeometry.left(), -widgetGeometry.top());

      item->update(widgetRect);
    }

    // Substract the painted area.
    backgroundArea -= widgetGeometry.intersect(e->rect());
  }

  // Paint the background.
  QPainter p(viewport());

  QVector<QRect> backgroundRects = backgroundArea.rects();

  for (int i = 0; i < backgroundRects.count(); i++)
    p.fillRect(backgroundRects[i], KGlobalSettings::baseColor());
}


void MarkList::setPageCache(DocumentPageCache* _pageCache)
{
  pageCache = _pageCache;
  connect(pageCache, SIGNAL(setThumbnail(PageNumber)), this, SLOT(slotSetThumbnail(PageNumber)));
}


MarkListWidget* MarkList::createWidget(const PageNumber& pageNumber)
{
  if (widgetMap.contains(pageNumber))
  {
    return widgetMap[pageNumber];
  }

  MarkListWidget* item;

  // If we have a widget in the widget cache use it,
  // otherwise we really need to create a new MarkListWidget
  if (!widgetCache.isEmpty())
  {
    item = widgetCache.back();
    item->setPageNumber(pageNumber);
    widgetCache.pop_back();
  }
  else
  {
    item = new MarkListWidget(viewport(), this, pageNumber, pageCache, dataModel->preferences()->showThumbnails());
    item->setupObservers(dataModel);
  }

  connect(item, SIGNAL(selected(const PageNumber&)), this, SLOT(thumbnailSelected(const PageNumber&)));
  connect(item, SIGNAL(showPopupMenu(const PageNumber&, const QPoint&)), this, SLOT(showPopupMenu(const PageNumber&, const QPoint&)));
  connect(item, SIGNAL(selectionToggled(const PageNumber&, bool)), this, SLOT(selectionToggled(const PageNumber&, bool)));

  if (dataModel->isPageBookmarked(pageNumber))
  {
    item->addBookmark();
    item->setBookmarkLabel(dataModel->bookmarkLabel(pageNumber));
  }

  if (dataModel->currentPageNumber() == pageNumber)
  {
    item->setSelected(true);
  }

  item->setNewWidth(visibleWidth());
  item->setChecked(dataModel->isSelected(pageNumber));
  widgetMap.insert(pageNumber, item);
  addChild(item, 0, widgetPositionList[pageNumber-1]);

  return item;
}


void MarkList::recalculateWidgetPositions()
{
  int y = 0;

  widgetPositionList.clear();

  for (unsigned int page = 1; page <= dataModel->numberOfPages(); page++)
  {
    testWidget->setPageNumber(page);
    int height = testWidget->setNewWidth(visibleWidth());
    widgetPositionList.append(y);

    y += height;
  }

  resizeContents(visibleWidth(), y);

  if (contentsY() > y)
    setContentsPos(0, 0);
}


void MarkList::rebuildThumbnailWidgets()
{
  clear();

  // We need to recreate the testWidget everytime, because
  // we currently cannot enable/disable the the ThumbnailWidget
  // for an already created MarkListWidget.
  if (testWidget)
  {
    delete testWidget;
    testWidget = 0;
  }

  if (!testWidget)
  {
    testWidget = new MarkListWidget(this, this, 1, pageCache, dataModel->preferences()->showThumbnails());
    testWidget->setupObservers(dataModel);
    testWidget->hide();
  }

  recalculateWidgetPositions();
  slotCreateWidgets();

  QMap<PageNumber, MarkListWidget*>::Iterator it;
  for (it = widgetMap.begin(); it != widgetMap.end(); ++it)
  {
    MarkListWidget* item = *it;
    item->setNewWidth(visibleWidth());
    moveChild(item, 0, widgetPositionList[item->getPageNumber() - 1]);
  }

  viewport()->update();

  setCurrentPage();
}


void MarkList::thumbnailSelected(const PageNumber& pageNumber)
{
  // This variable is set to remember that the next call to setCurrentPageNumber
  // has been initiated with a left click on the thumbnail of page pageNumber.
  clickedThumbnail = pageNumber;
  dataModel->setCurrentPageNumber(pageNumber);
}


void MarkList::selectionToggled(const PageNumber& page, bool on)
{
  if (on)
    dataModel->selectPage(page);
  else
    dataModel->deselectPage(page);
}


void MarkList::setCurrentPage()
{
  PageNumber pageNumber = dataModel->currentPageNumber();

  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    clickedThumbnail = 0;
    return;
  }

  if (currentPage == pageNumber)
    return;

  // Create the widget for the current page
  MarkListWidget* item = createWidget(pageNumber);

  viewport()->update();

  // Clear old selection
  if (currentPage.isValid() && currentPage <= dataModel->numberOfPages())
  {
    if (widgetMap.contains(currentPage))
    {
      item = widgetMap[currentPage];
      item->setSelected(false);
    }
  }

  // Draw new selection
  if (widgetMap.contains(pageNumber))
  {
    item = widgetMap[pageNumber];
    item->setSelected(true);

    // Make selected page visible if the current page has not been set with a mouseclick
    // in the thumbnail list. (We use this because it is a bit confusing if the element that
    // you have just clicked on, is scrolled away under the mouse cursor)
    if (clickedThumbnail != pageNumber && !item->isCompletelyVisible())
    {
      // Center the thumbnail of the current page vertically.
      int ycoord = childY(item) + item->height() / 2 - visibleHeight() / 2;
      setContentsPosSmooth(childX(item), ycoord);
    }
  }

  clickedThumbnail = 0;

  currentPage = pageNumber;
}


void MarkList::clear()
{
  resizeContents(0, 0);
  setContentsPos(0, 0);

  currentPage = PageNumber::invalidPage;
  widgetPositionList.clear();

  QList<PageNumber> pages = widgetMap.keys();
  for (unsigned int i = 0; i < pages.count(); i++)
  {
    MarkListWidget* item = widgetMap[pages[i]];
    removeChild(item);
    widgetMap.remove(pages[i]);
    delete item;
  }

  for (unsigned int i = 0; i < widgetCache.count(); i++)
  {
    delete widgetCache[i];
  }
  widgetCache.clear();
}


void MarkList::selectAll()
{
  for (unsigned int page = 1; page <= dataModel->numberOfPages(); page++)
  {
    dataModel->selectPage(page);
  }
}


void MarkList::selectEven()
{
  for (unsigned int page = 2; page <= dataModel->numberOfPages(); page += 2)
  {
    dataModel->selectPage(page);
  }
}


void MarkList::selectOdd()
{
  for (unsigned int page = 1; page < dataModel->numberOfPages(); page += 2)
  {
    dataModel->selectPage(page);
  }
}


void MarkList::toggleSelection()
{
  QList<PageNumber> selectedPages = dataModel->selectedPages();

  selectAll();

  for (unsigned int i = 0; i < selectedPages.count(); i++)
  {
    dataModel->deselectPage(selectedPages[i]);
  }
}


void MarkList::removeSelection()
{
  dataModel->deselectAllPages();
}


void MarkList::pageSelected(const PageNumber& pageNumber)
{
  // safety checks
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::pageSelected called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (widgetMap.contains(pageNumber))
  {
    widgetMap[pageNumber]->setChecked(true);
  }
}


void MarkList::pageDeselected(const PageNumber& pageNumber)
{
  // safety checks
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::pageDeselected called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (widgetMap.contains(pageNumber))
  {
    widgetMap[pageNumber]->setChecked(false);
  }
}


void MarkList::fitToWidth()
{
  recalculateWidgetPositions();
  slotCreateWidgets();

  QMap<PageNumber, MarkListWidget*>::Iterator it;
  for (it = widgetMap.begin(); it != widgetMap.end(); ++it)
  {
    MarkListWidget* item = *it;
    item->setNewWidth(visibleWidth());
    moveChild(item, 0, widgetPositionList[item->getPageNumber() - 1]);
  }

  // Temporarely disable smooth scrolling
  bool smooth = dataModel->preferences()->smoothScrolling();
  dataModel->preferences()->setSmoothScrolling(false);

  // Make sure the selected item is still visible.
  if (currentPage.isValid() && currentPage <= dataModel->numberOfPages())
  {
    MarkListWidget* item = createWidget(currentPage);

    if (!item->isCompletelyVisible())
    {
      // Center the thumbnail of the current page vertically.
      int ycoord = childY(item) + item->height() / 2 - visibleHeight() / 2;
      setContentsPosSmooth(childX(item), ycoord);
    }
    slotCreateWidgets();
  }

  // Set smooth scrolling to its old state
  dataModel->preferences()->setSmoothScrolling(smooth);

  //viewport()->update();
}


void MarkList::viewportResizeEvent(QResizeEvent* e)
{
  Q3ScrollView::viewportResizeEvent(e);

  emit viewSizeChanged(viewport()->size());
}


void MarkList::mousePressEvent(QMouseEvent* e)
{
  if (e->button() == Qt::RightButton) {
    // We call showPopupMenu with an invalid pageNumber to indicate that
    // the mouse does not point at a thumbnailWidget.
    showPopupMenu(PageNumber::invalidPage, e->globalPos());
  }
}


void MarkList::showEvent(QShowEvent*)
{
  slotCreateWidgets();

  // We need to realign the widgets, as their positions
  // might have been messed up while the thumbnaillist was
  // invisible.
  QMap<PageNumber, MarkListWidget*>::Iterator it;
  for (it = widgetMap.begin(); it != widgetMap.end(); ++it)
  {
    MarkListWidget* item = *it;
    item->setNewWidth(visibleWidth());
    moveChild(item, 0, widgetPositionList[item->getPageNumber() - 1]);
  }

  viewport()->update();
}

void MarkList::slotShowThumbnails()
{
  if (dataModel->numberOfPages() == 0)
    return;

  // Temporarely disable smooth scrolling
  bool smooth = dataModel->preferences()->smoothScrolling();
  dataModel->preferences()->setSmoothScrolling(false);

  // Rebuild thumbnail widgets.
  rebuildThumbnailWidgets();

  dataModel->preferences()->setSmoothScrolling(smooth);
}


void MarkList::showPopupMenu(const PageNumber& pageNumber, const QPoint& position)
{
  if (contextMenu == 0)
  {
    // Initialize Contextmenu
    contextMenu = new QMenu(this);

    Action_SelectCurrentPage = contextMenu->addAction(i18n("Select &Current Page"));
    Action_SelectAllPages    = contextMenu->addAction(i18n("Select &All Pages"));
    Action_SelectEvenPages   = contextMenu->addAction(i18n("Select &Even Pages"));
    Action_SelectOddPages    = contextMenu->addAction(i18n("Select &Odd Pages"));
    Action_InverSelection    = contextMenu->addAction(i18n("&Invert Selection"));
    Action_DeselectAllPages  = contextMenu->addAction(i18n("&Deselect All Pages"));
    contextMenu->insertSeparator();
    Action_BookmarkPage      = contextMenu->addAction(SmallIcon("bookmark_add"), i18n("Add &Bookmark"));
  }

  // TODO: use new slot-based API, once it works reliable
  if (dataModel->numberOfPages() == 0) {
    Action_SelectAllPages->setEnabled(false);
    Action_SelectEvenPages->setEnabled(false);
    Action_SelectOddPages->setEnabled(false);
    Action_InverSelection->setEnabled(false);
    Action_DeselectAllPages->setEnabled(false);
    Action_BookmarkPage->setEnabled(false);
  }
  else
  {
    Action_SelectAllPages->setEnabled(true);
    Action_SelectEvenPages->setEnabled(true);
    Action_SelectOddPages->setEnabled(true);
    Action_InverSelection->setEnabled(true);
    Action_DeselectAllPages->setEnabled(true);
    Action_BookmarkPage->setEnabled(true);
  }

  // Only allow to select the current page if we got a valid pageNumber.
  if (pageNumber.isValid() && pageNumber <= dataModel->numberOfPages())
  {
    Action_SelectCurrentPage->setEnabled(true);
    Action_BookmarkPage->setEnabled(true);

    if (dataModel->isPageBookmarked(pageNumber))
    {
      Action_BookmarkPage->setIcon(SmallIcon("bookmark"));
      Action_BookmarkPage->setText(i18n("Remove &Bookmark"));
    }
    else
    {
      Action_BookmarkPage->setIcon(SmallIcon("bookmark_add"));
      Action_BookmarkPage->setText(i18n("Add &Bookmark"));
    }
  }
  else
  {
    Action_SelectCurrentPage->setEnabled(false);
    Action_BookmarkPage->setEnabled(false);
  }

  // Show Contextmenu
  QAction *selected = contextMenu->exec(position);
  if (selected == Action_SelectCurrentPage)
    createWidget(pageNumber)->toggle();

  if (selected == Action_SelectAllPages)
    selectAll();

  if (selected == Action_SelectEvenPages)
    selectEven();

  if (selected == Action_SelectOddPages)
    selectOdd();

  if (selected == Action_InverSelection)
    toggleSelection();

  if (selected == Action_DeselectAllPages)
    removeSelection();

  if (selected == Action_BookmarkPage)
  {
    if (isPageBookmarked(pageNumber))
      dataModel->removeBookmark(pageNumber);
    else
      dataModel->addBookmark(pageNumber, QString::null);
  }
}


void MarkList::bookmarkAdded(const PageNumber& pageNumber, const QString& label)
{
  // safety checks
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::bookmarkAdded called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (widgetMap.contains(pageNumber))
  {
    widgetMap[pageNumber]->addBookmark();
    widgetMap[pageNumber]->setBookmarkLabel(label);
  }
}


void MarkList::bookmarkRenamed(const PageNumber& pageNumber, const QString& label)
{
  // safety checks
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::bookmarkRenamed called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (widgetMap.contains(pageNumber))
  {
    widgetMap[pageNumber]->setBookmarkLabel(label);
  }
}


void MarkList::bookmarkRemoved(const PageNumber& pageNumber)
{
  // safety checks
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::bookmarkRemoved called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (widgetMap.contains(pageNumber))
  {
    widgetMap[pageNumber]->removeBookmark();
  }
}


void MarkList::allBookmarksRemoved()
{
  QMap<PageNumber, MarkListWidget*>::Iterator it;
  for (it = widgetMap.begin(); it != widgetMap.end(); ++it)
  {
    (*it)->removeBookmark();
  }
}


void MarkList::slotSetThumbnail(PageNumber pageNumber)
{
  if (!pageNumber.isValid() || pageNumber > dataModel->numberOfPages())
  {
    kError(kvs::shell) << "MarkList::slotSetThumbnail called with invalid pageNumber " << pageNumber << endl;
    return;
  }

  if (dataModel->preferences()->showThumbnails())
  {
    createWidget(pageNumber)->setThumbnail();
  }
}


void MarkList::setupObservers(DataModel* _dataModel)
{
  SmoothScrollView::setupObservers(_dataModel);
  connect(dataModel, SIGNAL(currentPageNumberChanged()), this, SLOT(setCurrentPage()));
  connect(dataModel, SIGNAL(numberOfPagesChanged()), this, SLOT(rebuildThumbnailWidgets()));

  /*
  QList<PageNumber> keys = dataModel->bookmarks();
  for (int i = 0; i < keys.count(); i++)
  {
    bookmarkAdded(keys[i], dataModel->bookmarkLabel(keys[i]));
  }*/

  connect(dataModel, SIGNAL(bookmarkAdded(const PageNumber&, const QString&)), this, SLOT(bookmarkAdded(const PageNumber&, const QString&)));
  connect(dataModel, SIGNAL(bookmarkRenamed(const PageNumber&, const QString&)), this, SLOT(bookmarkRenamed(const PageNumber&, const QString&)));
  connect(dataModel, SIGNAL(bookmarkRemoved(const PageNumber&)), this, SLOT(bookmarkRemoved(const PageNumber&)));
  connect(dataModel, SIGNAL(allBookmarksRemoved()), this, SLOT(allBookmarksRemoved()));

  connect(dataModel, SIGNAL(pageSelected(const PageNumber&)), this, SLOT(pageSelected(const PageNumber&)));
  connect(dataModel, SIGNAL(pageDeselected(const PageNumber&)), this, SLOT(pageDeselected(const PageNumber&)));
}

