// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/* This file is part of the KDE project
   Copyright (C) 2001 Wilco Greven <greven@kde.org>
   Copyright (C) 2002-2004 Stefan Kebekus <kebekus@kde.org>
   Copyright (C) 2004-2006 Wilfried Huss <Wilfried.Huss@gmx.at>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <config.h>

#include "documentPageCache.h"
#include "pageView.h"
#include "ligaturePluginGUI.h"
#include "kvs_debug.h"
#include "kvsprefs.h"
#include "textBox.h"

#include <QFrame>
#include <QKeyEvent>
#include <QMouseEvent>
#include <QPainter>
#include <QRect>
#include <QTimer>

#include <cmath>


PageView::PageView( QWidget* parent, const char* name)
  : SmoothScrollView( parent, name, Qt::WStaticContents | Qt::WNoAutoErase)
{
  viewport()->setFocusPolicy(Qt::StrongFocus);

  setResizePolicy(Q3ScrollView::Manual);

  setVScrollBarMode(Q3ScrollView::Auto);
  setHScrollBarMode(Q3ScrollView::Auto);

  viewport()->setAttribute(Qt::WA_NoSystemBackground, true);

  setResizePolicy(Manual);
  setDragAutoScroll(false);

  enableClipper(true);
  nrCols = 2;
  nrRows = 1;
  continuousViewmode = true;
  fullScreen = false;

  connect(this, SIGNAL(contentsMoving(int, int)), this, SLOT(calculateCurrentPageNumber(int, int)));
}


void PageView::setPageCache(DocumentPageCache* _pageCache)
{
  pageCache = _pageCache;
  connect(pageCache, SIGNAL(updateWidget(const PageNumber&)), this, SLOT(updatePage(const PageNumber&)));
}


void PageView::setMultiPage(ligaturePluginGUI* _multiPage)
{
  multiPage = _multiPage;
}


void PageView::clear()
{
  for (int i = 0; i < widgetList.size(); i++)
    delete widgetList[i];
  widgetList.clear();
  layoutPages();
}


bool PageView::atTop() const
{
  return verticalScrollBar()->value() == verticalScrollBar()->minimum();
}


bool PageView::atBottom() const
{
  return verticalScrollBar()->value() == verticalScrollBar()->maximum();
}


void PageView::scrollRight()
{
  QScrollBar* sb = horizontalScrollBar();
  sb->setSliderPosition(sb->value() + sb->singleStep());
}

void PageView::scrollLeft()
{
  QScrollBar* sb = horizontalScrollBar();
  sb->setSliderPosition(sb->value() - sb->singleStep());
}

void PageView::scrollBottom()
{
  verticalScrollBar()->setValue( verticalScrollBar()->maximum() );
}

void PageView::scrollTop()
{
  verticalScrollBar()->setValue( verticalScrollBar()->minimum() );
}

void PageView::scrollUpPage()
{
  verticalScrollBar()->triggerAction(QAbstractSlider::SliderPageStepSub);
}


void PageView::scrollDownPage()
{
  verticalScrollBar()->triggerAction(QAbstractSlider::SliderPageStepAdd);
}


void PageView::scrollLeftPage()
{
  horizontalScrollBar()->triggerAction(QAbstractSlider::SliderPageStepSub);
}


void PageView::scrollRightPage()
{
  horizontalScrollBar()->triggerAction(QAbstractSlider::SliderPageStepAdd);
}


void PageView::scrollUp()
{
  QScrollBar* scrollBar = verticalScrollBar();
  scroll(-scrollBar->singleStep());
}


void PageView::scrollDown()
{
  QScrollBar* scrollBar = verticalScrollBar();
  scroll(scrollBar->singleStep());
}


void PageView::contentsMousePressEvent( QMouseEvent* e )
{
  if (e->button() == Qt::LeftButton)
  {
    setCursor(Qt::SizeAllCursor);
    dragGrabPos = e->globalPos();
  }
  else
  {
    setCursor(Qt::ArrowCursor);
  }
}

void PageView::contentsMouseReleaseEvent(QMouseEvent* e)
{
  setCursor(Qt::ArrowCursor);
}

void PageView::contentsMouseMoveEvent( QMouseEvent* e )
{
  QPoint newPos = e->globalPos();

  if (e->buttons() & Qt::LeftButton)
  {
    QPoint delta = dragGrabPos - newPos;
    scrollBy(delta.x(), delta.y());
  }
  dragGrabPos = newPos;
}

void PageView::viewportResizeEvent( QResizeEvent* e )
{
  Q3ScrollView::viewportResizeEvent( e );

  if (widgetList.isEmpty())
    return;

  layoutPages();

  emit viewSizeChanged( viewport()->size() );
}

void PageView::slotShowScrollbars(bool status)
{
  if (status == true) {
    setVScrollBarMode(Q3ScrollView::Auto);
    setHScrollBarMode(Q3ScrollView::Auto);
  } else {
    setVScrollBarMode(Q3ScrollView::AlwaysOff);
    setHScrollBarMode(Q3ScrollView::AlwaysOff);
  }
}

void PageView::setFullScreenMode(bool fullScreen)
{
  this -> fullScreen = fullScreen;
  if (fullScreen == true)
  {
    setVScrollBarMode(Q3ScrollView::AlwaysOff);
    setHScrollBarMode(Q3ScrollView::AlwaysOff);
    oldFrameStyle = frameStyle();
    setFrameStyle(QFrame::NoFrame);
  }
  else
  {
    setFrameStyle(oldFrameStyle);
  }
}

void PageView::layoutPages(bool zoomChanged)
{
  // If there are no widgets, e.g. because the last widget has been
  // removed, the matter is easy: set the contents size to 0. If there
  // are no widgets because previously existing widgets were removed
  // (we detect that by looking at the contentsWidth and -Height).
  if (widgetList.isEmpty()) {
    if ((contentsWidth() != 0) || (contentsHeight() != 0)) {
      Q3ScrollView::resizeContents(0,0);
    }
    return;
  }

  // Ok, now we are in a situation where we do have some widgets that
  // shall be centered.
  int distance = distanceBetweenWidgets;

  QVector<quint32> colWidth(nrCols);
  for(quint8 i=0; i<colWidth.size(); i++)
    colWidth[i] = 0;

  quint16 numRows;
  if(nrCols <= 2)
  {
    numRows = (widgetList.size()+2*nrCols-2) / nrCols;
  }
  else
  {
    numRows = (qint16)ceil(((double)widgetList.size()) / nrCols);
  }

  QVector<quint32> rowHeight(numRows);
  for(quint16 i=0; i<rowHeight.size(); i++)
    rowHeight[i] = 0;

  // Now find the widths and heights of the columns
  for(quint16 i=0; i<widgetList.size(); i++)
  {
    quint8 col;
    quint16 row;

    if (nrCols == 2) {
      // In two-column display, start with the right column
      col = (i+1+nrCols) % nrCols;
      row = (i+1+nrCols) / nrCols - 1;
    } else {
      col = (i+nrCols) % nrCols;
      row = (i+nrCols) / nrCols - 1;
    }

    colWidth[col] = qMax(colWidth[col], (quint32)widgetList[i]->pageSize().width());
    rowHeight[row] = qMax(rowHeight[row], (quint32)widgetList[i]->pageSize().height());
  }

  // Calculate the total width and height of the display
  quint32 totalHeight = 0;
  for(quint16 i=0; i<rowHeight.size(); i++)
    totalHeight += rowHeight[i];

  totalHeight += (numRows+1)*distance;
  quint32 totalWidth = 0;
  for(quint8 i=0; i<colWidth.size(); i++)
    totalWidth += colWidth[i];

  totalWidth += (nrCols+1)*distance;
  QSize newViewportSize = viewportSize( totalWidth, totalHeight );
  quint32 centeringLeft = 0;
  if( (quint32)newViewportSize.width() > totalWidth )
    centeringLeft = ( newViewportSize.width() - totalWidth )/2;
  quint32 centeringTop = 0;
  if( (quint32)newViewportSize.height() > totalHeight )
    centeringTop = ( newViewportSize.height() - totalHeight)/2;

  // Resize the viewport
  if (((quint32)contentsWidth() != totalWidth) || ((quint32)contentsHeight() != totalHeight))
  {
    // Calculate the point in the coordinates of the contents which is currently at the center of the viewport.
    QPoint midPoint = QPoint(visibleWidth() / 2 + contentsX(), visibleHeight() / 2 + contentsY());
    double midPointRatioX = (double)(midPoint.x()) / contentsWidth();
    double midPointRatioY = (double)(midPoint.y()) / contentsHeight();

    resizeContents(totalWidth,totalHeight);

    // If the zoom changed recenter the former midPoint
    if (zoomChanged)
      center((int)(contentsWidth() * midPointRatioX), (int)(contentsHeight() * midPointRatioY));
  }

  // Finally, calculate the left and top coordinates of each row and
  // column, respectively
  QVector<quint32> colLeft(nrCols);
  colLeft[0] = distance;
  for(quint8 i=1; i<colLeft.size(); i++)
    colLeft[i] = colLeft[i-1]+colWidth[i-1]+distance;

  QVector<quint32> rowTop(numRows);
  rowTop[0] = distance;
  for(quint16 i=1; i<rowTop.size(); i++)
    rowTop[i] = rowTop[i-1]+rowHeight[i-1]+distance;

  for(quint16 i=0; i<widgetList.size(); i++)
  {
    quint8 col;
    quint16 row;
    if (nrCols == 2)
    {
      // In two column-mode start with the right column.
      col = (i+nrCols-1) % nrCols;
      row = (i+nrCols-1) / nrCols;
    }
    else
    {
      col = (i+nrCols) % nrCols;
      row = i / nrCols;
    }
    if (nrCols == 2)
    {
      // in 2-column mode right justify the first column, and leftjustify the second column
      int width = widgetList[i]->width();
      int left;
      if (col == 0)
        left = centeringLeft + colLeft[col] + colWidth[col]-width + distance/2;
      else
        left = centeringLeft + colLeft[col];
      moveChild( widgetList[i], left, centeringTop+rowTop[row]);
    }
    else
    {
      // in single column and overview mode center the widgets
      int widgetWidth = widgetList[i]->width();
      int left = centeringLeft + colLeft[col] + ((int)colWidth[col]-widgetWidth)/2;
      moveChild(widgetList[i], left, centeringTop+rowTop[row]);
    }
  }
  calculateCurrentPageNumber();
}


PageNumber PageView::widestPage() const
{
  if (dataModel->numberOfPages() == 0)
    return PageNumber::invalidPage;

  Length maxWidth;
  PageNumber pageNumber = 1;

  for (unsigned int i = 1; i <= dataModel->numberOfPages(); i++)
  {
    Length width;

    // Since the width in this context is with respect to the pagelayout,
    // we need to consider the current layout rotation.
    if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
        dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
    {
      width = pageCache->sizeOfPage(i).height();
    }
    else
    {
      width = pageCache->sizeOfPage(i).width();
    }

    if (width > maxWidth)
    {
      maxWidth = width;
      pageNumber = i;
    }
  }

  return pageNumber;
}


double PageView::zoomForWidthColumns(unsigned int viewportWidth) const
{
  if (multiPage.isNull())
    return 1.0;

  Length maxLeftColumnWidth;
  Length maxRightColumnWidth;
  Length maxWidth;

  PageNumber widestPageLeft;
  PageNumber widestPageRight;

  for (unsigned int i = 1; i <= dataModel->numberOfPages(); i++)
  {
    Length width;

    // Since the width in this context is with respect to the pagelayout,
    // we need to consider the current layout rotation.
    if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
        dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
    {
      width = pageCache->sizeOfPage(i).height();
    }
    else
    {
      width = pageCache->sizeOfPage(i).width();
    }

    if ( i % 2 == 0) // page is in left column
    {
      if (width > maxLeftColumnWidth)
      {
        maxLeftColumnWidth = width;
        widestPageLeft = i;
      }
    }

    if ( i % 2 == 1) // page is in right column
    {
      if (width > maxRightColumnWidth)
        maxRightColumnWidth = width;
        widestPageRight = i;
    }
  }

  double ratio =  maxLeftColumnWidth / (maxLeftColumnWidth + maxRightColumnWidth);

  // This number is the amount of space the left column should occupy in the viewport.
  unsigned int leftTargetWidth = (unsigned int)(ratio * viewportWidth);

  const int dpix = viewport()->logicalDpiX();

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    return pageCache->sizeOfPage(widestPageLeft).zoomForHeight(leftTargetWidth, dpix);
  }
  else
  {

    return pageCache->sizeOfPage(widestPageLeft).zoomForWidth(leftTargetWidth, dpix);
  }
}


double PageView::calculateFitToHeightZoomValue() const
{
  if (multiPage.isNull())
    return 1.0;

  PageNumber pageNumber = 1;

  // See below, in the documentation of the method "calculatefitToWidthZoomLevel"
  // for an explanation of the complicated calculation we are doing here.
  int columns = getNrColumns();
  int rows = 1;
  int continuousViewmode = (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Continuous) ||
                           (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::ContinuousFacing);

  if (columns == 1 && rows == 1 && !continuousViewmode) // single page mode
  {
    pageNumber = dataModel->currentPageNumber();
    if (!pageNumber.isValid())
      pageNumber = 1;
  }

  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Overview)
  {
    rows = getNrRows();
  }

  int targetViewportHeight = viewportSize(0,0).height();
  // maximal height of a single page
  int targetPageHeight = (targetViewportHeight - (rows+1) * distanceBetweenWidgets) / rows;

  double aspectRatio = pageCache->sizeOfPage(pageNumber).aspectRatio();

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    aspectRatio = 1 / aspectRatio;
  }

  int targetPageWidth  = (int)(targetPageHeight * aspectRatio);
  int targetViewportWidth = targetPageWidth * columns + (columns+1) * distanceBetweenWidgets;
  targetViewportHeight = viewportSize(targetViewportWidth, targetViewportHeight).height();
  targetPageHeight = (targetViewportHeight - (rows+1) * distanceBetweenWidgets) / rows;

  const int dpix = viewport()->logicalDpiX();

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    return pageCache->sizeOfPage(pageNumber).zoomForWidth(targetPageHeight, dpix);
  }
  else
  {
    return pageCache->sizeOfPage(pageNumber).zoomForHeight(targetPageHeight, dpix);
  }
}


double PageView::calculateFitToWidthZoomValue() const
{
  if (multiPage.isNull())
    return 1.0;

  PageNumber pageNumber = 1;

  int columns = getNrColumns();
  int rows = getNrRows();
  int continuousViewmode = (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Continuous) ||
                           (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::ContinuousFacing);

  if (columns == 1 && rows == 1 && !continuousViewmode) // single page mode
  {
    // To calculate the zoom level in single page mode we need the size
    // of the current page. When a new document is opened this function
    // is called while the currentPageNumber is invalid. We use the size
    // of the first page of the document in this case.
    pageNumber = dataModel->currentPageNumber();
    if (!pageNumber.isValid())
      pageNumber = 1;
  }

  if (columns == 1 && rows == 1 && continuousViewmode) // continuous viewmode
  {
    pageNumber = widestPage();
    if (!pageNumber.isValid())
      pageNumber = 1;
  }

  // rows should be 1 for Single Page Viewmode,
  // the number of Pages in Continuous Viewmode
  // and number of Pages/2 in Continuous-Facing Viewmode
  if (continuousViewmode)
    rows = (int)(ceil(dataModel->numberOfPages() / (double)columns));

  // There is a slight complication here... if we just take the width
  // of the viewport and scale the contents by a factor x so that it
  // fits the viewport exactly, then, depending on chosen papersize
  // (landscape, etc.), the contents may be higher than the viewport
  // and the QScrollview may or may not insert a scrollbar at the
  // right. If the scrollbar appears, then the usable width of the
  // viewport becomes smaller, and scaling by x does not really fit
  // the (now smaller page) anymore.

  // Calculate the width and height of the view, disregarding the
  // possible complications with scrollbars, e.g. assuming the maximal
  // space is available.

  // width of the widget excluding possible scrollbars
  int targetViewportWidth  = viewportSize(0,0).width();

  // maximal width of a single page
  int targetPageWidth = (targetViewportWidth - (columns+1) * distanceBetweenWidgets) / columns;

  double aspectRatio = pageCache->sizeOfPage(pageNumber).aspectRatio();

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    aspectRatio = 1 / aspectRatio;
  }

  // maximal height of a single page
  int targetPageHeight = (int)(targetPageWidth/aspectRatio);
  // FIXME: this is only correct if all pages in the document have the same height
  int targetViewportHeight = rows * targetPageHeight + (rows+1) * distanceBetweenWidgets;

  // Think again, this time use only the area which is really
  // acessible (which, in case that targetWidth targetHeight don't fit
  // the viewport, is really smaller because of the scrollbars).
  targetViewportWidth = viewportSize(targetViewportWidth, targetViewportHeight).width();

  if (columns == 2 && continuousViewmode) // continuous facing
  {
    // TODO Generalize this for more than 2 columns
    return zoomForWidthColumns(targetViewportWidth - (columns+1) * distanceBetweenWidgets);
  }

  // maximal width of a single page (now the scrollbars are taken into account)
  targetPageWidth = (targetViewportWidth - (columns+1) * distanceBetweenWidgets) / columns;

  const int dpix = viewport()->logicalDpiX();

  if (dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Landscape ||
      dataModel->preferences()->rotation() == KVSPrefs::EnumRotation::Seascape)
  {
    return pageCache->sizeOfPage(pageNumber).zoomForHeight(targetPageWidth, dpix);
  }
  else
  {
    return pageCache->sizeOfPage(pageNumber).zoomForWidth(targetPageWidth, dpix);
  }
}


void PageView::contentsWheelEvent ( QWheelEvent * e )
{
  QScrollBar* sb = verticalScrollBar();

  // Zoom in/out
  if (e->modifiers() & Qt::ControlModifier)
  {
    if (e->delta() < 0)
      emit zoomOut();
    else
      emit zoomIn();
    return;
  }

  int pxl = -(e->delta()*sb->singleStep())/60;
  if (pxl == 0)
  {
    if (e->delta() > 0)
      pxl = -1;
    else
      pxl = 1;
  }

  // Faster scrolling
  if (e->modifiers() & Qt::ShiftModifier)
    pxl *= 10;

  scroll(pxl);
}


void PageView::moveViewportToWidget(QWidget* widget, int y, bool smoothMove)
{
  int verticalPos = 0;
  int verticalPosTop = 0;

  if (y != 0)
  {
    verticalPosTop = childY(widget) +  y - visibleHeight()/2;
    verticalPos = childY(widget) +  y;
  }
  else
  {
    verticalPos = childY(widget) - distanceBetweenWidgets;
    verticalPosTop = verticalPos;
  }

  if (nrCols == 1)
  {
    // In single column viewmodes, we change the vertical position only, to make it
    // easier to work with high zoomlevels where not the whole pagewidth is visible.
    // TODO: Smarter algorithm also for continuous facing viewmode.
    int top = (int)(contentsY() + 0.1 * visibleHeight());
    int bottom = (int)(contentsY() + 0.9 * visibleHeight());

    // Move the viewport if the target is currently not visible, or lies at the edge
    // of the viewport. If y = 0 always move the top of the targetpage to the top edge
    // of the viewport.
    if (verticalPos < top || verticalPos > bottom || y == 0)
    {
      if (smoothMove)
        setContentsPosSmooth(contentsX(), verticalPosTop);
      else
        setContentsPos(contentsX(), verticalPosTop);
    }
  }
  else
  {
    if (smoothMove)
      setContentsPosSmooth(childX(widget) - distanceBetweenWidgets, verticalPosTop);
    else
      setContentsPos(childX(widget) - distanceBetweenWidgets, verticalPosTop);
  }
}


void PageView::viewportPaintEvent(QPaintEvent* e)
{
  // Region from which rectangles occupied by child widgets will be
  // subtracted.
  QRegion backgroundArea(e->rect());

  if (!widgetList.isEmpty())
  {
    for (int i = 0; i < widgetList.size(); i++)
    {
      DocumentWidget* item = widgetList[i];

      QRect widgetGeometry = item->geometry();

      // Draw the widget.
      if (e->rect().intersects(widgetGeometry))
      {
        QRect widgetRect = e->rect().intersect(widgetGeometry);
        widgetRect.translate(-widgetGeometry.left(), -widgetGeometry.top());

        item->update(widgetRect);
      }

      // Substract the painted area.
      backgroundArea -= widgetGeometry.intersect(e->rect());
    }
  }

  // Paint the background.
  QPainter p(viewport());

  QVector<QRect> backgroundRects = backgroundArea.rects();

  for (int i = 0; i < backgroundRects.count(); i++)
    p.fillRect(backgroundRects[i], palette().mid());
}


void PageView::calculateCurrentPageNumber(int x, int y)
{
  // Safety check
  if (widgetList.isEmpty())
    return;

  // Don't update the current page number, when a smooth move is in progress.
  // At the beginning of a smooth move, the signal contentsMoving(int, int)
  // with the endcoordinates is emitted.
  if (moveInProgress())
    return;

  QRect viewportRect(x, y, visibleWidth(), visibleHeight());

  //kDebug(kvs::shell) << "viewportRect(" << viewportRect.x() << ", " << viewportRect.y() << ", "
  //          << viewportRect.width() << ", " << viewportRect.height() << ")" << endl;

  int maxVisiblePixels = 0;
  DocumentWidget* _currentWidget = 0;

  for (quint16 i = 0; i < widgetList.size(); i++)
  {
    DocumentWidget* documentWidget = widgetList[i];
    // Safety check
    if (documentWidget == 0)
      continue;

    // Check if the Widget is visible
    int cx = childX(documentWidget);
    int cy = childY(documentWidget);
    QRect widgetRect(cx, cy, documentWidget->width(), documentWidget->height());
    bool isVisible = widgetRect.intersects(viewportRect);

    if (!isVisible)
      continue;

    // Calculate the number of visible pixels of the widget
    QRect visibleRect = widgetRect.intersect(viewportRect);
    int visiblePixels = visibleRect.width() * visibleRect.height();

    //kDebug(kvs::shell) << visiblePixels << " pixels are visible of page " << documentWidget->getPageNumber() << endl;

    // If a bigger part of this widget as of the previous widgets is visible make it the current widget.
    if (maxVisiblePixels < visiblePixels)
    {
      maxVisiblePixels = visiblePixels;
      _currentWidget = documentWidget;
    }
  }

  // No page is visible
  if (_currentWidget == 0)
    return;

  // Update the number of the current page
  dataModel->setCurrentPageNumber(_currentWidget->getPageNumber(), true);
}

void PageView::calculateCurrentPageNumber()
{
  calculateCurrentPageNumber(contentsX(), contentsY());
}


void PageView::setViewMode()
{
  //FIXME might not be needed anymore
  // Save the current page number because when we are changing the columns
  // and rows in the scrollview the currently shown Page probably out of view.
  PageNumber currentPage = dataModel->currentPageNumber();	
  int viewmode = dataModel->preferences()->viewMode();

  switch (viewmode)
  {
    case KVSPrefs::EnumViewMode::SinglePage:
      // Don't do anything if the view mode is already set
      if (nrCols == 1 && nrRows == 1 && !continuousViewmode)
        return;

      nrCols = 1;
      nrRows = 1;
      continuousViewmode = false;
      // We scroll the view to the top, so that top and not the bottom
      // of the visible page is shown.
      scrollTop();
      break;
    case KVSPrefs::EnumViewMode::ContinuousFacing:
      // Don't do anything if the view mode is already set
      if (nrCols == 2 && nrRows == 1 && continuousViewmode)
        return;

      nrCols = 2;
      nrRows = 1;
      continuousViewmode = true;
      break;
    case KVSPrefs::EnumViewMode::Overview:
      // Don't do anything if the view mode is already set
      if ((int)nrCols == dataModel->preferences()->overviewModeColumns() &&
          (int)nrRows == dataModel->preferences()->overviewModeRows() && !continuousViewmode)
        return;

      nrCols = dataModel->preferences()->overviewModeColumns();
      nrRows = dataModel->preferences()->overviewModeRows();
      continuousViewmode = false;
      // We scroll the view to the top, so that top and not the bottom
      // of the visible tableau is shown.
      scrollTop();
      break;
    default:  //KVSPrefs::EnumViewMode::Continuous
      // Don't do anything if the view mode is already set
      if (nrCols == 1 && nrRows == 1 && continuousViewmode)
        return;

      nrCols = 1;
      nrRows = 1;
      continuousViewmode = true;
  }

  generateDocumentWidgets(currentPage);
}

bool PageView::isVisible(QWidget* child)
{
  QRect visibleRect(contentsX(), contentsY(), visibleWidth(), visibleHeight());
  QRect widgetRect(childX(child), childY(child), child->width(), child->height());
  return widgetRect.intersects(visibleRect);
}


void PageView::generateDocumentWidgets(const PageNumber& _startPage)
{
  PageNumber startPage = _startPage;
  
#ifdef DEBUG_KMULTIPAGE
  kDebug(kvs::shell) << "PageView::generateDocumentWidgets(" << startPage << ")" << endl;
#endif

  if (dataModel->numberOfPages() == 0)
  {
    clear();
    return;
  }

  // This function is only called with an invalid pagenumber, when
  // the file has been loaded or reloaded.
  bool reload = !startPage.isValid();

  if (reload)
  {
    // Find the number of the current page, for later use.
    startPage = dataModel->currentPageNumber();
  }

  int numberOfPages = dataModel->numberOfPages();

  // Make sure that startPage is in the permissible range.
  if (startPage < 1)
    startPage = 1;
  if (startPage > numberOfPages)
    startPage = numberOfPages;

  unsigned int tableauStartPage = startPage;

#ifdef __GNUC__
#warning TODO: move the RESIZE code to the end of this paragraph
#endif
  // Find out how many widgets are needed, and resize the widgetList accordingly
  quint16 oldwidgetListSize = widgetList.size();
  if (numberOfPages == 0) {
    for (int i = 0; i < widgetList.size(); i++)
      delete widgetList[i];
    widgetList.clear();
  } else {
    switch (dataModel->preferences()->viewMode()) {
    case KVSPrefs::EnumViewMode::SinglePage:
      // Resize the widgetList to contain exactly one element
      if (widgetList.isEmpty())
        widgetList.append(0); // The structure will later be constructed and a point inserted
      else {
        while(widgetList.size() > 1)
          delete widgetList.takeLast();
      }
      break;
    case KVSPrefs::EnumViewMode::Overview:
      {
        // Calculate the number of pages shown in overview mode.
        int visiblePages = dataModel->preferences()->overviewModeColumns() * dataModel->preferences()->overviewModeRows();
        // Calculate the number of the first page in the tableau.
        tableauStartPage = startPage - ((startPage - 1) % visiblePages);
        // We cannot have more widgets then pages in the document.
        visiblePages = qMin((unsigned int)visiblePages, numberOfPages - tableauStartPage + 1);
        if (widgetList.size() != visiblePages) {
          while(widgetList.size() > visiblePages)
            delete widgetList.takeLast();
          while(widgetList.size() < visiblePages)
            widgetList.append(0); // The structure will later be constructed and a point inserted
        }
        break;
      }
    default:
      // In KVS_Continuous and KVS_ContinuousFacing all pages in the document are shown.
      while(widgetList.size() > numberOfPages)
        delete widgetList.takeLast();
      while(widgetList.size() < numberOfPages)
        widgetList.append(0); // The structure will later be constructed and a point inserted
    }
  }
  bool isWidgetListResized = (widgetList.size() != oldwidgetListSize);

  // If the widgetList is empty, there is nothing left to do.
  if (widgetList.size() == 0) {
    return;
  }

  // Allocate DocumentWidget structures so that all entries of
  // widgetList point to a valid DocumentWidget.
  DocumentWidget *documentWidget;
  for(quint16 i=0; i<widgetList.size(); i++) {
    documentWidget = widgetList[i];
    if (documentWidget == 0) {
      documentWidget = createDocumentWidget();

      widgetList[i] = documentWidget;
      documentWidget->show();
    }
  }

  // Set the page numbers for the newly allocated widgets. How this is
  // done depends on the viewMode.
  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::SinglePage) {
    // In KVS_SinglePage mode, any number between 1 and the maximum
    // number of pages is acceptable. If an acceptable value is found,
    // nothing is done, and otherwise '1' is set as a default.
    documentWidget = widgetList[0];
    if (documentWidget != 0) { // Paranoia safety check
      documentWidget->setPageNumber(startPage);
      if (isVisible(documentWidget))
        documentWidget->update();
    } else
      kError(kvs::shell) << "Zero-Pointer in widgetList in ligaturePluginGUI::generateDocumentWidgets()" << endl;
  } else {
    // In all other modes, the widgets will be numbered continuously,
    // starting from firstShownPage.
    for(quint16 i=0; i<widgetList.size(); i++) {
      documentWidget = widgetList[i];
      if (documentWidget != 0) // Paranoia safety check
      {
        if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Overview)
          documentWidget->setPageNumber(i+tableauStartPage);
        else
          documentWidget->setPageNumber(i+1);
      }
      else
        kError(kvs::shell) << "Zero-Pointer in widgetList in ligaturePluginGUI::generateDocumentWidgets()" << endl;
    }
  }

  // Make the changes in the widgetList known to the scrollview. so
  // that the scrollview may update its contents.
  layoutPages();

  // If the number of widgets has changed, or the viewmode has been changed the widget
  // that displays the current page may not be visible anymore. Bring it back into focus.
  if (isWidgetListResized || !reload)
    dataModel->setCurrentPageNumber(startPage);
}


void PageView::gotoSelection(const TextSelection& selection)
{
  if (selection.isEmpty())
  {
    kDebug(kvs::shell) << "PageView::gotoSelection() called with empty TextSelection." << endl;
    return;
  }

  // We render the page asyncron, because we need to know the exact positon of the selection
  // at the current zoomlevel, before we can scroll to this position.
  RenderedDocumentPage* pageData = pageCache->getPage(selection.getPageNumber(), false);

  if (pageData == 0)
  {
    kDebug(kvs::shell) << "PageView::gotoSelection(): no documentPage generated" << endl;
    return;
  }

  switch (numberOfWidgets())
  {
    case 0:
      kError(kvs::shell) << "PageView::gotoSelection() while widgetList is empty" << endl;
      break;
    case 1:
      pageWidget(0)->select(selection);
      break;
    default:
      PageNumber pageNumber = selection.getPageNumber();
      if (numberOfWidgets() < pageNumber)
        kError(kvs::shell) << "PageView::gotoSelection() while widgetList.size()=" << numberOfWidgets() << "and selection.getPageNumber()=" << pageNumber << endl;
      else
      {
        DocumentWidget* pageWidget = widgetOfPageNumber(pageNumber);
        if (pageWidget)
        {
          pageWidget->select(selection);
        }
      }
  }

  QRect box = pageData->textBoxList[selection.getSelectedTextStart()].box;

  unsigned int y = 0;

  switch (dataModel->preferences()->rotation())
  {
    case KVSPrefs::EnumRotation::Landscape:
      y = box.left();
      break;
    case KVSPrefs::EnumRotation::Upsidedown:
      y = pageCache->sizeOfPageInPixel(selection.getPageNumber()).height() - box.bottom();
      break;
    case KVSPrefs::EnumRotation::Seascape:
      y = pageCache->sizeOfPageInPixel(selection.getPageNumber()).width() - box.right();
      break;
    case KVSPrefs::EnumRotation::Portrait:
    default:
      y = box.top();
  }

  Length l;
  l.setLength_in_pixel(y, dataModel->resolution());

  // Make the selection visible in the page view
  dataModel->setCurrentPageNumber(Anchor(selection.getPageNumber(), l, false));
}


void PageView::gotoAnchor(Anchor a)
{
  if (!a.page.isValid()) {
    kError(kvs::shell) << "PageView::gotoAnchor() call when anchor had invalid pageNumber." << endl;
    return;
  }
  if (dataModel->numberOfPages() == 0) {
    kError(kvs::shell) << "PageView::gotoAnchor() called with numberOfPages == 0" << endl;
    return;
  }
  if (numberOfWidgets() == 0) {
    kError(kvs::shell) << "PageView::gotoAnchor() called, but widgetList is empty" << endl;
    return;
  }

  PageNumber page = a.page;

  int y = (int)(a.distance_from_top.getLength_in_inch() * dataModel->resolution() + 0.5);
  if (!a.isHistoryLink)
  {
    dataModel->history()->add(a);
  }

  DocumentWidget* _pageWidget;

  // If we are in overview viewmode
  if (dataModel->preferences()->viewMode() == KVSPrefs::EnumViewMode::Overview) {
    unsigned int visiblePages = dataModel->preferences()->overviewModeColumns() * dataModel->preferences()->overviewModeRows();
    // Pagenumber of the first visibile Page in the current tableau
    unsigned int firstPage = pageWidget(0)->getPageNumber();
    // Pagenumber of the first page in the new tableau.
    unsigned int tableauStartPage = page + 1 - (page % visiblePages);
    // If these numbers arn't equal "page" is not in the current tableu.
    bool newTableu = false;
    if (firstPage != tableauStartPage) { // widgets need to be updated
      newTableu = true;
      if ((dataModel->numberOfPages() - tableauStartPage + 1 < visiblePages) || (numberOfWidgets() < visiblePages)) {
        // resize widgetList
        // the pages are also set correctly by "generateDocumentWidgets"
        generateDocumentWidgets(tableauStartPage);
      } else {
        // "page" is not shown in the scrollview, so we have to switch widgets.
        // Here we don't need to resize the widgetList.
        for (unsigned int i = 0; i < numberOfWidgets(); i++) {
          _pageWidget = pageWidget(i);
          if (_pageWidget != 0)
            _pageWidget->setPageNumber(tableauStartPage + i);
        }
        layoutPages();
      }
    }
    // move scrollview to "page".
    // Make the widget pageWidget visible in the scrollview. Somehow this
    // doesn't seem to trigger the signal contentsMoved in the
    // QScrollview, so that we better call setCurrentPage() ourselves.
    _pageWidget = widgetOfPageNumber(page);

    if (_pageWidget)
    {
      if (newTableu) {
        // If we have switched to a new tableu, we don't want smooth scrolling
        moveViewportToWidget(_pageWidget, y, false);
      }
      else
        moveViewportToWidget(_pageWidget, y);
    }
  } else if (numberOfWidgets() == 1) {
    // If the widget list contains only a single element, then either
    // the document contains only one page, or we are in "single page"
    // view mode. In either case, we set the page number of the single
    // widget to 'page'
    _pageWidget = pageWidget(0);

    // Paranoia security check
    if (_pageWidget == 0) {
      kError(kvs::shell) << "PageView::gotoAnchor() called with widgetList.size() == 1, but widgetList[0] == 0" << endl;
      return;
    }

    _pageWidget->setPageNumber(page);
    layoutPages();
    moveViewportToWidget(_pageWidget, y, false);
  } else {
    // There are multiple widgets, then we are either in the
    // "Continuous" or in the "Continouous-Facing" view mode. In that
    // case, we find the widget which is supposed to display page
    // 'page' and move the scrollview to make it visible

    // Paranoia security checks
    if (numberOfWidgets() < page) {
      kError(kvs::shell) << "PageView::gotoAnchor() called with widgetList.size()=" << numberOfWidgets() << ", and page=" << page << endl;
      return;
    }
    _pageWidget = widgetOfPageNumber(page);
    if (_pageWidget == 0) {
      kError(kvs::shell) << "PageView::gotoAnchor() called with widgetList.size() > 1, but widgetList[page] == 0" << endl;
      return;
    }

    moveViewportToWidget(_pageWidget, y);
  }

  if (a.isLinkTarget && y != 0 && _pageWidget )
    _pageWidget->flash(y);

  return;
}


void PageView::prevPage()
{
  PageNumber np = 1;
  if (nrCols*nrRows < dataModel->currentPageNumber())
  {
    np = dataModel->currentPageNumber() - nrCols*nrRows;
  }

  dataModel->setCurrentPageNumber(np);
}


void PageView::nextPage()
{
  PageNumber np = qMin(dataModel->currentPageNumber() + nrCols*nrRows, dataModel->numberOfPages());
  dataModel->setCurrentPageNumber(np);
}


void PageView::firstPage()
{
  dataModel->setCurrentPageNumber(Anchor(1));
}


void PageView::lastPage()
{
  dataModel->setCurrentPageNumber(Anchor(dataModel->numberOfPages()));
}


void PageView::readDown()
{
  if (atBottom())
  {
    if (continuousViewmode)
      return;

    if (dataModel->currentPageNumber() == dataModel->numberOfPages())
      return;

    nextPage();
    setContentsPos(contentsX(), 0);
  }
  else
  {
    // Coordinate of the bottom of the viewport
    int bottom = contentsY() + visibleHeight();

    DocumentWidget* widget = 0;
    // Find the widget(s) that intersect the bottom of the viewport
    // TODO: It would be better to use a binary search.
    for(int i=0; i<widgetList.size(); i++)
    {
      widget = widgetList[i];
      if (childY(widget) < bottom && childY(widget) + widget->height() > bottom)
      {
        // Draw scrollguide
        widget->drawScrollGuide(bottom - childY(widget));
      }
    }

    int newValue = qMin( verticalScrollBar()->value() + (int)(height() * 0.9),
                         verticalScrollBar()->maximum() );
    setContentsPosSmooth(contentsX(), newValue);
  }
}


void PageView::readUp()
{
  if (atTop())
  {
    if (continuousViewmode)
      return;

    if (dataModel->currentPageNumber() == 1)
      return;

    prevPage();
    setContentsPos(contentsX(), contentsHeight());
  }
  else
  {
    // Coordinate of the top of the viewport
    int top = contentsY();

    DocumentWidget* widget = 0;
    // Find the widget(s) that intersect the top of the viewport
    // TODO: It would be better to use a binary search.
    for(quint16 i=0; i<widgetList.size(); i++)
    {
      widget = widgetList.at(i);
      if (childY(widget) < top && childY(widget) + widget->height() > top)
      {
        // Draw scrollguide
        widget->drawScrollGuide(top - childY(widget));
      }
    }

    int newValue = qMax( verticalScrollBar()->value() - (int)(height() * 0.9),
                         verticalScrollBar()->minimum() );
    setContentsPosSmooth(contentsX(), newValue);
  }
}


void PageView::goBack()
{
  Anchor* anchor = dataModel->history()->back();
  if (anchor)
  {
    dataModel->setCurrentPageNumber(*anchor); // Do not add a history item.
  }
  else
    kDebug(kvs::shell) << "Faulty return -- bad history buffer" << endl;
}


void PageView::goForward()
{
  Anchor* anchor = dataModel->history()->forward();
  if (anchor)
  {
    dataModel->setCurrentPageNumber(*anchor); // Do not add a history item.
  }
  else
    kDebug(kvs::shell) << "Faulty return -- bad history buffer" << endl;
}


DocumentWidget* PageView::pageWidget(unsigned int i)
{
  if (i >= (unsigned int)widgetList.size())
    return 0;
  else
    return widgetList[i];
}


void PageView::resizeWidgets()
{
  bool everResized = false;

  // Go through the list of widgets and resize them, if necessary
  for(unsigned int i = 0; i < numberOfWidgets(); i++)
  {
    DocumentWidget* documentWidget = widgetList[i];
    if (documentWidget == 0)
      continue;

    // Resize, if necessary
    if(documentWidget->resizeWidget())
      everResized = true;
  }

  // If at least one widget was resized, all widgets should be
  // re-aligned. This will automatically update all necessary
  // widgets.
  if (everResized == true)
  {
    layoutPages(true);
  }
}


void PageView::updatePage(const PageNumber& pageNumber)
{
  DocumentWidget* pageWidget = widgetOfPageNumber(pageNumber);
  if (!pageWidget)
    return;

  if (isVisible(pageWidget))
  {
    pageWidget->update();
  }
}


void PageView::repaintPages()
{
  for (unsigned int i = 0; i < numberOfWidgets(); i++)
  {
    DocumentWidget* pageWidget = widgetList[i];
    if (isVisible(pageWidget))
      pageWidget->update();
  }
}


void PageView::scroll(int deltaInPixel)
{
  if (dataModel->numberOfPages() == 0)
  {
    kError(kvs::shell) << "PageView::scroll() called with numberOfPages == 0" << endl;
    return;
  }

  QScrollBar* scrollBar = verticalScrollBar();

  if (deltaInPixel < 0) {
    if (scrollBar->value() == scrollBar->minimum()) {
      if ( (dataModel->currentPageNumber() == 1) || (changePageDelayTimer.isActive()) )
        return;

      if (continuousViewmode)
        return;

      changePageDelayTimer.stop();
      prevPage();

      setContentsPos(contentsX(), scrollBar->maximum());
      return;
    }
  }

  if (deltaInPixel > 0) {
    if (scrollBar->value() == scrollBar->maximum()) {
      if ( (dataModel->currentPageNumber() == dataModel->numberOfPages()) || (changePageDelayTimer.isActive()) )
        return;

      if (continuousViewmode)
        return;

      changePageDelayTimer.stop();
      nextPage();

      setContentsPos(contentsX(), 0);
      return;
    }
  }

  scrollBar->setValue(scrollBar->value() + deltaInPixel);

  if ( (scrollBar->value() == scrollBar->maximum()) || (scrollBar->value() == scrollBar->minimum()) )
  {
    changePageDelayTimer.setSingleShot(true);
    changePageDelayTimer.start(200);
  }
  else
    changePageDelayTimer.stop();
}


DocumentWidget* PageView::createDocumentWidget()
{
  if (multiPage.isNull())
  {
    kError() << "PageView::createDocumentWidget() called with no ligaturePluginGUI set." << endl;
    return 0;
  }
  if (pageCache.isNull())
  {
    kError() << "PageView::createDocumentWidget() called with no DocumentPageCache set." << endl;
    return 0;
  }

  DocumentWidget* documentWidget = multiPage->createDocumentWidget(this, pageCache);
  documentWidget->setupObservers(dataModel);

  connect(documentWidget, SIGNAL(localLink(const Hyperlink&)), this, SIGNAL(localLink(const Hyperlink&)));
  connect(documentWidget, SIGNAL(setStatusBarText(const QString&)), this, SIGNAL(setStatusBarText(const QString&)));
  connect(documentWidget, SIGNAL(clearSelection()), this, SIGNAL(clearSelection()));
  connect(this, SIGNAL(switchTool()), documentWidget, SLOT(slotSwitchTool()));

  return documentWidget;
}


DocumentWidget* PageView::widgetOfPageNumber(const PageNumber& pageNumber)
{
  for (unsigned int i = 0; i < numberOfWidgets(); i++)
  {
    DocumentWidget* pageWidget = widgetList[i];
    if (pageWidget->getPageNumber() == pageNumber)
    {
      return pageWidget;
    }
  }

  return 0;
}


void PageView::slotSwitchTool()
{
  emit switchTool();
}


void PageView::setupObservers(DataModel* _dataModel)
{
  SmoothScrollView::setupObservers(_dataModel);
  connect(dataModel, SIGNAL(GUIGotoAnchor(Anchor)), this, SLOT(gotoAnchor(Anchor)));
  connect(dataModel, SIGNAL(numberOfPagesChanged()), this, SLOT(generateDocumentWidgets()));
  connect(dataModel, SIGNAL(layoutChanged()), this, SLOT(resizeWidgets()));
}


void PageView::gotoCurrentPage()
{
  dataModel->setCurrentPageNumber(Anchor(dataModel->currentPageNumber()));
}

#include "pageView.moc"
