// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/* This file is part of the KDE project
   Copyright (C) 2001 Wilco Greven <greven@kde.org>
   Copyright (C) 2004-2006 Wilfried Huss <Wilfried.Huss@gmx.at>
   Copyright (C) 2005 Stefan Kebekus <kebekus@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef PAGEVIEW_H
#define PAGEVIEW_H

#include "ligature_export.h"

#include "dataView.h"
#include "documentWidget.h"
#include "selection.h"
#include "smoothScrollView.h"

#include <QPointer>

class Anchor;
class DocumentPageCache;
class ligaturePluginGUI;
class PageNumber;

/*
 * PageView is a customized QScrollView, which can hold one
 * page. This page will be centered on the viewport.
 */
class LIGATURECORE_EXPORT PageView : public SmoothScrollView
{
    Q_OBJECT

public:
  PageView(QWidget* parent = 0, const char* name = 0);
  ~PageView() {}
  
  void setPageCache(DocumentPageCache*);
  void setMultiPage(ligaturePluginGUI*);
  
  /** Make the selection visible */
  void gotoSelection(const TextSelection&);

  /** removes all widgets from the PageView
      
  This method adds the widgets contained in wdgList to the
  PageView. The widgets are positioned on the PageView according to
  the size at the time when they were added. If the widget size
  changes later, the widgets should be re-added.
  
  The PageView does not assume ownership of the widgets in the
  list. When the method addChild is called another time, the PageView
  simply forgets about the widgets.
  
  @warning The PageView may access the widgets at any time. Thus, you
  must call either addChild() or clear() before deleting any of the
  widgets added here.
  */
  void clear();
  
  void setViewMode();

    /** Return true if the top resp. bottom of the page is visible. */
    bool atTop()    const;
    bool atBottom() const;

    /** Returns true if @param child is visible in the current viewport. */
    bool isVisible(QWidget* child);

    /** Returns the widgets of the current tableau, or 0 if the index @param i is out
        of range.

        @warning the parameter @param i does not correspond to the page of the same
        number, in all viewmodes. */
    DocumentWidget* pageWidget(unsigned int i);

    unsigned int numberOfWidgets() const { return widgetList.size(); }

    virtual void setupObservers(DataModel*);

    /** These methods calculate the Zoomfactor needed to fit the pages
      into the current viewport. Note that the return value need *not*
      be within the limits defined in "zoomLimits.h". If they are not,
      this indicates that fitting to width or height is currently not
      possible (e.g. because no document is loaded). The return value
      should then be ignored and any operation that relies on the
      return value should be aborted. */
    double calculateFitToWidthZoomValue() const;
    double calculateFitToHeightZoomValue() const;

public slots:
    void generateDocumentWidgets(const PageNumber& startPage = PageNumber::invalidPage);

    void readUp();
    void readDown();
    void scrollUp();
    void scrollDown();
    void scrollRight();
    void scrollLeft();
    void scrollBottom();
    void scrollTop();

    void scrollUpPage();
    void scrollDownPage();
    void scrollRightPage();
    void scrollLeftPage();

    void prevPage();
    void nextPage();
    void firstPage();
    void lastPage();

    void goBack();
    void goForward();

    void setFullScreenMode(bool fullScreen);
    /** Turn the scrollbars on/off. */
    void slotShowScrollbars(bool);

    /** Redraws the widget that shows the page @param pageNumber.

        If the page is not shown by any widget, the function does nothing. */
    void updatePage(const PageNumber& pageNumber);

    /** Redraws all widgets. */
    void repaintPages();

    void slotSwitchTool();

    /** Updates the widget sizes, and relayouts the pageview if necessary. */
    void resizeWidgets();

signals:
    void viewSizeChanged(const QSize& size);
    void pageSizeChanged(const QSize& size);

    void zoomIn();
    void zoomOut();


    /** The next three signals are relayed from the DocumentWidgets.
        TODO: find a better way to do this. */
    void localLink(const Hyperlink&);
    void setStatusBarText(const QString&);
    void clearSelection();

    /** The next signal is relayed to the DocumentWidgets.
        TODO: find a better way to do this. */
    void switchTool();

protected:
    /** Reimplemented from QScrollView to make sure that the page is
        centered when it fits in the viewport. */
    void viewportResizeEvent( QResizeEvent* );
    void viewportPaintEvent(QPaintEvent*);

    /** This implements scrolling and zooming with the mousewheel. */
    void contentsWheelEvent(QWheelEvent*);

    void contentsMousePressEvent(QMouseEvent*);
    void contentsMouseReleaseEvent(QMouseEvent*);
    void contentsMouseMoveEvent(QMouseEvent*);

private slots:
    /** \brief move the display so that the anchor is visible

    Move the display so that the anchor is visible. If a.isLinkTarget
    is true, the anchor will be visually highlighted for a moment
    */
    void gotoAnchor(Anchor a);

    void calculateCurrentPageNumber();
    void calculateCurrentPageNumber(int x, int y);
    void gotoCurrentPage();

    /** Set layout of the page widgets according to the current viewmode and zoomlevel.
        Set zoomChanged = true if the the layout needs updateing because the zoomlevel has changed. */
    void layoutPages(bool zoomChanged = false);

    /** Scrolls the main scrollview by deltaInPixel (positive values
        scroll DOWN). If the user tries to scroll past the beginning or
        the end of a page, then the method either returns without doing
        anything (if the current page is the first or last page, resp, or
        if the method is called within 200ms after the beg. or end of the
        page was reached), or goes the the next/previous page. The delay
        makes it a little easier for the user to scroll with the mouse
        wheel or the keyboard without involuntarily moving to another page. */
    void scroll(int deltaInPixel);

private:
    DocumentWidget* createDocumentWidget();

    /** @returns a pointer to the page widget which displays the page of the
        given number, or 0 if the page is not in the current tableau. */
    DocumentWidget* widgetOfPageNumber(const PageNumber&);

    /** Moves the viewport so that the widget is at the top left corner. */
    void moveViewportToWidget(QWidget* widget, int y = 0, bool smoothMove = true);

    /** Returns the number of columns into which the widgets are aligned.

    This method returns the number of colums actually used to display
    the widgets.

    @warning This method need not return the number columns set in the
    setViewMode() method. For instance, if the viewmode
    KVSPrefs::EnumViewMode::ContinuousFacing is set, but there is only
    one widget, then only one column is used, and the method returns
    the number one.

    If there aren't any widgets, the number 1 is returned.

    @returns Number of columns used, or 1 if there aren't any
    widgets. The number i returned always satisfies 1 <= i <= nrCols,
    where nrCols is the private variable of the same nane.
    */
    unsigned int getNrColumns() const { return  (widgetList.isEmpty()) ? 1 : qMin((int)nrCols, qMax(1, (int)widgetList.size())); }

    unsigned int getNrRows() const { return nrRows; }

    /** The next two functions are used for the autozoom feature
        TODO optimize (the calculation of the widest page needs to be done only once
        per document, not everytime calculateFitToWidthZoomValue() is called) */
    PageNumber widestPage() const;


    // TODO Generalize this for more than 2 columns
    double zoomForWidthColumns(unsigned int viewportWidth) const;

    /** Stores the mouse position between two mouse events. This is used
        to implement the "grab and drag the viewport contents" feature. */
    QPoint   dragGrabPos;

    QList<DocumentWidget *> widgetList;

    /** The number of columns and rows in the current layout. See @ref layoutPages() for details. */
    unsigned int nrCols;
    unsigned int nrRows;

    bool continuousViewmode;
    bool fullScreen;

    /** This int remembers the style of the frame of the centering
        scrollview when fullscreen mode is switched on. It is then
        restored when it is switched off. */
    int oldFrameStyle;

    /** Distance between pages in pixels
        (this is independent of the zoom level). */
    static const int distanceBetweenWidgets=10;

    QPointer<DocumentPageCache> pageCache;
    QPointer<ligaturePluginGUI> multiPage;

    /** This timer is used to implement a brief delay when the user
        scrolls past the beginning or the end of the page before a the
        program moves to a new page. That way, it is a little easier for
        the user to scroll with the mouse wheel or the keyboard without
        involuntarily moving to another page. The timer is used in the
        scroll() method. */
    QTimer changePageDelayTimer;
};

#endif
