// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: RenderedDocumentPage
//
// Widget for displaying TeX DVI files.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004-2006 Stefan Kebekus.
// (C) 2005-2006 Wilfried Huss.
//
// Distributed under the GPL.

#include <config.h>

#include "pagetransition.h"
#include "renderedDocumentPage.h"
#include "hyperlink.h"
#include "kvs_debug.h"
#include "selection.h"
#include "textBox.h"

#include <QRegion>


RenderedDocumentPage::RenderedDocumentPage()
  : _transition(0)
{
  textBoxList.reserve(250);
  pageNr = 0;
  isEmpty = true;
  pageText = QString::null;
  hyperlinks_are_marked = false;
}


RenderedDocumentPage::~RenderedDocumentPage()
{
  delete _transition;
}


void RenderedDocumentPage::setPageNumber(const PageNumber& pnr)
{
  pageNr = pnr;
  clear();
}


void RenderedDocumentPage::clear()
{
#ifdef DEBUG_DOCUMENTPAGE
  kDebug(kvs::shell) << "RenderedDocumentPage::clear() called for page #" << pageNumber << endl;
#endif

  textBoxList.clear();
  hyperLinkList.clear();
  pageText = QString::null;

  isEmpty = true;
}


QRegion RenderedDocumentPage::selectedRegion(const TextSelection& selection)
{
  if (selection.isEmpty() || selection.getPageNumber() != pageNr)
    return QRegion();

  int startIndex = selection.getSelectedTextStart();
  int endIndex = selection.getSelectedTextEnd();

  QVector<QRect> wordBox;

  QRect currentWordBox;

  // Merge character boxes into boxes containing complete words.
  // Note: A word in this context is defined as a string of boxes
  // with the same baseline.
  for (int i = startIndex; i <= endIndex; i++)
  {
    if (i == 0)
    {
      // start first word
      currentWordBox = textBoxList[i].box;
      //currentBaseline = textBoxList[i].baseline;
      continue;
    }

    /*if (currentBaseline == textBoxList[i].baseline)
    {
      currentWordBox = currentWordBox.unite(textBoxList[i].box);
    }
    else*/
    {
      // start next word
      wordBox.push_back(currentWordBox);
      currentWordBox = textBoxList[i].box;
      //currentBaseline = textBoxList[i].baseline;
    }
  }
  // we still need to store the last word
  wordBox.push_back(currentWordBox);

  QVector<QRect> lineBox;

  // Merge word boxes into boxes containing whole lines.
  // We start a new line if we encounter a wordbox which does not
  // vertically overlap which the current lineBox.
  QRect currentLineBox;
  
  for (int i = 0; i < wordBox.size(); i++)
  {
    if (!currentLineBox.isValid())
    {
      // start first line
      currentLineBox = wordBox[i];
      continue;
    }

    // check if there is vertical overlap
    if (wordBox[i].top() <= currentLineBox.bottom() && wordBox[i].bottom() >= currentLineBox.top())
    {
      // the word belongs to the current line
      currentLineBox = currentLineBox.unite(wordBox[i]);
    }
    else
    {
      // start next line
      //kDebug(kvs::shell) << "push line (" << currentLineBox.top() << ", " << currentLineBox.bottom() << ")" << endl;
      lineBox.push_back(currentLineBox);
      currentLineBox = wordBox[i];
    }
  }
  // we still need to store the last line
  //kDebug(kvs::shell) << "push line (" << currentLineBox.top() << ", " << currentLineBox.bottom() << ")" << endl;
  lineBox.push_back(currentLineBox);

  //kDebug(kvs::shell) << "Number of lineboxes = " << lineBox.size() << endl;

  // Now we increase the height of the lines if necessary to obtain a connected region
  // for our selection.
  for (int i = 0; i < lineBox.size() - 1; i++)
  {
      int dy = lineBox[i+1].top() - lineBox[i].bottom();
      int lineHeight = QMAX(lineBox[i].height(), lineBox[i+1].height());
      // If dy is negativ be expect a new column.
      // If dy is quite a bit bigger than the difference between the consecutive lines,
      // we expect a new paragraph. In these two cases we do not exand the lineboxes vertically.
      if (dy >= 0 && dy < 1.5 * lineHeight)
      {
        int midPoint = (lineBox[i].bottom() + lineBox[i+1].top()) / 2;

        lineBox[i].setBottom(midPoint);
        lineBox[i+1].setTop(midPoint+1);
      }
  }

  // Add the lineboxes to a Region
  QRegion selectedRegion;
  for (int i = 0; i < lineBox.size(); i++)
  {
    selectedRegion += QRegion(lineBox[i]);
  }

  return selectedRegion;
}


TextSelection RenderedDocumentPage::select(const QRect& selectedRectangle, bool normal)
{
  TextSelection selection;

  if (textBoxList.size() == 0)
    return selection;

  int selectedTextStart = -1;
  int selectedTextEnd   = -1;

  // Find the smallest and biggest index for which the corresponding
  // textBoxList entry intersects the selected rectangle.
  for (unsigned int i=0; i<textBoxList.size(); i++)
  {
    if (selectedRectangle.intersects(textBoxList[i].box))
    {
      if (selectedTextStart == -1)
        selectedTextStart = i;
      selectedTextEnd = i;
    }
  }

  if (!normal)
  {
    // If we are not in normal mode, things get a little more difficult.
    // See the diagram in selection.svg for an explanation of the desired results.

    // First we need to detact, if the selected text has one or more columns.
    bool multiColumn = false;
    int rightMargin = 0;
    for (unsigned int i = selectedTextStart; i < selectedTextEnd - 1; i++)
    {
      if (rightMargin < textBoxList[i].box.right())
        rightMargin = textBoxList[i].box.right();

      if (textBoxList[i+1].box.bottom() < textBoxList[i].box.top() && textBoxList[i+1].box.left() > rightMargin)
      {
        multiColumn = true;
        break;
      }
    }

    selectedTextStart = -1;
    selectedTextEnd   = -1;
      
    if (multiColumn)
    {
      // The start of the selection, is the texBox that is closest
      // to the bottom left point of the selectedRectangle
      int minLength = -1;
      for (unsigned int i = 0; i < textBoxList.size(); i++)
      {
        if (selectedRectangle.intersects(textBoxList[i].box))
        {
          QRect temp = textBoxList[i].box;
          QPoint midPoint(temp.x() + temp.width()/2, temp.y() + temp.height()/2);
          QPoint bottomLeft(selectedRectangle.x(), selectedRectangle.bottom());
          QPoint difference = midPoint - bottomLeft;
          int length = difference.manhattanLength();

          if (minLength == -1 || minLength > length)
          {
            selectedTextStart = i;
            minLength = length;
          }
        }
      }

      // We only need to search for the endpoint of the selection,
      // if we have found a starting point.
      if (selectedTextStart != -1)
      {
        // The end of the selection, is the texBox that is closest
        // to the top right point of the selectedRectangle
        minLength = -1;
        for (unsigned int i = selectedTextStart; i < textBoxList.size(); i++)
        {
          if (selectedRectangle.intersects(textBoxList[i].box))
          {
            QRect temp = textBoxList[i].box;
            QPoint midPoint(temp.x() + temp.width()/2, temp.y() + temp.height()/2);
            QPoint topRight(selectedRectangle.right(), selectedRectangle.top());
            QPoint difference = midPoint - topRight;
            int length = difference.manhattanLength();
  
            if (minLength == -1 || minLength > length)
            {
              selectedTextEnd = i;
              minLength = length;
            } 
          }
        }
      }
    }
    if (!multiColumn)
    {
      bool startSelection = false;
      for (unsigned int i = 0; i < textBoxList.size(); i++)
      {
        if (selectedRectangle.intersects(textBoxList[i].box))
        {
          if (selectedTextStart == -1)
            startSelection = true;
  
          if (startSelection)
            selectedTextStart = i;
        }
        else
        {
          startSelection = false;
        }
      }
  
      bool endSelection = false;
      for (int i = int(textBoxList.size()) - 1; i >= 0; i--)
      {
        if (selectedRectangle.intersects(textBoxList[i].box))
        {
          if (selectedTextEnd == -1)
            endSelection = true;
  
          if (endSelection)
            selectedTextEnd = i;
        }
        else
        {
          endSelection = false;
        }
      }
    }
  }
  QString selectedText;

  if (selectedTextStart != -1)
  {
    for (int i = selectedTextStart; (i <= selectedTextEnd) && (i < (int)textBoxList.size()); i++)
    {
      selectedText += textBoxList[i].text;
    }
    selection.set(pageNr, selectedTextStart, selectedTextEnd, selectedText);
    return selection;
  }
  // return empty selection
  return selection;
}

TextSelection RenderedDocumentPage::select(const QPoint& point)
{
  int selectedTextStart = -1;
  int selectedTextEnd   = -1;

  for (int i=0; i<textBoxList.size(); i++)
  {
    if (textBoxList[i].box.contains(point))
    {
      selectedTextStart = i;
      selectedTextEnd = i;
      break;
    }
  }

  TextSelection selection;

  QString selectedText;

  if (selectedTextStart != -1)
  {
    selectedText = textBoxList[selectedTextStart].text;
    selection.set(pageNr, selectedTextStart, selectedTextEnd, selectedText);
    return selection;
  }
  // return empty selection
  return selection;
}

TextSelection RenderedDocumentPage::find(const QString& str, int index, bool caseSensitive)
{
  if (pageText.isNull())
  {
    // Create the pageText by joining all entries of textBoxList.
    for (QVector<TextBox>::Iterator i = textBoxList.begin(); i != textBoxList.end(); i++)
    {
      pageText = pageText + i->text;
    }
  }

  // Create empty selection;
  TextSelection selection;

  // If the page contains no searchable text
  if (pageText.isNull())
    return selection;

  // Compute the corresponding pageText index
  unsigned int subIndex = 0;
  for (int i = 0; i < index; i++)
  {
    subIndex += textBoxList[i].text.length();
  }


  int textIndex;
  if (caseSensitive)
    textIndex = pageText.indexOf(str, subIndex, Qt::CaseSensitive);
  else
    textIndex = pageText.indexOf(str, subIndex, Qt::CaseInsensitive);

  if (textIndex == -1)
    return selection;

  // Because a single Hyperlink structure can possible store more then
  // one character we now have to search for the Indices in the
  // textBoxList Vector which corresponds to the found index in the
  // String pageText.  FIXME: It would be faster to search directly in
  // the textBoxList.
  int counter = 0;
  int startIndex = 0;
  while (counter < textIndex)
  {
    counter += textBoxList[startIndex].text.length();
    // If the string we searched for starts in the middle of some text element we better return a
    // selection that it somewhat bigger.
    if (counter > textIndex)
      break;

    startIndex++;

    // safety check
    if (startIndex >= (int)textBoxList.size())
      return selection;
  }

  // Search for the end index.
  // TODO: This algorithm is not entirely correct if str does not start exactly at the beginning of some text element.
  counter = 0;
  int endIndex = startIndex;
  while (counter < (int)str.length())
  {
    counter += textBoxList[endIndex].text.length();
    if (counter >= (int)str.length())
      break;

    endIndex++;

    // safety check
    if (endIndex >= (int)textBoxList.size())
      return selection;
  }

  // Set the selection
  selection.set(pageNr, startIndex, endIndex, str);
  return selection;
}


QString RenderedDocumentPage::rectangleSelect(const QRect& selectedRect) const
{
  QString selectedText;

  for (int i=0; i<textBoxList.size(); i++)
  {
    if (textBoxList[i].box.intersects(selectedRect))
    {
      selectedText += textBoxList[i].text;
    }
  }
  return selectedText;
}


TextSelection RenderedDocumentPage::findRev(const QString& str, int index, bool caseSensitive)
{
  // Negative index means we start the search at the end of the text.
  if (index < 0)
  {
    index = textBoxList.size();
  }

  if (pageText.isNull())
  {
    // Create the pageText by joining all entries of textBoxList.
    for (QVector<TextBox>::Iterator i = textBoxList.begin(); i != textBoxList.end(); i++)
    {
      pageText = pageText + i->text;
    }
  }

  // Create empty selection;
  TextSelection selection;

  // If the page contains no searchable text
  if (pageText.isNull())
    return selection;

  // Compute the corresponding pageText index
  unsigned int subIndex = 0;
  for (int i = 0; i < index; i++)
  {
    subIndex += textBoxList[i].text.length();
  }

  int textIndex;
  if (caseSensitive)
    textIndex = pageText.lastIndexOf(str, subIndex, Qt::CaseSensitive);
  else
    textIndex = pageText.lastIndexOf(str, subIndex, Qt::CaseInsensitive);

  if (textIndex == -1)
    return selection;

  // Because a single Hyperlink structure can possible store more then
  // one character we now have to search for the Indices in the
  // textBoxList Vector which corresponds to the found index in the
  // String pageText.  FIXME: It would be faster to search directly in
  // the textBoxList.
  int counter = 0;
  int startIndex = 0;
  while (counter < textIndex)
  {
    counter += textBoxList[startIndex].text.length();
    // If the string we searched for starts in the middle of some text element we better return a
    // selection that it somewhat bigger.
    if (counter > textIndex)
      break;

    startIndex++;

    // safety check
    if (startIndex >= (int)textBoxList.size())
      return selection;
  }

  // Search for the end index.
  // TODO: This algorithm is not entirely correct if str does not start exactly at the beginning of some text element.
  counter = 0;
  int endIndex = startIndex;
  while (counter < (int)str.length())
  {
    counter += textBoxList[endIndex].text.length();
    if (counter >= (int)str.length())
      break;

    endIndex++;

    // safety check
    if (endIndex >= (int)textBoxList.size())
      return selection;
  }

  // Set the selection
  selection.set(pageNr, startIndex, endIndex, str);
  return selection;
}


KPDFPageTransition* RenderedDocumentPage::transition()
{
  return _transition;
}


void RenderedDocumentPage::setTransition(KPDFPageTransition* pageTransition)
{
  _transition = pageTransition;
}

#include "renderedDocumentPage.moc"

