// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2004-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <config.h>

#include "smoothScrollView.h"
#include "kvs_debug.h"
#include "kvsprefs.h"

#include <cmath>

SmoothScrollView::SmoothScrollView(QWidget* parent, const char* name, Qt::WFlags f)
  : Q3ScrollView(parent, name, f), viewportMoveActive(false), viewportMoveTimer(0)
{
  connect(&viewportMoveTimer, SIGNAL(timeout()), this, SLOT(slotMoveViewport()));
  viewportMoveTime.start();
}


SmoothScrollView::~SmoothScrollView()
{
}


void SmoothScrollView::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);
}


bool SmoothScrollView::isSmoothScrollDistance(double scrollDistance)
{
  double viewportDiagonal = sqrt((double)(visibleHeight()) * (double)(visibleHeight()) +
                                 (double)(visibleWidth()) * (double)(visibleWidth()));
  return (scrollDistance < 3 * viewportDiagonal);
}


void SmoothScrollView::setContentsPosSmooth(int x, int y)
{
  // Measure the time between the last call of this function.
  int timeBetweenFunctionCalls = viewportMoveTime.elapsed();
  viewportMoveTime.start();

  double scrollDistance = sqrt((double)(contentsY() - y) * (double)(contentsY() - y) +
                               (double)(contentsX() - x) * (double)(contentsX() - x));

  // Only use smooth scrolling if we move the viewport a moderate distance.
  if (dataModel->preferences()->smoothScrolling() && isSmoothScrollDistance(scrollDistance))
  {
    // If a smooth scrolling operation is active, or the time between two invocations of this
    // function is small, we cancel the smooth scroll and jump to the requested position.
    if (viewportMoveActive || timeBetweenFunctionCalls <= 667)
    {
      viewportMoveTime.start();
      viewportMoveTimer.stop();
      viewportMoveActive = false;
      verticalScrollBar()->setEnabled(true);
      horizontalScrollBar()->setEnabled(true);
      setContentsPos(x, y);
      return;
    }

    viewportMoveActive = false;
    emit contentsMoving(x, y);
    viewportMoveActive = true;
    viewportMoveTime.start();
    viewportMoveDest.setX(x);
    viewportMoveDest.setY(y);

    viewportMoveTimer.start(25);
    verticalScrollBar()->setEnabled(false);
    horizontalScrollBar()->setEnabled(false);
  }
  else
  {
    setContentsPos(x, y);
  }
}


void SmoothScrollView::slotMoveViewport()
{
  // converge to viewportMoveDest in 1 second
  int diffTime = viewportMoveTime.elapsed();
  if (diffTime >= 667 || !viewportMoveActive)
  {
    setContentsPos(viewportMoveDest.x(), viewportMoveDest.y());
    viewportMoveTimer.stop();
    viewportMoveActive = false;
    verticalScrollBar()->setEnabled(true);
    horizontalScrollBar()->setEnabled(true);
    return;
  }

  // move the viewport smoothly (kmplot: p(x)=x+x*(1-x)*(1-x))
  double convergeSpeed = (double)diffTime / 667.0;
  int x = contentsX();
  int y = contentsY();
  int diffX = viewportMoveDest.x() - x;
  int diffY = viewportMoveDest.y() - y;

  convergeSpeed *= convergeSpeed * (1.4 - convergeSpeed);

  setContentsPos((int)(x + diffX * convergeSpeed), (int)(y + diffY * convergeSpeed));
}

#include "smoothScrollView.moc"
