// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2005 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   Copyright (C) 2006 by Stefan Kebekus                                  *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <config.h>

#include "tableOfContents.h"
#include "bookmark.h"
#include "kvs_debug.h"
#include "kvsprefs.h"

#include <kiconloader.h>
#include <klocale.h>

#include <Q3PtrList>


TocItem::TocItem(TocItem* parent)
  : K3ListViewItem(parent)
{
}

TocItem::TocItem(Q3ListView* parent)
  : K3ListViewItem(parent)
{
}


TableOfContents::TableOfContents(QWidget* parent)
  : K3ListView(parent),
    DataView(),
    currentChapter(0)
{
  addColumn(i18n("Topic"));
  addColumn(i18n("Page"));

  setSorting(-1);
  setRootIsDecorated(true);
  setSelectionMode(Q3ListView::NoSelection);
  setResizeMode(AllColumns);
  setColumnWidthMode(0, Manual);
  setColumnWidthMode(1, Manual);
  setFullWidth(true);

  connect(this, SIGNAL(executed(Q3ListViewItem*)), this, SLOT(itemClicked(Q3ListViewItem*)));

  connect(this, SIGNAL(expanded(Q3ListViewItem*)), this, SLOT(computeCurrentChapter()));
  connect(this, SIGNAL(collapsed(Q3ListViewItem*)), this, SLOT(computeCurrentChapter()));

  KIconLoader* iconLoader = KIconLoader::global();
  currentChapterIcon = iconLoader->loadIcon("1rightarrow", K3Icon::Small, K3Icon::SizeSmall);
}

TableOfContents::~TableOfContents()
{
  writeSettings();
}

void TableOfContents::writeSettings()
{
  if (dataModel.isNull()) {
    kError(kvs::shell) << "TableOfContents::readSettings() called when dataModel == 0" << endl;
    return;
  }

  saveLayout(dataModel->preferences()->config(), "tocLayout");
}

void TableOfContents::readSettings()
{
  if (dataModel.isNull()) {
    kError(kvs::shell) << "TableOfContents::readSettings() called when dataModel == 0" << endl;
    return;
  }

  restoreLayout(dataModel->preferences()->config(), "tocLayout");
}

void TableOfContents::clear()
{
  currentChapter = 0;
  K3ListView::clear();
}

void TableOfContents::setContents(const Q3PtrList<Bookmark>& bookmarks)
{
  clear();
  addItems(bookmarks);
}

void TableOfContents::addItems(const Q3PtrList<Bookmark>& _bookmarks, TocItem* parent)
{
  kDebug(kvs::shell) << "TableOfContents::setContents()" << endl;
  if (_bookmarks.isEmpty())
    return;
  kDebug(kvs::shell) << "Bookmarks are not empty" << endl;

  // Why isn't QPtrList const-correct?
  Q3PtrList<Bookmark> bookmarks = _bookmarks;
  for (Bookmark* current = bookmarks.last(); current; current = bookmarks.prev() ) {
    TocItem* item;
    if (!parent)
      item = new TocItem(this);
    else
      item = new TocItem(parent);

    item->setText(0, current->bookmarkText);
    if (current->position.page != 0)
      item->setText(1, QString().setNum(current->position.page));
    else
      item->setText(1, "--");
    item->setAnchor(current->position);

    if (!current->subordinateBookmarks.isEmpty())
      addItems(current->subordinateBookmarks, item);
  }
 }

void TableOfContents::itemClicked(Q3ListViewItem* _item)
{
  TocItem* item = static_cast<TocItem*>(_item);

  Anchor destination = item->getAnchor();
  dataModel->setCurrentPageNumber(destination);
}


void TableOfContents::setCurrentPage()
{
  if (dataModel.isNull()) {
    kError(kvs::shell) << "TableOfContents::setCurrentPage() called when dataModel == 0" << endl;
    return;
  }

  currentPageNumber = dataModel->currentPageNumber();

  TocItem* item = static_cast<TocItem*>(firstChild());
  TocItem* oldChapter = currentChapter;
  currentChapter = 0;

  while (item)
  {
    if (item->getAnchor().page <= currentPageNumber)
    {
      currentChapter = item;
    }
    else
    {
      // The items are sorted, so we don't need to check any further.
      break;
    }
    item = static_cast<TocItem*>(item->itemBelow());
  }

  if (oldChapter != currentChapter)
  {
    if (oldChapter)
    {
      oldChapter->setPixmap(0, QPixmap());
    }
    if (currentChapter)
    {
      currentChapter->setPixmap(0, currentChapterIcon);
    }
  }
}


void TableOfContents::computeCurrentChapter()
{
  setCurrentPage();
}


void TableOfContents::setupObservers(DataModel* _dataModel)
{
  if (_dataModel == 0) {
    kError(kvs::shell) << "TableOfContents::setupObservers() called when dataModel == 0)" << endl;
    return;
  }

  DataView::setupObservers(_dataModel);
  connect(dataModel, SIGNAL(currentPageNumberChanged()), this, SLOT(setCurrentPage()));

  readSettings();
}

#include "tableOfContents.moc"
