/* This file is part of the KDE libraries
    Copyright (C) 1999 Reginald Stadlbauer <reggie@kde.org>
              (C) 1999 Simon Hausmann <hausmann@kde.org>
              (C) 2000 Nicolas Hadacek <haadcek@kde.org>
              (C) 2000 Kurt Granroth <granroth@kde.org>
              (C) 2000 Michael Koch <koch@kde.org>
              (C) 2001 Holger Freyther <freyther@kde.org>
              (C) 2002 Ellis Whitehead <ellis@kde.org>
              (C) 2005-2006 Hamish Rodda <rodda@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KACTIONCOLLECTION_H
#define KACTIONCOLLECTION_H

#include <qobject.h>
#include <kstandardaction.h>
#include <kdelibs_export.h>
#include <kcomponentdata.h>

class QAction;
class KConfigBase;
class KXMLGUIClient;

class QActionGroup;
class QString;

/**
 * \short A container for a set of QAction objects.
 *
 * KActionCollection acts as the owning QObject for a set of QAction objects.  It
 * allows them to be grouped for organized presentation of configuration to the user,
 * saving + loading of configuration, and optionally for automatic plugging into
 * specified widget(s).
 *
 * Additionally, KActionCollection provides several convenience functions for locating
 * named actions, and actions grouped by QActionGroup.
 */
class KDEUI_EXPORT KActionCollection : public QObject
{
  friend class KXMLGUIClient;

  Q_OBJECT

  Q_PROPERTY( bool enabled READ isEnabled WRITE setEnabled RESET forgetEnabled )
  Q_PROPERTY( QString configGroup READ configGroup WRITE setConfigGroup )
  Q_PROPERTY( bool configIsGlobal READ configIsGlobal WRITE setConfigGlobal )

public:
  /**
   * Constructor.  Allows specification of a KComponentData other than the default
   * global KComponentData, where needed.
   */
  explicit KActionCollection(QObject *parent, const KComponentData &cData = KComponentData());

  /**
   * Destructor.
   */
  virtual ~KActionCollection();

  /**
   * Access the list of all action collections in existence for this app
   */
  static const QList<KActionCollection*>& allCollections();

  /**
   * Clears the entire action collection, deleting all actions.
   */
  void clear();

  /**
   * Returns whether this action collection as a whole has been disabled via
   * setEnabled() or not.
   *
   * \note this does not check each action to see if its state has been changed
   *       elsewhere.
   */
  bool isEnabled() const;

  /**
   * Allows you to enable or disable all actions in this action collection.
   * The state is remembered, and returned in isEnabled().
   */
  void setEnabled(bool enable);

  /**
   * Forgets the enabled state of the action collection.  Actions added
   * to the collection will not have their enabled state changed, unless there
   * is a subsequent call to setEnabled().
   */
  void forgetEnabled();

  /**
   * This sets the default shortcut context for new actions created in this
   * collection.  The default is not to assign a default context, and this
   * can be restored by passing -1.
   *
   * If this collection has an associated widget, and there is no default
   * shortcut context specified, all actions' contexts are
   * automatically changed to Qt::WidgetShortcut.
   */
  void setDefaultShortcutContext(Qt::ShortcutContext context);

  /**
   * Retrieves the default shortcut context for new actions created in this
   * collection.  The default is -1, ie. do not assign a default context to
   * added actions.
   *
   * If this collection has an associated widget, and there is no default
   * shortcut context specified, all actions will be changed to
   * Qt::WidgetShortcut.  However, this function will not return Qt::WidgetShortcut
   * when no widget is associated with this action collection and no default has
   * been set.
   */
  Qt::ShortcutContext defaultShortcutContext() const;

  /**
   * Applies the current default context (if there is one set) to all of the
   * actions in this action collection.
   */
  void applyDefaultShortcutContext();

  /**
   * Set an associated widget (clears any others).  Associated widgets automatically have all actions
   * in the action collection added to themselves.
   *
   * Shortcut context will automatically be set to Qt::WidgetShortcut, if no defaultShortcutContext() has been set.
   *
   * \sa addAssociatedWidget(), removeAssociatedWidget(), clearAssociatedWidgets() and associatedWidgets().
   */
  void setAssociatedWidget(QWidget* widget);

  /**
   * Add an associated widget.  Associated widgets automatically have all actions
   * in the action collection added to themselves.
   *
   * Shortcut context will automatically be set to Qt::WidgetShortcut, if no defaultShortcutContext() has been set.
   *
   * \sa setAssociatedWidget(), removeAssociatedWidget(), clearAssociatedWidgets() and associatedWidgets().
   */
  void addAssociatedWidget(QWidget* widget);

  /**
   * Remove an associated widget.  Removes all actions in this collection from
   * the removed associated widget.
   *
   * Shortcut context will not be reverted from Qt::WidgetShortcut, which would have been
   * assigned if no defaultShortcutContext() was set for this action collection.
   *
   * \sa addAssociatedWidget(), setAssociatedWidget(), clearAssociatedWidgets(), and associatedWidgets().
   */
  void removeAssociatedWidget(QWidget* widget);

  /**
   * Clears all associated widgets.  All actions in this collection will be removed
   * from associated widgets.
   *
   * Shortcut contexts for all actions will automatically be reverted from Qt::WidgetShortcut,
   * to the defaultShortcutContext() for this action collection.
   *
   * \sa addAssociatedWidget(), setAssociatedWidget(), removeAssociatedWidget(), and associatedWidgets().
   */
  void clearAssociatedWidgets();

  /**
   * Returns a list of widgets currently associated with this action collection.
   *
   * Associations are created to enable custom widgets to provide keyboard interactivity
   * via KActions without having to use QWidget::grabShortcut().  An example of its use
   * is katepart, which creates actions for each editor command and then sets its view
   * as an associated widget.
   *
   * \sa addAssociatedWidget(), setAssociatedWidget(), removeAssociatedWidget(), and clearAssociatedWidgets().
   */
  const QList<QWidget*>& associatedWidgets() const;

  /**
   * Returns the KConfig group with which settings will be loaded and saved.
   */
  const QString& configGroup() const;

  /**
   * Returns whether this action collection's configuration should be global to KDE ( \e true ),
   * or specific to the application ( \e false ).
   */
  bool configIsGlobal() const;

  /**
   * Sets \a group as the KConfig group with which settings will be loaded and saved.
   */
  void setConfigGroup( const QString& group );

  /**
   * Set whether this action collection's configuration should be global to KDE ( \e true ),
   * or specific to the application ( \e false ).
   */
  void setConfigGlobal( bool global );

  /**
    * Read all key associations from @p config, or (if @p config
    * is zero) from the application's configuration file
    * KGlobal::config().
    *
    * The group in which the configuration is stored can be
    * set with setConfigGroup().
    */
  void readSettings( KConfigBase* config = 0 );

  /**
    * Write the current configurable key associations to @p config,
    * or (if @p config is zero) to the application's
    * configuration file.
    *
    * \param config Config object to save to, or null to use the application's config object.
    * \param writeDefaults set to true to write settings which are already at defaults.
    * \param oneAction pass an action here if you just want to save the values for one action, eg.
    *                  if you know that action is the only one which has changed.
    */
  void writeSettings( KConfigBase* config = 0, bool writeDefaults = false, QAction* oneAction = 0L ) const;

  /**
   * Doc/View model.  This lets you add the action collection of a document
   * to a view's action collection.
   */
  void addDocCollection( KActionCollection* pDoc );

  /**
   * Returns the number of actions in the collection.
   *
   * This is equivalent to actions().count().
   */
  int count() const;

  /**
   * Returns whether the action collection is empty or not.
   */
  inline bool isEmpty() const { return count() == 0; }

  /**
   * Return the QAction* at position "index" in the action collection.
   *
   * This is equivalent to actions().value(index);
   */
  QAction *action(int index) const;

  /**
   * Find the first action with a given \a name in the action collection.
   *
   * @param name Name of the KAction, or null to match all actions
   * @return A pointer to the first KAction in the collection which matches the parameters or
   * null if nothing matches.
   */
  QAction* action( const QString& name ) const;

  /**
   * Returns the list of KActions which belong to this action collection.
   */
  QList<QAction*> actions() const;

  /**
   * Returns the list of KActions without an QAction::actionGroup() which belong to this action collection.
   */
  const QList<QAction*> actionsWithoutGroup() const;

  /**
   * Returns the list of all QActionGroups associated with actions in this action collection.
   */
  const QList<QActionGroup*> actionGroups() const;

  /**
   * Set the \a componentData associated with this action collection.
   *
   * \param componentData the KComponentData which is to be associated with this action collection, or to null to indicate the default KComponentData.
   */
  void setComponentData(const KComponentData &componentData);

  /** The KComponentData with which this class is associated. */
  const KComponentData &componentData() const;

  /**
   * The parent KXMLGUIClient, or null if not available.
   */
  const KXMLGUIClient *parentGUIClient() const;

Q_SIGNALS:
  /**
   * Indicates that \a action was inserted into this action collection.
   */
  void inserted( QAction* action );

  /**
   * Indicates that \a action was removed from this action collection.
   */
  void removed( QAction* action );

  /**
   * Indicates that \a action was highlighted
   */
  void actionHighlighted(QAction* action);

  /**
   * Indicates that \a action was triggered
   */
  void actionTriggered(QAction* action);

protected:
  /// Overridden to perform connections when someone wants to know whether an action was highlighted or triggered
  virtual void connectNotify ( const char * signal );

  virtual void slotActionTriggered();
  virtual void slotActionHighlighted();

public:
  /**
   * Add an action under the given name to the collection.
   *
   * Inserting an action that was previously inserted under a different name will replace the
   * old entry, i.e. the action will not be available under the old name anymore but only under
   * the new one.
   *
   * Inserting an action under a name that is already used for another action will replace
   * the other action in the collection.
   *
   * @param name The name by which the action be retrieved again from the collection.
   * @param action The action to add.
   */
  QAction *addAction(const QString &name, QAction *action);

  /**
   * Removes an action from the collection and deletes it.
   * @param action The action to remove.
   */
  void removeAction(QAction *action);

  /**
   * Removes an action from the collection.
   * @param action the action to remove.
   */
  QAction* takeAction(QAction *action);

  /**
   * Creates a new standard action, adds it to the collection and connects the action's triggered() signal to the
   * specified receiver/member. The newly created action is also returned.
   *
   * The action can be retrieved later from the collection by its standard name as per
   * KStandardAction::stdName.
   */
  QAction *addAction(KStandardAction::StandardAction actionType, const QObject *receiver = 0, const char *member = 0);
  /**
   * Creates a new standard action, adds to the collection under the given name and connects the action's triggered() signal to the
   * specified receiver/member. The newly created action is also returned.
   *
   * The action can be retrieved later from the collection by the specified name.
   */
  QAction *addAction(KStandardAction::StandardAction actionType, const QString &name,
                     const QObject *receiver = 0, const char *member = 0);

  /**
   * Creates a new action under the given name to the collection and connects the action's triggered()
   * signal to the specified receiver/member. The newly created action is returned.
   *
   * Inserting an action that was previously inserted under a different name will replace the
   * old entry, i.e. the action will not be available under the old name anymore but only under
   * the new one.
   *
   * Inserting an action under a name that is already used for another action will replace
   * the other action in the collection.
   *
   * @param name The name by which the action be retrieved again from the collection.
   * @param action The action to add.
   */
  QAction *addAction(const QString &name, const QObject *receiver = 0, const char *member = 0);

  /**
   * Creates a new action under the given name, adds it to the collection and connects the action's triggered()
   * signal to the specified receiver/member. The type of the action is specified by the template
   * parameter ActionType.
   */
  template<class ActionType>
  ActionType *add(const QString &name, const QObject *receiver = 0, const char *member = 0)
  {
      ActionType *a = new ActionType(this);
      if (receiver && member)
          connect(a, SIGNAL(triggered()), receiver, member);
      addAction(name, a);
      return a;
  }

private:
  Q_PRIVATE_SLOT(d, void _k_widgetDestroyed(QObject *))
  Q_PRIVATE_SLOT(d, void _k_actionDestroyed(QObject *))

  KActionCollection( const KXMLGUIClient* parent ); // used by KXMLGUIClient

  class KActionCollectionPrivate* const d;

  static QList<KActionCollection*> s_allCollections;
};

#endif
