/* This file is part of Noatun

  Copyright 2000-2001 Neil Stevens <neil@qualityassistant.com>
  Copyright 2000-2003 Charles Samuels <charles@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "pluginmodule.h"
#include "ui_pluginmodulewidget.h"

#include "noatun/global.h"
#include "noatun/plugin.h"
#include "noatun/pluginloader.h"

#include <qstyle.h>
#include <qstyleoption.h>
#include <qpainter.h>
#include <qfont.h>

#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kplugininfo.h>

using namespace Noatun;


PluginListItem::PluginListItem(const KPluginInfo *info, bool isRadio, Q3ListView *parent)
	: Q3ListViewItem(parent, QString::null, info->name(), info->comment())
{
	mPluginName = info->pluginName();
	mIsRadio = isRadio;
	mPluginEnabled = !info->isPluginEnabled();
	setPluginEnabled(info->isPluginEnabled());
}

bool PluginListItem::pluginEnabled() const
{
	return mPluginEnabled;
}

void PluginListItem::setPluginEnabled(bool val)
{
	if (val != mPluginEnabled)
	{
		kDebug(66666) << k_funcinfo <<
			"Plugin " << mPluginName << " is now " << (val ? "enabled" : "disabled") << endl;
		mPluginEnabled = val;
		repaint();
	}
}

QString PluginListItem::pluginCaption() const
{
	return text(1);
}

const QString &PluginListItem::pluginName() const
{
	return mPluginName;
}

void PluginListItem::paintCell(QPainter *p, const QColorGroup &cg,
	int column, int width, int alignment)
{
	QFont oldFont = p->font();

	if (mPluginEnabled)
	{
		QFont newFont(oldFont);
		newFont.setBold(true);
		p->setFont(newFont);
	}

	if(column == 0)
	{
		QStyle *style = qApp->style();
		QStyleOption styleOpt;

		styleOpt.state |= QStyle::State_Enabled;
		styleOpt.state |= mPluginEnabled ? QStyle::State_On : QStyle::State_Off;

		int boxW = QMIN(
			width,
			style->pixelMetric(mIsRadio ? QStyle::PM_ExclusiveIndicatorWidth : QStyle::PM_IndicatorWidth)
		);

		int boxH = QMIN(
			height(),
			style->pixelMetric(mIsRadio ? QStyle::PM_ExclusiveIndicatorHeight :QStyle::PM_IndicatorHeight)
		);

		int boxLeft = (width - boxW) / 2;
		int boxTop = (height() - boxH) / 2;

		styleOpt.rect = QRect(boxLeft, boxTop, boxH, boxW);

		style->drawPrimitive(
			mIsRadio ? QStyle::PE_IndicatorRadioButton : QStyle::PE_IndicatorCheckBox,
			&styleOpt,
			p);
	}
	else
	{
		Q3ListViewItem::paintCell(p, cg, column, width, alignment);
	}

// restore old values (just for safety)
	p->setFont(oldFont);
}


// ----------------------------------------------------------------------------


PluginSelectorPage::PluginSelectorPage(Global *nInst)
	: PreferencesPage(new QWidget(), i18n("Plugin Selection")),
	nInstance(nInst),
	mChanged(false),
	mUpdating(false),
	mWidget(new Ui::PluginModuleWidget())
{
	setHeader(i18n("Select the plugins you wish to use"));
	setIcon(KIcon("gear"));

	mWidget->setupUi(widget());

	/// check, name, comment
	mWidget->lvUserinterface->addColumn(i18n("Enabled"));
	mWidget->lvUserinterface->addColumn(i18n("Name"));
	mWidget->lvUserinterface->addColumn(i18n("Comment"));

	mWidget->lvPlaylist->addColumn(i18n("Enabled"));
	mWidget->lvPlaylist->addColumn(i18n("Name"));
	mWidget->lvPlaylist->addColumn(i18n("Comment"));

	mWidget->lvVis->addColumn(i18n("Enabled"));
	mWidget->lvVis->addColumn(i18n("Name"));
	mWidget->lvVis->addColumn(i18n("Comment"));

	mWidget->lvOthers->addColumn(i18n("Enabled"));
	mWidget->lvOthers->addColumn(i18n("Name"));
	mWidget->lvOthers->addColumn(i18n("Comment"));

	//mWidget->lvUserinterface->header()->setResizeEnabled(false, 0);
	//mWidget->lvPlaylist->header()->setResizeEnabled(false, 0);

	connect(mWidget->lvUserinterface, SIGNAL(clicked(Q3ListViewItem *, const QPoint &, int)),
		SLOT(slotFrontendClicked(Q3ListViewItem *, const QPoint &, int)));

	connect(mWidget->lvPlaylist, SIGNAL(clicked(Q3ListViewItem *, const QPoint &, int)),
		SLOT(slotPLClicked(Q3ListViewItem *, const QPoint &, int)));

	connect(mWidget->lvVis, SIGNAL(clicked(Q3ListViewItem *, const QPoint &, int)),
		SLOT(slotMultiSelClicked(Q3ListViewItem *, const QPoint &, int)));

	connect(mWidget->lvOthers, SIGNAL(clicked(Q3ListViewItem *, const QPoint &, int)),
		SLOT(slotMultiSelClicked(Q3ListViewItem *, const QPoint &, int)));

	connect(nInst->pluginHandler(), SIGNAL(pluginUnloaded(Plugin *)), SLOT(slotDeselectPlugin(Plugin *)));
	connect(nInst->pluginHandler(), SIGNAL(pluginLoaded(Plugin *)), SLOT(slotSelectPlugin(Plugin *)));
}


void PluginSelectorPage::slotFrontendClicked(Q3ListViewItem *qli, const QPoint &, int col)
{
	if (col == 0)
	{
		PluginListItem *pli = static_cast<PluginListItem *>(qli);
		if (pli && !pli->pluginEnabled())
		{
			clearPluginSelection(mWidget->lvUserinterface);
			pli->setPluginEnabled(true);

			if (pli->pluginName() != mCurrentPL)
			{
				mNewUI = pli->pluginName();
				kDebug(66666) << k_funcinfo << "Will change to frontend " << mNewUI << " on Ok/Apply" << endl;
			}
			else
			{
				mNewUI = QString::null;
				kDebug(66666) << k_funcinfo << "Sticking to current frontend." << endl;
			}
			emit changed(true);
		}
	}
}


void PluginSelectorPage::slotPLClicked(Q3ListViewItem *qli, const QPoint &, int col)
{
	if (col == 0)
	{
		PluginListItem *pli = static_cast<PluginListItem *>(qli);
		if (pli && !pli->pluginEnabled())
		{
			clearPluginSelection(mWidget->lvPlaylist);
			pli->setPluginEnabled(true);

			if (pli->pluginName() != mCurrentPL)
			{
				mNewPL = pli->pluginName();
				kDebug(66666) << k_funcinfo << "Changing to playlist " << mNewPL << " on ok/apply." << endl;
			}
			else
			{
				mNewPL = QString::null;
				kDebug(66666) << k_funcinfo << "Sticking to currently loaded playlist." << endl;
			}
			emit changed(true);
		}
	}
}


void PluginSelectorPage::clearPluginSelection(Q3ListView *lv)
{
	PluginListItem *itPli = 0;
	Q3ListViewItemIterator it(lv);
	while (it.current())
	{
		itPli = static_cast<PluginListItem *>(it.current());
		if (itPli)
			itPli->setPluginEnabled(false);
		++it;
	}
}


void PluginSelectorPage::slotMultiSelClicked(Q3ListViewItem *qli, const QPoint &, int col)
{
	if (col == 0)
	{
		PluginListItem *pli = static_cast<PluginListItem *>(qli);
		if (!pli)
			return;

		kDebug(66666) << k_funcinfo << "toggling enabled state of plugin " <<
				pli->pluginName() << "." << endl;

		pli->setPluginEnabled(!pli->pluginEnabled());
		emit changed(true);
	}
}


void PluginSelectorPage::slotDeselectPlugin(Plugin *p)
{
	kDebug(66666) << k_funcinfo << p->pluginName() << endl;
	updateSelectionFor(p->pluginName(), p->pluginInterfaces(), false);
}


void PluginSelectorPage::slotSelectPlugin(Plugin *p)
{
	kDebug(66666) << k_funcinfo << p->pluginName() << endl;
	updateSelectionFor(p->pluginName(), p->pluginInterfaces(), true);
}


void PluginSelectorPage::updateSelectionFor(const QString &pluginName, const QStringList &interfaces, bool on)
{
	/*kDebug(66666) << k_funcinfo <<
		"Updating plugin '" << pluginName << "' selected = " << on <<
		", interfaces: " << interfaces <<  endl;*/

	mUpdating = true;
	PluginListItem *pli = 0;

	if (interfaces.contains("none"))
	{
		pli = mOthersDict[pluginName];
	}
	else
	{
		if (interfaces.contains("visualization"))
		{
			pli = mVisDict[pluginName];
		}

		if (interfaces.contains("userinterface"))
		{
			mCurrentUI = pluginName;
			pli = mUIDict[pluginName];
		}

		if (interfaces.contains("playlist"))
		{
			mCurrentPL = pluginName;
			pli = mPLDict[pluginName];
		}
	}

	if (pli)
	{
		//kDebug(66666) << k_funcinfo << "setting pixmap for plugin..." << endl;
		pli->setPluginEnabled(on);
	}
	mUpdating = false;
}


void PluginSelectorPage::fillList(K3ListView *lv, bool radio, QMap<QString, PluginListItem*>  &dict, const QString &interfacename)
{
	lv->clear();
	dict.clear();

	PluginListItem *lvItem = 0;
	PluginInfoList avail =
		nInstance->pluginHandler()->availablePluginsByInterface(interfacename);

	lv->setUpdatesEnabled(false);
	PluginInfoList::iterator it;
	for (it = avail.begin(); it != avail.end(); ++it)
	{
		if ((*it)->service()->noDisplay())
			continue;

		lvItem = new PluginListItem(*it, radio, lv);
		dict.insert((*it)->pluginName(), lvItem);
		QStringList ifaces(interfacename);
		updateSelectionFor((*it)->pluginName(), ifaces, (*it)->isPluginEnabled());
	}
	lv->setUpdatesEnabled(true);
}


void PluginSelectorPage::load()
{
	kDebug(66666) << k_funcinfo << endl;
	mCurrentUI = mNewUI = mCurrentPL = mNewPL = QString::null;

	fillList(mWidget->lvUserinterface, true, mUIDict, "userinterface");
	fillList(mWidget->lvPlaylist, true, mPLDict, "playlist");
	fillList(mWidget->lvVis, false, mVisDict, "visualization");
	fillList(mWidget->lvOthers, false, mOthersDict, "none");
	mChanged = false;
	emit changed(false);
}


void PluginSelectorPage::save()
{
	kDebug(66666) << k_funcinfo << endl;

	if (!mNewPL.isEmpty() && (mNewPL != mCurrentPL))
	{
		if (KMessageBox::shouldBeShownContinue("PlaylistPluginChange"))
		{
			KMessageBox::information(widget(),
				i18n("<qt>Changing your playlist plugin will stop playback. " \
					"Different playlists may use different methods of storing " \
					"information, so after changing playlist-plugins you may have " \
					"to recreate your playlist.</qt>"),
				i18n("Changing playlist-plugin"),
				"PlaylistPluginChange");
		}

		nInstance->pluginHandler()->loadPlugin(mNewPL);
	}

	if (!mNewUI.isEmpty() && mNewUI != mCurrentUI)
	{
		nInstance->pluginHandler()->loadPlugin(mNewUI);
	}

	QStringList mToBeLoaded, mToBeUnloaded;
	getChanged(mVisDict, "visualization", mToBeLoaded, mToBeUnloaded);
	getChanged(mOthersDict, "none", mToBeLoaded, mToBeUnloaded);


	kDebug(66666) << k_funcinfo << "mToBeLoaded   = " << mToBeLoaded << endl;
	kDebug(66666) << k_funcinfo << "mToBeUnloaded = " << mToBeUnloaded << endl;

	// Unload all plugins that are neither playlist nor frontend and have
	// been disabled by the user
	if (!mToBeUnloaded.empty())
	{
		nInstance->pluginHandler()->unloadPlugin(mToBeUnloaded);
		mToBeUnloaded.clear();
	}

	// Load all plugins that are neither playlist nor frontend and have
	// been enabled by the user
	if (!mToBeLoaded.empty())
	{
		nInstance->pluginHandler()->loadPlugin(mToBeLoaded);
		mToBeLoaded.clear();
	}

	mCurrentUI = mNewUI = mCurrentPL = mNewPL = QString::null;
	emit changed(false);
	//emit saved();
}


void PluginSelectorPage::defaults()
{
	kDebug(66666) << k_funcinfo << "NOT IMPLEMENTED YET" << endl;
}


void PluginSelectorPage::getChanged(QMap<QString, PluginListItem*> &dict, const QString &interfacename,
	QStringList &load, QStringList &unload)
{
	const PluginInfoList avail = nInstance->pluginHandler()->availablePluginsByInterface(interfacename);

	foreach(KPluginInfo *p, avail)
	{
		PluginListItem *pli = dict[p->pluginName()];
		if (pli)
		{
			if (p->isPluginEnabled() && !pli->pluginEnabled())
			{
				unload.append(p->pluginName());
				emit changed(true);
			}
			else if (!p->isPluginEnabled() && pli->pluginEnabled())
			{
				load.append(p->pluginName());
				emit changed(true);
			}
		}
	}
}


#include "pluginmodule.moc"
