/* This file is part of Noatun

  Copyright 2005-2006 by Charles Samuels <charles@kde.org>
  Copyright 2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef NOATUN__PLAYLISTITEM_H
#define NOATUN__PLAYLISTITEM_H

#include <noatun_export.h>

#include <qstringlist.h>
#include <qmap.h>

#include <kurl.h>


namespace Noatun
{

class Global;

class PlaylistInterfaceGeneric;

/**
 * A QMap of key-value-pairs, used for properties.
 **/
typedef QMap<QString, QString> PropertyMap;


/**
 * @short A reference to a PlaylistItemData object.
 *
 * All methods here should have the same behavior
 * as they do for PlaylistItemData.
 *
 * If you're a playlist, you should inherit
 * from PlaylistItemData.
 *
 * It is client code's responsibility to ensure that
 * PlaylistItem is not null by using either the boolean
 * conversion or PlaylistItem::isNull()
 *
 * @author Charles Samuels
 **/
class NOATUN_EXPORT PlaylistItem
{
	friend class PlaylistInterfaceGeneric;
	struct PlaylistItemShared;
	mutable PlaylistItemShared *shared;

	static PlaylistItem fromGeneric(PlaylistInterfaceGeneric *playlist, void *source);

	void *obj() const;

public:
	/**
	 * Copy constructor
	 **/
	PlaylistItem(const PlaylistItem &source);

	/**
	 * Constructor
	 *
	 * Creates a null-reference, i.e. isNull() will return true
	 **/
	PlaylistItem();

	/**
	 * Destructor
	 **/
	~PlaylistItem();

	PlaylistItem &operator =(const PlaylistItem &source);
	const PlaylistItem &operator =(const PlaylistItem &source) const;

	operator bool() const;
	/**
	 * @return true if we do not reference any PlaylistItemData, false otherwise.
	 **/
	bool isNull() const;

	bool operator ==(const PlaylistItem &i) const;
	bool operator !=(const PlaylistItem &i) const;

	/// See PlaylistItemData::property()
	QString property(const QString &key, const QString &def = 0) const;
	/// See PlaylistItemData::setProperty()
	void setProperty(const QString &key, const QString &property);
	/// See PlaylistItemData::setProperties()
	void setProperties(const PropertyMap &properties);
	/// See PlaylistItemData::clearProperty()
	void clearProperty(const QString &key);
	/// See PlaylistItemData::properties()
	QStringList properties() const;
	/// See PlaylistItemData::hasProperty()
	bool hasProperty(const QString &key) const;

	/// See PlaylistItemData::url()
	KUrl url() const;
	/// See PlaylistItemData::setUrl()
	void setUrl(const KUrl &url);

	/// See PlaylistItemData::mimetype()
	QString mimetype() const;

	/// See PlaylistItemData::length()
	int length() const;
	/// See PlaylistItemData::setLength(int)
	void setLength(int ms) const;

	/// See PlaylistItemData::lengthString()
	QString lengthString() const;
	/// See PlaylistItemData::remove()
	void remove();
	/// See PlaylistItemData::streamable()
	bool streamable() const;

public:
	/**
	 * Call this when you want to signal
	 * the given item has been added to the list
	 **/
	void added(Global *global);

	/**
	 * Your playlist must call this when the item
	 * is removed from the playlist
	 **/
	void removed(Global *global);

	/**
	 * Your playlist must call this when the item
	 * is modified
	 **/
	void modified(Global *global);

private:
	void removeRef() const;
	void addRef() const;
};


}

#endif
