/* This file is part of Noatun

  Copyright 2005-2006 by Charles Samuels <charles@kde.org>
  Copyright 2005-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef N_PLUGININTERFACES_H
#define N_PLUGININTERFACES_H

#include <kurl.h>

#include <noatun/player.h>
#include <noatun/playlist.h>
#include <noatun_export.h>

class KMainWindow;

namespace Noatun
{
class PlaylistItem;
class Global;

/**
 * Base class for all interfaces, its whole purpose is to group all interfaces
 **/
class NOATUN_EXPORT Interface
{
public:
	/// Constructor
	Interface();
	/// Destructor
	virtual ~Interface();
};

/**
 * Implement this interface if your plugin is a noatun-mainwindow
 * \todo Use QWidget instead of KMainWindow if possible, KParts usually do not
 *       have a mainwindow of their own
 **/
class NOATUN_EXPORT FrontendInterface : public Interface
{
public:
	FrontendInterface();
	virtual ~FrontendInterface();
	virtual KMainWindow *mainWindow() = 0;
};


// ----------------------------------------------------------------------------


/**
 * The playlist interface, which you derive from when creating
 * your own playlist.
 **/
class NOATUN_EXPORT PlaylistInterfaceGeneric : public Interface
{
	friend class PlaylistItem;

public:
	explicit PlaylistInterfaceGeneric(Global *glob);
	virtual ~PlaylistInterfaceGeneric();

	Global *global() { return mGlobal; }

	/**
	 * @return the window for this playlist. Might return 0.
	 **/
	virtual KMainWindow *mainWindow() const = 0;

	/**
	 * Add a file to the playlist
	 * @p file the file to add (does not have to be a local file)
	 * @p purgeList empty playlist before adding
	 * @p autoplay start playing the added file
	 */
	virtual void addFile(const KUrl &file, bool purgeList = false,
		bool autoplay = false) = 0;

	/**
	 * get the first item (not necessarily the first item played).
	 * See getAfter() as well.
	 **/
	virtual PlaylistItem getFirst() = 0;

	/**
	 * Get the item after @p item.
	 * Note that getFirst() and getAfter() do not have to follow play order
	 * since they are used solely to iterate over the entire collection in some
	 * order.
	 * The playlist does not have to duplicate the play order (by looking into
	 * the future)!
	 **/
	virtual PlaylistItem getAfter(const PlaylistItem &item) = 0;

	/**
	 * @return a list of songs in which at least one of the keys matches at
	 * least one of the values.
	 *
	 * the default implementation will call getFirst() and getAfter() which could
	 * be potentially slow, depending on how the playlist is designed. You're
	 * free to reimplement this for performance reasons.
	 *
	 * A value of "" is equal to an unset value.
	 *
	 * @p limit is the maximum amount of items to return, or -1 if you want as
	 * many as possible.
	 *
	 * @p exact if true, a match is only made if the string is identical to a
	 * value. if false a match is made if the string contains a value.
	 *
	 * @p caseSensitive, if false, the given @p values are compared case
	 * insensitively to to the items in the playlist. The @p keys are always
	 * compared with case sensitivity.
	 **/
	virtual QList<PlaylistItem> select(
			const QStringList &keys, const QStringList &values,
			int limit=-1, bool exact=false, bool caseSensitive=false
		);

	/**
	 * Determines if the window returned by mainWindow() is visible or not.
	 * Usually you don't need to reimplement this.
	 *
	 * @return true if the window is visible, false otherwise.
	 **/
	virtual bool listVisible();

	/**
	 * Makes the playlist-window visible if not visible already.
	 * Usually you don't need to reimplement this.
	 **/
	virtual void showList();

	/**
	 * Hides the playlist-window.
	 * Usually you don't need to reimplement this.
	 **/
	virtual void hideList();

	/**
	 * Call this when the list was hidden
	 */
	void listHidden();

	/**
	 * Call this when the list is shown
	 */
	void listShown();

protected: // all the playlistitem functions
	/**
	 * Noatun asks your playlist for properties.  It is your
	 * responsiblity to store the information. But usually a
	 * QMap\<QString,QString\> is enough.
	 *
	 * If you return the default value, the default should not
	 * be written.
	 *
	 * This returns the property, or @p def if such a property doesn't exist
	 **/
	virtual QString property_g(void *item, const QString &key, const QString &def=0) =0;

	/**
	 * This sets the property with the given key and value.
	 *
	 * Important: If you use a QMap, you'll have to remove the current
	 * item before adding a new one
	 **/
	virtual void setProperty_g(void *item, const QString &key, const QString &value) = 0;

	/**
	 * Similar to @see setProperty(const QString &key, const QString &value) but
	 * for more than one key-value-pair.
	 *
	 * This sets a bunch of properties with their given keys and values
	 * @p properties contains all properties to set
	 *
	 **/
	virtual void setProperties_g(void *item, const PropertyMap &properties)=0;

	/**
	 * remove the item with given key
	 **/
	virtual void clearProperty_g(void *item, const QString &key) = 0;

	/**
	 * return a list of property keys
	 **/
	virtual QStringList properties_g(void *item)=0;

	/**
	 * return whether if the given key exists
	 **/
	virtual bool hasProperty_g(void *item, const QString &key)=0;

	/**
	 * the true filename of the song, remote or local
	 **/
	virtual KUrl url_g(void *item) = 0;
	/**
	 * set the true filename of the song, remote or local
	 **/
	virtual void setUrl_g(void *item, const KUrl &url)=0;

	/**
	 * first, this checks for the property "mimetype", else
	 * it'l ask KMimeType based on file()
	 **/
	virtual QString mimetype_g(void *item)=0;

	/**
	 * @returns the length of the song, in milliseconds?
	 **/
	virtual int length_g(void *item)=0;

	/**
	 * sets the length of the song, in milliseconds
	 **/
	virtual void setLength_g(void *item, int ms)=0;

	/**
	 * compare yourself with the given PlaylistItemData
	 * This is implemented in the slow fashion of
	 * comparing all the properties.  You may
	 * have a more accurate way of implementing this
	 *
	 * if one==two, this will not be called, normally
	 **/
	virtual bool compare_g(void *one, void *two)=0;

	/**
	 * remove this item from the playlist, this does not necessarily include
	 * deleting this object right now as something might still reference it.
	 **/
	virtual void remove_g(void *data) = 0;

	/**
	 * Playlists should not download files if this returns true
	 **/
	bool streamable_g(void *data);

	/// @todo rename this function
	void *item_g(const PlaylistItem &i)
	{
		return i.obj();
	}

	/// @todo rename this function
	PlaylistItem item_g(void *obj)
	{
		return PlaylistItem::fromGeneric(this, obj);
	}

private:
	virtual void deleteItem_g(void *data)=0;

private:
	Global *mGlobal;
};





template <typename ItemType>
class NOATUN_EXPORT PlaylistInterface : public PlaylistInterfaceGeneric
{
public:
	explicit PlaylistInterface(Global *glob) : PlaylistInterfaceGeneric(glob)
	{
	}

	/**
	 * @return the window for this playlist. Might return 0.
	 **/
	virtual KMainWindow *mainWindow() const = 0;

	/**
	 * Add a file to the playlist
	 * @p file the file to add (does not have to be a local file)
	 * @p purgeList empty playlist before adding
	 * @p autoplay start playing the added file
	 */
	virtual void addFile(const KUrl &file, bool purgeList = false,
		bool autoplay = false) = 0;

	/**
	 * get the first item (not necessarily the first item played).
	 * See getAfter() as well.
	 **/
	virtual PlaylistItem getFirst() = 0;

	/**
	 * Get the item after @p item.
	 * Note that getFirst() and getAfter() do not have to follow play order
	 * since they are used solely to iterate over the entire collection in some
	 * order.
	 * The playlist does not have to duplicate the play order (by looking into
	 * the future)!
	 **/
	virtual PlaylistItem getAfter(const PlaylistItem &item) = 0;

	/**
	 * Determines if the window returned by mainWindow() is visible or not.
	 * Usually you don't need to reimplement this.
	 *
	 * @return true if the window is visible, false otherwise.
	 **/
	virtual bool listVisible() { return PlaylistInterfaceGeneric::listVisible(); }

	/**
	 * Ma
	 kes the playlist-window visible if not visible already.
	 * Usually you don't need to reimplement this.
	 **/
	virtual void showList() { PlaylistInterfaceGeneric::showList(); }

	/**
	 * Hides the playlist-window.
	 * Usually you don't need to reimplement this.
	 **/
	virtual void hideList() { PlaylistInterfaceGeneric::hideList(); }

	/**
	 * Call this when the list was hidden
	 */
	void listHidden()
	{
		PlaylistInterfaceGeneric::listHidden();
	}

	/**
	 * Call this when the list is shown
	 */
	void listShown()
	{
		PlaylistInterfaceGeneric::listShown();
	}

private: // all the playlistitem functions
	/**
	 * Playlists should not download files if this returns true
	 **/
	bool streamable(const ItemType &item)
	{
		return PlaylistInterfaceGeneric::streamable_g(toGeneric(item));
	}

	/**
	 * Noatun asks your playlist for properties.  It is your
	 * responsiblity to store the information. But usually a QMap<QString,QString>
	 * is enough.
	 *
	 * If you return the default value, the default should not
	 * be written.
	 *
	 * This returns the property, or @p def if such a property doesn't exist
	 **/
	virtual QString property(ItemType item, const QString &key, const QString &def=0)=0;

	/**
	 * This sets the property with the given key and value.
	 *
	 * Important: If you use a QMap, you'll have to remove the current
	 * item before adding a new one
	 **/
	virtual void setProperty(ItemType item, const QString &key, const QString &value) = 0;

	/**
	 * Similar to @see setProperty(const QString &key, const QString &value) but
	 * for more than one key-value-pair.
	 *
	 * This sets a bunch of properties with their given keys and values
	 * @p properties contains all properties to set
	 *
	 **/
	virtual void setProperties(ItemType item, const PropertyMap &properties)
	{
		PlaylistInterfaceGeneric::setProperties_g(toGeneric(item), properties);
	}

	/**
	 * remove the item with given key
	 **/
	virtual void clearProperty(ItemType item, const QString &key) = 0;

	/**
	 * return a list of property keys
	 **/
	virtual QStringList properties(ItemType item)=0;

	/**
	 * return whether if the given key exists
	 **/
	virtual bool hasProperty(ItemType item, const QString &key)=0;

	/**
	 * the true filename of the song, remote or local
	 **/
	virtual KUrl url(ItemType *item)
	{
		return KUrl(property(item, "url"));
	}

	/**
	 * set the true filename of the song, remote or local
	 **/
	virtual void setUrl(ItemType *item, const KUrl &url)
	{
		setProperty(item, "url", url.url());
	}

	/**
	 * first, this checks for the property "mimetype", else
	 * it'll ask KMimeType based on file()
	 **/
	virtual QString mimetype(const ItemType &item)
	{
		return PlaylistInterfaceGeneric::mimetype_g(toGeneric(item));
	}

	/**
	 * @returns the length of the song, in milliseconds?
	 **/
	virtual int length(const ItemType &item)
	{
		return property(item, "length", "-1").toInt();
	}

	/**
	 * sets the length of the song, in milliseconds
	 **/
	virtual void setLength(const ItemType &item, int ms)
	{
		setProperty(item, "length", QString::number(ms));
	}

	/**
	 * compare yourself with the given PlaylistItemData
	 * This is implemented in the slow fashion of
	 * comparing all the properties.  You may
	 * have a more accurate way of implementing this
	 *
	 * if one==two, this will not be called, normally
	 **/
	virtual bool compare(const ItemType &one, const ItemType &two)=0;

	/**
	 * remove this item from the playlist, this does not necessarily include
	 * deleting this object right now as something might still reference it.
	 **/
	virtual void remove(const ItemType &data) = 0;


public:
	ItemType item(const PlaylistItem &i)
	{
		return fromGeneric(PlaylistInterfaceGeneric::item_g(i));
	}

	PlaylistItem item(const ItemType &obj)
	{
		return PlaylistInterfaceGeneric::item_g(new ItemType(obj));
	}


private: // all the playlistitem functions
	/**
	 * Noatun asks your playlist for properties.  It is your
	 * responsiblity to store the information. But usually a QMap<QString,QString>
	 * is enough.
	 *
	 * If you return the default value, the default should not
	 * be written.
	 *
	 * This returns the property, or @p def if such a property doesn't exist
	 **/
	virtual QString property_g(void *item, const QString &key, const QString &def)
	{
		return property(fromGeneric(item), key, def);
	}

	/**
	 * This sets the property with the given key and value.
	 *
	 * Important: If you use a QMap, you'll have to remove the current
	 * item before adding a new one
	 **/
	virtual void setProperty_g(void *item, const QString &key, const QString &value)
	{
		setProperty(fromGeneric(item), key, value);
	}

	/**
	 * Similar to @see setProperty(const QString &key, const QString &value) but
	 * for more than one key-value-pair.
	 *
	 * This sets a bunch of properties with their given keys and values
	 * @p properties contains all properties to set
	 *
	 **/
	virtual void setProperties_g(void *item, const PropertyMap &properties)
	{
		setProperties(fromGeneric(item), properties);
	}

	/**
	 * remove the item with given key
	 **/
	virtual void clearProperty_g(void *item, const QString &key)
	{
		clearProperty(fromGeneric(item), key);
	}

	/**
	 * return a list of property keys
	 **/
	virtual QStringList properties_g(void *item)
	{
		return properties(fromGeneric(item));
	}

	/**
	 * return whether if the given key exists
	 **/
	virtual bool hasProperty(void *item, const QString &key)
	{
		return hasProperty_g(fromGeneric(item), key);
	}

	/**
	 * the true filename of the song, remote or local
	 **/
	virtual KUrl url_g(void *item)
	{
		return url_g(fromGeneric(item));
	}

	/**
	 * set the true filename of the song, remote or local
	 **/
	virtual void setUrl_g(void *item, const KUrl &url)
	{
		setUrl(fromGeneric(item), url);
	}

	/**
	 * first, this checks for the property "mimetype", else
	 * it'l ask KMimeType based on file()
	 **/
	virtual QString mimetype_g(void *item)
	{
		return mimetype(fromGeneric(item));
	}

	/**
	 * @returns the length of the song, in milliseconds?
	 **/
	virtual int length_g(void *item)
	{
		return length(fromGeneric(item));
	}

	/**
	 * sets the length of the song, in milliseconds
	 **/
	virtual void setLength_g(void *item, int ms)
	{
		return setLength(fromGeneric(item), ms);
	}

	/**
	 * compare yourself with the given PlaylistItemData
	 * This is implemented in the slow fashion of
	 * comparing all the properties.  You may
	 * have a more accurate way of implementing this
	 *
	 * if one==two, this will not be called, normally
	 **/
	virtual bool compare_g(void *one, void *two)
	{
		return compare(fromGeneric(one), fromGeneric(two));
	}

	/**
	 * remove this item from the playlist, this does not necessarily include
	 * deleting this object right now as something might still reference it.
	 **/
	virtual void remove_g(void *item)
	{
		remove(fromGeneric(item));
	}

private:
	virtual void deleteItem_g(void *data)
	{
		delete static_cast<ItemType*>(data);
	}

	void *toGeneric(ItemType &type) const
	{
		return &type;
	}

	ItemType &fromGeneric(void *g)
	{
		return *static_cast<ItemType*>(g);
	}


};


/**
 * this is a specialization for pointer ItemTypes
 * @internal
 **/
template <typename ItemType>
class NOATUN_EXPORT PlaylistInterface<ItemType*> : public PlaylistInterfaceGeneric
{
public:
	explicit PlaylistInterface(Global *glob) : PlaylistInterfaceGeneric(glob)
	{
	}

	virtual KMainWindow *mainWindow() const = 0;

	virtual void addFile(const KUrl &file, bool purgeList = false,
		bool autoplay = false) = 0;

	virtual PlaylistItem getFirst() = 0;

	virtual PlaylistItem getAfter(const PlaylistItem &item) = 0;

	virtual bool listVisible() { return PlaylistInterfaceGeneric::listVisible(); }

	virtual void showList() { PlaylistInterfaceGeneric::showList(); }

	virtual void hideList() { PlaylistInterfaceGeneric::hideList(); }

	void listHidden()
	{
		PlaylistInterfaceGeneric::listHidden();
	}

	void listShown()
	{
		PlaylistInterfaceGeneric::listShown();
	}

public: // all the playlistitem functions
	bool streamable(ItemType *item)
	{
		return PlaylistInterfaceGeneric::streamable_g(toGeneric(item));
	}

	virtual QString property(ItemType *item, const QString &key, const QString &def=0)=0;

	virtual void setProperty(ItemType* item, const QString &key, const QString &value) = 0;

	virtual void setProperties(ItemType *item, const PropertyMap &properties)
	{
		PlaylistInterfaceGeneric::setProperties_g(toGeneric(item), properties);
	}

	virtual void clearProperty(ItemType *item, const QString &key) = 0;
	virtual QStringList properties(ItemType *item)=0;

	virtual bool hasProperty(ItemType *item, const QString &key)=0;
	virtual KUrl url(ItemType *item)
	{
		return KUrl(property(item, "url"));
	}

	virtual void setUrl(ItemType *item, const KUrl &url)
	{
		setProperty(item, "url", url.url());
	}

	virtual QString mimetype(ItemType *item)
	{
		return PlaylistInterfaceGeneric::mimetype_g(toGeneric(item));
	}

	virtual int length(ItemType *item)
	{
		return property(item, "length", "-1").toInt();
	}

	virtual void setLength(ItemType *item, int ms)
	{
		setProperty(item, "length", QString::number(ms));
	}

	virtual bool compare(ItemType *one, ItemType *two)=0;

	virtual void remove(ItemType *data) = 0;


public:
	ItemType *item(const PlaylistItem &i)
	{
		return fromGeneric(PlaylistInterfaceGeneric::item_g(i));
	}

	PlaylistItem item(ItemType *obj)
	{
		return PlaylistInterfaceGeneric::item_g(toGeneric(obj));
	}

private: // all the playlistitem functions
	virtual QString property_g(void *item, const QString &key, const QString &def)
	{
		return property(fromGeneric(item), key, def);
	}

	virtual void setProperty_g(void *item, const QString &key, const QString &value)
	{
		setProperty(fromGeneric(item), key, value);
	}

	virtual void setProperties_g(void *item, const PropertyMap &properties)
	{
		setProperties(fromGeneric(item), properties);
	}

	virtual void clearProperty_g(void *item, const QString &key)
	{
		clearProperty(fromGeneric(item), key);
	}

	virtual QStringList properties_g(void *item)
	{
		return properties(fromGeneric(item));
	}

	virtual bool hasProperty_g(void *item, const QString &key)
	{
		return hasProperty(fromGeneric(item), key);
	}

	virtual KUrl url_g(void *item)
	{
		return url(fromGeneric(item));
	}

	virtual void setUrl_g(void *item, const KUrl &url)
	{
		setUrl(fromGeneric(item), url);
	}

	virtual QString mimetype_g(void *item)
	{
		return mimetype(fromGeneric(item));
	}

	virtual int length_g(void *item)
	{
		return length(fromGeneric(item));
	}

	virtual void setLength_g(void *item, int ms)
	{
		return setLength(fromGeneric(item), ms);
	}

	virtual bool compare_g(void *one, void *two)
	{
		return compare(fromGeneric(one), fromGeneric(two));
	}

	virtual void remove_g(void *item)
	{
		remove(fromGeneric(item));
	}

private:
	virtual void deleteItem_g(void *g)
	{
		delete fromGeneric(g);
	}

	void *toGeneric(ItemType *type) const
	{
		return type;
	}

	ItemType* fromGeneric(void *g)
	{
		return static_cast<ItemType*>(g);
	}


};


// ----------------------------------------------------------------------------

/**
 * Implement this interface if your plugin is a playlist.
 *
 * @todo add methods for implementing loop-modes
 **/
class NOATUN_EXPORT ItemSourceInterface : public Interface
{
public:
	ItemSourceInterface();
	virtual ~ItemSourceInterface();
	virtual PlaylistItem forward() = 0;
	virtual PlaylistItem backward() = 0;
	virtual void setCurrent(const PlaylistItem &item) = 0;
};


// ----------------------------------------------------------------------------

/**
 * Implement this if your plugin is a playback backend.
 *
 * @todo Check if API can support gapless/crossfading playback easily
 * @todo Add support for effects, most notably an equalizer
 **/
class NOATUN_EXPORT EngineInterface : public Interface
{
public:
	explicit EngineInterface(Global *nInst);
	virtual ~EngineInterface();
	/**
	 * @return engine state
	 **/
	virtual Player::State state() const = 0;
	/**
	 * @return position in milliseconds or -1 if not playing/pausing
	 **/
	virtual int position() const = 0;
	/**
	 * @return track length in millisecond or -1 if not playing/pausing
	 **/
	virtual int length() const = 0;
	/**
	 * @return a list of supported mimetypes
	 **/
	virtual QStringList mimeTypes() const = 0;
	/**
	 * @return Software volume in percent
	 *
	 * @note An engine does not have to implement this, Noatun will use hardware
	 *       volume-control in this case
	 **/
	virtual unsigned int softwareVolume() const;
	/**
	 * @brief Set software volume to @p percent percent
	 *
	 * @note An engine does not have to implement this, Noatun will use hardware
	 *       volume-control in this case
	 */
	virtual void setSoftwareVolume(unsigned int percent);
	/**
	 * @brief Continues or starts playing the current item
	 * @param url the item that is to be played or an empty KUrl to resume from
	 *            paused playback
	 * @return true if starting/continuing worked, false otherwise.
	 * @todo Remove return value in favor of some async api
	 **/
	virtual bool play(const KUrl &url) = 0;
	/**
	 * @brief Pauses playback keeping the currently playing item loaded
	 *
	 * For items that are not seekable (i.e. streams) this method should behave
	 * like stop().
	 **/
	virtual void pause() = 0;
	/**
	 * @brief Stops playback if state is either Player::PlayingState or
	 *        Player::PausedState
	 *
	 * The current playback position should be reset to 0 if the played item is
	 * seekable
	 **/
	virtual void stop() = 0;
	/**
	 * @brief Set a new playback position (i.e. seek)
	 * @param msec new position given in milliseconds
	 **/
	virtual void setPosition(int msec) = 0;

protected:
	/**
	 * @brief Call this whenever playback started/stopped/paused
	 *
	 * @todo implement using postEvent() to allow calling from another thread
	 **/
	void stateChanged(Player::State newState);
	/**
	 * @brief Call this whenever playback of the current item ended gracefully
	 *
	 * @todo implement using postEvent() to allow calling from another thread
	 **/
	void playbackFinished();
	/**
	 * @brief Call this when playback somehow failed
	 *
	 * This Should be called whenever your backend cannot continue with playback
	 * (file not found, stream is down, socket timeout for streams etc.)
	 *
	 * @note The engine should be in state Player::Stopped when calling this method
	 *
	 * @todo implement using postEvent() to allow calling from another thread
	 **/
	void errorOccurred(Player::ErrorType errorType, const QString &errorDescription);

private:
	Global *nInstance;
}; // class EngineInterface


} // namespace Noatun

#endif
