/* This file is part of Noatun

  Copyright 2001-2006 by Charles Samuels <charles@kde.org>
  Copyright 2002 by Stefan Westerfeld <stefan@space.twc.de>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <noatun/playlist.h>
#include <noatun/global.h>
#include <noatun/plugin.h>
#include <noatun/pluginloader.h>
#include <noatun/plugininterfaces.h>

#include <qevent.h>

#include <kdebug.h>
#include <kmainwindow.h>


namespace Noatun
{

class PlaylistPrivate
{
public:
	PlaylistPrivate() : iface(0), plugin(0)
	{
	}

	QList<PlaylistNotifier *>  notifiers;
	PlaylistInterfaceGeneric  *iface;
	Plugin                    *plugin;
};


Playlist::Playlist(Global *nInstance)
	: QObject(nInstance), d(new PlaylistPrivate)
{
	setObjectName("Noatun_Playlist");
	kDebug(66666) << k_funcinfo << endl;

	connect(nInstance->pluginHandler(), SIGNAL(pluginLoaded(Plugin *)),
		SLOT(slotPluginLoaded(Plugin *)));
	connect(nInstance->pluginHandler(), SIGNAL(pluginUnloaded(Plugin *)),
		SLOT(slotPluginUnloaded(Plugin *)));
}


Playlist::~Playlist()
{
	kDebug(66666) << k_funcinfo << endl;
}


KMainWindow *Playlist::mainWindow() const
{
	Q_ASSERT(d->iface);
	return d->iface->mainWindow();
}


void Playlist::addFile(const KUrl::List &f, bool purgeList, bool autoplay)
{
	bool play = autoplay;
	bool purge = purgeList;
	for (KUrl::List::ConstIterator i(f.begin()); i != f.end(); ++i)
	{
		addFile(*i, purge, play);
		purge = false;
		play = false;
	}
}


void Playlist::addFile(const KUrl &url, bool purgeList, bool autoplay)
{
	Q_ASSERT(d->iface);
	d->iface->addFile(url, purgeList, autoplay);
}


PlaylistItem Playlist::getFirst() const
{
	Q_ASSERT(d->iface);
	return d->iface->getFirst();
}


PlaylistItem Playlist::getAfter(const PlaylistItem &item) const
{
	Q_ASSERT(d->iface);
	return d->iface->getAfter(item);
}


bool Playlist::listVisible() const
{
	if (d->iface)
		return d->iface->listVisible();
	return false;
}


QList<PlaylistItem> Playlist::select(const QStringList &keys,
	const QStringList &values, int limit, bool exact, bool caseSensitive)
{
	if (d->iface)
		return d->iface->select(keys, values, limit, exact, caseSensitive);
	return QList<PlaylistItem>();
}


QList<PlaylistItem> Playlist::select(
		const QString &key, const QString &value,
		int limit, bool exact, bool caseSensitive)
{
	QStringList keys(key);
	QStringList values(value);
	return select(keys, values, limit, exact, caseSensitive);
}


void Playlist::showList()
{
	if (d->iface)
		d->iface->showList();
}


void Playlist::hideList()
{
	if (d->iface)
		d->iface->hideList();
}


void Playlist::toggleList()
{
	if (d->iface)
		d->iface->listVisible() ? d->iface->hideList() : d->iface->showList();
}


void Playlist::slotPluginLoaded(Plugin *plug)
{
	if (PlaylistInterfaceGeneric *f = plug->interface<PlaylistInterfaceGeneric>())
	{
		// Only one playlist at a time please, unload the old one now
		if (d->plugin)
		{
			kDebug(66666) << k_funcinfo << "Unloading old playlist..." << endl;
			d->plugin->requestUnload();
		}
		d->plugin = plug;
		d->iface = f;
		if (d->iface->mainWindow())
		{
			kDebug(66666) << k_funcinfo << "Installing even filter on playlist mainwindow" << endl;
			d->iface->mainWindow()->installEventFilter(this);
		}
		emit newPlaylist();
	}
}


void Playlist::slotPluginUnloaded(Plugin *plug)
{
	if (plug->interface<PlaylistInterfaceGeneric>() && plug == d->plugin)
	{
		kDebug(66666) << k_funcinfo << "Playlist is GONE!" << endl;
		d->plugin = 0;
		d->iface = 0;
	}
}


bool Playlist::eventFilter(QObject *o, QEvent *e)
{
	if (o && d->iface && o == d->iface->mainWindow())
	{
		if (e->type() == QEvent::Hide)
		{
			kDebug(66666) << k_funcinfo << "QEvent::Hide for playlist window seen" << endl;
			if (!d->iface->mainWindow()->isVisible() || d->iface->mainWindow()->isMinimized())
				emit listHidden();
		}
		else if(e->type() == QEvent::Show)
		{
			kDebug(66666) << k_funcinfo << "QEvent::Show for playlist window seen" << endl;
			if (d->iface->mainWindow()->isVisible() && !d->iface->mainWindow()->isMinimized())
				emit listShown();
		}
	}
	return false;
}


QList<PlaylistNotifier *> *Playlist::notifiers()
{
	return &d->notifiers;
}


// ----------------------------------------------------------------------------


PlaylistNotifier::PlaylistNotifier(Noatun::Global *glob)
{
	mGlobal = glob;
	mGlobal->playlist()->notifiers()->append(this);
}


PlaylistNotifier::~PlaylistNotifier()
{
	mGlobal->playlist()->notifiers()->removeAll(this);
}


void PlaylistNotifier::added(PlaylistItem &item)
{
	Q_UNUSED(item)
}


void PlaylistNotifier::removed(PlaylistItem &item)
{
	Q_UNUSED(item)
}


void PlaylistNotifier::modified(PlaylistItem &item)
{
	Q_UNUSED(item)
}


} // namespace Noatun
#include "playlist.moc"
