/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/playlistsaver.h"

#include <qdom.h>
#include <qfile.h>
#include <qregexp.h>
#include <qxml.h>
#include <qtextcodec.h>

#include <kio/netaccess.h>
#include <ksimpleconfig.h>
#include <kmimetype.h>
#include <klocale.h>
#include <ksavefile.h>
#include <ktemporaryfile.h>

#include <kdebug.h>

namespace Noatun
{

PlaylistSaver::PlaylistSaver()
{
}

PlaylistSaver::~PlaylistSaver()
{
}

QStringList PlaylistSaver::mimetypes()
{
	return QStringList() << "audio/x-scpls" << "audio/mpegurl" << "audio/x-mpegurl" << "text/xml";
}

QStringList PlaylistSaver::extensions()
{
	return QStringList() << "xml" << "pls" << "m3u" << "wax" << "wvx" << "asx";
}

bool PlaylistSaver::save(const KUrl &url, const PlaylistType opt)
{
	bool ret = false;

	kDebug(66666) << k_funcinfo << "url:" << url << "; opt:" << opt << endl;

	if (url.isEmpty() || !url.isValid())
		return false;

	if ((opt == PLS) || (opt == ASX))
	{
		kWarning(66666) << "Saving PLS and ASX playlists is not supported" << endl;
		return false;
	}

	if (url.isLocalFile())
	{ // local files
		KSaveFile saveFile;
		saveFile.setFileName(url.path());

		if (saveFile.open())
		{
			if (saveLocal(saveFile, opt)) // saving worked
			{
				ret = saveFile.finalize();
				//TODO: check if finalize worked and use saveFile.errorString();
			}
			else
			{
				saveFile.abort();
			}
		}
		else
		{
			kWarning(66666) << "Couldn't open playlist file" << endl;
			//TODO: make use of saveFile.errorString();
		}
	}
	else
	{ // remote files
		KTemporaryFile tmpFile;
		tmpFile.setSuffix(".noatunplaylist.xml");

		if (tmpFile.open())
		{
			if (saveLocal(tmpFile, opt)) // saving worked
			{
				ret = KIO::NetAccess::upload(tmpFile.fileName(), url, 0L);
			}
		}
		else
		{
			kWarning(66666) << "Couldn't open tempfile " <<
				tmpFile.fileName() << endl;
		}
	}
	return ret;
}


bool PlaylistSaver::saveLocal(QFile &file, const PlaylistType opt)
{
	QTextStream stream(&file);
	if (opt == M3U || opt == EXTM3U)
		return saveM3U(stream, opt);
	else  // It's a XMLPlaylist
		return saveXML(stream);
}



bool PlaylistSaver::load(const KUrl &url, const PlaylistType opt)
{
	bool res = false;
	QString localTempFile;

	kDebug(66666) << k_funcinfo << "url:" << url << "; opt:" << opt << endl;

	if(!KIO::NetAccess::download(url, localTempFile, 0L))
		return false;

	switch (opt)
	{
		default:
		case Default:
		{
			/*
			QString end = url.filename().toLower().section('.', -1);
			QStringList knownExt = extensions();
			if (knownExt.find(end) == knownExt.end() && url.protocol().toLower() == "http")
			{
				Noatun::PropertyMap map;
				KMimeType::Ptr mimetype = KMimeType::findByURL(url, 0, false, false);
				if (mimetype->name() == "application/octet-stream")
				{
					map["title"] = i18n("Stream from %1").arg(url.host());

					KUrl u(url);
					if (!u.hasPath())
						u.setPath("/");

					map["stream_"] = map["url"] = u.url();
					reset();
					readItem(map);
					res = true;
				}
			}*/
			res = metalist(localTempFile, url);
			break;
		}
		case XMLPlaylist:
			res = loadXML(localTempFile, opt);
			break;
		case ASX:
			res = loadASX(localTempFile, url, opt);
			break;
		case M3U:
		case EXTM3U:
			res = loadM3U(localTempFile, url, opt);
			break;
		case PLS:
			res = loadPLS(localTempFile, opt);
			break;
	}
	KIO::NetAccess::removeTempFile(localTempFile);
	return res;
}


bool PlaylistSaver::metalist(const QString &file, const KUrl &originalUrl)
{
	if(loadXML(file, XMLPlaylist))
		return true;
	if(loadASX(file, originalUrl, ASX))
		return true;
	if(loadPLS(file, PLS))
		return true;
	if(loadM3U(file, originalUrl))
		return true;
	return false;
}


bool PlaylistSaver::saveXML(QTextStream &stream)
{
	const PlaylistItem &i = PlaylistItem();
	QDomDocument doc("playlist");
	doc.setContent(QString("<!DOCTYPE XMLPlaylist><playlist version=\"2.0\" client=\"noatun\"/>"));
	QDomElement docElem = doc.documentElement();

	reset();

	while ((i = writeItem()))
	{
		// write all properties
		const QStringList &props = i.properties();
		QDomElement elem = doc.createElement("item");

		foreach(const QString &propertyName, props)
		{
			QString propertyValue = i.property(propertyName);
			elem.setAttribute(propertyName, propertyValue);

			/*if (propertyName == "url")
			{
				KUrl u(val);
				if (u.isLocalFile())
					elem.setAttribute("local", u.path());
			}*/
		}
		docElem.appendChild(elem);
	}

	stream.setCodec(QTextCodec::codecForName("UTF-8"));
	stream << doc.toString();
	return true;
}


class NoatunXMLStructure : public QXmlDefaultHandler
{
public:
	PlaylistSaver *saver;
	bool fresh;

	NoatunXMLStructure(PlaylistSaver *s)
		: saver(s), fresh(true)
	{
	}

	bool startElement(const QString&, const QString &, const QString &name,
		const QXmlAttributes &attr)
	{
		if (fresh)
		{
			fresh = (name != "playlist");
			// if we are still marked as "fresh" then the xml file did not start
			// with a playlist-tag and is thus INVALID.
			return (!fresh);
		}

		if (name == "item")
		{
			Noatun::PropertyMap propMap;
			int attrCount = attr.count();
			for (int i = 0; i < attrCount; i++)
			{
				propMap[attr.qName(i)] = attr.value(i);
			}
			saver->readItem(propMap);
		}

		return true;
	}
};


class MSASXStructure : public QXmlDefaultHandler
{
public:
	PlaylistSaver *saver;
	bool fresh;
	bool inEntry, inTitle;
	Noatun::PropertyMap propMap;
	QString mAbsPath;

	MSASXStructure(PlaylistSaver *s, const QString &absPath)
		: saver(s), fresh(true), inEntry(false),
		inTitle(false), mAbsPath(absPath)
	{
		//kDebug(66666) << k_funcinfo << endl;
	}

	bool startElement(const QString&, const QString &,
		const QString &name, const QXmlAttributes &a)
	{
		if (fresh)
		{
			fresh = (name.toLower() != "asx");
			// if we are still marked as "fresh" then the xml file did not start
			// with an asx-tag and is thus INVALID.
			return (!fresh);
		}

		if (name.toLower() == "entry")
		{
			if(inEntry) // WHOOPS, we are already in an entry, this should NEVER happen
			{
				kDebug(66666) << "STOP, ENTRY INSIDE ENTRY!" << endl;
				return false;
			}
			inEntry=true;
		}
		else
		{
			if (inEntry) // inside entry block
			{
				// known stuff inside an <entry> ... </entry> block:
				// <title>blah</title>
				// <param album="blub" />
				// <param artist="blah" />
				// <ref HREF="file:/something" />
				if(name.toLower()=="ref")
				{
					for (int i=0; i<a.count(); i++)
					{
						if(a.qName(i).toLower()=="href")
						{
							QString filename = a.value(i);
							if (filename.contains(QRegExp("^[a-zA-Z0-9]+:/")))
							{
								KUrl url(filename);
								KMimeType::Ptr mimetype = KMimeType::findByUrl(url);
								QString type = mimetype->name();
								if (type != "application/octet-stream")
								{
									propMap["url"]=filename;
								}
								else
								{
									//propMap["playObject"]="SplayPlayObject";
									propMap["title"] = i18n("Stream from %1",url.host());
									if (!url.hasPath())
										url.setPath("/");
									propMap["url"] = url.url();
									propMap["stream_"]=propMap["url"];
								}
							}
							else
							{
								KUrl u1;
								// we have to deal with a relative path
								if (filename.contains('/'))
								{
									u1.setPath(mAbsPath); //FIXME: how to get the path in this place?
									u1.setFileName(filename);
								}
								else
								{
									u1.setPath(filename);
								}
								propMap["url"]=u1.url();
							}
						}
					}
				}
				else if(name.toLower()=="param")
				{
					QString keyName="", keyValue="";

					for (int i=0; i<a.count(); i++)
					{
						if(a.value(i).toLower()=="album")
							keyName="album";
						else if(a.value(i).toLower()=="artist")
							keyName="author";
						else if(!keyName.isEmpty()) // successfully found a key, the next key=value pair has to be the value
						{
							keyValue=a.value(i);
						}
					}

					if (!keyName.isEmpty() && !keyValue.isEmpty())
					{
						propMap[keyName]=keyValue;
					}
				}
				else if(name.toLower()=="title")
				{
					if(inTitle) // WHOOPS, we are already in an entry, this should NEVER happen
					{
						kDebug(66666) << "STOP, TITLE INSIDE TITLE!" << endl;
						return false;
					}
//					kDebug(66666) << "<TITLE> ======" << endl;
					inTitle=true;
				}
/*				else
				{
					kDebug(66666) << "Unknown/unused element inside ENTRY block, NAME='" << name << "'" << endl;
					for (int i=0; i<a.count(); i++)
						kDebug(66666) << " | " << a.qName(i) << " = '" << a.value(i) << "'" << endl;
				}*/
			}
/*			else
			{
				kDebug(66666) << "Uninteresting element, NAME='" << name << "'" << endl;
				for (int i=0; i<a.count(); i++)
					kDebug(66666) << "  | " << a.qName(i) << " = '" << a.value(i) << "'" << endl;
			}*/
		}

		return true;
	}

	bool endElement(const QString&,const QString&, const QString& name)
	{
		if (!inEntry)
			return false;

		if (name.toLower() == "entry")
		{
			saver->readItem(propMap);
			propMap.clear();
			inEntry = false;
		}
		else if (name.toLower() == "title")
		{
			if(inTitle)
			{
				inTitle = false;
			}
			else if (inTitle) // found </title> without a start or not inside an <entry> ... </entry>
			{
				//kDebug(66666) << "STOP, </TITLE> without a start" << endl;
				return false;
			}
		}
		return true;
	}

	bool characters(const QString &ch)
	{
		if(inTitle && !ch.isEmpty())
			propMap["title"] = ch;
		return true;
	}
};



bool PlaylistSaver::loadASX(const QString &file, const KUrl &originalUrl, const PlaylistType opt)
{
	Q_UNUSED(opt);
	kDebug(66666) << k_funcinfo << "url:" << originalUrl << "; opt:" << opt << endl;
	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	reset();

	QXmlInputSource source(&f);
	QXmlSimpleReader reader;

	MSASXStructure ASXparser(this, originalUrl.path()); // path() argument? TODO KDE4
	reader.setContentHandler(&ASXparser);
	reader.parse(source);
	return !ASXparser.fresh;
}

bool PlaylistSaver::loadXML(const QString &file, const PlaylistType opt)
{
	Q_UNUSED(opt);
	kDebug(66666) << k_funcinfo << "file:" << file << "; opt:" << opt << endl;

	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	reset();

	QXmlInputSource source(&f);
	QXmlSimpleReader reader;

	NoatunXMLStructure parser(this);
	reader.setContentHandler(&parser);
	reader.parse(source);
	return !parser.fresh;
}

bool PlaylistSaver::loadM3U(const QString &file, const KUrl &originalUrl, const PlaylistType opt)
{
	Q_UNUSED(opt);
	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	QTextStream t(&f);

	bool isExt = false; // flag telling if we load an EXTM3U file
	QString filename;
	QString extinf;
	Noatun::PropertyMap prop;

	reset();

	while (!t.atEnd())
	{
		if (isExt)
		{
			extinf = t.readLine();
			if (!extinf.startsWith("#EXTINF:"))
			{
				filename = extinf;
				extinf="";
			}
			else
			{
				filename = t.readLine(); // read in second line containing the filename
			}
		}
		else // old style m3u
		{
			filename = t.readLine();
		}

		if (filename == "#EXTM3U") // on first line
		{
			isExt = true;
			continue; // skip parsing the first (i.e. this) line
		}

		if (filename.isEmpty())
			continue;

		if (filename.contains(QRegExp("^[a-zA-Z0-9]+:/")))
		{
			//kDebug(66666) << k_funcinfo << "url filename = " << filename << endl;

			KUrl protourl(filename);
			KMimeType::Ptr mimetype = KMimeType::findByUrl(protourl);

			if (mimetype->name() != "application/octet-stream")
			{
				prop["url"] = filename;
			}
			else
			{
				//prop["playObject"]="SplayPlayObject";
				// Default title, might be overwritten by #EXTINF later
				prop["title"] = i18n("Stream from %1",protourl.host());

				if (!protourl.hasPath())
					protourl.setPath("/");

				prop["url"] = protourl.url();
				prop["stream_"] = prop["url"];
			}
		}
		else // filename that is not of URL style (i.e. NOT "proto:/path/somefile")
		{
			KUrl u1;
			// we have to deal with a relative path
			if (filename.contains('/'))
			{
				u1.setPath(originalUrl.path()); // KDE4 TODO
				u1.setFileName(filename);
			}
			else
			{
				u1.setPath(filename);
			}
			prop["url"] = u1.url();
		}

		// parse line of the following format:
		//#EXTINF:length,displayed_title
		if (isExt)
		{
			extinf.remove(0,8); // remove "#EXTINF:"
			int timeTitleSep = extinf.indexOf(',');

			int length = (extinf.left(timeTitleSep)).toInt();
			if (length > 0)
				prop["length"] = QString::number(length * 1000);

			QString displayTitle = extinf.mid(timeTitleSep + 1);
			if (!displayTitle.isEmpty())
			{
				int artistTitleSep = displayTitle.indexOf(" - ");
				if (artistTitleSep == -1) // no "artist - title" like format, just set it as title
				{
					prop["title"] = displayTitle;
				}
				else
				{
					prop["author"] = displayTitle.left(artistTitleSep);
					prop["title"] = displayTitle.mid(artistTitleSep+3);
				}
			} // END !displayTitle.isEmpty()
		} // END if(isExt)

		readItem(prop);
		prop.clear();
	} // END while()

	return true;
}


bool PlaylistSaver::saveM3U(QTextStream &stream, PlaylistType opt)
{
	bool isExt = (opt==EXTM3U);

	reset();
	PlaylistItem i;
	QStringList props;

	// this is more code but otoh faster than checking for isExt inside the loop
	if(isExt)
	{
		stream << "#EXTM3U" << '\n';

		while ((i = writeItem()))
		{
			int length = static_cast<int>(((i.property("length")).toInt())/1000);

			// special value in an EXTM3U file, means "unknown length"
			if (length == 0)
				length = -1;

			KUrl u(i.property("url"));
			QString title;

		// if a playlistitem is without a tag or ONLY title is set
			if((i.property("author").isEmpty() && i.property("title").isEmpty())
				|| (i.property("author").isEmpty() && !i.property("title").isEmpty()) )
			{
				title = u.fileName().left(u.fileName().length()-4);
			}
			else
			{
				title = i.property("author") + " - " + i.property("title");
			}

			stream << "#EXTINF:"<< QString::number(length) << "," << title << '\n';

			if (u.isLocalFile())
				stream << u.path() << '\n';
			else
				stream << u.url() << '\n';
		}
	}
	else
	{
		while ((i = writeItem()))
		{
			KUrl u(i.property("url"));
			if (u.isLocalFile())
				stream << u.path() << '\n';
			else
				stream << u.url() << '\n';
		}
	}

	return true;
}

static QString findNoCase(const Noatun::PropertyMap &map, const QString &key)
{
	Noatun::PropertyMap::ConstIterator it(map.begin());
	Noatun::PropertyMap::ConstIterator end(map.end());
	const QString lowerKey = key.toLower();
	for ( ; it != end; ++it)
	{
		if (it.key().toLower() == lowerKey)
			return it.value();
	}
	return QString();
}

bool PlaylistSaver::loadPLS(const QString &file, const PlaylistType /*opt*/)
{
	QFile checkFile(file);
	if (!checkFile.open(QIODevice::ReadOnly))
		return false;
	QTextStream t(&checkFile);
	QString firstLine = t.readLine();
	if(firstLine.toLower() != "[playlist]")
	{
		kWarning(66666) <<
			"PLS file '" << file <<
			"' did not start with '[playlist]', aborting loading" << endl;
		return false;
	}
	checkFile.close();


	KSimpleConfig list(file, true);
	//list.setGroup("playlist");
	// some windows users like to be case insensitive, oh my
	QStringList groups = list.groupList().filter(QRegExp("^playlist$", Qt::CaseInsensitive));
	QMap<QString, QString> group = list.entryMap(groups[0]);

	QString numOfEntries = findNoCase(group, "numberofentries");
	if(numOfEntries.isEmpty())
		return false;

	reset();

	unsigned int nEntries = numOfEntries.toUInt();
	for(unsigned int entry = 1; entry <= nEntries; ++entry )
	{
		Noatun::PropertyMap map;
		QString str;
		str.sprintf("file%d", entry);
		QString cast = findNoCase(group, str.toUtf8());
		str.sprintf("title%d", entry);
		QString title = findNoCase(group, str.toUtf8());

		// TODO: This assumes that everything in a pls is a streamable file
		KUrl url(cast);
		if (!url.hasPath())
			url.setPath("/");

		if (title.isEmpty())
			map["title"] = i18n("Stream from %1 (port: %2)",url.host(), QString::number(url.port()));
		else
			map["title"] = i18n("Stream from %1, (ip: %2, port: %3)",title, url.host(), QString::number(url.port()));

		map["url"] = map["stream_"] = url.url();

		readItem(map);
	}
	return true;
}

} // namespace Noatun
