/* This file is part of Noatun

  Copyright 2005-2006 by Charles Samuels <charles@kde.org>
  Copyright 2005-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/plugininterfaces.h"
#include "noatun/plugin.h"
#include "noatun/playlist.h"
#include "noatun/global.h"

#include <qstringlist.h>
#include <kmainwindow.h>
#include <kwin.h>
#include <kmimetype.h>

namespace Noatun
{

// ---------------------------------------------------------------------------


Interface::Interface()
{ }

Interface::~Interface()
{ }


// ---------------------------------------------------------------------------


template <>
PlaylistInterfaceGeneric *Plugin::interface<PlaylistInterfaceGeneric>()
{
	return static_cast<PlaylistInterfaceGeneric*>(getInterface("PlaylistInterface"));
}


PlaylistInterfaceGeneric::PlaylistInterfaceGeneric(Global *glob)
{
	mGlobal = glob;
}


PlaylistInterfaceGeneric::~PlaylistInterfaceGeneric()
{
}


QList<PlaylistItem> PlaylistInterfaceGeneric::select(
		const QStringList &keys,
		const QStringList &values, int limit, bool exact, bool caseSensitive
	)
{
	QList<PlaylistItem> list;
	QString k;
	QString v;
	QStringList::ConstIterator key, val;
	for (PlaylistItem i(getFirst()); i && limit; i=getAfter(i))
	{
		for (key = keys.begin(); key != keys.end() && limit ; ++key)
		{
			k=*key;
			v=i.property(k);

			for (val = values.begin(); val != values.end() && limit; ++val)
			{
				if ((*val).length()==0 && v.length()==0)
				{
					list.append(i);
					limit--;
					goto nextSong;
				}
				else if (exact)
				{
					if (
							(!caseSensitive && (*val).toLower() == v.toLower())
							|| (caseSensitive && (*val) == v)
						)
					{
						list.append(i);
						limit--;
						goto nextSong;
					}
				}
				else
				{
					if ((*val).contains(v, Qt::CaseSensitive))
					{
						list.append(i);
						limit--;
						goto nextSong;
					}
				}
			}
		}
	nextSong:
		;
	}
	return list;
}


void PlaylistInterfaceGeneric::listHidden()
{
	emit mGlobal->playlist()->listHidden();
}


void PlaylistInterfaceGeneric::listShown()
{
	emit mGlobal->playlist()->listShown();
}


bool PlaylistInterfaceGeneric::listVisible()
{
	QWidget *w = mainWindow();
	if (w)
	{
		KWin::WindowInfo info(w->winId(), NET::WMState | NET::WMDesktop, 0);
		return !(info.hasState(NET::Shaded) || info.hasState(NET::Hidden) || !info.valid() || !info.isOnCurrentDesktop());
	}
	return false;
}


void PlaylistInterfaceGeneric::showList()
{
	QWidget *w = mainWindow();
	if (w)
	{
		KWin::WindowInfo info(w->winId(), NET::WMDesktop, 0);
		// move to current desktop
		if (!info.onAllDesktops() && !info.isOnCurrentDesktop())
			KWin::setOnDesktop(w->winId(), KWin::currentDesktop());

		w->show();
		if (w->isMinimized())
			w->showNormal();
	}
}


void PlaylistInterfaceGeneric::hideList()
{
	QWidget *w = mainWindow();
	if (w)
		w->hide();
}


QString PlaylistInterfaceGeneric::mimetype_g(void *item)
{
	if (hasProperty_g(item, "mimetype"))
		return property_g(item, "mimetype");
	KMimeType::Ptr mimetype = KMimeType::findByUrl(url_g(item));

	return mimetype->name();
}


int PlaylistInterfaceGeneric::length_g(void *item)
{
	return property_g(item, "length", "-1").toInt();
}


void PlaylistInterfaceGeneric::setLength_g(void *item, int ms)
{
	setProperty_g(item, "length", QString::number(ms));
}


void PlaylistInterfaceGeneric::setProperties_g(void *item, const PropertyMap &properties)
{
	PropertyMap::ConstIterator it;
	for(it = properties.begin(); it != properties.end(); ++it)
		setProperty_g(item, it.key(), it.value());
}


bool PlaylistInterfaceGeneric::streamable_g(void *item)
{
	return hasProperty_g(item, "stream_");
}


// ---------------------------------------------------------------------------


template <>
FrontendInterface *Plugin::interface<FrontendInterface>()
{
	return static_cast<FrontendInterface*>(getInterface("FrontendInterface"));
}

FrontendInterface::FrontendInterface()
{
}

FrontendInterface::~FrontendInterface()
{
}


// ----------------------------------------------------------------------------


template <>
ItemSourceInterface *Plugin::interface<ItemSourceInterface>()
{
	return static_cast<ItemSourceInterface*>(getInterface("ItemSourceInterface"));
}


ItemSourceInterface::ItemSourceInterface()
{
}


ItemSourceInterface::~ItemSourceInterface()
{
}


// ----------------------------------------------------------------------------


template <>
EngineInterface *Plugin::interface<EngineInterface>()
{
	return static_cast<EngineInterface*>(getInterface("EngineInterface"));
}


EngineInterface::EngineInterface(Global *nInst) : nInstance(nInst)
{
}


EngineInterface::~EngineInterface()
{
}


unsigned int EngineInterface::softwareVolume() const
{
	return 100;
}


void EngineInterface::setSoftwareVolume(unsigned int percent)
{
	Q_UNUSED(percent);
}


void EngineInterface::stateChanged(Player::State newState)
{
	nInstance->player()->engineStateChanged(newState);
}


void EngineInterface::playbackFinished()
{
	nInstance->player()->enginePlaybackFinished();
}


void EngineInterface::errorOccurred(Player::ErrorType errorType, const QString &errorDescription)
{
	nInstance->player()->engineErrorOccurred(errorType, errorDescription);
}


} // namespace Noatun
