/* This file is part of Noatun

  Copyright 2004-2007 by Stefan Gehn <mETz81@web.de>
  Copyright 2005-2006 by Charles Samuels <charles@kde.org>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/stdaction.h"
#include "noatun/global.h"
#include "noatun/player.h"
#include "noatun/playlist.h"
#include "noatun/preferencesdialog.h"
#include "noatun/stereobuttonaction.h"

#include <kactioncollection.h>
#include <kdebug.h>
#include <klocalizedstring.h>
#include <kstandardaction.h>


namespace Noatun
{


PlayPauseAction::PlayPauseAction(Global *nInst, QObject *parent, const char *name)
	: KAction(parent)
{
	setObjectName(name);

	Player *p = nInst->player();
	connect(p, SIGNAL(playing()), SLOT(playing()));
	connect(p, SIGNAL(paused()), SLOT(notplaying()));
	connect(p, SIGNAL(stopped()), SLOT(notplaying()));
	QObject::connect(this, SIGNAL(triggered()), p, SLOT(playpause()));

	if (p->isPlaying())
		playing();
	else
		notplaying();
}

void PlayPauseAction::playing()
{
	setText(i18n("Play"));
	setIcon(KIcon("player_play"));
}

void PlayPauseAction::notplaying()
{
	setText(i18n("Pause"));
	setIcon(KIcon("player_pause"));
}


// ----------------------------------------------------------------------------


PlaylistAction::PlaylistAction(Global *nInst, QObject *parent, const char *name)
	: KToggleAction(parent), mGlobal(nInst)
{
	setObjectName(name);

	// normal state
	setText(i18n("Show Playlist"));
	setIcon(KIcon("noatun_playlist"));

	// checked state (when noatun is playing)
	setCheckedState(KGuiItem(i18n("Hide Playlist")));

	connect(nInst->playlist(), SIGNAL(listShown()),  SLOT(slotShown()));
	connect(nInst->playlist(), SIGNAL(listHidden()), SLOT(slotHidden()));

	setChecked(nInst->playlist()->listVisible());
}

void PlaylistAction::slotToggled(bool on)
{
	KToggleAction::slotToggled(on);
	if (on)
		mGlobal->playlist()->showList();
	else
		mGlobal->playlist()->hideList();
}

void PlaylistAction::slotShown()
{
	setChecked(true);
}

void PlaylistAction::slotHidden()
{
	setChecked(false);
}

// ----------------------------------------------------------------------------

LoopActionMenu::LoopActionMenu(Global *glob, QObject *parent, const char *name)
	: KActionMenu(parent), mGlobal(glob)
{
	setObjectName(name);
	setText(i18n("&Loop"));

	QActionGroup *loopGroup = new QActionGroup(this);
	loopGroup->setExclusive(true);

	mLoopNone = new KAction(loopGroup);
	mLoopNone->setObjectName("loop_none");
	mLoopNone->setText(i18n("&None"));
	mLoopNone->setIcon(KIcon("noatun_loop_none"));
	connect(mLoopNone, SIGNAL(triggered()), this, SLOT(loopNoneSelected()));
	addAction(mLoopNone);

	mLoopSong = new KAction(loopGroup);
	mLoopSong->setObjectName("loop_song");
	mLoopSong->setText(i18n("&Song"));
	mLoopSong->setIcon(KIcon("noatun_loop_song"));
	connect(mLoopSong, SIGNAL(triggered()), this, SLOT(loopSongSelected()));
	addAction(mLoopSong);

	mLoopPlaylist = new KAction(loopGroup);
	mLoopPlaylist->setObjectName("loop_playlist");
	mLoopPlaylist->setText(i18n("&Playlist"));
	mLoopPlaylist->setIcon(KIcon("noatun_loop_playlist"));
	connect(mLoopPlaylist, SIGNAL(triggered()), this, SLOT(loopPlaylistSelected()));
	addAction(mLoopPlaylist);

	mLoopRandom = new KAction(loopGroup);
	mLoopRandom->setObjectName("loop_random");
	mLoopRandom->setText(i18n("&Random"));
	mLoopRandom->setIcon(KIcon("noatun_loop_random"));
	connect(mLoopRandom, SIGNAL(triggered()), this, SLOT(loopRandomSelected()));
	addAction(mLoopRandom);

	//connect(mGlobal->player(), SIGNAL(loopTypeChange(int)), this, SLOT(updateLooping(int)));
	//updateLooping(static_cast<int>(mGlobal->player()->loopStyle()));

	mLoopNone->setChecked(true); // TODO: remove these if loop is implemented again
	setIcon(mLoopNone->icon());  //
}

/*void LoopActionMenu::updateLooping(int loopType)
{
	switch(loopType)
	{
	case Player::None:
		mLoopNone->setChecked(true);
		setIcon(mLoopNone->icon());
		break;
	case Player::Song:
		mLoopSong->setChecked(true);
		setIcon("noatun_loop_song");
		break;
	case Player::Playlist:
		mLoopPlaylist->setChecked(true);
		setIcon("noatun_loop_playlist");
		break;
	case Player::Random:
		mLoopRandom->setChecked(true);
		setIcon("noatun_loop_random");
		break;
	}
}*/

void LoopActionMenu::loopNoneSelected()
{
	// mGlobal->player()->loop(Player::None);
}

void LoopActionMenu::loopSongSelected()
{
	// mGlobal->player()->loop(Player::Song);
}

void LoopActionMenu::loopPlaylistSelected()
{
	// mGlobal->player()->loop(Player::Playlist);
}

void LoopActionMenu::loopRandomSelected()
{
	// mGlobal->player()->loop(Player::Random);
}

// ----------------------------------------------------------------------------

#if 0
VisActionMenu::VisActionMenu(Global *glob, QObject *parent, const char *name)
	: KActionMenu(i18n("&Visualizations"), parent, name)
{
	connect(popupMenu(), SIGNAL(aboutToShow()), this, SLOT(fillPopup()));
	connect(popupMenu(), SIGNAL(activated(int)), this, SLOT(toggleVisPlugin(int)));
	PluginInfoList avail = glob->pluginHandler()->availablePlugins("visualization");
	setEnabled(!avail.empty());

	if (KActionCollection *collection = qobject_cast<KActionCollection *>(parent))
		collection->addAction(pAction->objectName(), pAction);
}

void VisActionMenu::fillPopup()
{
	int id;
	popupMenu()->clear();
	mVisPluginMap.clear();

	PluginInfoList avail = glob->pluginHandler()->availablePlugins("visualization");

	for(PluginInfoList::Iterator i = avail.begin(); i != avail.end(); ++i)
	{
		id = popupMenu()->insertItem((*i)->name());
		mVisPluginMap[id] = (*i)->pluginName();
		popupMenu()->setItemChecked(id, (*i)->isPluginEnabled());
	}
}

void VisActionMenu::toggleVisPlugin(int id)
{
	if(!mVisPluginMap.contains(id))
		return;

	QString pluginName = mVisPluginMap[id];

	if(popupMenu()->isItemChecked(id))
	{
		if(glob->pluginHandler()->unloadPlugin(pluginName))
			popupMenu()->setItemChecked(id, false);
	}
	else
	{
		glob->pluginHandler()->loadPlugin(pluginName);
		popupMenu()->setItemChecked(id, true);
	}
}
#endif


// -----------------------------------------------------------------------------


namespace StdAction
{

/// @internal
static void setupStdAction(KAction *a, const char *name)
{
	a->setObjectName(name);
	if (KActionCollection *coll = qobject_cast<KActionCollection *>(a->parent()))
		coll->addAction(a->objectName(), a);
}


KToggleAction *playlist(Global *nInstance, QObject *parent, const char *name)
{
	KToggleAction *a = new PlaylistAction(nInstance, parent, name);
	setupStdAction(a, name);
	return a;
}


KAction *effects(Global */*nInstance*/, QObject *parent, const char *name)
{
	KAction *a = new KAction(parent);
	setupStdAction(a, name);
	a->setText(i18n("&Effects..."));
	//a->setIcon(KIcon("noatun_effects")); // TODO: Needs an icon
	a->setEnabled(false);
	return a;
}


KAction *equalizer(Global */*nInstance*/, QObject *parent, const char *name)
{
	KAction *a = new KAction(parent);
	setupStdAction(a, name);
	a->setText(i18n("E&qualizer..."));
	a->setIcon(KIcon("noatun_equalizer"));
	a->setEnabled(false); //TODO: no equalizer in Noatun3 yet
	return a;
}


KAction *back(Global *nInstance, QObject *parent, const char *name)
{
	KAction *a = new KAction(parent);
	setupStdAction(a, name);
	a->setText(i18n("&Back"));
	a->setIcon(KIcon("player_start"));
	QObject::connect(a, SIGNAL(triggered()), nInstance->player(), SLOT(previous()));
	return a;
}


KAction *stop(Global *nInstance, QObject *parent, const char *name)
{
	Player *p = nInstance->player();
	StereoButtonAction *a = new StereoButtonAction(KIcon("player_stop"), i18n("Stop"), parent);
	QObject::connect(a, SIGNAL(triggered()), p, SLOT(stop()));
	QObject::connect(p, SIGNAL(playing()), a, SLOT(enable()));
	QObject::connect(p, SIGNAL(paused()) , a, SLOT(enable()));
	QObject::connect(p, SIGNAL(stopped()), a, SLOT(disable()));
	a->setEnabled(!p->isStopped());
	setupStdAction(a, name); // name and reparent as needed
	return a;
}


KAction *forward(Global *nInstance, QObject *parent, const char *name)
{
	KAction *a = new KAction(parent);
	setupStdAction(a, name);
	a->setText(i18n("&Forward"));
	a->setIcon(KIcon("player_end"));
	QObject::connect(a, SIGNAL(triggered()), nInstance->player(), SLOT(next()));
	return a;
}


KAction *playpause(Global *nInstance, QObject *parent, const char *name)
{
	KAction *a = new PlayPauseAction(nInstance, parent, name);
	setupStdAction(a, name);
	return a;
}


KAction *play(Global *nInstance, QObject *parent, const char *name)
{
	Player *p = nInstance->player();
	StereoButtonAction *a = new StereoButtonAction(KIcon("player_play"), i18n("Play"), parent);
	QObject::connect(a, SIGNAL(triggered()), p, SLOT(play()));
	QObject::connect(p, SIGNAL(playing()), a, SLOT(disable()));
	QObject::connect(p, SIGNAL(paused()) , a, SLOT(enable()));
	QObject::connect(p, SIGNAL(stopped()), a, SLOT(enable()));
	a->setEnabled(!p->isPlaying()); // initial state
	setupStdAction(a, name); // name and reparent as needed
	return a;
}


KAction *pause(Global *nInstance, QObject *parent, const char *name)
{
	Player *p = nInstance->player();
	StereoButtonAction *a = new StereoButtonAction(KIcon("player_pause"), i18n("Pause"), parent);
	QObject::connect(a, SIGNAL(triggered()), p, SLOT(pause()));
	QObject::connect(p, SIGNAL(playing()), a, SLOT(enable()));
	QObject::connect(p, SIGNAL(paused()) , a, SLOT(disable()));
	QObject::connect(p, SIGNAL(stopped()), a, SLOT(disable()));
	a->setEnabled(p->isPlaying()); // initial state
	setupStdAction(a, name); // name and reparent as needed
	return a;
}


LoopActionMenu *loop(Global *glob, QObject *parent, const char *name)
{
	LoopActionMenu *a = new LoopActionMenu(glob, parent, name);
	setupStdAction(a, name);
	return a;
}


QAction *quit(Global *nInstance, QObject *parent, const char *name)
{
	KAction *a = KStandardAction::quit(nInstance, SLOT(quit()), parent);
	a->setObjectName(name);
	return a;
}


QAction *preferences(Global *nInstance, QObject *parent, const char *name)
{
	KAction *a = KStandardAction::preferences(nInstance->preferencesDialog(), SLOT(show()), parent);
	a->setObjectName(name);
	return a;
}


#if 0
VisActionMenu *visualizations(Global *glob, QObject *parent, const char *name)
{
	return new VisActionMenu(parent, name);
}
#endif

} // END namespace StdAction

} // END namespace Noatun

#include "stdaction.moc"
