/***************************************************************************
                     A simple database-driven playlist for Noatun
                    ---------------------------------------------
    begin                : 24.05.2005
    copyright            : (C) 2005 by Stefan Gehn
    email                : Stefan Gehn <mETz81@web.de>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "simpleplaylist.h"
#include "simpledb.h"
#include "simplemainwindow.h"
#include "simplelistview.h"
#include "itemadder.h"

#include <noatun/player.h>
#include <noatun/global.h>

#include <kmainwindow.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kio/netaccess.h>

using namespace Noatun;

K_EXPORT_COMPONENT_FACTORY(noatun_simpleplaylist, PluginFactory<SimplePlaylist>("noatun_simpleplaylist"))

SimplePlaylist::SimplePlaylist(const KComponentData &inst, Global *global, const char* name)
	: Plugin(inst, global, name), Noatun::PlaylistInterface(global), Noatun::ItemSourceInterface()
{
	//kDebug(66666) << k_funcinfo << endl;
	mItemAdder = 0;
	mDb = new SimpleDB(this);
	mWin = new SimpleMainWindow(this);
	connect(mWin->listView(), SIGNAL(executedItem(int)),
		this, SLOT(slotPlayItem(int)));
}

SimplePlaylist::~SimplePlaylist()
{
	delete mWin;
	componentData().config()->sync();
	//kDebug(66666) << k_funcinfo << "Tha playlist finished" << endl;
}

void SimplePlaylist::init()
{
	//kDebug(66666) << k_funcinfo << endl;
	KSharedConfig::Ptr c = componentData().config();

	mWin->listView()->refresh(); // fill listview from db

	c->setGroup(QString::null);
	int id = c->readNumEntry("CurrentItemId");
	if (id > 0)
		mWin->listView()->setCurrentByItemId(id);
}

void SimplePlaylist::requestUnload()
{
	KSharedConfig::Ptr c = componentData().config();
	c->setGroup(QString::null);
	c->writeEntry("CurrentItemId", mWin->listView()->currentItemId());
	debugDbMap();
	emit readyForUnload(this);
}

KMainWindow *SimplePlaylist::mainWindow() const
{
	return mWin;
}

Interface *SimplePlaylist::getInterface(const QString &name)
{
	if (name == "PlaylistInterface")
		return static_cast<Noatun::PlaylistInterface *>(this);
	if (name == "ItemSourceInterface")
		return static_cast<Noatun::ItemSourceInterface *>(this);
	return 0;
}

void SimplePlaylist::addFile(const KURL &url, bool purgeList, bool autoplay) // from PlaylistInterface
{
	Q_UNUSED(purgeList);
	kDebug(66666) << k_funcinfo << "url = " << url.prettyURL() << endl;
	addSingleFile(url, -1, autoplay);
}

void SimplePlaylist::addSingleFile(const KURL &url, int afterId, bool autoplay) // public
{
	if (!mItemAdder)
	{
		mItemAdder = new ItemAdder(this, url, afterId, autoplay);
		connect (mItemAdder, SIGNAL(finished()), this, SLOT(slotItemAdderFinished()));
		mItemAdder->start();
	}
	else
		kDebug(66666) << k_funcinfo << "OOPS, already running an ItemAdder" << endl;
}

void SimplePlaylist::addFiles(const KURL::List &urls, int afterId, bool autoplay) // public
{
	if (!mItemAdder)
	{
		mItemAdder = new ItemAdder(this, urls, afterId, autoplay);
		connect (mItemAdder, SIGNAL(finished()), this, SLOT(slotItemAdderFinished()));
		mItemAdder->start();
	}
	else
		kDebug(66666) << k_funcinfo << "OOPS, already running an ItemAdder" << endl;
}

void SimplePlaylist::slotItemAdderFinished()
{
	kDebug(66666) << k_funcinfo << endl;
	delete mItemAdder;
	mItemAdder = 0;
}

int SimplePlaylist::addFile(const KURL &url, int length, const Noatun::PropertyMap &properties, int afterId)
{
	//kDebug(66666) << k_funcinfo << "url = " << url.prettyURL() << ", afterId = " << afterId << endl;
	int id;
	SimplePlaylistItem *item = new SimplePlaylistItem(this, url, length, properties, afterId);
	PlaylistItem ref(item); // just to ref/unref this item
	ref.added(global());
	id = item->id();
	return id;
}

Noatun::PlaylistItem SimplePlaylist::getFirst() const // from PlaylistInterface
{
	Noatun::PlaylistItem ret;
	int firstId = mDb->getFirstItemId();
	// first added item will always get id = 1, 0 means no items
	if (firstId != 0)
	{
		SimplePlaylistItem *item = getById(firstId);
		if (item)
			ret = Noatun::PlaylistItem(item);
	}
	//kDebug(66666) << k_funcinfo << "Returning PlaylistItem for id " << firstId << endl;
	return ret;
}

Noatun::PlaylistItem SimplePlaylist::getAfter(const PlaylistItem &item) const // from PlaylistInterface
{
	Noatun::PlaylistItem ret;
	const SimplePlaylistItem *pli = static_cast<const SimplePlaylistItem *>(item.data());
	if (pli)
	{
		int nextId = mDb->getNextItemId(pli->id());
		if (nextId != -1)
			ret = Noatun::PlaylistItem(getById(nextId));
		//kDebug(66666) << k_funcinfo << "Returning PlaylistItem for id " << nextId << endl;
	}
	return ret;
}

Noatun::PlaylistItem SimplePlaylist::forward() // from ItemSourceInterface
{
	Noatun::PlaylistItem ret;
	int id;
	SimpleListView *lv;

	lv = mWin->listView();
	id = lv->nextItemId();

	if (id == -1) // wrap to first item
		id = lv->firstItemId();
	if (id != -1)
		ret = Noatun::PlaylistItem(getById(id));

	kDebug(66666) << k_funcinfo << "Returning PlaylistItem for id " << id << endl;
	return ret;
}

Noatun::PlaylistItem SimplePlaylist::backward() // from ItemSourceInterface
{
	Noatun::PlaylistItem ret;
	int id;
	SimpleListView *lv;

	lv = mWin->listView();
	id = lv->previousItemId();
	if (id != -1)
		ret = Noatun::PlaylistItem(getById(id));

	kDebug(66666) << k_funcinfo << "Returning PlaylistItem for id " << id << endl;
	return ret;
}

void SimplePlaylist::setCurrent(const Noatun::PlaylistItem &item) // from ItemSourceInterface
{
	const SimplePlaylistItem *pli;

	pli = static_cast<const SimplePlaylistItem *>(item.data());
	if (pli)
		mWin->listView()->setCurrentByItemId(pli->id());
}

SimplePlaylistItem *SimplePlaylist::getById(int id) const
{
	SimplePlaylistItem *plitem = 0;
	QMapConstIterator<int, SimplePlaylistItem*> iter = mDbMap.find(id);
	if (iter != mDbMap.end())
	{
		plitem = iter.data();
	}
	else
	{
		/*kDebug(66666) << k_funcinfo <<
			"Could not find a SimplePlaylistItem for item with db id " <<
			id << ", creating new one" << endl;*/
		plitem = new SimplePlaylistItem(const_cast<SimplePlaylist *>(this), id);
	}
	return plitem;
}

void SimplePlaylist::debugDbMap() // public slot
{
	QMapConstIterator<int, SimplePlaylistItem*> iter;
	kDebug(66666) << k_funcinfo << "Printing list of 'known' playlist items in memory -------------" << endl;
	for (iter = mDbMap.begin(); iter != mDbMap.end(); ++iter)
		kDebug(66666) << "id " << iter.key() << "; url '" << iter.data()->url() << "'" << endl;
	kDebug(66666) << k_funcinfo << "---------------------------------------------------------------" << endl;
}

SimpleDB *SimplePlaylist::db()
{
	return mDb;
}

// a fresh SimplePlaylistItem was created in memory
void SimplePlaylist::itemCreated(SimplePlaylistItem *item) // protected
{
	mDbMap.insert(item->id(), item);
}

// Some SimplePlaylistItem is gone from memory
void SimplePlaylist::itemDestroyed(SimplePlaylistItem *item) // protected
{
	QMapIterator<int, SimplePlaylistItem*> iter = mDbMap.find(item->id());
	if (iter != mDbMap.end())
		mDbMap.remove(iter);
}

void SimplePlaylist::slotPlayItem(int id) // private slot
{
	SimplePlaylistItem *item = getById(id);
	if (item)
	{
		PlaylistItem pl(item);
		global()->player()->play(pl);
	}
}

void SimplePlaylist::purgeList() // public slot
{
	KURL url;
	int currentId = mDb->getFirstItemId();
	while(currentId > 0)
	{
		SimplePlaylistItem *item = getById(currentId);
		if (!item)
			break;

		url = item->url();
		if(!KIO::NetAccess::exists(url, true, mWin))
			item->remove();

		currentId = mDb->getNextItemId(item->id());
	}
}


// ============================================================================
// ============================================================================


SimplePlaylistItem::SimplePlaylistItem(SimplePlaylist *pl,
	const KURL &url,
	int length,
	const Noatun::PropertyMap &props,
	int insertAfterId) : PlaylistItemData(pl), bRemoved(false), playlist(pl)
{
	mItem.url = url;
	mItem.properties = props;
	mItem.length = length;
	// sets mItem.id automatically
	if (playlist->db()->insertItem(mItem, insertAfterId))
		playlist->itemCreated(this); // inform playlist about new data in memory
}

SimplePlaylistItem::SimplePlaylistItem(SimplePlaylist *pl, int id)
	: PlaylistItemData(pl), bRemoved(false), playlist(pl)
{
	mItem.id = id;
	kDebug(66666) << k_funcinfo << "Item with id " << mItem.id << " created" << endl;
	mItem = playlist->db()->getItem(id);
	playlist->itemCreated(this); // inform playlist about new data in memory
}

SimplePlaylistItem::~SimplePlaylistItem()
{
	kDebug(66666) << k_funcinfo << "item with id " << mItem.id << " is gone" << endl;
	playlist->itemDestroyed(this); // inform playlist about data vanishing from memory
}

QString SimplePlaylistItem::property(const QString &key, const QString &def) const
{
	Noatun::PropertyMap::ConstIterator valIt = mItem.properties.find(key);
	if (valIt == mItem.properties.end())
		return def;
	return *valIt;
}

void SimplePlaylistItem::setProperty(const QString &key, const QString &val)
{
	if (bRemoved)
		return;

	if (val != property(key, QString::null))
	{
		kDebug(66666) << k_funcinfo <<
			"id = " << mItem.id << ": " << key << " => " << val << endl;

		mItem.properties.insert(key, val, true);
		playlist->db()->setProperty(mItem.id, key, val);
		//playlist->updatedItem(this);
		PlaylistItem item(this);
		item.modified(playlist->global());
	}
}

void SimplePlaylistItem::setProperties(const Noatun::PropertyMap &properties)
{
	if (bRemoved)
		return;
	PropertyMap changedProperties;
	PropertyMap::ConstIterator it;
	QString key, val;
	for(it = properties.begin(); it != properties.end(); ++it)
	{
		key = it.key();
		val = it.data();
		if (val != property(key, QString::null))
		{
			/*kDebug(66666) << k_funcinfo <<
				"id = " << mItem.id << ": " << key << " => " << val << endl;*/
			mItem.properties.insert(key, val, true); // update internal item
			changedProperties.insert(key, val); // save in db later on
		}
	}

	if (!changedProperties.empty())
	{
		playlist->db()->setProperties(mItem.id, changedProperties);
		//playlist->updatedItem(this);
		PlaylistItem item(this);
		item.modified(playlist->global());
	}
}

void SimplePlaylistItem::clearProperty(const QString &key)
{
	Noatun::PropertyMap::Iterator valIt = mItem.properties.find(key);
	if (valIt != mItem.properties.end())
	{
		mItem.properties.erase(valIt);
		playlist->db()->deleteProperty(mItem.id, key);
		//playlist->updatedItem(this);
		PlaylistItem item(this);
		item.modified(playlist->global());
	}
}

QStringList SimplePlaylistItem::properties() const
{
	return mItem.properties.keys();
}

bool SimplePlaylistItem::hasProperty(const QString &key) const
{
	return mItem.properties.contains(key);
}

KURL SimplePlaylistItem::url() const
{
	return mItem.url;
}

void SimplePlaylistItem::setUrl(const KURL &url)
{
	if (mItem.url == url)
		return;
	if (playlist->db()->setUrl(mItem.id, url))
	{
		mItem.url = url;
		//playlist->updatedItem(this);
		PlaylistItem item(this);
		item.modified(playlist->global());
	}
}

int SimplePlaylistItem::length() const
{
	return mItem.length;
}

void SimplePlaylistItem::setLength(int ms)
{
	if (mItem.length == ms)
		return;

	/*kDebug(66666) << k_funcinfo <<
		"Length for item " << mItem.id << " is now " << ms << " msecs" << endl;*/

	if (playlist->db()->setLength(mItem.id, ms))
	{
		mItem.length = ms;
		//playlist->updatedItemLength(this);
		PlaylistItem item(this);
		item.modified(playlist->global());
	}
}

void SimplePlaylistItem::remove()
{
	kDebug(66666) << k_funcinfo << "REMOVING item with id " << mItem.id << endl;
	bRemoved = true;
	//playlist->deletedItem(this);
	playlist->db()->deleteItem(mItem.id);
	PlaylistItem item(this);
	item.removed(playlist->global());
}

int SimplePlaylistItem::id() const
{
	return mItem.id;
}

#include "simpleplaylist.moc"
