/*  This file is part of the KDE project
    Copyright (C) 2006 Matthias Kretz <kretz@kde.org>
    Copyright (C) 2006-2007 Tim Beaulen <tbscope@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "backend.h"
#include "mediaobject.h"
#include "mediaqueue.h"
#include "avcapture.h"
#include "bytestream.h"
#include "audiopath.h"
#include "audioeffect.h"
#include "audiooutput.h"
#include "audiodataoutput.h"
#include "visualization.h"
#include "videopath.h"
#include "videoeffect.h"
#include "brightnesscontrol.h"
#include "volumefadereffect.h"
#include "videodataoutput.h"
#include "pipeline.h"

#include "qbtgstreamer/qbtgstreamerelementfactory.h"

#include <QSet>
#include <QVariant>

#include <kgenericfactory.h>
#include <kdebug.h>

#include <phonon/audiodevice.h>
#include <phonon/audiodeviceenumerator.h>

typedef KGenericFactory<Phonon::GStreamer::Backend, Phonon::GStreamer::Backend> GStreamerBackendFactory;
K_EXPORT_COMPONENT_FACTORY(phonon_gst, GStreamerBackendFactory("gstreamerbackend"))

namespace Phonon
{
namespace GStreamer
{

Backend *Backend::m_instance;

Backend::Backend(QObject *parent, const QStringList &)
    : QObject(parent)
{
    m_instance = this;

    QbtGStreamer::init(0, 0);

    kDebug(611) << "Using " << QbtGStreamer::versionString() << endl;

    m_player = new PlayBin;

    m_player->setConfig(GStreamerBackendFactory::componentData().config());

    // Fill the list with supported mimetypes
    m_supportedMimeTypes = m_player->supportedMimeTypes();

    new Pipeline(GStreamerBackendFactory::componentData().config());

    // For testing (looking at the available mimetypes)
    //knownMimeTypes();
}

Backend::~Backend()
{
}

QObject* Backend::createObject0(BackendInterface::Class0 c, QObject *parent)
{
    switch (c) {
    case MediaObjectClass:
        return new MediaObject(this, parent);
    case MediaQueueClass:
        return 0;
    case AvCaptureClass:
        return 0;
    case ByteStreamClass:
        return 0;
    case AudioPathClass:
        return new AudioPath(parent);
    case VolumeFaderEffectClass:
        return 0;
    case AudioOutputClass:
        return new AudioOutput(parent);
    case AudioDataOutputClass:
        return new AudioDataOutput(parent);
    case VisualizationClass:
        return 0;
    case VideoPathClass:
        return new VideoPath(parent);
    case BrightnessControlClass:
        return 0;
    case VideoDataOutputClass:
        return 0;
    }

    return 0;
}

QObject *Backend::createObject1(BackendInterface::Class1 c, QObject *parent, QVariant arg1)
{
    Q_UNUSED(parent)
    Q_UNUSED(arg1)

    switch (c) {
    case AudioEffectClass:
        return 0;
    case VideoEffectClass:
        return 0;
    }

    return 0;
}

bool Backend::supportsVideo() const
{
    return true;
}

bool Backend::supportsOSD() const
{
    return false;
}

bool Backend::supportsFourcc(quint32 fourcc) const
{
    switch (fourcc) {
    case 0x00000000:
        return true;
    default:
        return false;
    }
}

bool Backend::supportsSubtitles() const
{
    return false;
}

QStringList Backend::knownMimeTypes() const
{
    if (m_supportedMimeTypes.isEmpty()) {
        const_cast<Backend *>(this)->m_supportedMimeTypes
                << QLatin1String( "audio/vorbis" )
                << QLatin1String( "audio/x-mp3" )
                << QLatin1String( "audio/x-wav" )
                << QLatin1String( "video/x-ogm" );
    }

    return m_supportedMimeTypes;
}

QSet<int> Backend::objectDescriptionIndexes(ObjectDescriptionType type) const
{
    QSet<int> set;

    switch (type) {
    case Phonon::AudioOutputDeviceType:
        return m_player->audioOutputIndexes();
    case Phonon::AudioCaptureDeviceType:
    {
        QList<AudioDevice> devlist = AudioDeviceEnumerator::availableCaptureDevices();
        foreach (AudioDevice dev, devlist) {
            set << dev.index();
        }
        break;
    }
    case Phonon::VideoOutputDeviceType:
        set << 40000 << 40001 << 40002 << 40003;
        break;
    case Phonon::VideoCaptureDeviceType:
        set << 30000 << 30001;
        break;
    case Phonon::VisualizationType:
    case Phonon::AudioCodecType:
    case Phonon::VideoCodecType:
    case Phonon::ContainerFormatType:
        break;
    case Phonon::AudioEffectType:
        set << 0x7F000001;
        break;
    case Phonon::VideoEffectType:
        set << 0x7E000001;
        break;
    }

    return set;
}

QHash<QByteArray, QVariant> Backend::objectDescriptionProperties(ObjectDescriptionType type, int index) const
{
    QHash<QByteArray, QVariant> ret;

    switch (type) {
    case Phonon::AudioOutputDeviceType:
    {
        ret.insert("name", m_player->audioOutputName(index));
        ret.insert("description", m_player->audioOutputDescription(index));
        QString icon = m_player->audioOutputIcon(index);

        if (!icon.isEmpty()) {
            ret.insert("icon", icon);
        }
    
        ret.insert("available", m_player->audioOutputAvailable(index));
        break;
    }
    case Phonon::AudioCaptureDeviceType:
    {
        QList<AudioDevice> devlist = AudioDeviceEnumerator::availableCaptureDevices();
            
        foreach (AudioDevice dev, devlist) {
            if (dev.index() == index) {
                ret.insert("name", dev.cardName());
                
                 switch (dev.driver()) {
                 case Solid::AudioHw::Alsa:
                     ret.insert("description", i18n("ALSA Capture Device"));
                     break;
                 case Solid::AudioHw::OpenSoundSystem:
                     ret.insert("description", i18n("OSS Capture Device"));
                     break;
                 case Solid::AudioHw::UnknownAudioDriver:
                     break;
                 }

                 ret.insert("icon", dev.iconName());
                 ret.insert("available", dev.isAvailable());
                 break;
             }
         }
         break;
    }
    case Phonon::VideoOutputDeviceType:
        switch (index) {
        case 40000:
            ret.insert("name", QLatin1String("XVideo"));
            break;
        case 40001:
            ret.insert("name", QLatin1String("XShm"));
            break;
        case 40002:
            ret.insert("name", QLatin1String("X11"));
            break;
        case 40003:
            ret.insert("name", QLatin1String("SDL"));
            break;
        }
        break;
    case Phonon::VideoCaptureDeviceType:
        switch (index) {
        case 30000:
            ret.insert("name", QLatin1String("USB Webcam"));
            ret.insert("description", QLatin1String("first description"));
            break;
        case 30001:
            ret.insert("name", QLatin1String("DV"));
            ret.insert("description", QLatin1String("second description"));
            break;
        }
        break;
   case Phonon::VisualizationType:
        break;
   case Phonon::AudioCodecType:
        break;
   case Phonon::VideoCodecType:
        break;
   case Phonon::ContainerFormatType:
        break;
   case Phonon::AudioEffectType:
        switch (index) {
        case 0x7F000001:
            ret.insert("name", QLatin1String("Delay"));
            ret.insert("description", QLatin1String("Simple delay effect with time, feedback and level controls."));
            break;
        }
        break;
    case Phonon::VideoEffectType:
        switch (index) {
        case 0x7E000001:
            ret.insert("name", QLatin1String("VideoEffect1"));
            ret.insert("description", QLatin1String("Description 1"));
            break;
        }
        break;
    }

    return ret;
}

char const *Backend::uiLibrary() const
{
    return "phonon_gstui";
}

QbtGStreamerElement *Backend::sourceElement()
{
    return m_sourceElement;
}

QbtGStreamerPipeline *Backend::pipeline()
{
    return m_pipeline;
}

PlayBin *Backend::player()
{
    return m_player;
}

void Backend::slotHaveType(QbtGStreamerTypeFindElement *element, uint probability, QbtGStreamerCapabilities *caps, QbtGStreamerDataPointer *data)
{
    Q_UNUSED(data)
    
    kDebug(611) << "Detected media type " << caps->toString() <<" (probability " << probability << "%)" << endl;

    QbtGStreamerPad *pad;

    pad = element->pad("src");
    tryToPlug(pad, caps);

    delete pad;
} 

void Backend::tryToPlug(QbtGStreamerPad *pad, QbtGStreamerCapabilities *caps)
{
    Q_UNUSED(pad)
    /* don't plug if we're already plugged*/
    QbtGStreamerPad *audioPad = m_audioSink->pad("sink");
    if (audioPad->linked()) {
        kDebug(611) << "Already linked!" << endl;
        return;
    }

    QbtGStreamerCapabilities *audioCapabilities = audioPad->capabilities();
    QbtGStreamerCapabilities *res = caps->intersect(audioCapabilities);

    if (res && !res->isEmpty()) {
         kDebug(611) << "Found pad to link to audiosink - plugging is now done" << endl;
    }

    delete audioCapabilities;
    delete res;
}

void Backend::freeSoundcardDevices()
{
}

}}

#include "backend.moc"

// vim: sw=4 ts=4 et
