/*  This file is part of the KDE project
    Copyright (C) 2006 Matthias Kretz <kretz@kde.org>
    Copyright (C) 2006-2007 Tim Beaulen <tbscope@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "mediaobject.h"
#include <QTimer>
#include <kdebug.h>

namespace Phonon
{
namespace GStreamer
{
MediaObject::MediaObject(Backend *backend, QObject *parent)
    : AbstractMediaProducer(backend, parent)
    , m_aboutToFinishNotEmitted(true)
    , m_backend(backend)
{
    //kDebug(611) << k_funcinfo << endl;
}

MediaObject::~MediaObject()
{
    //kDebug(611) << k_funcinfo << endl;
}

KUrl MediaObject::url() const
{
    //kDebug(611) << k_funcinfo << endl;
    return m_url;
}

qint64 MediaObject::totalTime() const
{
    //kDebug(611) << k_funcinfo << endl;
    return m_backend->player()->duration();
}

qint32 MediaObject::aboutToFinishTime() const
{
    //kDebug(611) << k_funcinfo << endl;
    return m_aboutToFinishTime;
}

void MediaObject::setUrl(const KUrl &url)
{
    kDebug(611) << k_funcinfo << endl;

    stop();
    m_url = url;
    emit length(totalTime());

    m_backend->player()->setUrl(url);

    //kDebug(611) << "##### " << url.url().toLocal8Bit().data() << endl;
    /*QMultiMap<QString, QString> metaData;
    metaData.insert("TITLE", "Fake video");
    metaData.insert("ARTIST", "Matthias Kretz");
    emit metaDataChanged(metaData);*/
}

void MediaObject::setAboutToFinishTime(qint32 newAboutToFinishTime)
{
    //kDebug(611) << k_funcinfo << endl;

    m_aboutToFinishTime = newAboutToFinishTime;

    if (currentTime() < totalTime() - m_aboutToFinishTime) // not about to finish
        m_aboutToFinishNotEmitted = true;
}

void MediaObject::play()
{
    kDebug(611) << k_funcinfo << endl;

    AbstractMediaProducer::play();
    m_backend->player()->play();
}

void MediaObject::pause()
{
    //kDebug(611) << k_funcinfo << endl;

    if (state() == PlayingState || state() == BufferingState) {
        AbstractMediaProducer::pause();
    }

    m_backend->player()->pause();
}

void MediaObject::stop()
{
    //kDebug(611) << k_funcinfo << endl;

    AbstractMediaProducer::stop();
    m_aboutToFinishNotEmitted = true;
    m_backend->player()->stop();
}

void MediaObject::seek(qint64 time)
{
    //kDebug(611) << k_funcinfo << endl;

    AbstractMediaProducer::seek(time);
    m_backend->player()->seek(time);

    if (currentTime() < totalTime() - m_aboutToFinishTime) // not about to finish
        m_aboutToFinishNotEmitted = true;
}

void MediaObject::emitTick()
{
    if (currentTime() >= totalTime() - m_aboutToFinishTime) { // about to finish
        if (m_aboutToFinishNotEmitted) {
            m_aboutToFinishNotEmitted = false;
            emit aboutToFinish(totalTime() - currentTime());
        }
    }

    if (currentTime() >= totalTime()) { // finished
        stop();
        emit finished();
    }
}

}}

#include "mediaobject.moc"
// vim: sw=4 ts=4 noet
