/*  This file is part of the KDE project
    Copyright (C) 2006 Matthias Kretz <kretz@kde.org>
    Copyright (C) 2006-2007 Tim Beaulen <tbscope@gmail.com>

    Based on code by Fluendo:
    Copyright (C) 2006 Wim Taymans <wim@fluendo.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "pipelineitem.h"

#include <QString>
#include <QByteArray>
#include <QGraphicsScene>
#include <QRectF>
#include <QFontMetrics>
#include <QFont>
#include <QtGlobal>
#include <QPixmap>
#include <QPixmapCache>
#include <QImage>
#include <QPainterPath>

#include "paramitem.h"
#include "childitem.h"

#include <kdebug.h>

namespace Phonon
{
namespace GStreamer
{
QPixmap PipelineItem::cached(const QString &img)
{
    if (QPixmap *p = QPixmapCache::find(img))
        return *p;

    QPixmap pm;
    pm = QPixmap::fromImage(QImage(img), Qt::OrderedDither | Qt::OrderedAlphaDither);
    if (pm.isNull())
        return QPixmap();

    QPixmapCache::insert(img, pm);
    return pm;
}

PipelineItem::PipelineItem(const QDomElement &element)
{
    m_pipelineName = element.firstChildElement("gst:name").text();

    int offset = 0;

    QDomElement child = element.firstChildElement("gst:param");
    while (!child.isNull()) {
        ParamItem *p = new ParamItem(child, this);
        p->setPos(10, 30 + offset);
        offset += 25;
        child = child.nextSiblingElement("gst:param");
    }   

    offset += 50;
    int offsetx = 0;

    QDomElement childroot = element.firstChildElement("gst:children");
    //while (!childroot.isNull()) {
        kDebug() << "         ##############################################" << endl;
        child = childroot.firstChildElement("gst:element");
       // while (!child.isNull()) {
            kDebug() << "##############################################" << endl;
            ChildItem *p = new ChildItem(child, this);
            p->setPos(10 + offsetx, 30 + offset);
            offsetx += 200;
            offset += 200;
            child = child.nextSiblingElement("gst:element");
        //}
        childroot = childroot.nextSiblingElement("gst:children");
    //} 
}

QRectF PipelineItem::boundingRect() const
{
    QFont font;
    QFontMetrics fm(font);

    QRectF f;

    if (children().isEmpty()) {
        f.setWidth(fm.boundingRect(m_pipelineName).width() + 150);
        f.setHeight(fm.boundingRect(m_pipelineName).height() + 450);
    } else {
        f = childrenBoundingRect();
        f.setX(0);
        f.setY(0);
        f.adjust(0,0,10,10); 
    }

    return f;
}

void PipelineItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QRect r = boundingRect().toRect();

    painter->save();

    painter->setRenderHint(QPainter::Antialiasing);

    int radius = 14;
    int radius2 = radius*2;
    QPainterPath clipPath;
    clipPath.moveTo(radius, 0);
    clipPath.arcTo(r.right() - radius2, 0, radius2, radius2, 90, -90);
    clipPath.arcTo(r.right() - radius2, r.bottom() - radius2, radius2, radius2, 0, -90);
    clipPath.arcTo(r.left(), r.bottom() - radius2, radius2, radius2, 270, -90);
    clipPath.arcTo(r.left(), r.top(), radius2, radius2, 180, -90);
    painter->setClipPath(clipPath);
    QPixmap titleStretch = cached(":/images/title_stretch.png");
    QPixmap topLeft = cached(":/images/groupframe_topleft.png");
    QPixmap topRight = cached(":/images/groupframe_topright.png");
    QPixmap bottomLeft = cached(":/images/groupframe_bottom_left.png");
    QPixmap bottomRight = cached(":/images/groupframe_bottom_right.png");
    QPixmap leftStretch = cached(":/images/groupframe_left_stretch.png");
    QPixmap topStretch = cached(":/images/groupframe_top_stretch.png");
    QPixmap rightStretch = cached(":/images/groupframe_right_stretch.png");
    QPixmap bottomStretch = cached(":/images/groupframe_bottom_stretch.png");
    QLinearGradient lg(0, 0, 0, r.height());
    lg.setColorAt(0, QColor(224,224,224));
    lg.setColorAt(1, QColor(255,255,255));
    painter->setPen(Qt::NoPen);
    painter->setBrush(lg);
    painter->drawRect(r.adjusted(0, titleStretch.height()/2, 0, 0));
    painter->setClipping(false);

    int topFrameOffset = titleStretch.height()/2 - 2;
    painter->drawPixmap(r.topLeft() + QPoint(0, topFrameOffset), topLeft);
    painter->drawPixmap(r.topRight() - QPoint(topRight.width()-1, 0)
                        + QPoint(0, topFrameOffset), topRight);
    painter->drawPixmap(r.bottomLeft() - QPoint(0, bottomLeft.height()-1), bottomLeft);
    painter->drawPixmap(r.bottomRight() - QPoint(bottomRight.width()-1,
                        bottomRight.height()-1), bottomRight);

    QRect left = r;
    left.setY(r.y() + topLeft.height() + topFrameOffset);
    left.setWidth(leftStretch.width());
    left.setHeight(r.height() - topLeft.height() - bottomLeft.height() - topFrameOffset);
    painter->drawTiledPixmap(left, leftStretch);

    QRect top = r;
    top.setX(r.x() + topLeft.width());
    top.setY(r.y() + topFrameOffset);
    top.setWidth(r.width() - topLeft.width() - topRight.width());
    top.setHeight(topLeft.height());
    painter->drawTiledPixmap(top, topStretch);

    QRect right = r;
    right.setX(r.right() - rightStretch.width()+1);
    right.setY(r.y() + topRight.height() + topFrameOffset);
    right.setWidth(rightStretch.width());
    right.setHeight(r.height() - topRight.height()
                    - bottomRight.height() - topFrameOffset);
    painter->drawTiledPixmap(right, rightStretch);

    QRect bottom = r;
    bottom.setX(r.x() + bottomLeft.width());
    bottom.setY(r.bottom() - bottomStretch.height()+1);
    bottom.setWidth(r.width() - bottomLeft.width() - bottomRight.width());
    bottom.setHeight(bottomLeft.height());
    painter->drawTiledPixmap(bottom, bottomStretch);

    QPixmap titleLeft = cached(":/images/title_cap_left.png");
    QPixmap titleRight = cached(":/images/title_cap_right.png");

    QFont font;
    QFontMetrics fm(font);

    int txt_width = fm.width("Pipeline") + 20;
    painter->drawPixmap(r.center().x() - txt_width/2, 0, titleLeft);

    QRect tileRect = QRect(r.center().x() - txt_width/2 + titleLeft.width(), 0,
                             txt_width - titleLeft.width() - titleRight.width(),
                             titleStretch.height());

    painter->drawTiledPixmap(tileRect, titleStretch);
    painter->drawPixmap(tileRect.x() + tileRect.width(), 0, titleRight);
    int opacity = 31;
    painter->setPen(QColor(0, 0, 0, opacity));
    painter->drawText(tileRect.translated(0, 1),
                      Qt::AlignVCenter | Qt::AlignHCenter, "Pipeline");
    painter->drawText(tileRect.translated(2, 1),
                      Qt::AlignVCenter | Qt::AlignHCenter, "Pipeline");
    painter->setPen(QColor(0, 0, 0, opacity * 2));
    painter->drawText(tileRect.translated(1, 1),
                      Qt::AlignVCenter | Qt::AlignHCenter, "Pipeline");
    painter->setPen(Qt::white);
    painter->drawText(tileRect, Qt::AlignVCenter | Qt::AlignHCenter, "Pipeline");

    painter->restore();
}

}} // Phonon::GStreamer
